/*
 * Copyright (C) 2006 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif /* HAVE_FCNTL_H */

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif /* HAVE_SYS_IOCTL_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "sensors-applet.h"
#include "sonypi-sensors-interface.h"

/* These values are taken from spicctrl by Stelian Pop */
#define SONYPI_DEV       "/dev/sonypi"
#define SONYPI_IOCGFAN   _IOR('v', 10, guint8)
#define SONYPI_IOCGTEMP  _IOR('v', 12, guint8)
#define SONYPI_TEMP "sonypi_temp"

/* for error handling */
#define SONYPI_DEVICE_FILE_ERROR (sonypi_sensors_interface_device_file_error_quark())

enum {
	SONYPI_DEVICE_FILE_OPEN_ERROR,
	SONYPI_DEVICE_FILE_READ_ERROR
};


void sonypi_sensors_interface_init(SensorsApplet *sensors_applet) {
        int fd;

        sensors_applet_register_sensors_interface(sensors_applet,
                                                  SONYPI, 
                                                  sonypi_sensors_interface_get_sensor_value);

        if ( (fd = open(SONYPI_DEV, O_RDONLY)) != -1 ) {
                if ( close(fd) != -1 ) {
                        sensors_applet_add_sensor(sensors_applet,
                                                  SONYPI_DEV,
                                                  SONYPI_TEMP,
                                                  _("CPU TEMP"),
                                                  SONYPI,
                                                  TRUE,
                                                  TEMP_SENSOR,
                                                  CPU_ICON);
                }
        }
}

/* for error handling */
static GQuark sonypi_sensors_interface_device_file_error_quark(void) {
	static GQuark quark = 0;
	gchar *string;

	if (quark == 0) {
		string = g_strdup_printf("%s-device-file-error", SENSORS_INTERFACE_STRING(SONYPI));
		quark = g_quark_from_string(string);
		g_free(string);
	}

	return quark;
}

gdouble sonypi_sensors_interface_get_sensor_value(const gchar *path, 
                                                  const gchar *id, 
                                                  SensorType type,
                                                  GError **error) {
        int fd;
        guint8 value8;

        gdouble sensor_value = -1.0;

        if ((fd = open(path, O_RDONLY)) != -1) {
                /* only use temp sensor */
                if (g_ascii_strcasecmp(id, SONYPI_TEMP) == 0) {
                        if (ioctl(fd, SONYPI_IOCGTEMP, &value8) != -1) {
                                sensor_value = (gdouble)value8;
                        } else {
                                g_set_error(error, SONYPI_DEVICE_FILE_ERROR, SONYPI_DEVICE_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
                        }        
                }
                close(fd);
        } else {
		g_set_error(error, SONYPI_DEVICE_FILE_ERROR, SONYPI_DEVICE_FILE_OPEN_ERROR, "Error opening from sensor device file %s", path);
        }
                        
        return sensor_value;
}

