# Copyright (C) 2004 Tiago Cogumbreiro <cogumbreiro@users.sf.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors: Tiago Cogumbreiro <cogumbreiro@users.sf.net>

import nautilusburn
import os
import gconf
import gtk

from gtk import glade
from os import path
from types import StringType
from gettext import gettext as _

# Local imports
import gaw
import xspf
import gtkutil
import release
import urlutil
import tempfile

from serpentine.converting import GvfsMusicPool
from serpentine.common import (SafeFileWrite, KEY_TEMP_ISO_DIR, GCONF_DIR,
                               RAT_GCONF_DIR, NCB_GCONF_DIR)


# Get system's temporary dir
TEMP_DIR = tempfile.gettempdir()
HOME_DIR = path.expanduser("~")

def _is_dir_writable(dirname):
    # Try to open the local file
    try:
        is_ok = path.isdir(dirname) and os.access(dirname, os.W_OK)
    except OSError, err:
        is_ok = False
            
    return is_ok


for gconf_dir in (GCONF_DIR, RAT_GCONF_DIR, NCB_GCONF_DIR):
    gconf.client_get_default ().add_dir (gconf_dir, gconf.CLIENT_PRELOAD_NONE)

def recordingPreferencesWindow (preferences):
    prefs_widget = gtkutil.find_child_widget (preferences.dialog, "preferences")
    prefs_widget.show ()
    
    assert prefs_widget is not None
    parent = prefs_widget.get_parent ()
    parent.remove (prefs_widget)
    
    win = gtk.Window (gtk.WINDOW_TOPLEVEL)
    win.set_border_width (12)
    win.set_title (_("Serpentine Preferences"))
    
    vbox = gtk.VBox ()
    vbox.set_spacing (18)
    vbox.show ()
    win.add (vbox)
    
    vbox.pack_start (prefs_widget)
    
    # Add a close button
    bbox = gtk.HButtonBox ()
    bbox.set_layout (gtk.BUTTONBOX_END)
    bbox.show ()
    vbox.add (bbox)
    
    close_btn = gtk.Button (stock = "gtk-close")
    close_btn.show ()
    close_btn.connect ("clicked", gtk.main_quit)
    bbox.add (close_btn)
    
    return win

class HideCloseButton:
    """
    Monitors the 'rat' key for showing/hiding the close button.
    """
    def __init__(self, close_button):
        self.button = close_button
        self.use_button = gaw.GConfValue (
            key = RAT_GCONF_DIR + "/use_close_button",
            data_spec = gaw.Spec.BOOL,
            default = True
        )
        self.use_button.set_callback(self.on_update)
        self.on_update()
    
    def on_update(self, *args):
        if not self.use_button.data:
            self.button.hide()
        else:
            self.button.show()

class WriteSpeed(object):
    """
    Handles and monitors writing speed related widgets state.
    """
    def __init__(self, g, get_drive):
        self.get_drive = get_drive
        
        self._speed = gaw.data_spin_button (g.get_widget("speed"),
                                             GCONF_DIR + "/write_speed")
        
        self.__specify_speed = g.get_widget ("specify_speed_wrapper")
                                             
        specify_speed = g.get_widget ("specify_speed")
        
        self._speed_select = gaw.RadioButtonData (
            widgets = dict (
                specify_speed = specify_speed,
                use_max_speed = g.get_widget ("use_max_speed")
            ),
            key = GCONF_DIR + "/speed_select"
        )
        self._speed_select.seleted_by_default = "use_max_speed"
        
        specify_speed.connect ("toggled", self.__on_specify_speed)
        g.get_widget ("refresh_speed").connect ("clicked", self.__on_refresh_speed)
        
        # No default value set, set it to 99
        if self._speed.data == 0:
            self._speed.data = 99

        self.__update_speed ()
        self._speed.sync_widget()
        
        # init specify speed box sensitivity
        self.__on_specify_speed (specify_speed)
        
    def get(self):
        assert self.get_drive() is not None
        self.__update_speed()

        if self._speed_select.data == "use_max_speed":
            return self.get_drive().get_max_speed_write()
        return self._speed.data

    def __on_refresh_speed(self, *args):
        self.__update_speed()

    def __update_speed(self):
        drive = self.get_drive()
        if drive is None:
            self._speed.widget.set_sensitive(False)
            return
            
        max_speed = drive.get_max_speed_write()
        assert max_speed >= 0, max_speed

        val = self._speed.data

        low = min(1, max_speed)

        self._speed.widget.set_range(low, max_speed)
        self._speed.data = val
        
    def __on_specify_speed (self, widget, *args):
        # Sometimes the spinbox would get insensitive, so make sure it's not
        self._speed.widget.set_sensitive(widget.get_active())
        self.__specify_speed.set_sensitive(widget.get_active())
    



class RecordingPreferences(object):
    debug = False
    overburn = False
    simulate = False

    signals = dict(
        # eject checkbox
        eject = dict(key = GCONF_DIR + "/eject",
                     data_spec = gaw.Spec.BOOL
        ),
        
        # use gap checkbox
        use_gap = dict(
            key = GCONF_DIR + "/use_gap",
            default = True,
            data_spec = gaw.Spec.BOOL,
        ),
        
        # temp
        temp_iso_dir = dict(
            key = KEY_TEMP_ISO_DIR,
            data_spec = gaw.Spec.STRING,
            default = TEMP_DIR
        ),
        
        # debug
        debug = dict(
            key = GCONF_DIR + "/debug_mode",
            data_spec = gaw.Spec.BOOL,
            default = False
        ),
        
        # size
        x = dict(
            key = GCONF_DIR + "/x",
            data_spec = gaw.Spec.INT,
            default = 0,
        ),
        
        y = dict(
            key = GCONF_DIR + "/y",
            data_spec = gaw.Spec.INT,
            default = 0,
        ),

        width = dict(
            key = GCONF_DIR + "/width",
            data_spec = gaw.Spec.INT,
            default = 0,
        ),

        height = dict(
            key = GCONF_DIR + "/height",
            data_spec = gaw.Spec.INT,
            default = 0,
        ),
        
        # disc size
        disc_size = dict(
            key = GCONF_DIR + "/disc_size",
            data_spec = gaw.Spec.INT,
            default = 1,
        ),

        # device
        device = dict(
            key = GCONF_DIR + '/device',
            data_spec = gaw.Spec.STRING,
            default = '',
        ),
    )

    def __init__ (self, locations):
        # By default use burnproof
        self.__write_flags = nautilusburn.RECORDER_WRITE_BURNPROOF
        # Sets up data dir and version
        self.version = release.version

        # setup ui
        filename = locations.get_data_file("serpentine.glade")
        g = glade.XML (filename, "preferences_dialog")
        self.__dialog = g.get_widget ("preferences_dialog")
        self.dialog.connect("destroy-event", self.__on_destroy)
        self.dialog.set_title(_("Serpentine Preferences"))

        # Drive selection
        drv = g.get_widget ("drive")
        cmb_drv = nautilusburn.DriveSelection ()
        cmb_drv.set_property ("show-recorders-only", True)
        cmb_drv.show ()
        cmb_drv.connect('device-changed', self._on_device_changed)
        
        self.__drive_selection = cmb_drv
        drv.pack_start (cmb_drv, False, False)
        
        # Speed selection
        self._speed = WriteSpeed(g, self.__drive_selection.get_drive)
        
        # eject checkbox
        self._eject = gaw.data_toggle_button (g.get_widget ("eject"),
                                              self.signals['eject']['key'])
        
        # use gap checkbox
        use_gap = self.signals['use_gap']
        
        self._use_gap = gaw.data_toggle_button (
            g.get_widget ("use_gap"),
            use_gap['key'],
            default = use_gap['default']
        )
        
        # temp
        ncb_temp_dir = NCB_GCONF_DIR + "/temp_iso_dir"
        gconf.client_get_default ().add_dir (ncb_temp_dir, gconf.CLIENT_PRELOAD_NONE)
        for prop in self.signals:
            setattr(self, '_' + prop, gaw.GConfValue(**self.signals[prop]))

        # Setup device
        self._device.set_callback(self._on_gconf_device_changed)
        self._on_gconf_device_changed()
        
        # Pool
        self.__pool = GvfsMusicPool ()
        
        # Close button
        self.__close_button_handler = HideCloseButton(g.get_widget("close_btn"))

    
    ############################################################################
    # Properties
    
    ############
    # configDir
    __config_dir = path.join (path.expanduser ("~"), ".serpentine")
    def getConfigDir (self):
        return self.__config_dir
        
    configDir = property (lambda self: self.__config_dir)
    
    #########
    # dialog
    def getDialog (self):
        return self.__dialog
        
    dialog = property (getDialog)
    
    ##########
    # version
    def setVersion (self, version):
        assert isinstance (version, StringType)
        self.__version = version
        
    def getVersion (self):
        return self.__version
        
    version = property (getVersion, setVersion)

    def get_device(self):
        return self.__drive_selection.get_device()

    def set_device(self, device):
        return self.__drive_selection.set_device(device)

    device = property(get_device, set_device)
    
    ########
    # drive
    def getDrive (self):
        return self.__drive_selection.get_drive()
        
    drive = property (getDrive)
    
    ##############
    # useGnomeVfs
    def getUseGnomeVfs(self):
        return self.__pool.use_gnomevfs
    
    def setUseGnomeVfs(self, use_gnomevfs):
        self.__pool.use_gnomevfs = use_gnomevfs
    
    useGnomeVfs = property(getUseGnomeVfs, setUseGnomeVfs)
    
    ###############
    # temporary_dir
    
    def get_temporary_dir(self):
        return self._temp_iso_dir.data
    
    temporary_dir = property(get_temporary_dir)
    
    def get_temporary_dirs(self):
        # First try the GConf temporary dir, then try the system TEMP dir
        # finally try on the HOME
        temps = (self.temporary_dir, TEMP_DIR, HOME_DIR)

        for temp in temps:
            if urlutil.is_local(temp) and _is_dir_writable(temp):
                yield urlutil.get_path(temp)

    temporary_dirs = property(get_temporary_dirs)
    
    def get_window_position(self):
        return self._x.data, self._y.data
    
    def set_window_position(self, x_y):
        self._x.data, self._y.data = x_y
    
    window_position = property(get_window_position, set_window_position)
    
    def get_window_size(self):
        return self._width.data, self._height.data
    
    def set_window_size(self, width_height):
        self._width.data, self._height.data = width_height
    
    window_size = property(get_window_size, set_window_size)
    
    def get_disc_size(self):
        return self._disc_size.data
    
    def set_disc_size(self, disc_size):
        self._disc_size.data = disc_size
    
    disc_size = property(get_disc_size, set_disc_size)
    
    ##############
    # useGap
    def getUseGap(self):
        return self._use_gap.data
    
    useGap = property(getUseGap)

    #########
    # pool
    def getPool (self):
        return self.__pool
        
    pool = property (getPool)

    ##############
    # speedWrite
    def getSpeedWrite (self):
        return self._speed.get()
        
    speedWrite = property (getSpeedWrite)
    
    ################
    # writeFlags
    def getWriteFlags (self):
        ret = self.__write_flags
        if not self._use_gap.data:
            ret |= nautilusburn.RECORDER_WRITE_DISC_AT_ONCE
        if self._eject.data:
            ret |= nautilusburn.RECORDER_WRITE_EJECT
        if self.debug:
            ret |= nautilusburn.RECORDER_WRITE_DEBUG
        if self.overburn:
            ret |= nautilusburn.RECORDER_WRITE_OVERBURN
        if self.simulate:
            ret |= nautilusburn.RECORDER_WRITE_DUMMY_WRITE

        return ret
        
    writeFlags = property (getWriteFlags)
    
    
    #############################
    # Methods
    def _on_device_changed(self, widget, device):
        self._device.data = device

    def _on_gconf_device_changed(self, *args):
        self.__drive_selection.set_device(self._device.data)
        self._device.data = self.__drive_selection.get_device()
        
    def __on_destroy (self, *args):
        self.dialog.hide ()
        return False

    def savePlaylist (self, source):
        if not path.exists (self.configDir):
            os.makedirs (self.configDir)
        p = xspf.Playlist (title=_("Serpentine's playlist"), creator="Serpentine " + self.version)
        source.to_playlist (p)
        doc = p.toxml()
        
        out = SafeFileWrite (path.join (self.configDir, "playlist.xml"))
        try:
            doc.writexml (out, addindent = "\t", newl = "\n")
            del p
            out.close()
        except:
            out.abort ()
            return False
            
        return True
    
    def loadPlaylist (self, source):
        if not path.exists (self.configDir):
            os.makedirs (self.configDir)
        p = xspf.Playlist (title=_("Serpentine's playlist"), creator="Serpentine " + self.version)
        
        try:
            p.parse (path.join (self.configDir, "playlist.xml"))
        except IOError:
            return
            
        source.from_playlist (p)

    def connect(self, signal, callback):
        """
        Connect to one of the availabel signals (attribute 'signals')
        in order observe when a certain property has changed.

        The callback has the following signature:
        
        callback(signal, value)

        signal - the name of the property
        value - the value of the property

        The returned object is an opaque data that should be unreferenced
        when the callback is no longer needed.
        """
        signal = signal.replace("-", "_").lower()
        try:
            spec = self.signals[signal]
            val = gaw.GConfValue(**spec)
            val.set_callback(lambda *args: callback(signal, val.data))
            return val
        except KeyError:
            raise ValueError("Unknown signal %r" % signal)
