/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* compressedfile.cpp */

#include "compressedfile.h"
#include "extract.h"
#include "directoryview.h" 

#include <qfileinfo.h>
#include <qapplication.h>

#include <kiconloader.h>
#include <kio/job.h>


CompressedFileItem::CompressedFileItem (Directory *parent, const char *filename, const char *path,
					MainWindow *mw,	DirectoryView* dirView, 
					ImageViewer * imv, ImageListeView * imageList):
QListViewItem (parent), f (filename)
{
	this->mw=mw;
	this->dirView=dirView;
	this->iv = imv;
	this->imageList = imageList;
	
	readable = TRUE;
	full.append (path);
	full.append (filename);
	list.setAutoDelete (TRUE);
	QFileInfo * info = new QFileInfo (full);
	size = new QCString ();
	size->sprintf ("%10d", info->size ());
	extension = new QCString ();
	extension->setStr (info->extension ());
	p_open = new QPixmap (BarIcon ("tgz", 16));
	p_none = new QPixmap ();
}


QString
CompressedFileItem::key (int column, bool ascending) const
{
	if(column==0)
		return text(0).lower()+text(1).lower()+text(2).lower();
	else
	if(column==1)
		return text(1).lower()+text(0).lower()+text(2).lower();
	else
		return text(2).lower()+text(0).lower()+text(1).lower();
}


const QPixmap *
CompressedFileItem::pixmap (int column) const
{
	if (column == 0)
	{
		return p_open;
	}
	else
		return p_none;
}

QString
CompressedFileItem::fullName ()
{
	return full;
}

QString
CompressedFileItem::text (int column) const
{     
	if (column == 0)
	{
		return f.name ();
	}
	else
	if (column == 1)
		return (*extension);
	else
		return (*size);
}

void
CompressedFileItem::removeImage (CompressedImageFileItem * imf)
{
	if (list.findRef (imf))
	{
		list.removeRef (imf);
	}
}

void
CompressedFileItem::loadArchive (bool refresh)
{
	QApplication::setOverrideCursor (waitCursor); // this might take time
	mw->setMessage(i18n("Loading '")+text(0)+"'...");

	Extract *extract = new Extract (full);
	QString *  s;
	QString  ext;

	for (s = extract->files.first (); s != 0; s = extract->files.next ())
	{
		ext = s->right (3).lower();
		if ((ext == "jpg") || (ext == "jpeg") || (ext == "jpe") ||
		      (ext == "xpm") ||
		      (ext == "gif") ||
		      (ext == "png") ||
		      (ext == "bmp") ||
		      (ext == "pbm") ||
		      (ext == "pgm") ||
		      (ext == "ppm") ||
		      (ext == "xbm") ||
		      (ext == "tif"))
		    {
		    	    list.append (new
		    	              CompressedImageFileItem ((QIconView*)imageList, this, full,
		    	        			       (const char *) (*s),
		    	        			       (const char *) (*s), iv));		    	    
		    	    mw->slotAddImage();
		    }
	}
	if(refresh)
	{
		imageList->slotLoadFirst();
		iv->updateStatus();
	}

	mw->setMessage(i18n("Ready"));
	QApplication::restoreOverrideCursor ();       // restore original cursor
}

void
CompressedFileItem::unLoadArchive ()
{
	KShellProcess *proc = new KShellProcess ();

	QString dest;
	dest.setNum (getpid ());
	dest.prepend ("/tmp/.qlist");
	dest.append ("/");
	dest.append (text (0));

	*proc << "rm -f -r";
	*proc << (const char *) dest;
	proc->start (KShellProcess::Block);

	mw->slotRemoveImage(list.count());
	list.clear ();
	iv->updateStatus ();
}

void
CompressedFileItem::setSelected (bool s)
{
	QListViewItem::setSelected (s);
	listView()->triggerUpdate();
	qApp->processEvents();
	
	if (!dirView->isDropping())
	{
		if (s)
		{
			loadArchive ();
			dirView->startWatchDir(QString(fullName()));	
		}
		else
		{
		 	 unLoadArchive();
		 	 dirView->stopWatchDir(QString(fullName()));
		}	
	}
}
