/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "mainwindow.h"

#include "formatconversion.h"
#include "rename.h"
#include "mybookmarkmanager.h"
#include "printImageDialog.h"
#include "confshowimg.h"

#include <kaboutdata.h>
#include <khelpmenu.h>
#include <kio/job.h>
#include <kaboutapplication.h>
#include <ktoolbarbutton.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <kstdaccel.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <konq_operations.h>
#include <kglobalsettings.h>
#include <kurlrequesterdlg.h>
#include <kdialogbase.h>
#include <kdockwidget.h>
#include <kcursor.h>
#include <kurldrag.h>
#include <kfiledialog.h>

#if KDE_VERSION > 220
#include <kprinter.h>
#endif

#include <qclipboard.h>
#include <qpainter.h>

#include <stdlib.h>  

class KDialogBase;

MainWindow::MainWindow (const char *name):KDockMainWindow (0, name)
{
	init ();
	show();
	open (QDir ().homeDirPath ());

	hasimageselected=true;
	setHasImageSelected(false);
}


MainWindow::MainWindow (const char *pic, const char *name):
KDockMainWindow (0, name),
KBookmarkOwner()
{
	init ();
	show();
	if (QFileInfo (pic).isDir ())
	{
	    open (QDir (pic).absPath ()+"/");
	    
	    hasimageselected=true;
	    setHasImageSelected(false);	   
	}
	else
	if (QString (pic).right (3) != "zip")
	{
	 	 show ();
	 	 QString rep = QDir (pic).absPath ();
	 	 int pos = rep.findRev ("/");
	 	 open (rep.left (pos));

	 	 iv->loadImage (pic);
	 	 slotFullScreen();
		 
		 hasimageselected=false;
		 setHasImageSelected(true);
	}
	else
	{
		QString rep = QDir (pic).absPath ();
		open (rep);
		
		hasimageselected=true;
		setHasImageSelected(false);
	}
}


MainWindow::~MainWindow ()
{

}


void
MainWindow::createActions()
{
	aBack		=new HistoryAction(i18n("Back"),"back", KStdAccel::key(KStdAccel::Back), this,SLOT(slotBack()), actionCollection(),"back");
	connect(aBack->popupMenu(), SIGNAL(activated(int)),this, SLOT(backMenuActivated(int)));
	connect(aBack->popupMenu(), SIGNAL( aboutToShow() ), SLOT( slotBackAboutToShow() ) );	
	aBack->setEnabled(false);
	
	aForward	=new HistoryAction(i18n("Forward"),"forward", KStdAccel::key(KStdAccel::Forward) , this,SLOT(slotForward()), actionCollection(),"forward");
	connect(aForward->popupMenu(), SIGNAL(activated(int)),this, SLOT(forwardMenuActivated(int)));
	connect(aForward->popupMenu(), SIGNAL( aboutToShow() ), SLOT( slotForwardAboutToShow() ) );
	aForward->setEnabled(false);
	
	aCut		=new KAction(i18n("Cut"),"editcut", KStdAccel::key(KStdAccel::Cut), this,SLOT(slotcut()),actionCollection() ,"editcut");
	aCut->setEnabled(false);
	aCopy		=new KAction(i18n("Copy"),"editcopy", KStdAccel::key(KStdAccel::Copy), this,SLOT(slotcopy()), actionCollection(),"editcopy");
	aCopyPixmap	=new KAction(i18n("Copy Image"), Key_Shift+KStdAccel::key(KStdAccel::Copy), this,SLOT(slotcopyPixmap()), actionCollection(),"editcopypixmap");
	aPaste		=new KAction(i18n("Paste"),"editpaste", KStdAccel::key(KStdAccel::Paste), this,SLOT(slotpaste()),actionCollection(),"editpaste" );
	
	aPrevious	=new KAction(i18n("Previous Image"),"1leftarrow", Key_PageUp, imageList,SLOT(previous()),actionCollection(),"Previous Image" );
	aNext		=new KAction(i18n("Next Image"),"1rightarrow", Key_PageDown, imageList,SLOT(next()), actionCollection(),"Next Image");
	
	aFirst		=new KAction(i18n("First Image"),"top", KStdAccel::key(KStdAccel::Home), imageList,SLOT(first()),actionCollection(),"First Image" );
	aLast		=new KAction(i18n("Last Image"),"bottom", KStdAccel::key(KStdAccel::End),imageList ,SLOT(last()),actionCollection(),"Last Image" );

	aWallpaper_CENTER	=new KAction(i18n("Centered"), 0, iv, SLOT(wallpaperC()), actionCollection(),"Center");
	aWallpaper_MOSAIC	=new KAction(i18n("Tiled"), 0, iv, SLOT(wallpaperM()), actionCollection(),"Mosaic");
	aWallpaper_CENTER_MOSAIC=new KAction(i18n("Center Tiled"), 0, iv, SLOT(wallpaperCM()), actionCollection(),"Center adapt");
	aWallpaper_CENTER_MAX	=new KAction(i18n("Centered Maxpect"), 0, iv, SLOT(wallpaperCMa()), actionCollection(),"Center max");
	aWallpaper_ADAPT	=new KAction(i18n("Scaled"), 0, iv, SLOT(wallpaperA()), actionCollection(),"Adapt");
	aWallpaper_LOGO		=new KAction(i18n("Logo"), 0, iv, SLOT(wallpaperL()), actionCollection(),"Logo");

	aNewWindow	=new KAction(i18n("New window"),"window_new", KStdAccel::key(KStdAccel::New), this, SLOT(slotNewWindow()),actionCollection(),"window_new" );
	
	aPrint		=new KAction(i18n("Print..."),"fileprint", KStdAccel::key(KStdAccel::Print), this, SLOT(slotPrint()), actionCollection(),"fileprint");
	aPrint->setEnabled(false);
	
	aOpenLocation	=new KAction(i18n("Open Location"),"fileopen", KStdAccel::key(KStdAccel::Open), this, SLOT(slotOpenLocation()),actionCollection(),"fileopen" );
	
	aQuit		=new KAction(i18n("Quit"),"exit", KStdAccel::key(KStdAccel::Quit), this, SLOT(quit()),actionCollection() ,"exit");
	aClose		=new KAction(i18n("Close"),"close", KStdAccel::key(KStdAccel::Close), this, SLOT(close()),actionCollection() ,"close");
	
	aUndo		=new KAction(i18n("Undo"),"undo", KStdAccel::key(KStdAccel::Undo), this, SLOT(slotUndo()),actionCollection() ,"undo");
	aUndo->setEnabled(false);
	aRedo		=new KAction(i18n("Redo"),"redo", KStdAccel::key(KStdAccel::Redo), this, SLOT(slotRedo()),actionCollection() ,"redo");
	aRedo->setEnabled(false);
	
	aRename		=new KAction(i18n("Rename"),"item_rename", Key_F2, imageList, SLOT(slotRename()),actionCollection() ,"rename");
	aDelete		=new KAction(i18n("Delete"),"editdelete", SHIFT+Key_Delete, imageList,SLOT(slotSupprimmer()), actionCollection(),"editdelete");
	aTrash		=new KAction(i18n("Move to trash"),"edittrash", Key_Delete, imageList,SLOT(slotMoveToTrash()), actionCollection(),"edittrash");
	aShred		=new KAction(i18n("Shred"),"editshred", SHIFT+CTRL+Key_Delete, imageList,SLOT(slotShred()), actionCollection(),"editshred");
	
	aEditType	=new KAction(i18n("Edit File Type"), 0, this,SLOT(slotEditFileType()), actionCollection(),"Edit File Type");
	aFileProperties	=new KAction(i18n("Properties"), 0, imageList,SLOT(slotFileProperty()), actionCollection(),"Properties");
	aImageInfo	=new KAction(i18n("Image Info"), 0, imageList,SLOT(slotImageInfo()), actionCollection(),"Image Info");
	aDirInfo	=new KAction(i18n("Describe directory"), 0, dirView,SLOT(slotDirInfo()), actionCollection(),"Dir Info");
	aDirProperties	=new KAction(i18n("Properties"), 0, dirView, SLOT(slotDirProperty()), actionCollection(),"Dir Properties");

	aSelect		=new KAction(i18n("SelectAll"), KStdAccel::key(KStdAccel::SelectAll) , imageList,SLOT(slotSelectAll()), actionCollection(),"SelectAll");
	aUnselectAll	=new KAction(i18n("Unselect All"),0, imageList ,SLOT(slotUnselectAll()), actionCollection(),"Unselect All");
	aInvertSelection=new KAction(i18n("Invert Selection"), CTRL+Key_I,imageList ,SLOT(slotInvertSelection()), actionCollection(),"Invert Selection");
	
	aConfigureKey	=new KAction(i18n("Configure &Key Bindings..."),"key_bindings",0 , this,SLOT(configureKey()), actionCollection(),"key_bindings");
	aConfigureToolbars=new KAction(i18n("Configure &Toolbars..."),"style", 0, this,SLOT(configureToolbars()), actionCollection(),"style");
	aConfigureShowImg=new KAction(i18n("Configure &showimg..."),"configure",0 , this,SLOT(configureShowImg()), actionCollection(),"Configure showimg");
		
	aRenameSeries	=new KAction(i18n("&Rename series..."), 0, this,SLOT(renameSeries()), actionCollection(),"Rename series");
	aSlideshow	=new KToggleAction(i18n("&Slide show"),"run", 0, this,SLOT(slotSlideShow()), actionCollection(),"Slideshow");
	aSmooth		=new KToggleAction(i18n("Smooth scaling"), 0, this,SLOT(slotSmooth()), actionCollection(),"Smooth scaling");

	aCompareFast	=new KAction(i18n("&Fast comparison"), 0, this,SLOT(slotCompareFast()), actionCollection(),"Compare fast");
	aCompareAlmost	=new KAction(i18n("&Approximate comparison"), 0, this,SLOT(slotCompareAlmost()), actionCollection(),"Compare almost");
	aCovert		=new KAction(i18n("Conver&t..."), 0, this,SLOT(convert()), actionCollection(),"convert");

	aRotLeft	=new KAction(i18n("Rotate Left"),"rotation_acw", Key_L, this,SLOT(slotRotateLeft()), actionCollection(),"Rotate Left");
	aRotRight	=new KAction(i18n("Rotate Right"),"rotation_cw", Key_R, this,SLOT(slotRotateRight()), actionCollection(),"Rotate Right");
	aHMirror	=new KAction(i18n("Vertical flip"),"flip", 0, this,SLOT(slotMirrorH()), actionCollection(),"Flip");
	aVMirror	=new KAction(i18n("Horizontal flip"),"miror", 0, this,SLOT(slotMirrorV()), actionCollection(),"Mirror");

	aReloadDir	=new KAction(i18n("Refresh"),"reload", 0, this,SLOT(slotRefresh()), actionCollection(),"Refresh");
	aPreview	=new KToggleAction(i18n("Toggle Thumbnails"),"thumbnail", 0, this,SLOT(slotPreview()), actionCollection(),"Preview");
	aStop		=new KAction(i18n("Stop"),"stop", 0, this,SLOT(slotStop()), actionCollection(),"Stop");
	aStop->setEnabled(false);

	aIconSmall	=new KRadioAction(i18n("Small Icons"),"smallthumbnails", 0, this,SLOT(slotIconSize()), actionCollection(),"Small Icons");
	aIconMed	=new KRadioAction(i18n("Medium Icons"),"medthumbnails", 0, this,SLOT(slotIconSize()), actionCollection(),"Medium Icons");
	aIconBig	=new KRadioAction(i18n("Big Icons"),"largethumbnails", 0, this,SLOT(slotIconSize()), actionCollection(),"Big Icons");
	aIconSmall->setExclusiveGroup("IconSize");
	aIconMed->setExclusiveGroup("IconSize");aIconMed->setChecked(true);
	aIconBig->setExclusiveGroup("IconSize");
	
	aArrangementLR	=new KRadioAction(i18n("from left to right"), 0, this, SLOT(slotArrangement()), actionCollection(),"From left to right");
	aArrangementTB	=new KRadioAction(i18n("from top to bottom"), 0, this, SLOT(slotArrangement()), actionCollection(),"From top to bottom");
	aArrangementLR->setExclusiveGroup("Arrangement");aArrangementLR->setChecked(true);	
	aArrangementTB->setExclusiveGroup("Arrangement");
	aArrangementR	=new KRadioAction(i18n("on the right"), 0, this, SLOT(slotTxtPos()), actionCollection(),"on the right");
	aArrangementB	=new KRadioAction(i18n("at the bottom"), 0, this, SLOT(slotTxtPos()), actionCollection(),"at the bottom");
	aArrangementB->setExclusiveGroup("TxtPos");aArrangementB->setChecked(true);	
	aArrangementR->setExclusiveGroup("TxtPos");
	
	aFullScreen	=new KAction(i18n("Full Screen"),"window_fullscreen", Key_Return, this, SLOT(slotFullScreen()),actionCollection(),"FullScreen" );
	aZoomIn		=new KAction(i18n("Zoom in"),"viewmag_bis+", KStdAccel::key(KStdAccel::ZoomIn), this,SLOT(slotZoomIn()), actionCollection(),"Zoom in");
	aZoomOut	=new KAction(i18n("Zoom out"),"viewmag_bis-", KStdAccel::key(KStdAccel::ZoomOut), this,SLOT(slotZoomOut()), actionCollection(),"Zoom out");
	aZoomFit	=new KAction(i18n("Fit to Screen"),"viewmag_full", Key_Slash, this,SLOT(slotZoom()), actionCollection(),"Fit to Screen");
	aZoomFitWidth=new KAction(i18n("Fit width"),"viewmag_w", 0, iv, SLOT(fitWidth()), actionCollection(),"Fit the width");
	aZoomFitHeight=new KAction(i18n("Fit height"),"viewmag_h", 0, iv, SLOT(fitHeight()), actionCollection(),"Fit the height");
	aZoomNo		=new KAction(i18n("Original size"),"viewmag_no", Key_Asterisk, this,SLOT(slotZoomNo()), actionCollection(),"Originale size");
	aZoomLock	=new KToggleAction(i18n("Lock zoom"),"viewmag_lock",0, this,SLOT(slotZoomLock()), actionCollection(),"ZoomLock");
	
	aEnlarge	=new KToggleAction(i18n("Enlarge if smaller"), 0, this,SLOT(slotEnlarge()), actionCollection(),"Enlarge");
	aShrink		=new KToggleAction(i18n("Shrink if bigger"), 0, this,SLOT(slotShrink()), actionCollection(),"Shrink");
	
	aSortByName	=new KRadioAction(i18n("by name"), 0, imageList,SLOT(slotByName()), actionCollection(),"by name");
	aSortByType	=new KRadioAction(i18n("by extension"), 0, imageList,SLOT(slotByExtension()), actionCollection(),"by extension");
	aSortBySize	=new KRadioAction(i18n("by size"), 0, imageList,SLOT(slotBySize()), actionCollection(),"by size");
	aSortByDate	=new KRadioAction(i18n("by date"), 0, imageList,SLOT(slotByDate()), actionCollection(),"by date");
	aSortBySize->setExclusiveGroup("sort mode");aSortByType->setExclusiveGroup("sort mode");
	aSortByName->setExclusiveGroup("sort mode");aSortByDate->setExclusiveGroup("sort mode");
	aSortByName->setChecked(true);
	
	aOpenWithGimp	=new KAction(i18n("Open with Gimp"),"gimp", 0, imageList,SLOT(slotGimp()), actionCollection(),"Open with Gimp");
	aOpenWith	=new KAction(i18n("Open with ..."), 0, imageList,SLOT(slotOpenWith()), actionCollection(),"Open with");

	aDirNewFolder	=new KAction(i18n("New..."),"folder_new", 0, dirView,SLOT(slotNew()),actionCollection() ,"editdirnew");
	
	aDirCut		=new KAction(i18n("Cut"),"editcut", 0, this,SLOT(slotDirCut()),actionCollection() ,"editdircut");
	aDirCopy	=new KAction(i18n("Copy folder to ..."),"editcopy", 0, dirView, SLOT(slotDirCopy()),actionCollection() ,"editdircopy");
	aDirMove	=new KAction(i18n("Move folder to ..."), 0, dirView, SLOT(slotDirMove()),actionCollection() ,"editdirmove");
	aDirPaste	=new KAction(i18n("Paste"),"editpaste", 0, this,SLOT(slotDirPaste()),actionCollection() ,"editdirpaste");
	aDirPasteFiles	=new KAction(i18n("Paste files"),"editpaste", 0, dirView, SLOT(slotDirPasteFiles()),actionCollection() ,"editdirpaste files");
	
	aDirRename=new KAction(i18n("&Rename"),"item_rename", 0, dirView,SLOT(slotRename()),actionCollection() ,"editdirrename");
	
	aDirTrash=new KAction(i18n("&Move to trash"),"edittrash", 0, dirView,SLOT(slotTrash()),actionCollection() ,"editdirtrash");
	aDirDelete=new KAction(i18n("&Delete"),"editdelete", 0, dirView,SLOT(slotSuppr()),actionCollection() ,"editdirdelete");

	aFilesMoveTo	=new KAction(i18n("Move to ..."), 0, imageList,SLOT(slotFilesMoveTo()),actionCollection() ,"moveFilesTo");
	aFilesCopyTo	=new KAction(i18n("Copy to ..."), 0, imageList, SLOT(slotFilesCopyTo()),actionCollection() ,"copyFilesTo");
	
	aScrollXR	=new KAction(i18n("Scroll on the right"), Key_Right, iv, SLOT(scrolldxR()),actionCollection() ,"ScrollXR");
	aScrollYB	=new KAction(i18n("Scroll at the bottom"),Key_Down , iv, SLOT(scrolldyB()),actionCollection() ,"ScrollYB");
	aScrollXL	=new KAction(i18n("Scroll on the left"), Key_Left, iv, SLOT(scrolldxL()),actionCollection() ,"ScrollXL");
	aScrollYT	=new KAction(i18n("Scroll on the top"), Key_Up, iv, SLOT(scrolldyT()),actionCollection() ,"ScrollYT");

	aToolsRotateLeft=new KAction(i18n("Rotate Left"), "rotation_acw", 0, this, SLOT(toolsRotateLeft()),actionCollection() ,"aToolsRotateLeft");
	aToolsRotateRight=new KAction(i18n("Rotate Right"), "rotation_cw", 0, this, SLOT(toolsRotateRight()),actionCollection() ,"aToolsRotateRight");
}

void
MainWindow::setActionsEnabled(bool enable)
{
	int count=actionCollection()->count();
	for (int pos=0;pos<count;++pos)
	{
		actionCollection()->action(pos)->setEnabled(enable);
	}
}

void
MainWindow::slotIconSize()
{
	slotStop();	
	if(aIconSmall->isChecked())
	{
		imageList->setThumbnailSize(QSize(40, 30));
	}
	else
	if(aIconMed->isChecked ())
	{
		imageList->setThumbnailSize(QSize(80, 60));	
	}
	else
	if(aIconBig->isChecked ())
	{
		imageList->setThumbnailSize(QSize(100, 80));	
	}
	slotRefresh ();
}

void
MainWindow::createAccels()
{
	mAccel=new KAccel(this);
	int count=actionCollection()->count();
	for (int pos=0;pos<count;++pos)
	{
		KAction *action = actionCollection()->action(pos);
		action->plugAccel(mAccel, true);
	}
	mAccel->readSettings();	

	imageList->initMenu(actionCollection());
	dirView->initMenu(actionCollection());
	iv->initMenu(actionCollection());
}

void
MainWindow::createMenus()
{
	file = new QPopupMenu ();
	menuBar ()->insertItem (i18n("&File"), file);

	aNewWindow->plug(file);
	file->insertSeparator ();
	aPrint->plug(file);
	
	file->insertSeparator ();
	aOpenLocation->plug(file);
	file->insertSeparator ();
	
	aQuit->plug(file);

	//////////
	QPopupMenu *edition = new QPopupMenu ();
	menuBar ()->insertItem (i18n("&Edit"), edition);

	aCut->plug(edition);
	aCopy->plug(edition);
	aCopyPixmap->plug(edition);
	aPaste->plug(edition);
	edition->insertSeparator ();
	//aUndo->plug(edition);
	//aRedo->plug(edition);
	edition->insertSeparator ();
	aRename->plug(edition);
	edition->insertSeparator ();
	aTrash->plug(edition);
	aDelete->plug(edition);
	aShred->plug(edition);
	edition->insertSeparator ();
	aEditType->plug(edition);
	aFileProperties->plug(edition);
	aImageInfo->plug(edition);
	edition->insertSeparator ();
	aSelect->plug(edition);
	aUnselectAll->plug(edition);
	aInvertSelection->plug(edition);

	////////////
	QPopupMenu*  goMenu = new QPopupMenu ();
	menuBar ()->insertItem (i18n("&Go"), goMenu);
	aPrevious->plug(goMenu);
	aNext->plug(goMenu);
	goMenu->insertSeparator ();
	aFirst->plug(goMenu);
	aLast->plug(goMenu);

	/////////////
	QPopupMenu*  settings = new QPopupMenu ();
	menuBar()->insertItem (i18n("&Settings"), settings);
	aConfigureKey->plug(settings);
	aConfigureShowImg->plug(settings);

	/////////////
	options = new QPopupMenu ();
	menuBar ()->insertItem (i18n("&View"), options);
	options->insertItem(i18n("&Windows"), dockHideShowMenu());
	QPopupMenu *sort=new QPopupMenu ();
	options->insertItem (i18n("Sorting"), sort);
	aSortBySize->plug(sort);
	aSortByType->plug(sort);
	aSortByName->plug(sort);
	aSortByDate->plug(sort);

	QPopupMenu *zoom=new QPopupMenu ();
	options->insertItem (i18n("Zoom"), zoom);
	aZoomIn->plug(zoom);
	aZoomOut->plug(zoom);
	aZoomFit->plug(zoom);
	aZoomFitWidth->plug(zoom);
	aZoomFitHeight->plug(zoom);
	aZoomNo->plug(zoom);
	aZoomLock->plug(zoom);
	zoom->insertSeparator ();
	aEnlarge->plug(zoom);
	aShrink->plug(zoom);

	QPopupMenu *iconssize=new QPopupMenu ();
	options->insertItem (i18n("Icons size"), iconssize);
	aIconSmall->plug(iconssize);
	aIconMed->plug(iconssize);
	aIconBig->plug(iconssize);

	QPopupMenu *wallPaper=new QPopupMenu ();
	options->insertItem(i18n("&Wallpaper"), wallPaper);
	aWallpaper_CENTER->plug(wallPaper);
	aWallpaper_MOSAIC->plug(wallPaper);
	aWallpaper_CENTER_MOSAIC->plug(wallPaper);
	aWallpaper_CENTER_MAX->plug(wallPaper);
	aWallpaper_ADAPT->plug(wallPaper);
	aWallpaper_LOGO->plug(wallPaper);

	aFullScreen->plug(options);
	aSlideshow->plug(options);

	QPopupMenu* tools = new QPopupMenu ();
	menuBar ()->insertItem (i18n("&Tools"), tools);

	QPopupMenu *cmp = new QPopupMenu ();
	tools->insertItem (BarIcon ("filefind", 16), i18n("Compare"), cmp);
	aCompareFast->plug(cmp);
	aCompareAlmost->plug(cmp);
	
	QPopupMenu *conv = new QPopupMenu ();
	tools->insertItem (i18n("Conver&t..."), conv);		
	conv->insertItem (i18n("Type"), this,  SLOT (convert()));
	aToolsRotateLeft->plug(conv);
	aToolsRotateRight->plug(conv);
	
	aRenameSeries->plug(tools);

	///////////////
	MyBookmarkManager::self();

	abookmarkmenu =new KActionMenu(i18n("&Bookmark"), actionCollection());
	abookmarkmenu->plug(menuBar());

	mBookMenu=new KBookmarkMenu(this, abookmarkmenu->popupMenu(),
		       actionCollection(), true);

	KConfig *config = new KConfig("showimgrc");
	config->setGroup("Options");
	iv->nbImg = config->readNumEntry( "nbImg", 0 );

	KHelpMenu *helpMenu = new KHelpMenu( this );
	menuBar()->insertItem(i18n("&Help"),helpMenu->menu());
	connect( helpMenu, SIGNAL(showAboutApplication()),
		this, SLOT(myDialogSlot()));
}

#define SB_ITEMS    1
#define SB_SCALE    2
#define SB_NAME     3
#define SB_TYPE     4
#define SB_IMG_SIZE 5
#define SB_BYTES    6
#define SB_DATE     7
#define SB_MSG      8
#define SB_PATH     9
void
MainWindow::createToolbar()
{
	KToolBar *toolbar = toolBar("mainToolBar");

	aBack->plug(toolbar);
	aForward->plug(toolbar);
	toolbar->insertSeparator ();
	aCut->plug(toolbar);
	aCopy->plug(toolbar);
	aPaste->plug(toolbar);
	toolbar->insertSeparator ();

	toolbar->insertLineSeparator ();

	aRotLeft->plug(toolbar);
	aRotRight->plug(toolbar);
	aHMirror->plug(toolbar);
	aVMirror->plug(toolbar);

	toolbar->insertLineSeparator ();

	aSlideshow->plug(toolbar);aSlideshow->setChecked(false);
	
	aPrevious->plug(toolbar);
	aNext->plug(toolbar);

	toolbar->insertLineSeparator ();

	aReloadDir->plug(toolbar);
	aPreview->plug(toolbar);
	aStop->plug(toolbar);

	toolbar->insertLineSeparator ();

	aFullScreen->plug(toolbar);
	aZoomIn->plug(toolbar);
	aZoomOut->plug(toolbar);
	aZoomFit->plug(toolbar);
	aZoomNo->plug(toolbar);
}


void
MainWindow::createStatusbar()
{	
	statusBar()->insertItem("", SB_MSG, 1);	
	statusBar()->insertItem("", SB_ITEMS, 1);
	statusBar()->insertItem("", SB_SCALE, 1);
	statusBar()->insertItem("", SB_NAME, 2);
	statusBar()->insertItem("", SB_TYPE, 1);
	statusBar()->insertItem("", SB_IMG_SIZE, 1);
	statusBar()->insertItem("", SB_BYTES, 1);
	statusBar()->insertItem("", SB_DATE, 2);
	
	statusBar()->setFixedHeight(25);

	progress = new QProgressBar (statusBar(), "QProgressBar de chargement des images de MainWindow");
	progress->setCenterIndicator (true);
	statusBar()->addWidget (progress, 1, FALSE);
	progress->hide();

}
void
MainWindow::setMessage(QString msg)
{
	statusBar()->changeItem(msg,SB_MSG);
	qApp->processEvents();
}

void
MainWindow::setMessage (const char *txt)
{
	setMessage(QString(txt));
};

void
MainWindow::setNbrItems (int nbr)
{
	QString msg;
	msg.sprintf(i18n("%d item(s)"), nbr);
	statusBar()->changeItem(msg, SB_ITEMS);
};

void
MainWindow::setZoom (int zoom)
{
	QString msg;
	msg.sprintf(("%d%%"), zoom);
	statusBar()->changeItem(msg,SB_SCALE );
};

void
MainWindow::setImagename (const char *name)
{
	if(name)
	{
		QString msg(name);
		statusBar()->changeItem(msg,SB_NAME );
	}
	else
		statusBar()->changeItem("", SB_NAME );	
};

void
MainWindow::setImagetype (const char *type)
{
	if(type)
	{
		QString msg(type);
		statusBar()->changeItem(msg.upper(),SB_TYPE );
	}
	else
		statusBar()->changeItem("", SB_TYPE );	
	
};

void
MainWindow::setDim (QSize *size)
{
	if(size)
	{
		QString msg;
		msg.sprintf(("%dx%d"), size->width(), size->height());
		statusBar()->changeItem(msg,SB_IMG_SIZE );
	}
	else
		statusBar()->changeItem("", SB_IMG_SIZE );
};

void
MainWindow::setSize (int size)
{
	if(size>=0)
	{
		QString msg;
		msg.sprintf(("%d"), size);
		statusBar()->changeItem(msg, SB_BYTES);
	}
	else
		statusBar()->changeItem("", SB_BYTES);
};

void
MainWindow::setDate (QDate *date)
{
	if(date)
		statusBar()->changeItem(date->toString(),SB_DATE);
	else
		statusBar()->changeItem("",SB_DATE);	
};



void
MainWindow::changeDirectory(QString dir)
{
	setCaption(dir);
	currentDir=dir;
	
	updateHistory();	
	
	bool hasImg=imageList->hasImages();
	aPrevious->setEnabled(hasImg);
	aNext->setEnabled(hasImg);	
	aFirst->setEnabled(hasImg);	
	aLast->setEnabled(hasImg);	
}


void
MainWindow::backMenuActivated(int item)
{
	go(-(aBack->popupMenu()->indexOf(item)+1));
}

void
MainWindow::forwardMenuActivated(int item)
{
	go(aForward->popupMenu()->indexOf(item)+1);
}


void
MainWindow::createMainView()
{
	dockIL = createDockWidget ("Directory", BarIcon("folder"), 0L, i18n("Directory"), i18n("Directory"));
	imageList = new ImageListeView (dockIL, "ImageList", this);
	imageList->setThumbnailSize(QSize(80, 60));
	QWhatsThis::add(imageList, i18n( "Images of the selected directories" ) );
	dockIL->setWidget( imageList );
	dockIL->setToolTipString(i18n("Images in the selected directories"));
	
	dockIV = createDockWidget ("Preview", BarIcon("image"),0L, i18n("Preview"), i18n("Preview"));
	iv = new ImageViewer (dockIV, imageList, this, "ImageViewer");
	dockIV->setWidget( iv );
	KCursor::setAutoHideCursor(iv, true);	
	QWhatsThis::add(iv  , i18n( "Image preview" ) );
	dockIV->setToolTipString(i18n("Image preview"));
	
	dockDir = createDockWidget ("Tree", BarIcon("view_tree"),0L, i18n("Treeview"), i18n("Treeview"));
	dirView = new DirectoryView (dockDir, iv, imageList, this);		
	QWhatsThis::add(dirView, i18n( "List of directories" ) );	
	dockDir->setWidget( dirView );
	dockDir->setToolTipString(i18n("The directory tree"));	
	
	setMainDockWidget( dockIV );
	setView (dockIV);
	
	dockDir->manualDock(dockIV, KDockWidget::DockLeft, 35);	
	dockIL->manualDock(dockDir, KDockWidget::DockBottom, 35);
	
	root = new Directory (dirView, iv, imageList, this);
	root->setOpen (TRUE);

}

void
MainWindow::initSize()
{
	KConfig *config = new KConfig("showimgrc");
	config->setGroup("Options");
	if(config->readBoolEntry("quitOK", false))
	{
		readDockConfig ();
	}

	config->writeEntry("quitOK",false);
	config->sync();

	int w = config->readNumEntry( "Width", 0 );  // largeur de la fentre
	int h = config->readNumEntry( "Height", 0 ); // hauteur de la fentre

	if (w != 0 && h != 0)
		resize( w, h );
	else
		resize(640,480);
}


void
MainWindow::init ()
{
	total = 0;
	done = 0;
	inFullScreen = false;
	inInterface=false;

	createStatusbar();
	createMainView();

	createActions();
	createAccels();

	createMenus();
	createToolbar();

	initSize();	

	/////////////////////
	KConfig *config = new KConfig("showimgrc");
	
	config->setGroup("Options");	
	aPreview->setChecked(config->readBoolEntry("preview", false));
	bgColor=config->readColorEntry("bgcolor", new QColor("black"));
	iv->setBackgroundColor(bgColor);
	grayscale=config->readNumEntry("grayscale", -1);
	iv->setToGrayscale(grayscale);
	storeth=config->readBoolEntry("storeth", true);
	smooth=config->readBoolEntry("smooth", true);
	iv->setSmooth(smooth);
	dirView->setShowHiddenDir(config->readBoolEntry("showhiddenDir", false));
	dirView->setShowHiddenFile(config->readBoolEntry("showhiddenFile", false));
	
	config->setGroup("Slideshow");
	slideshowTime=config->readNumEntry("time", 2);
	slideshowType=config->readNumEntry("type", 0);
	
	config->setGroup("Zoom");
	aShrink->setChecked(config->readBoolEntry("shrink", false));
	slotShrink();
	aEnlarge->setChecked(config->readBoolEntry("enlarge", false));
	slotEnlarge();
	aZoomLock->setChecked(config->readBoolEntry("lock", false));
	slotZoomLock();

	config->setGroup("confirm");
	checkTrash=config->readBoolEntry("trash", true);
	checkDelete=config->readBoolEntry("delete", true);
	checkShred=config->readBoolEntry("shred", true);
	dirView->setCheckDelete(checkDelete);dirView->setCheckTrash(checkTrash);
	imageList->setCheckDelete(checkDelete);imageList->setCheckTrash(checkTrash); imageList->setCheckShred(checkShred);

	config->writeEntry("quitOK",false);

	timer = new QTimer (this);
	connect (timer, SIGNAL (timeout ()), imageList, SLOT (next ()));
}

void
MainWindow::myDialogSlot()
{
	static const char *description = I18N_NOOP("Image viewer for your desktop");
	QString txt = QString(i18n("Viewed : "))+QString().setNum(iv->nbImg)+QString(i18n(" pictures\n\n(Please send me e-mail for bugs report... or encouragements :)"));
	KAboutData*  aboutData = new KAboutData("showimg", I18N_NOOP ("showimg"),
	        	      "0.7", description, KAboutData::License_GPL,
	        	      "(c) 2001, Richard Groult",txt,
	        	      "http://www.jalix.org/projects/showimg/",
	        	      "rgroult@jalix.org" );

	aboutData->addAuthor ("Richard Groult", I18N_NOOP("Developer"), "rgroult@jalix.org", "http://ric.jalix.org");
	
	aboutData->addCredit ("Valerie", I18N_NOOP("for the logo"));
	aboutData->addCredit ("Benoist", I18N_NOOP("BETA testor, translation, and help for the zoom feature"), "b.gaston@laposte.net");
	aboutData->addCredit ("Romain Vallet", I18N_NOOP("Documentation and spelling corrector"), "rom@jalix.org");
	
	aboutData->addCredit ("Alain Bidaud and the Jalix team", I18N_NOOP("for the web site"));
	
	aboutData->addCredit ("Hape Schaal ", I18N_NOOP("german translator and great help for debug"), "hp.news@gmx.de");
	aboutData->addCredit ("Jean-Philippe Martin",I18N_NOOP("for icons and advices about features and design") , "jeanphilippemartin@club-internet.fr");
	aboutData->addCredit ("Andrea Tavazzani",I18N_NOOP("Italian translation") , "andreatavazzani@libero.it");
	
	aboutData->addCredit ("Dominik Seichter", I18N_NOOP("for allowing me to use his great software krename for showimg"), "domseichter@web.de", "http://www.freshmeat.net/projects/krename" );
	aboutData->addCredit ("Matthias Wandel", I18N_NOOP("for allowing me to use jhead for jpeg-exif format support"), "MWandel@rim.net");
	aboutData->addCredit ("Andrew Richards", I18N_NOOP("for allowing me to use his printImageDialog class"), "ajr@users.sourceforge.net");
	
	aboutData->addCredit ("Lot's-of people", I18N_NOOP("for bugs reports, advices,\nand encouragements :)"));

	KAboutApplication(aboutData, this).show();
}


bool
MainWindow::preview()
{
	return aPreview->isChecked();
}


QString
MainWindow::getFileName(QString *fullName)
{
		int debut = fullName->findRev ("/");
		int fin = fullName->findRev (".");
		return fullName->mid(debut+1, fin-debut-1);
}


QString 
MainWindow::getFileExt(QString *fullName)
{
		int pos = fullName->findRev (".");
		return  fullName->right (fullName->length () - pos-1);
}

QString 
MainWindow::getFullName(QString *fullName)
{
		int pos = fullName->findRev ("/");
		return fullName->right (fullName->length () - pos-1);
}

QString 
MainWindow::getFullPath(QString *fullName)
{
		int pos = fullName->findRev ("/");
		return fullName->left(pos+1);
}


DirectoryView*
MainWindow::getDirectoryView()
{
	return dirView;
}


void
MainWindow::convert()
{
	FormatConversion *formatConver = new FormatConversion(this);
	switch(formatConver->exec())
	{
		case QDialog::Accepted :
			break;
		default : return;
	}
	setMessage(i18n("Conversion in progress..."));
	
	QString ext=formatConver->getType();
	QString opt=formatConver->getOptions();
	bool replace = formatConver->replace();
	QList <QString> list;
	list.setAutoDelete (false);

	for (ImageFileItem * item = (ImageFileItem *)((QIconView*)imageList)->firstItem ();
	     item != 0;
	     item = (ImageFileItem *) item->nextItem ())
	{
		if (item->isSelected () && item->text(3)=="file"  )
	        {
			list.append(new QString(item->fullName()));
		}
	}
	if(list.isEmpty())
	{
		KMessageBox::error(this, i18n("You have to select at least one file"));
		return;
	}
	
	stopWatchDir();
	
	QProgressDialog*  progres =  new QProgressDialog ("Conversion", "&Cancel", list.count(), this);
	progres->setMinimumDuration(1);
	progres->show();

	QString *name;
	int done=0;

	KShellProcess* proc = new KShellProcess ();
	connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(slotEndConvert(KProcess *)));
	
	QString  msg, com;
	
	for ( name=list.first(); name != 0; name=list.next() )
	{
		msg.sprintf ("Conversion of <b>%s</b>\n(%d/%d)",
					 (const char*)*name,
					 done,
		  	     list.count ());
		done++;
		qApp->processEvents();
		if ( progres->wasCancelled() )
			break;
		progres->setLabelText (msg);
		progres->setProgress (done-1);

		proc->clearArguments();
		com.sprintf ("convert %s '%s' '%s'",
		  	     (const char *)opt, (const char *)*name, (const char *)(getFullPath(name)+getFileName(name)+"."+ext));
		*proc << com;
		proc->start (KShellProcess::Block);
		qApp->processEvents ();

		if(replace && ext!=getFileExt(name))
		{
			FileItem *item=imageList->findItem(getFileName(name));
			if(item) item->suppression();
		}
	}
	delete(progres);
	delete(proc);
	setMessage(i18n("Conversion done"));
	startWatchDir();
	slotRefresh ();
}

void
MainWindow::slotEndConvert(KProcess *proc)
{
	if(proc->exitStatus()!=0)
	{
		qWarning(i18n("ERROR : Can't find the 'convert' program !!!"));
	}
}


int
MainWindow::getRed(QImage* im, int x, int y)
{
	return qRed(im->pixel(x, y));
}
int
MainWindow::getGreen(QImage* im, int x, int y)
{
	return qGreen(im->pixel(x, y));
}

int
MainWindow::getBlue(QImage* im, int x, int y)
{
	return qBlue(im->pixel(x, y));
}

#define  PAS 32

ImageSimilarityData*
MainWindow::image_sim_fill_data(QImage *pixbuf)
{
	
	int w, h; 	
	uchar *pix;
	int has_alpha;
	int p_step;

	int i;
	int j;
	int x_inc, y_inc;
	int xs, ys;
	
     const int INC=1;

    ImageSimilarityData *sd = new ImageSimilarityData();

	if (!sd || !pixbuf) return 0L;

	w = pixbuf->width();
	h = pixbuf->height();
	
	pix = pixbuf->bits();
	has_alpha = pixbuf->hasAlphaBuffer();

	p_step = has_alpha ? 4 : 3;
	x_inc = w / PAS;
	y_inc = h / PAS;

	if (x_inc < 1 || y_inc < 1) return 0L;

	j = 0;

	for (ys = 0; ys < PAS; ys++)
		{
		i = 0;

		for (xs = 0; xs < PAS; xs++)
			{
			int x, y;
			int r, g, b;

			r = g = b = 0;

			for (y = j; y < j + y_inc; y+=INC)
				{
				
				for (x = i; x < i + x_inc; x+=INC)
					{
                  	r +=getRed(pixbuf, x, y);
 					g +=getGreen(pixbuf, x, y);
 					b +=getBlue(pixbuf, x, y);
                	}
				}
			r /= x_inc * y_inc;
			g /= x_inc * y_inc;
			b /= x_inc * y_inc;

			sd->avg_r[ys * PAS + xs] = r;
			sd->avg_g[ys * PAS + xs] = g;
			sd->avg_b[ys * PAS + xs] = b;

			i += x_inc;
			}

		j += y_inc;
		}

	sd->filled = TRUE;
	
	return sd;
}



float 
MainWindow::image_sim_compare(ImageSimilarityData *a, ImageSimilarityData *b)
{
	float sim;
	int i;

	if (!a || !b || !a->filled || !b->filled) return 0.0;

	sim = 0.0;

	for (i = 0; i < PAS*PAS; i++)
	{
		sim += (float)abs(a->avg_r[i] - b->avg_r[i]) / 255.0;
		sim += (float)abs(a->avg_g[i] - b->avg_g[i]) / 255.0;
		sim += (float)abs(a->avg_b[i] - b->avg_b[i]) / 255.0;
	}

	sim /= (1024.0 * 3.0);

	return 1.0 - sim;
}


float 
MainWindow::image_sim_compare_fast(ImageSimilarityData *a, ImageSimilarityData *b, float min)
{
	float sim;
	int i;
	int j;

	if (!a || !b || !a->filled || !b->filled) return 0.0;

	min = 1.0 - min;
	sim = 0.0;

	for (j = 0; j < PAS*PAS; j+= PAS)
	{
		for (i = j; i < j + PAS; i++)
		{
			sim += (float)abs(a->avg_r[i] - b->avg_r[i]) / 255.0;
			sim += (float)abs(a->avg_g[i] - b->avg_g[i]) / 255.0;
			sim += (float)abs(a->avg_b[i] - b->avg_b[i]) / 255.0;
		}
		/* check for abort, if so return 0.0 */
		if (sim / (PAS*PAS * 3.0) > min) return 0.0;
}

	sim /= (PAS*PAS * 3.0);

	return 1.0 - sim;
}


////////////////////////////////////////////
void
MainWindow::compareAlmost ()
{
	setMessage(i18n("Comparisons in progress..."));

	QProgressDialog * progres =  new QProgressDialog (i18n("Comparisons"), i18n("&Cancel"), ((QIconView*)imageList)->count (), this,  0, true, 0);

	QDict < QVector < QFile > >*dict = new QDict < QVector < QFile > >;
	QDict < QVector < QFile > >*res = new QDict < QVector < QFile > >;
	QVector < QFile > *list;
	long total = 0;

	dict->setAutoDelete(TRUE);

	///////////////
	FileItem *item = (ImageFileItem *) ((QIconView*)imageList)->firstItem ();
	QString size;
	QFile *file;

	list = new QVector < QFile >;
	list->setAutoDelete(TRUE);
	dict->insert (0, list);

	while (item)
	{
		//creation d un file
		file = new QFile (item->fullName ());

		//on ajoute le file
		list->resize (list->size () + 1);
		list->insert (list->size () - 1, file);

		//on passe au suivant
		item = (FileItem *) item->nextItem ();
	}

	/////////////
	//comparaison des fichiers
	progres->setTotalSteps(((QVector < QFile > *)dict->find (0))->size());
	progres->setLabelText (i18n("Approximate comparison in process..."));
	progres->show();

	QDictIterator < QVector < QFile > >it (*dict);        // iterator for dict
	while (it.current ())
	{
		QDict < QFile > *fait = new QDict < QFile >;
		list = (QVector < QFile > *)it.current ();

		if (list->size () != 1)
		{
			for (unsigned int i = 0; i < list->size (); i++)
			{
				progres->setProgress (total ++);
		      		qApp->processEvents();
		        	if ( progres->wasCancelled() )
				{
					delete (it);
					delete (progres);
					delete(res);
					setMessage(i18n("Ready"));
					
					return;
				}

				QFile *file1 = (QFile *) (list->at (i));

				QImage *im1 = new QImage(file1->name());
				ImageSimilarityData *i1 = image_sim_fill_data(im1);

				if (!fait->find (file1->name()))
				{
					for (unsigned int j = i + 1; j < list->size (); j++)
		        		{
						qApp->processEvents();
		        			if ( progres->wasCancelled() )
						{
							delete (it);
							delete (progres);
							delete(res);
							setMessage(i18n("Ready"));
							QApplication::restoreOverrideCursor ();
							return;
						}
		        			QFile *file2 = (QFile *) (list->at (j));

						QImage *im2 = new QImage(file2->name());
		      				ImageSimilarityData *i2 = image_sim_fill_data(im2);
						float eq = image_sim_compare_fast(i1,i2,0.94);
		        	
						if (eq>=0.95)   //les fic sont egaux
		        			{
		        			 	 QVector < QFile > *vect;

		        			 	 //on ajoute le file
		        			 	 if (!res->find (file1->name ()))
		        			 	 {
		        			 	 	 vect = new QVector < QFile >;
		        			 	 	 vect->setAutoDelete(TRUE);
		        			 	 	 res->insert (file1->name (), vect);
		        			 	 }
		        			 	 else
		        			 	 {
		        			 		vect = (QVector < QFile > *)res->find (file1->name ());
		        			 	 }
		        			 	 vect->resize (vect->size () + 1);
		        			 	 vect->insert (vect->size () - 1, file2);

				    			fait->insert(file2->name(), file2);
				  		}
						delete(i2);
						delete(im2);
			   		}
				delete(i1);
				delete(im1);
				}
			}
		}
		delete(fait);
		++it;
	}

	delete (it);
	delete (progres);

	setMessage(i18n("Ready"));

	if(!res->isEmpty())
	{
		DisplayCompare(this, res).exec();
		//slotRefresh();
	}
	delete(res);
}


void
MainWindow::compareFast ()
{
	setMessage(i18n("Fast comparisons in progress..."));
	QProgressDialog * progres = new QProgressDialog (i18n("Comparisons"), i18n("&Cancel"), ((QIconView*)imageList)->count (), this,  0, true, 0);
	progres->setLabelText (i18n("Comparison in progress..."));
	progres->show();
	
	QDict < QVector < QFile > >*dict = new QDict < QVector < QFile > >;
	dict->setAutoDelete(TRUE);
	QDict < QVector < QFile > >*res = new QDict < QVector < QFile > >;
	QVector < QFile > *list;
	long total = 0;

	FileItem *item = (ImageFileItem *) ((QIconView*)imageList)->firstItem ();
	QString size;
	QFile *file;

	while (item)
	{
		//creation d un file
		file = new QFile (item->fullName ());
		//on recupere la taille sous forme de chaine
		size = QString::number (QFileInfo (*file).size ());
		//si pas dans la table, on creer
		if (!dict->find (size))
		{
			list = new QVector < QFile >;
			list->setAutoDelete(TRUE);
			dict->insert (size, list);
		}
		//on recupere la liste
		list = (QVector < QFile > *)dict->find (size);
		//on ajoute le file
		list->resize (list->size () + 1);
		list->insert (list->size () - 1, file);
		//on passe au suivant
		item = (FileItem *) item->nextItem ();
	}

	/////////////
	//comparaison des fichiers
	QDictIterator < QVector < QFile > >it (*dict);        // iterator for dict
	while (it.current ())
	{
		QDict < QFile > *fait = new QDict < QFile >;
		list = (QVector < QFile > *)it.current ();
		if ( progres->wasCancelled() )
			break;
		progres->setProgress (total += list->size ());
		if ( progres->wasCancelled() )
		{
			delete (it);
			delete (progres);
			delete (res);
			QApplication::restoreOverrideCursor ();
			return;
		}
		if (list->size () != 1)
		{
			for (unsigned int i = 0; i < list->size (); i++)
			{
				QFile *file1 = (QFile *) (list->at (i));
				if (!fait->find (file1->name()))
				{
					for (unsigned int j = i + 1; j < list->size (); j++)
					{
						QFile *file2 = (QFile *) (list->at (j));
						if (equals (file1, file2))	  //les fic sont egaux
						{
							QVector < QFile > *vect;
							//on ajoute le file
							if (!res->find (file1->name ()))
							{
								vect = new QVector < QFile >;
								vect->setAutoDelete(TRUE);
								res->insert (file1->name (), vect);
							}
							else
							{
								vect = (QVector < QFile > *)res->find (file1->name ());
							}
							vect->resize (vect->size () + 1);
							vect->insert (vect->size () - 1, file2);

							fait->insert(file2->name(), file2);
						}	
					}
				}
			}				
		}		
		delete(fait);
		++it;
	}

	delete (it);
	delete (progres);
	setMessage(i18n("Ready"));
	if(!res->isEmpty())
	{
		DisplayCompare(this, res).exec();
		//slotRefresh();
	}
	else
	{
		KMessageBox::information(this, i18n("No identical files found"));
	}
	delete(res);
}


bool
MainWindow::equals (QFile * f1, QFile * f2)
{
	if (QFileInfo (*f1).size () != QFileInfo (*f2).size ())
	{
		return false;
	}

	f1->open (IO_ReadOnly);
	f2->open (IO_ReadOnly);

	QDataStream s1 (f1);
	QDataStream s2 (f2);

	Q_INT8 b1, b2;
	bool eq = true;

	while (!s1.atEnd () && eq)
	{
		s1 >> b1;
		s2 >> b2;

		eq = (b1 == b2);
	}

	f1->close ();
	f2->close ();

	return eq;
}


Directory*
MainWindow::findDir(const char *dir)
{
	QCString *res=new QCString(dir);;
	while(res->at(res->length()-1)=='/')
		res->setStr(res->left(res->length()-1));
	
	int pos = res->find("/");
	Directory *ssrep;

	res->setStr (res->right (res->length () - pos - 1));
	pos = res->find("/");

	ssrep = root;
	while (pos != -1)
	{
		ssrep = ssrep->find (res->left (pos));
		res->setStr (res->right (res->length () - pos - 1));

		if (ssrep)
		{
			pos = res->find ("/");
		}
		else
			return NULL;
	}

	ssrep = ssrep->find (*res);
	if (ssrep)
	{
		return ssrep;
    }
	else
	{
		return NULL;
	}
}


bool
MainWindow::open (const char *dir)
{
	QCString *res = new QCString (dir);
	if((*res)[res->length()-1]=='/')
		res=new QCString(res->left(res->length()-1));

	int pos = res->find ("/");
	Directory *ssrep;

	res->setStr (res->right (res->length () - pos - 1));
	pos = res->find ("/");

	ssrep = root;
	while ((pos!=-1) && (res->left(pos)[0]!='.' || dirView->showHiddenDir()))
	{
		ssrep = ssrep->find (res->left (pos));
		res->setStr (res->right (res->length () - pos - 1));
		if (ssrep)
		{
			pos = res->find ("/");
			ssrep->setOpen (true);
		}
		else
			break;
	 }
	if(!res->isEmpty())
		ssrep = ssrep->find(*res);
	if (ssrep)
	{
		dirView->clearSelection();
		dirView->slotShowItem(ssrep);
		dirView->setSelected (ssrep, true);
		dirView->setCurrentItem (ssrep);
		return true;
	}
	KMessageBox::error(this, QString().sprintf(i18n("Can't open the directory <b>%s<b>"),dir));
	return false;
}


void
MainWindow::keyPressEvent (QKeyEvent * e)
{
	KDockMainWindow::keyPressEvent(e);
}


void
MainWindow::nextDir (Directory * r)
{
	if (r == 0)
	{
		return;
	}

	if(!r->fullName ())
	{
		return;
	}
	QListViewItem *dir = r;
	while (dir != 0)
	{
		if (dir->text (1) != i18n("Directory"))
		{
			dir = 0;
		}
		else
		{
			if (dir->isSelected ())
			{
				((Directory *) dir)->unLoadDir ();
				((Directory *) dir)->loadDir ();
			}

			if (dir->firstChild ())
			{
				nextDir ((Directory *) (dir->firstChild ()));
			}
		}
		dir = dir->itemBelow ();
	}
}


void
MainWindow::renameSeries()
{
	QApplication::setOverrideCursor (waitCursor);	
	RenameSeries rs(this,"RenameSeries",true);
	QString fullName, name;
	bool hasFiles=false;
	for (FileItem * item = (FileItem *) ((QIconView*)imageList)->firstItem (); item != 0; item = (FileItem *)item->nextItem ())
	{
		if (item->isSelected () )
		{
			rs.addFile(item->fullName());
			hasFiles=true;
		}
	}
	if(!hasFiles)
	{
		QApplication::restoreOverrideCursor ();
		KMessageBox::error(this, i18n("You have to select one file at least"));
		return;
	}
	QApplication::restoreOverrideCursor ();
	if(rs.exec())
		slotRefresh();
	
}


void
MainWindow::slotRefresh (const char *dir)
{
	setMessage(i18n("Refreshing..."));

	QCString *res = new QCString (dir);
	int  pos = res->find ("/");
	Directory *ssrep;

	res->setStr (res->right (res->length () - pos - 1));
	pos = res->find ("/");

	ssrep = root;
	while (pos != -1)
	{
		ssrep = ssrep->find (res->left (pos));
		res->setStr (res->right (res->length () - pos - 1));

		if (ssrep)
		{
			pos = res->find ("/");
		}
		else
			break;
	}
	ssrep = ssrep->find (*res);
	if (ssrep && ssrep->isSelected ())
	{
		ssrep->unLoadDir ();
		ssrep->loadDir ();
	}
	setMessage(i18n("Ready"));
}


void
MainWindow::slotRefresh ()
{
	setMessage(i18n("Refreshing..."));
	QList<Directory> dirList;
	QList<CompressedFileItem> compList;	

	QListViewItemIterator it (root);
	for (; it.current (); ++it)
	{
		if (it.current ()->text (1) == i18n("Directory"))
		{
			if (it.current ()->isSelected ())
			{
				dirList.append((Directory *)it.current());
			}
		}
		else
		{
			if (it.current ()->isSelected ())
			{
				compList.append((CompressedFileItem *) it.current());
			}
		}
	}
	
	QString current=imageList->currentItemName();
	
	Directory *dir;
	for ( dir=dirList.first(); dir != 0; dir=dirList.next() )	
		dir->refresh();
	CompressedFileItem *comp;
	for ( comp=compList.first();  comp!= 0; comp=compList.next() )	
		comp->unLoadArchive();
	//	
	for ( comp=compList.first();  comp!= 0; comp=compList.next() )	
		comp->loadArchive(false);
	
	imageList->setCurrentItemName(current, false);
	imageList->slotLoadFirst();	

	dirView->sort();
	setMessage(i18n("Ready"));

}



void
MainWindow::setHasImageSelected(bool selected)
{
	if(hasimageselected==selected)
		return;
	else
		hasimageselected=selected;	

	//aCut->setEnabled(selected);
	aCopyPixmap->setEnabled(selected);
	aWallpaper_CENTER->setEnabled(selected);
	aWallpaper_MOSAIC->setEnabled(selected);
	aWallpaper_CENTER_MOSAIC->setEnabled(selected);
	aWallpaper_CENTER_MAX->setEnabled(selected);	
	aWallpaper_ADAPT->setEnabled(selected);
	aWallpaper_LOGO->setEnabled(selected);	
	aPrint->setEnabled(selected);
	aRename->setEnabled(selected);
	aDelete	->setEnabled(selected);	
	aTrash	->setEnabled(selected);	
	aShred	->setEnabled(selected);	
	aEditType->setEnabled(selected);		
	aFileProperties->setEnabled(selected);	
	aImageInfo->setEnabled(selected);	
	aRotLeft->setEnabled(selected);
	aRotRight->setEnabled(selected);
	aHMirror->setEnabled(selected);
	aVMirror->setEnabled(selected);
	aFullScreen->setEnabled(selected);
	aZoomIn	->setEnabled(selected);
	aZoomOut->setEnabled(selected);
	aZoomFit->setEnabled(selected);
	aZoomFitWidth->setEnabled(selected);
	aZoomFitHeight->setEnabled(selected);
	aZoomNo	->setEnabled(selected);
	aZoomLock	->setEnabled(selected);
	aOpenWithGimp->setEnabled(selected);
	aOpenWith->setEnabled(selected);
	aFilesMoveTo->setEnabled(selected);
	aFilesCopyTo->setEnabled(selected);
}


void
MainWindow::slotPreview ()
{
	if(aPreview->isChecked())
		imageList->slotLoadFirst();
	else
	{
		slotStop();
		imageList->slotResetThumbnail();
	}
}

void
MainWindow::slotKeepFit ()
{
}

void
MainWindow::slotShrink ()
{
	iv->setShrink(aShrink->isChecked());
}

void
MainWindow::slotEnlarge ()
{
	iv->setEnlarge(aEnlarge->isChecked());
}


void
MainWindow::slotSmooth ()
{
	iv->setSmooth(aSmooth->isChecked());
}


void
MainWindow::slotSlideShow ()
{
	if (!timer->isActive())
	{
		int time=0;
		switch(slideshowTime)
		{
			case 1 : time =   1000; break;
			case 2 : time =   5000; break;
			case 3 : time =  10000; break;
			case 4 : time =  30000; break;
			case 5 : time =  60000; break;
			case 6 : time = 120000; break;
		}
		timer->start (time, FALSE);
	}
	else
	{
		timer->stop ();
	}
}


void
MainWindow::setOnTop(bool onTop)
{
	if(onTop)
	{
		KWin::setState (dockIV->winId (), NET::StaysOnTop);
	}
	else
	{
		KWin::clearState(dockIV->winId (), NET::StaysOnTop);
	}
}


void
MainWindow::slotFullScreen()
{
	if(! iv->hasImage())
		return;
  	setUpdatesEnabled(true);

	if (!inFullScreen )
	{
		imageList->inFullScreen = true;
		inFullScreen = true;
		oldPos=pos();
		oldSize=size();
		writeDockConfig ();
		QWidget *d = QApplication::desktop ();
		dockIV->manualDock (0, KDockWidget::DockFullDocking);
		KWin::setType (dockIV->winId (), NET::Override);
		dockIV->setGeometry (0, -10, d->size ().width (),d->size ().height () + 30);
		KWin::setState (dockIV->winId (), NET::StaysOnTop);
		inInterface=true;
		setUpdatesEnabled(true);
	}
	else
	{
		hide();
		inFullScreen = false;
		imageList->inFullScreen = false;		
		readDockConfig ();
		setUpdatesEnabled(true);
		KWin::setType (winId (), NET::Normal);
		move(oldPos);
		resize(oldSize);
		show();
	}
	setFocus();
	KWin::setActiveWindow(winId());
}


void
MainWindow::closeEvent( QCloseEvent* ce )
{
	quit();	
}

bool
MainWindow::close()
{	
	quit();	
	return true;
}

void
MainWindow::quit()
{
	if(inFullScreen)
		slotFullScreen();
	/////
	KConfig *config = new KConfig("showimgrc");
	config->setGroup("Options");
	config->writeEntry("quitOK",true);
	config->writeEntry( "Width", width() ); // largeur de la fentre
	config->writeEntry( "Height", height() ); // hauteur de la fentre
	config->writeEntry( "nbImg", iv->nbImg );
	config->writeEntry( "preview", aPreview->isChecked() );
	config->writeEntry( "bgcolor", bgColor);
	config->writeEntry( "grayscale", grayscale );
	config->writeEntry( "storeth", storeth );
	config->writeEntry( "smooth", smooth );
	config->writeEntry( "showhiddenDir", dirView->showHiddenDir() );
	config->writeEntry( "showhiddenFile", dirView->showHiddenFile() );

	config->setGroup("Slideshow");
	config->writeEntry( "time", slideshowTime );
	config->writeEntry( "type", slideshowType );

	config->setGroup("Zoom");
	config->writeEntry( "shrink", aShrink->isChecked() );
	config->writeEntry( "enlarge", aEnlarge->isChecked() );
	config->writeEntry( "lock", aZoomLock->isChecked() );

	config->setGroup("confirm");
	config->writeEntry( "trash", checkTrash);
	config->writeEntry( "delete", checkDelete );
	config->writeEntry( "shred", checkShred );
	
	config->sync();
	writeDockConfig();
	////
	qApp->quit();
}


void 
MainWindow::slotAddImage (int number)
{
	total+=number;
	setNbrItems(total);
	progress->setTotalSteps (total);
	if(aPreview->isChecked())
		progress->show();
}

void 
MainWindow::slotRemoveImage ()
{
	total--;setNbrItems(total);
	progress->setTotalSteps (total);
}

void 
MainWindow::slotPreviewDone ()
{
	done++;
	progress->setProgress (done);
};

void 
MainWindow::slotReset (bool init)
{
	aStop->setEnabled(true);
	if(init)
	{	
		done = 0;
		progress->reset ();
	}
	progress->show();
};

void
MainWindow::slotDone()
{
	aStop->setEnabled(false);
 	done = total;
 	progress->setProgress (done);
	progress->hide();
};

void 
MainWindow::slotRemoveImage (int val)
{
	total -= val;setNbrItems(total);
	progress->setTotalSteps (total);
	progress->setProgress (total);
};


void  
MainWindow::slotRotateLeft ()
{
	setMessage(i18n("Rotating..."));
	iv->rotateLeft();
	setMessage(i18n("Ready"));
}

void  
MainWindow::slotRotateRight ()
{
	setMessage(i18n("Rotating..."));
	iv->rotateRight();
	setMessage(i18n("Ready"));
}

void  
MainWindow::slotMirrorH ()
{
	setMessage(i18n("Flip..."));
	iv->mirror (false, true);
	setMessage(i18n("Ready"));
}


void 
MainWindow::slotMirrorV ()
{
	setMessage(i18n("Flip..."));
	iv->mirror (true, false);
	setMessage(i18n("Ready"));
}

void 
MainWindow::slotDirChange (const QString & dir)
{
	findDir(dir)->refresh();
};

void 
MainWindow::slotFileChange (const QString & file)
{
};

void 
MainWindow::slotPreferences()
{
}


ImageSimilarityData::ImageSimilarityData()
{
	avg_r=(uint*)malloc(PAS*PAS*sizeof(uint));
	avg_g=(uint*)malloc(PAS*PAS*sizeof(uint));
	avg_b=(uint*)malloc(PAS*PAS*sizeof(uint));
}

ImageSimilarityData::~ImageSimilarityData()
{
	delete(avg_r);
	delete(avg_g);
	delete(avg_b);
}

void 
MainWindow::slotTODO ()
{
	KMessageBox::sorry(this,
		i18n("Not yet implemented !!!\nSorry ;("),
		i18n("Fonctionnality"));
}

void 
MainWindow::slotcopyPixmap()
{
	QApplication::setOverrideCursor (waitCursor); // this might take time
	QApplication::clipboard()->setPixmap(iv->getPixmap());
	QApplication::restoreOverrideCursor ();
}

void 
MainWindow::slotcopy ()
{
	QString files, name;
	QStrList uris;

	for (FileItem* item = (FileItem *)((QIconView*)imageList)->firstItem (); item != 0; item = (FileItem *) item->nextItem ())
	{
		if (item->isSelected () )
		{
			uris.append(QUriDrag::localFileToUri(item->fullName()));
		}
	}
	QClipboard *cb = QApplication::clipboard();
	cb->setData(new QUriDrag(uris, this));
}


void 
MainWindow::slotcut ()
{
	slotTODO ();
}

void 
MainWindow::slotpaste ()
{
	QStrList uris;
	if(QUriDrag::decode(QApplication::clipboard()->data(), uris))
	{
		if(!uris.isEmpty())
			dirView->copy(uris, currentTitle());
	}
}


void 
MainWindow::slotNewWindow()
{
	new MainWindow("new Main Frame", currentDir);
}

void 
MainWindow::slotPrint()
{
	int res;
	
#if KDE_VERSION > 220
	KPrinter printer;
#else
	QPrinter printer;
#endif
	printer.setFullPage( true );
	do {
		if (!printer.setup(this))
			return;	
		printImageDialog printDlg(this, iv->getPixmap(), iv->getFilename(), &printer);
		res=printDlg.exec();
	}
	while(res==2);

}


void 
MainWindow::slotQuickPrint()
{

}


void 
MainWindow::slotOpenLocation()
{
	QString destDir=KFileDialog::getExistingDirectory(currentDir,
								this,
								i18n("Open Location"));

	if(!destDir.isEmpty())
	{
		if(!QFileInfo(destDir).exists())
		{
			KMessageBox::error(this, QString().sprintf(i18n("The directory '<b>%s</b>' doesn't exist"),(const char*)destDir));
			return;
		}
		open(destDir);
		changeDirectory(destDir);
	}
}

void 
MainWindow::slotUndo()
{
	slotTODO ();
}


void 
MainWindow::slotRedo()
{
	slotTODO ();
}


void 
MainWindow::slotMovetotrash()
{
}

void 
MainWindow::slotShred()
{
}

void 
MainWindow::slotEditFileType()
{
	if(!imageList->currentItem())
		return;
	KonqOperations::editMimeType( ((FileItem*)imageList->currentItem())->mimetype());
}


void 
MainWindow::slotStop()
{
	imageList->stopLoading ();
	slotDone();
}


void 
MainWindow::slotZoomIn()
{
	setMessage(i18n("Zooming In..."));
	iv->zoomIn(1.5);
	setMessage(i18n("Ready"));
}



void 
MainWindow::slotZoomOut()
{
	setMessage(i18n("Zooming Out..."));
	iv->zoomOut(1.5);
	setMessage(i18n("Ready"));
}


void 
MainWindow::slotZoom()
{
        setMessage(i18n("Toggle fit to screen..."));
	iv->setFit(true);
	setMessage(i18n("Ready"));
}


void 
MainWindow::slotZoomNo()
{
	setMessage(i18n("Original size..."));
	iv->originalSize();	
	setMessage(i18n("Ready"));
}



void 
MainWindow::configureKey()
{
	KKeyDialog::configureKeys(mAccel, true, this);
}


void 
MainWindow::configureToolbars()
{
	saveMainWindowSettings( KGlobal::config(), "MainWindow" );
	KEditToolbar dlg(actionCollection());
	if (dlg.exec())
	{
		createGUI();
	}

}


void 
MainWindow::configureShowImg()
{
	ConfShowImg conf(this);
	conf.initColor(bgColor, grayscale);
	conf.initFiling(checkTrash, checkDelete, checkShred);
	conf.initMiscellaneous(storeth, smooth, dirView->showHiddenDir(), dirView->showHiddenFile());
	conf.initSlideshow(slideshowType, slideshowTime);
	if(conf.exec())
	{
		checkDelete=conf.checkDelete();
		checkShred=conf.checkShred();
		checkTrash=conf.checkTrash();
		dirView->setCheckDelete(checkDelete);dirView->setCheckTrash(checkTrash);
		dirView->setShowHiddenDir(conf.getShowDir());dirView->setShowHiddenFile(conf.getShowFile());
		imageList->setCheckDelete(checkDelete);imageList->setCheckTrash(checkTrash); imageList->setCheckShred(checkShred);
		imageList->setRandom(conf.getSlideshowType()==2);
		
		bgColor = conf.getColor();
		iv->setBackgroundColor(bgColor);
		
		grayscale=conf.getGrayscale();
		iv->setToGrayscale(grayscale);
		
		smooth=conf.getSmooth();
		iv->setSmooth(smooth);
		
		slideshowTime=conf.getSlideshowTime();
		slideshowType=conf.getSlideshowType();
		
		storeth=conf.getStoreth();
		
		setLayout(conf.getLayout());
	}
}

void
MainWindow::setLayout(int layout)
{
	switch(layout)
	{
		case 1:
				dockDir->manualDock(dockIV, KDockWidget::DockLeft, 35);	
				dockIL->manualDock(dockDir, KDockWidget::DockBottom, 35);
				aArrangementB->setChecked(true);
				aArrangementLR->setChecked(true);
			break;
		case 2:
				dockDir->manualDock(dockIV, KDockWidget::DockTop, 35);	
				dockIL->manualDock(dockDir, KDockWidget::DockRight, 50);
				aArrangementB->setChecked(true);
				aArrangementLR->setChecked(true);
			break;
		case 3:
				dockIL->manualDock(dockIV, KDockWidget::DockRight, 35);
				dockDir->manualDock(dockIV, KDockWidget::DockTop, 35);	
				aArrangementB->setChecked(true);
				aArrangementLR->setChecked(true);
			break;
		case 4:
				dockDir->manualDock(dockIV, KDockWidget::DockLeft, 35);	
				dockIL->manualDock(dockIV, KDockWidget::DockTop, 10);
				aArrangementB->setChecked(true);
				aArrangementTB->setChecked(true);
			break;
		default:
			return;
			break;
	}
	slotArrangement();
	slotTxtPos();
}

void
MainWindow::slotBookmark()
{
}



void
MainWindow::openBookmarkURL(const QString& _url)
{
	KURL url(_url);
	if(!url.isLocalFile())
	{
		KMessageBox::error(this, QString().sprintf(i18n("The directory '<b>%s</b>' is not local"),(const char*)url.url()));
		return;
	}
	else
	{
		if(!QFileInfo(url.path()).exists())
		{
			KMessageBox::error(this, QString().sprintf(i18n("The directory '<b>%s</b>' doesn't exist"),(const char*)url.url()));
			return;
		}
		open(url.path());
	}
}


QString
MainWindow::currentTitle() const
{
	return currentDir;
}


QString
MainWindow::currentURL() const
{
	return currentDir;
}



void
MainWindow::slotBack()
{
	go(-1);
}



void
MainWindow::slotForward()
{
	go(+1);
}

void
MainWindow::updateHistory()
{
	HistoryEntry * current = m_lstHistory.current();
	HistoryEntry* newEntry = new HistoryEntry;
	newEntry->filePath = currentDir;
	if (current && current->filePath == newEntry->filePath) {
		delete newEntry;
		return;
	}
	if (current)
	{
		m_lstHistory.at( m_lstHistory.count() - 1 ); // go to last one
		for ( ; m_lstHistory.current() != current ; )
		{
			m_lstHistory.removeLast();
		}
	}
	m_lstHistory.append(newEntry);
	aBack->setEnabled( m_lstHistory.at() > 0 );
	aForward->setEnabled( m_lstHistory.at() != ((int)m_lstHistory.count())-1 );
}



void MainWindow::go( int steps )
{
	updateHistory();
	int newPos = m_lstHistory.at() + steps;
	HistoryEntry* l = m_lstHistory.at( newPos );
	if(open(l->filePath))
	{
		aBack->setEnabled( m_lstHistory.at() > 0 );
		aForward->setEnabled( m_lstHistory.at() != ((int)m_lstHistory.count())-1 );
	}
}

void
MainWindow::slotForwardAboutToShow()
{
	aForward->popupMenu()->clear();
	HistoryAction::fillHistoryPopup( m_lstHistory, aForward->popupMenu(), false, true );
}


void
MainWindow::slotBackAboutToShow()
{
	aBack->popupMenu()->clear();
	HistoryAction::fillHistoryPopup( m_lstHistory, aBack->popupMenu(), true, false );
}


void
MainWindow::slotDirCut()
{
	slotTODO();
}

void
MainWindow::slotDirCopy()
{
	slotTODO();
}

void
MainWindow::slotDirPaste()
{
	slotTODO();
}

void
MainWindow::slotDirMove()
{
	slotTODO();	}

void
MainWindow::slotArrangement()
{
	if(aArrangementLR->isChecked())
		imageList->setArrangement(ImageListeView::LeftToRight);	
	else
		imageList->setArrangement(ImageListeView::TopToBottom);
}

void
MainWindow::slotTxtPos()
{
	if(aArrangementR->isChecked())
		imageList->setItemTextPos(ImageListeView::Right);
	else
		imageList->setItemTextPos(ImageListeView::Bottom);
}

void
MainWindow::slotZoomLock()
{
	iv->setZoomLock(aZoomLock->isChecked());
}

bool
MainWindow::fullScreen()
{
	return inFullScreen;
}

void
MainWindow::setEmptyImage()
{
	iv->loadImage(NULL);
	iv->repaint();
}

void
MainWindow::copyFilesTo(QStrList uris, QString dest)
{
	dirView->copy(uris, dest);
}

void
MainWindow::moveFilesTo(QStrList uris, QString dest)
{
	dirView->move(uris, dest);
}


void
MainWindow::startWatchDir()
{
	dirView->startWatchDir();
}
void
MainWindow::stopWatchDir()
{
	dirView->stopWatchDir();
}



void 
MainWindow::toolsRotateLeft()
{
	setMessage(i18n("Rotating..."));
	QList <QString> list;
	list.setAutoDelete (false);

	for (ImageFileItem * item = (ImageFileItem *)((QIconView*)imageList)->firstItem ();
	     item != 0;
	     item = (ImageFileItem *) item->nextItem ())
	{
		if (item->isSelected () && item->text(3)=="file"  )
	        {
			list.append(new QString(item->fullName()));
		}
	}
	if(list.isEmpty())
	{
		KMessageBox::error(this, i18n("You have to select at least one file"));
	}
	else
	{
	stopWatchDir();
	
	QProgressDialog*  progres =  new QProgressDialog ("Conversion", "&Cancel", list.count(), this);
	progres->setMinimumDuration(1);
	progres->show();

	QString *name;
	int done=0;

	KShellProcess* proc = new KShellProcess ();
	connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(slotEndConvert(KProcess *)));
	
	QString  msg, com;
	
	for ( name=list.first(); name != 0; name=list.next() )
	{
		msg.sprintf ("Conversion of <b>%s</b>\n(%d/%d)",
					 (const char*)*name,
					 done+1,
		  	     list.count ());
		done++;
		if ( progres->wasCancelled() )
			break;
		progres->setLabelText (msg);
		progres->setProgress (done-1);
		qApp->processEvents();

		proc->clearArguments();
		com.sprintf ("convert -rotate 270 '%s' '%s'",
		  	     (const char *)*name, (const char *)*name);
		*proc << com;
		proc->start (KShellProcess::Block);
		qApp->processEvents ();
	}
	delete(progres);
	delete(proc);
	}
	setMessage(i18n("Ready"));
	startWatchDir();
	slotRefresh ();
}


void 
MainWindow::toolsRotateRight()
{
	setMessage(i18n("Rotating..."));
	QList <QString> list;
	list.setAutoDelete (false);

	for (ImageFileItem * item = (ImageFileItem *)((QIconView*)imageList)->firstItem ();
	     item != 0;
	     item = (ImageFileItem *) item->nextItem ())
	{
		if (item->isSelected () && item->text(3)=="file"  )
	        {
			list.append(new QString(item->fullName()));
		}
	}
	if(list.isEmpty())
	{
		KMessageBox::error(this, i18n("You have to select at least one file"));
	}
	else
	{
	stopWatchDir();
	
	QProgressDialog*  progres =  new QProgressDialog ("Conversion", "&Cancel", list.count(), this);
	progres->setMinimumDuration(1);
	progres->show();

	QString *name;
	int done=0;

	KShellProcess* proc = new KShellProcess ();
	connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(slotEndConvert(KProcess *)));
	
	QString  msg, com;
	
	for ( name=list.first(); name != 0; name=list.next() )
	{
		msg.sprintf ("Conversion of <b>%s</b>\n(%d/%d)",
					 (const char*)*name,
					 done+1,
		  	     list.count ());
		done++;
		if ( progres->wasCancelled() )
			break;
		progres->setLabelText (msg);
		progres->setProgress (done-1);
		qApp->processEvents();

		proc->clearArguments();
		com.sprintf ("convert -rotate 90 '%s' '%s'",
		  	     (const char *)*name, (const char *)*name);
		*proc << com;
		proc->start (KShellProcess::Block);
		qApp->processEvents ();
	}
	delete(progres);
	delete(proc);
	}
	setMessage(i18n("Ready"));
	startWatchDir();
	slotRefresh ();
}
