/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "showimg.h"

#include "mainwindow.h"

#include <stdlib.h>
#include <math.h>

#include <qmenubar.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qkeycode.h>
#include <qapplication.h>
#include <qdatetime.h>  

#include <kmessagebox.h>
#include <kiconloader.h>
#include <kimageeffect.h>
#include <krun.h>
#include <kalphapainter.h>

#define ZOOM_MAX 150.0

int min(int a, int b)
{
	return a<b?a:b;
}
int max(int a, int b)
{
	return a>b?a:b;
}
/* XPM */
static const char *bg_xpm[] = {
"16 16 3 1",
"       c None",
".      c #656765",
"+      c #9B999B",
"++++++++........",
"++++++++........",
"++++++++........",
"++++++++........",
"++++++++........",
"++++++++........",
"++++++++........",
"++++++++........",
"........++++++++",
"........++++++++",
"........++++++++",
"........++++++++",
"........++++++++",
"........++++++++",
"........++++++++",
"........++++++++"};

static int nbrMU;

ImageViewer::ImageViewer (QWidget * parent,
			ImageListeView * il,
			MainWindow *mw,
			const char *name, int wFlags):
QWidget (parent, name, wFlags),
filename (0)
{
	this->il = il;
	this->mw = mw;
  
	total = 0;
	ss = true;	
	fit = false;
	shrink=false;
	enlarge=false;
	
	preloadedImage = new QImage ();
	image = new QImage ();
	
	sp=ep=NULL;
	movie = 0L;
	hasimage = false;
	nbImg=0;
	//
	dragStartPosX = -1;
	dragStartPosY = -1;
	difTopPosX = -1;
	difTopPosX = -1;
	realPosX=-1,
	realPosY=-1;
	button=NoButton;
	scale=1;
	setToGrayscale(-1);
	//
	pIO=new KPixmapIO();
}


void
ImageViewer::initMenu(KActionCollection *actionCollection)
{
	popup = new QPopupMenu();
	actionCollection->action("FullScreen")->plug(popup);
		
	popup->insertSeparator();
	KPopupMenu *popupsort = new KPopupMenu ();
	actionCollection->action("Zoom in")->plug(popupsort);
	actionCollection->action("Zoom out")->plug(popupsort);
	actionCollection->action("Fit to Screen")->plug(popupsort);
	actionCollection->action("Fit the width")->plug(popupsort);
	actionCollection->action("Fit the height")->plug(popupsort);
	actionCollection->action("Originale size")->plug(popupsort);
	actionCollection->action("ZoomLock")->plug(popupsort);
	popupsort->insertSeparator();
	actionCollection->action("Enlarge")->plug(popupsort);
	actionCollection->action("Shrink")->plug(popupsort);
	popup->insertItem(i18n("Zoom"), popupsort);

	KPopupMenu *popupmove = new KPopupMenu ();
	actionCollection->action("First Image")->plug(popupmove);
	actionCollection->action("Previous Image")->plug(popupmove);
	actionCollection->action("Next Image")->plug(popupmove);
	actionCollection->action("Last Image")->plug(popupmove);
	actionCollection->action("Slideshow")->plug(popupmove);
	popup->insertItem(i18n("Go to..."), popupmove);

	popup->insertSeparator();
	actionCollection->action("editcopypixmap")->plug(popup);
	actionCollection->action("editcopy")->plug(popup);
	actionCollection->action("Open with")->plug(popup);
	actionCollection->action("Open with Gimp")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("editdelete")->plug(popup);
	actionCollection->action("edittrash")->plug(popup);
	actionCollection->action("editshred")->plug(popup);

	popup->insertSeparator();
	actionCollection->action("Properties")->plug(popup);
	actionCollection->action("Image Info")->plug(popup);
}


QPixmap
ImageViewer::getPixmap()
{
	return pIO->convertToPixmap(*image);
}


QString
ImageViewer::getFilename()
{
	return imageName;
}


void
ImageViewer::setToGrayscale(int togray)
{
	grayscale=togray;
}


int
ImageViewer::toGrayscale()
{
	return grayscale;
}

bool
ImageViewer::preloadImage (const char *fileName)
{
	preimageName.setStr(fileName);
	preloadedImage->load(fileName);
	return preloadedImage->load(fileName);
}


/////////////////////////////////
/////////////////////////////////
/*
  This function loads an image from a file and resizes the widget to
  exactly fit the image size. If the file was not found or the image
  format was unknown it will resize the widget to fit the errorText
  message (see above) displayed in the current font.

  Returns TRUE if the image was successfully loaded.
*/

bool
ImageViewer::loadImage (const char *fileName)
{
	mw->setMessage(i18n("Loading image..."));
	bool ok = false;
	if (fileName)
	{
		FileItem *item=il->findItem(QFileInfo(fileName).fileName());
		if(item) {if(!item->isSelected()) return false;} else return false;
				
		filename = strdup(fileName);
		nbImg++;
		if (fileName == preimageName)
		{
			delete (image);image = new QImage (*preloadedImage);
			ok=reconvertImage();
		}
		else
		{
			delete(image); image = new QImage ();
			ok=image->load(filename);
			if(ok) ok=reconvertImage();
		}
		
	}
	if(ok)
	{
		
		if(movie)
		{
			movie->disconnectUpdate(this);
			movie->disconnectStatus(this);
			movie->pause();
		}
			
		doScale(false);
		imageName=fileName;
		mw->setZoom((int)(scale*100));
		//
		QString ext=QString(filename).right(3).lower();
		if( (ext == "mng")  ||
			(ext == "gif")   )
		{
			startMovie();
		}
		else
		{
			movie=0L;
			repaint();
		}
	}
	else
	{
		 filename=strdup("(none)");		
		 delete(image);image=new QImage();
		 repaint();
	}
	updateStatus ();
	hasimage = ok;
	mw->setMessage(i18n("Ready"));	
	return ok;
}


bool
ImageViewer::hasImage()
{
	return hasimage;
}


bool
ImageViewer::reconvertImage()
{
	if (image->isNull ())
		return FALSE;
	
	
	if(image->hasAlphaBuffer())
	{
		QPixmap pix(image->width(), image->height(),QPixmap::defaultDepth () );
		QPainter p;
			p.begin(&pix);
			p.drawTiledPixmap (0, 0, image->width(), image->height(), QPixmap(bg_xpm));
			p.end();
		QImage bg=pix.convertToImage();
		QPainter pI(&pix);
		KAlphaPainter::draw(&pI, *image, bg, 0,0);
		*image=pix.convertToImage();
	}
	
	if(toGrayscale()!=-1)
		*image=KImageEffect::desaturate(*image, toGrayscale()/100.0);
	
	return true;
}


bool
ImageViewer::smooth () const
{
	return ss;
}


void
ImageViewer::scalePreload ()
{
}


void
ImageViewer::startMovie()
{
	delete(movie); movie=0L;
	if ( filename ) // Start a new movie - have it delete when closed.
		initMovie();
}
void
ImageViewer::initMovie()
{
	movie = new QMovie(filename);
	nbrMU=-1;
	QPixmap pix(image->width(), image->height()); pix.fill(bgBrush.color()); *image=pIO->convertToImage(pix);repaint();
	movie->setBackgroundColor(bgBrush.color());
	movie->connectUpdate(this, SLOT(movieUpdated(const QRect&)));
	movie->connectStatus(this, SLOT(movieStatus(int)));
}
void
ImageViewer::movieUpdated(const QRect& area)
{
	if(! movie)	
		return;
	nbrMU++;
	if(nbrMU>movie->frameNumber())
	{
		movie->disconnectUpdate(this);
		movie->disconnectStatus(this);
		movie->pause();
		movie=0L;
		image=new QImage(filename);
		reconvertImage();
		repaint();
		return;
	}
	*image = pIO->convertToImage(movie->framePixmap());
	repaint();
}
void
ImageViewer::movieStatus(int status)
{
	if (movie && status<0)
	{
		QString msg;
		msg.sprintf(i18n("Could not play movie \"%s\""), (const char*)filename);
		KMessageBox::error(this, msg);
	}
	if(status==3) nbrMU=-1;
}
void
ImageViewer::movieResized(const QSize& size)
{
}




void
ImageViewer::mousePressEvent (QMouseEvent * e)
{
	QWidget::mousePressEvent(e);
	button= e->button ();
	if(movie)
		return;
	if (button==RightButton)
	{
    }
	else
	if (button==LeftButton)
	{
		QApplication::setOverrideCursor (sizeAllCursor);	
		dragStartPosX=e->pos().x();
		dragStartPosY=e->pos().y();	
		difTopPosX = getVirtualPosX()-dragStartPosX;
		difTopPosY = getVirtualPosY()-dragStartPosY;
	}
	else
	{
		delete(sp); sp = new QPoint(e->pos()) ;
		ep = new QPoint(*sp);
	}
}


void
ImageViewer::mouseReleaseEvent (QMouseEvent * e)
{
	if (e->button()==LeftButton)
	{
		QApplication::restoreOverrideCursor ();
		QWidget::mouseReleaseEvent(e);
		dragStartPosX=-1;
		dragStartPosY=-1;	
		repaint();
	}
	else
	if (e->button()==RightButton)
	{
		popup->exec(e->globalPos());
	}
	else
	if( (movie) || (e->button()!=MidButton) )
	{
	}
	else
	if(image)
	{
		delete(ep); ep = new QPoint(e->pos()) ;
		if(*sp==*ep)
		{
			ep=NULL;
			QApplication::setOverrideCursor (waitCursor);	// this might take time
			doScale();placeImage();repaint();
			QApplication::restoreOverrideCursor ();	// restore original cursor
			return;
		}
		else
		{
			QRect rectDraged= QRect(QPoint(min(sp->x(), ep->x()), min(sp->y(), ep->y())),
					 QPoint(max(sp->x(), ep->x()), max(sp->y(), ep->y())));
			rectDraged.moveBy(-getVirtualPosX(), -getVirtualPosY());
			QPoint oldCenter=QPoint( (int)(((max(sp->x(), ep->x())+min(sp->x(), ep->x()))/2-getVirtualPosX())/scale),
						 (int)(((max(sp->y(), ep->y())+min(sp->y(), ep->y()))/2-getVirtualPosY())/scale));			
			float
				sx=width()/rectDraged.width(),
				sy=height()/rectDraged.height();
			if(scale*((float)sx+(float)sy)/2 > ZOOM_MAX)	
				scale=ZOOM_MAX;
			else
				scale*=((float)sx+(float)sy)/2;
			
			mw->setZoom((int)(scale*100));
			oldCenter*=scale;			
			ep=NULL;
			
			centerImage(oldCenter.x(), oldCenter.y(), true);
		}
		ep=NULL;
		QApplication::restoreOverrideCursor ();
	}
	ep=NULL;
	button=NoButton;
}


void
ImageViewer::mouseMoveEvent(QMouseEvent * e)
{
	if (button==LeftButton)
	{
		QWidget::mouseMoveEvent(e);
		if ( (dragStartPosX+dragStartPosY!=-2.0))
		{
			double
				diffX=e->pos().x()-dragStartPosX,
				diffY=e->pos().y()-dragStartPosY,
				panX=0, panY=0;
				
			if (virtualPictureWitdth()>width())
			{
				if(fabs(diffX)>=scale)
				{
					panX=(int)diffX;
					dragStartPosX+=panX;
					if(!posXForTopXIsOK(difTopPosX+dragStartPosX))
					{	
						dragStartPosX-=panX;
						panX=0;
					}
				}
			}
			if (virtualPictureHeight()>height())
			{
				if(fabs(diffY)>=scale)
				{
					panY=diffY;
					dragStartPosY+=panY;
					if(!posYForTopYIsOK(difTopPosY+dragStartPosY))
					{
						dragStartPosY-=panY;
						panY=0;
					}
				}
			}
			if(panX!=0 || panY!=0)
				scroll((int)panX, (int)panY);
		}
		return;
	}
	else
	if(movie || !ep)
		return;

	QPainter p(this); p.setPen(QColor("black"));
	lp = new QPoint(*ep);
	ep = new QPoint(e->pos()) ;
	int
		lx = lp->x(),
		ly = lp->y(),
		ex = ep->x(),
		ey = ep->y(),
		sx = sp->x(),
		sy = sp->y();
		
	int tlx,tly,brx,bry;

		tlx=(sx<ex?sx:ex);
		tly=(ly<ey?ly:ey);
		brx=(sx>ex?sx:ex);
		bry=(ly>ey?ly:ey);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));
		
		tlx=(lx<ex?lx:ex);
		tly=(sy<ey?sy:ey);
		brx=(lx>ex?lx:ex);
		bry=(sy>ey?sy:ey);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));

		tlx=(lx<ex?lx:ex);
		tly=(ly<ey?ly:ey);
		brx=(lx>ex?lx:ex);
		bry=(ly>ey?ly:ey);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));

		tlx=(lx<sx?lx:sx);
		tly=(sy<ly?sy:ly);
		brx=(lx>sx?lx:sx);
		bry=(sy>ly?sy:ly);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));

	p.drawRect(QRect(*sp,*ep));
}


void
ImageViewer::focusInEvent ( QFocusEvent *fe )
{
	KWin::setActiveWindow(mw->winId());
}
void
ImageViewer::focusOutEvent ( QFocusEvent *fe )
{
	QWidget::focusOutEvent(fe);
}
void
ImageViewer::keyPressEvent (QKeyEvent * e)
{
	QWidget::keyPressEvent(e);
}

void
ImageViewer::wheelEvent (QWheelEvent * e)
{
	e->accept();
	if (e->delta () < 0)
		il->next ();
	else
		il->previous ();
}

#define CENTER 1
#define MOSAIC 2
#define CENTER_MOSAIC 3
#define CENTER_MAX 4
#define ADAPT 5
#define LOGO 6
			
void
ImageViewer::wallpaperC(){wallpaper(CENTER);};
void
ImageViewer::wallpaperM(){wallpaper(MOSAIC);};
void
ImageViewer::wallpaperCM(){wallpaper(CENTER_MOSAIC);};
void
ImageViewer::wallpaperCMa(){wallpaper(CENTER_MAX);};
void
ImageViewer::wallpaperA(){wallpaper(ADAPT);};
void
ImageViewer::wallpaperL(){wallpaper(LOGO);};


void
ImageViewer::wallpaper(int mode)
{
	/*
	1:center
	2:mosaic
	3:center mosaic
	4:center max
	5:adapt
	6:logo
	*/
	if(mode>LOGO)
		return
	
	mw->setMessage(i18n("Set as Wallpaper"));	
	QString com;
	com.sprintf ("dcop kdesktop KBackgroundIface setWallpaper '%s' %d >/dev/null 2>/dev/null", (const char *)filename, mode);
    KRun::runCommand(com);
	mw->setMessage(i18n("Ready"));
}



void
ImageViewer::setFit(bool fit, bool keep)
{
	if(keep)
		this->fit=fit;
	if(fit)
		scaleFit();
	else
		doScale();
	repaint();
}

void
ImageViewer::setSmooth(bool s)
{
	ss=s;
	doScale();
}


void
ImageViewer::scroll( int dx, int dy )
{
	QWidget::scroll(dx,dy);	
}

void
ImageViewer::mirror (bool horizontal, bool vertical)
{
	QApplication::setOverrideCursor (waitCursor);	// this might take time

	QPixmap pix(image->width(), image->height());
	QPainter p;
	p.begin(&pix);
	if(vertical)
	{
		p.scale(1,-1);
		p.drawImage(0, -image->height(), *image);
	}
	else
	if(horizontal)
	{
		p.scale(-1,1);	
		p.drawImage(-image->width(), 0, *image);
	}
	p.end();
	*image=pIO->convertToImage(pix);
	repaint();	
	placeImage(true);
	QApplication::restoreOverrideCursor ();	// restore original cursor
}


void
ImageViewer::rotateLeft()
{
	QApplication::setOverrideCursor (waitCursor);	// this might take time

	QPixmap pix(image->height(), image->width());
	QPainter p;
		p.begin(&pix);
		p.rotate(-90);
		p.drawImage(-image->width(), 0, *image);
	p.end();

	*image=pIO->convertToImage(pix);
	
	if(virtualPictureWitdth()<width())
		centerXImage();
	if(virtualPictureHeight()<height())
		centerYImage();
	placeImage();
	
	repaint();
	QApplication::restoreOverrideCursor ();	// restore original cursor
}


void
ImageViewer::rotateRight()
{
	QApplication::setOverrideCursor (waitCursor);	// this might take time

	QPixmap pix(image->height(), image->width());
	QPainter p;
	p.begin(&pix);
		p.rotate(90);
		p.drawImage( 0, -image->height(),*image);
	p.end();
	*image=pIO->convertToImage(pix);
	
	if(virtualPictureWitdth()<width())
		centerXImage();
	if(virtualPictureHeight()<height())
		centerYImage();
	placeImage();
	repaint();	

	QApplication::restoreOverrideCursor ();	// restore original cursor
}


void
ImageViewer::zoomIn(float rate)
{
	if(scale<ZOOM_MAX)
	{	
		QApplication::setOverrideCursor (waitCursor);	// this might take time
		
		QPoint center(width()/2, height()/2);
		center/=scale;
		center+=QPoint(getPosX(), getPosY());		
		if(scale*rate>ZOOM_MAX)
			scale=ZOOM_MAX;
		else
			scale*=rate;
		centerImage((int)(center.x()*scale), (int)(center.y()*scale), true);
		
		QApplication::restoreOverrideCursor ();	// restore original cursor
		mw->setZoom((int)(scale*100));	
	}
}


void
ImageViewer::zoomOut(float rate)
{
	if(scale>1.0/ZOOM_MAX)
	{	
		QApplication::setOverrideCursor (waitCursor);	// this might take time
		QPoint center(width()/2, height()/2);
		center/=scale;
		center+=QPoint(getPosX(), getPosY());		
		if(scale/rate<=1.0/ZOOM_MAX)
			scale=1.0/ZOOM_MAX;
		else
			scale/=rate;
		centerImage((int)(center.x()*scale), (int)(center.y()*scale), true);
		QApplication::restoreOverrideCursor ();	// restore original cursor
		mw->setZoom((int)(scale*100));
	}
}


void
ImageViewer::updateStatus ()
{
	if (image->size() == QSize(0,0))
	{
		if (filename!=QString("(none)"))
		{
			image = new QImage(BarIcon("file_broken",48).convertToImage());
			reconvertImage();
			
			mw->setNbrItems(total);
			mw->setZoom((int)(scale*100));			
			QString *fileName=new QString(filename);
			int pos = fileName->findRev ("/");
			mw->setImagename(fileName->right (fileName->length () - pos-1));
			pos = fileName->findRev(".");
			mw->setImagetype(fileName->right(fileName->length () - pos-1));
			mw->setDim(new QSize(0,0));
			QFileInfo fi(filename);
			mw->setSize(fi.size());
			mw->setDate(new QDate(fi.lastModified().date()));
		}
		else
		{
			mw->setZoom((int)(scale*100));
			mw->setImagename(NULL);
			mw->setImagetype(NULL);
			mw->setDim(NULL);
			mw->setSize(-1);
			mw->setDate(NULL);
		}
	}
	else
	{		
		if (filename!=QString("(none)"))
		{
			mw->setZoom((int)(scale*100));
			//
			QString *fileName=new QString(filename);
			int pos = fileName->findRev ("/");
			mw->setImagename(fileName->right (fileName->length () - pos-1));
			//
			pos = fileName->findRev(".");
			mw->setImagetype(fileName->right(fileName->length () - pos-1));
			//
			mw->setDim(new QSize(image->size()));
			//
			QFileInfo fi(filename);
			mw->setSize(fi.size());
			//
			mw->setDate(new QDate(fi.lastModified().date()));
		}
		else
		{
			mw->setZoom((int)(scale*100));
			mw->setImagename("(none)");
			mw->setImagetype("");
			mw->setDim(new QSize(0,0));
			mw->setSize(0);		
			mw->setDate(NULL);
		}
	}	
}



int ImageViewer::virtualScreenWitdth(){return (int)floor(width()/scale);}
int ImageViewer::virtualScreenHeight(){return (int)floor(height()/scale);}

int ImageViewer::virtualPictureWitdth(){ return (int)floor(image->width()*scale);}
int ImageViewer::virtualPictureHeight(){return (int)floor(image->height()*scale);}

void ImageViewer::setVirtualPosX(double posX){topPosX=posX; realPosX=-(int)floor(posX/scale);}
void ImageViewer::setVirtualPosY(double posY){topPosY=posY; realPosY=-(int)floor(posY/scale);}
int ImageViewer::getVirtualPosX(){return (int)topPosX;/*return -(int)floor(getPosX()*scale);*/};
int ImageViewer::getVirtualPosY(){return (int)topPosY;/*return -(int)floor(getPosY()*scale);*/};
QPoint ImageViewer::getVirtualPos(){return QPoint(getVirtualPosX(), getVirtualPosY());}
void ImageViewer::setVirtualPos(QPoint pos){setVirtualPosX(pos.x());setVirtualPosY(pos.y());}

void ImageViewer::setPosX(double posX){realPosX= (int)((posX/fabs(posX)) * floor(fabs(posX)));};
void ImageViewer::setPosY(double posY){realPosY= (int)((posY/fabs(posY)) * floor(fabs(posY)));};
void ImageViewer::setPos(QPoint pos){setPosX(pos.x()); setPosY(pos.y());};
int ImageViewer::getPosX(){return (int)floor(realPosX);};
int ImageViewer::getPosY(){return (int)floor(realPosY);};
QPoint ImageViewer::getPos(){return QPoint(getPosX(), getPosY());};

void
ImageViewer::placeImage(bool redraw)
{
	int
		oldVPX=getVirtualPosX(),
		oldPY=getVirtualPosY();
	//	
	if(!posXForTopXIsOK(getVirtualPosX()))
		setVirtualPosX(0);
	if(virtualPictureWitdth()<=width())
		centerXImage();
	//	
	if(!posYForTopYIsOK(getVirtualPosY()))
		setVirtualPosY(0);		
	if(virtualPictureHeight()<=height())
		centerYImage();
	//
	if(redraw &&
		((oldVPX!=getVirtualPosX()) ||
		 (oldPY!=getVirtualPosY())))
		repaint();
}


void
ImageViewer::centerImage(int posX, int posY, bool redraw)
{
	int
		oldVPX=getVirtualPosX(),
		oldVPY=getVirtualPosY();
	
	if (virtualPictureWitdth()>width())
	{
		int possibleX = width()/2 - posX;
		if(posXForTopXIsOK(possibleX))
		{
			setVirtualPosX(possibleX);
		}
		else
		{
			posX-=getVirtualPosX();
			if ( (posX>width()/2) &&
				(virtualPictureWitdth()+getVirtualPosX()-width()<abs(width()-posX)))
			{
				posX=width()-virtualPictureWitdth();
				setVirtualPosX(posX);
			}
			else
			{
				posX=0;
				setVirtualPosX(posX);
			}
		}
	}
	else
		centerXImage();
	//
	if(virtualPictureHeight()>height())
	{
		int possibleY = height()/2 - posY;
		
		if(posYForTopYIsOK(possibleY))
		{
			setVirtualPosY(possibleY);
		}
		else
		{
			posY-=getVirtualPosY();	
			if ((posY>height()/2) &&
				(virtualPictureHeight()+getVirtualPosY()-height()<abs(height()-posY)))
			{
				posY=height()-virtualPictureHeight();
				setVirtualPosY(posY);
			}
			else
			{
				posY=0;
				setVirtualPosY(posY);
			}
		}
	}
	else
		centerYImage();	
	
	if(redraw &&
		((oldVPX!=getVirtualPosX()) ||
		 (oldVPY!=getVirtualPosY())))
		repaint();
}

void
ImageViewer::centerImage(bool redraw)
{
	centerXImage();
	centerYImage();
	if(redraw)
		repaint();
}

void
ImageViewer::centerXImage(bool redraw)
{
	int oldVPX=getVirtualPosX();
	setVirtualPosX((width () - virtualPictureWitdth()) / 2);
	
	if(redraw &&
		oldVPX!=getVirtualPosX())
		repaint();
}

void
ImageViewer::centerYImage(bool redraw)
{
	int oldVPY=getVirtualPosX();
	setVirtualPosY((height() - virtualPictureHeight()) / 2);
	
	if(redraw &&
		oldVPY!=getVirtualPosY())
		repaint();
}


void
ImageViewer::originalSize()
{
	scale=1;
	placeImage(true);
	repaint();
	mw->setZoom((int)(scale*100));
}

void
ImageViewer::enlargeSize()
{
}

/**
	reduce size if larger than the viewer
*/
void
ImageViewer::reduceSize()
{
	if (!image)
		return;
	else
	if (image->isNull ())
		return;
	else
	{
		float s;
		if ((((double)height()) / image->height ()) <  (((double)width ()) / image->width ()))
			s = ((double)height()) / image->height ();
		else
			s = ((double)width ()) / image->width ();

		if ( (s < 1) )
			scale=s;
		else
			scale=1;	
	}
	placeImage();
	mw->setZoom((int)(scale*100));
}

void
ImageViewer::fitSize()
{
	if (!image)
		return;
	if (image->isNull ())
		return;

	float s;
	if ((((double)height()) / image->height ()) < (((double) width ()) / image->width ()))
		s = ((double)height()) / image->height ();
	else
		s = ((double) width ()) / image->width ();
	
	scale=s;
	placeImage();
	mw->setZoom((int)(scale*100));
}

void
ImageViewer::fitWidth()
{
	if (!image)
		return;
	if (image->isNull ())
		return;
	scale=((double)width()) / image->width ();
	placeImage(true);
	mw->setZoom((int)(scale*100));
}

void
ImageViewer::fitHeight()
{
	if (!image)
		return;
	if (image->isNull ())
		return;
	scale=((double)height()) / image->height();
	placeImage(true);
	mw->setZoom((int)(scale*100));
}

void
ImageViewer::doScale(bool repaint)
{
	float s;
	if ((((double) height ()) / image->height ()) < (((double) width ()) / image->width ()))
		s = ((double) height ()) / image->height ();
	else
		s = ((double) width ()) / image->width ();
		
	if(!lock)
	{
		if(s>1 && enlarge)
			scaleFit();
		else
		if(s<1 && shrink)
			scaleFit();
		else
			scale=1;
	}
	centerImage(repaint);
}


void
ImageViewer::scaleFit()
{
	fitSize();
}

bool
ImageViewer::posXForTopXIsOK(double posX)
{				
	int maxX=width();
	return !(
			(
				(posX+virtualPictureWitdth()>maxX) &&
				(posX>=0)
			)
		||
			(
				(posX<0) &&
				(posX+virtualPictureWitdth()<maxX)
			)
		);		
}

bool
ImageViewer::posYForTopYIsOK(double posY)
{
	int maxY=height();
	return !(
			(
				(posY+virtualPictureHeight()>maxY) &&
				(posY>=0)
			)
		||
			(
				(posY<0) &&
				(posY+virtualPictureHeight()<maxY)
			)
		);		
}



void
ImageViewer::resizeEvent (QResizeEvent *e)
{
	QWidget::resizeEvent(e);
	if (image->size () == QSize (0, 0))	// we couldn't load the image
		return;
	
	float s;
	if ((((double)height()) / image->height ()) < (((double) width ()) / image->width ()))
		s = ((double)height()) / image->height ();
	else
		s = ((double)width ()) / image->width ();
	if(s>1 && enlarge)
		scaleFit();
	else
	if(s<1 && shrink)
		scaleFit();
	else
			scale=1;
	placeImage();
	repaint();
}


void
ImageViewer::setBackgroundColor(QColor col)
{
	bgBrush = QBrush(col);
	QWidget::setBackgroundColor(col);
	setBackgroundMode(NoBackground);
	repaint();
}

void
ImageViewer::paintEvent (QPaintEvent * e)
{
	if (image->size () !=QSize (0, 0))
	{
		int options=ColorOnly||ThresholdDither||ThresholdAlphaDither||AvoidDither;
		if(dragStartPosX + dragStartPosY != -2)
		{
			setVirtualPosX(difTopPosX+dragStartPosX);
			setVirtualPosY(difTopPosY+dragStartPosY);
		}
		
		//////////////////
		QPoint rtp(	(int)floor(e->rect().topLeft().x()/scale),
					(int)floor(e->rect().topLeft().y()/scale));
		QPoint rbp(	(int)ceil(e->rect().bottomRight().x()/scale),
					(int)ceil(e->rect().bottomRight().y()/scale));
		QRect redraw(rtp,rbp);
		redraw.moveBy(getPosX(), getPosY());
		
		int
			cx=max(0,redraw.topLeft().x()),
			cy=max(0,redraw.topLeft().y()),
			cw=min(image->width(),  redraw.width() +min(0,redraw.topLeft().x())+1),
			ch=min(image->height(), redraw.height()+min(0,redraw.topLeft().y())+1);
		if(image->hasAlphaBuffer()){cw++; ch++;}
		
		int
			x= e->rect().topLeft().x()-min(0, (int)(ceil(redraw.topLeft().x()*scale))),
			y= e->rect().topLeft().y()-min(0, (int)(ceil(redraw.topLeft().y()*scale)));
		
		////////////////
		QPainter painter;
		painter.begin(this);
		
		if(cw>0 && ch>0)
		{
			if( (!smooth()) ||
				(scale==1.0)  ||
				(dragStartPosX + dragStartPosY != -2) ||
				(ep!=NULL))
			{
					QTime debut = QTime::currentTime ();
			   	
				QImage copiedImage=image->copy(cx, cy, cw, ch, options);
				QPixmap scaleCopy((int)ceil(cw*scale), (int)ceil(ch*scale));
				QPainter pC(&scaleCopy);
					pC.scale(scale, scale);
					pC.drawImage(0, 0, copiedImage);
					pC.end();
				painter.drawPixmap(x, y, scaleCopy);
			}			
			else
			{
				QTime debut = QTime::currentTime ();
				painter.drawImage (x,y, image->copy(cx, cy, cw, ch, options).smoothScale((int)ceil(cw*scale), (int)ceil(ch*scale)));
			}
		}
		
		if(getVirtualPosX()>0)
			{
				painter.fillRect(0,0,
						getVirtualPosX(), height(),
						bgBrush);				
				painter.fillRect(getVirtualPosX()+virtualPictureWitdth(), 0,
						width()-(getVirtualPosX()+virtualPictureWitdth()), height(),
						bgBrush);
				painter.flush();
			}
			if(getVirtualPosY()>0)
			{
				painter.fillRect(0,0,
						width(), getVirtualPosY(),
						bgBrush);
				painter.fillRect(0,getVirtualPosY()+virtualPictureHeight(),
						width(), height()-(getVirtualPosY()+virtualPictureHeight()),
						bgBrush);
				painter.flush();
			}
		painter.flush();
		painter.end();
	}
	else
	{
		QPainter painter;
		painter.begin(this);
		painter.fillRect(0,0,width(), height(), bgBrush);
		painter.end();
	}
	
}

void ImageViewer::setShrink(bool shrink)
{
	this->shrink=shrink;
	doScale();
}

/** Read property of bool enlarge. */
const bool&
ImageViewer::getEnlarge()
{
	return enlarge;
}

/** Write property of bool enlarge. */
void
ImageViewer::setEnlarge( bool enlarge)
{
	this->enlarge = enlarge;
	doScale();
}

void
ImageViewer::setZoomLock(bool lock)
{
	this->lock=lock;
}
void
ImageViewer::applyFilter(int filter)
{
}


void
ImageViewer::scrolldxR()
{
	if (virtualPictureWitdth()<width())
		return;
	dragStartPosX=-floor(10*scale);
	dragStartPosY=0;
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posXForTopXIsOK(difTopPosX+dragStartPosX))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}

void
ImageViewer::scrolldyB()
{
	if (virtualPictureHeight()<height())
		return;
	dragStartPosX=0;
	dragStartPosY=-floor(10*scale);
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posYForTopYIsOK(difTopPosY+dragStartPosY))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}
 void
ImageViewer::scrolldxL()
{
	if (virtualPictureWitdth()<width())
		return;
	dragStartPosX=floor(10*scale);
	dragStartPosY=0;	
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posXForTopXIsOK(difTopPosX+dragStartPosX))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}

void
ImageViewer::scrolldyT()
{
	if (virtualPictureHeight()<height())
		return;
	dragStartPosX=0;
	dragStartPosY=floor(10*scale);
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posYForTopYIsOK(difTopPosY+dragStartPosY))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}
