/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef SHOWIMG_H
#define SHOWIMG_H

#include <qcanvas.h>
#include <qimage.h>
#include <qpainter.h>
#include <qapplication.h>
#include <qmovie.h>

#include <kaction.h>
#include <kpixmapio.h>

#include "imagelisteview.h"
#include "mainwindow.h"

class QLabel;
class QMenuBar;
class QPopupMenu;

class ImageViewer:public QWidget
{
Q_OBJECT

public:
  
  /**
  	create the imageViewer
	@param parent the parent widget
	@param ImageListeView the list of the images
	@param MainWindow the mainwindow
	@param name
	@param wFlags
  */
  ImageViewer (QWidget * parent = NULL,
  		ImageListeView * il = NULL,
  		MainWindow *mw = NULL,
  		const char *name = NULL, int wFlags = 0);
  
  /**
  	init the menu given the actionCollection
  */
  void initMenu(KActionCollection *actionCollection);

  /**
  	set the grayscale level
	@param togray between 0 and 100
  */
  void setToGrayscale(int togray);
  int toGrayscale();

  /**
  	load and display an image given the full filaname
	@param fileName the full filaname of the picture to display
  */
  bool loadImage (const char *fileName);
  /**
  	preload in memory an image given the full filaname
	@param fileName the full filaname of the picture to load
  */
  bool preloadImage (const char *fileName);
  
  /**
  	upadate the status of the mainwindow
  */
  void updateStatus ();
  
  /**
  	@return true if an image is loaded
  */
  bool hasImage();

  /**
  	flip or miror the diplayed image
  */
  void mirror (bool horizontal, bool vertical);

  /**
  	rotate to the left
  */  
  void rotateLeft();
  /**
  	rotate to the right
  */  
  void rotateRight();

  /**
  	display the image with its original size (i.e zoom=1)
  */
  void originalSize();
  
  /**
  	fit or unfit the image ti the screen
	@param fit if true, fit the image
	@keep if true, store this param for te next images
  */
  void setFit(bool fit, bool keep=false);

  /**
  	set smooth for the next image, required for memory and time
  */
  void setSmooth(bool s);
  
  /**
  	zoom into the image given the rate
  */
  void zoomIn(float rate);
  
  /**
  	zoom out the image given the rate
  */
  void zoomOut(float rate);
  
  /**
  	if true, store the zoom rate
  */
  void setZoomLock(bool lock);
  
  /**
  	scroll the image of dx and dy pixels
  */
  void scroll ( int dx, int dy ) ;
  
  void applyFilter(int filter);
  
  /**
  	@return a pixmap of the displayed image
  */
  QPixmap getPixmap();
  
  /**
  	@return the fullname of the displayed image
  */
  QString getFilename();

  /**
  	@param shrink if true, shrink the image if it's larger than the screen
  */
  void setShrink(bool shrink);
  
  /**
  	@param enlarge if true, enlarge the image if it's smaller than the screen
  */
  void setEnlarge( bool enlarge);
  const bool& getEnlarge();

  /**
  	set the background color
  */
  void setBackgroundColor(QColor col);

  QLabel *status;
  int total;
  int nbImg;

public slots:
	void wallpaperC();
	void wallpaperM();
	void wallpaperCM();
	void wallpaperCMa();
	void wallpaperA();
	void wallpaperL();

	void wallpaper (int mode=1);

	void fitHeight();
	void fitWidth();

	void scrolldxR();
	void scrolldyB();
	void scrolldxL();
	void scrolldyT();
	
protected slots:
  void focusInEvent ( QFocusEvent *);
  void focusOutEvent ( QFocusEvent *);

  void paintEvent (QPaintEvent *);
  void resizeEvent (QResizeEvent *);

  void keyPressEvent (QKeyEvent * e);
  void wheelEvent (QWheelEvent * e);

  void mousePressEvent (QMouseEvent * e);
  void mouseReleaseEvent (QMouseEvent * e);
  void mouseMoveEvent (QMouseEvent * e);

  void movieUpdated(const QRect& area);
  void movieResized(const QSize& size);
  void movieStatus(int status);
  void startMovie();

private:
  int virtualScreenWitdth();
  int virtualScreenHeight();
  
  int virtualPictureWitdth();
  int virtualPictureHeight();
  
  void setVirtualPosX(double posX);
  void setVirtualPosY(double posY);
  int getVirtualPosX();
  int getVirtualPosY();
  QPoint getVirtualPos();
  void setVirtualPos(QPoint pos);

  void setPosX(double posX);
  void setPosY(double posY);
  void setPos(QPoint pos);
  int getPosX();
  int getPosY();
  QPoint getPos();

  bool posXForTopXIsOK(double posX);
  bool posYForTopYIsOK(double posY);
  
  void placeImage(bool redraw=false);
  void centerImage(int posX, int posY, bool redraw=false);
  void centerImage(bool redraw=false);
  void centerXImage(bool redraw=false);
  void centerYImage(bool redraw=false);
  
  void doScale (bool repaint=true);
  void scaleFit();
  void fitSize();
  void scalePreload ();
  void reduceSize();
  void enlargeSize();
  bool smooth () const;
  bool reconvertImage ();

  const char *filename;
  bool ss; //true if smooth image
  QImage *image;// the loaded image
  QString imageName;
  KPixmapIO *pIO;

  QCString preimageName;
  QImage *preloadedImage;	// the preloaded image

  QPopupMenu *file;
  KActionCollection *actionCollection;
  QPopupMenu *popup;
  int grayscale;
  ImageListeView *il;
  MainWindow *mw;

  QPoint *sp, *ep, *lp; //start, end, and last Points
  QBrush bgBrush;

  float scale;

  double realPosX, realPosY;
  double dragStartPosX, dragStartPosY;
  double topPosX, topPosY;
  double difTopPosX, difTopPosY;

  ButtonState button;
  QMovie* movie;
  void initMovie();

  bool fit;
  bool hasimage;


protected:
  bool shrink;
  bool enlarge;
  bool lock;
};


#endif // SHOWIMG_H
