%  VWhere: an SLgtk guilet providing a visual S-Lang 'where' function {{{
%
%  Copyright (C) 2003-2006 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
% 
%  SLgtk was partially developed at the MIT Center for Space Research,
%  under contract SV1-61010 from the Smithsonian Institution.
% 
%  Permission to use, copy, modify, distribute, and sell this software
%  and its documentation for any purpose is hereby granted without fee,
%  provided that the above copyright notice appear in all copies and
%  that both that copyright notice and this permission notice appear in
%  the supporting documentation, and that the name of the Massachusetts
%  Institute of Technology not be used in advertising or publicity
%  pertaining to distribution of the software without specific, written
%  prior permission.  The Massachusetts Institute of Technology makes
%  no representations about the suitability of this software for any
%  purpose.  It is provided "as is" without express or implied warranty.
%  
%  THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
%  WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
%  MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
%  INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
%  CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
%  OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
%  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
%  WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
% }}}

% Namespace setup {{{
eval("variable vw_private_ns = \"vw\"", "vw");
private variable vw_eval_ns = current_namespace();

#ifeval _slang_version >= 20000
if (vw_eval_ns == "") {
   vw_eval_ns =  _get_frame_info( _get_frame_depth()-1).namespace;
   if (vw_eval_ns == NULL)
	vw_eval_ns = "";
}
#endif
% }}}

require("gtkplot");
() = evalfile("vwutils", vw->vw_private_ns);
() = evalfile("vwdraw", vw->vw_private_ns);

private variable VWhereDescriptor = struct { % {{{
	fprefs,			% Filtering preferences
	selector,		% axis expression window
	expressors,		% axis expression, one per axis 2b plotted
	prompter,		% S-Lang prompt
	point_drawer,
	menus,			% dropdown menus, for selecting data to plot
	plotwin,		% toplevel window in which plots will be drawn
	oplotter,		% overplot button
	vectors,		% data vectors, one per axis per plot
	veclen,			% length of vectors (all must be same length)
   	names,			% and their "names"
	filtd,			% current plot filter descriptor
	filtds,			% array of plot filter descriptors, one per plot
	tbox,			% graphical toolbox
	panes,			% tabbed window panes, one for each plot
	zoom_area,		% display curent zoom factor
	coords_area,		% display mousepos, in data-relative coords
	ttips,			% tooltips (popup help windows)
	active_child,
	print_ctx,		% context for print jobs 
	status,			% statusbar
	result			% the end product of vwhere()
};
% }}}

% Misc support funcs {{{ 
private define destroy(vwd,terminate)
{
   variable w;
   if (vwd.plotwin != NULL)  {			% Destroy the plots window
	w = vwd.plotwin;
	vwd.plotwin = NULL;
	vwd.filtd = NULL;
	gtk_widget_set_sensitive(vwd.oplotter, FALSE);
	gtk_widget_destroy(w);
   }

   if (terminate) {				% And optionally the axis
	if (vwd.selector != NULL) {		% expression window
	   w = vwd.selector;
	   vwd.selector = NULL;
	   gtk_widget_destroy(w);
	}
	if (gtk_main_level())
	   gtk_main_quit ();			% reduce loop nesting by 1
   }
}

private define delete_plot(vwd)
{
   variable which = gtk_notebook_get_current_page(vwd.panes);
   gtk_notebook_remove_page(vwd.panes,which);
   vwd.filtds = vwd.filtds [ where ( [0:length(vwd.filtds)-1] != which) ];
   if (length(vwd.filtds) == 0)
	destroy(vwd, FALSE);
}

private define switch_plot(notebook,notebook_page_dummy,which,vwd)
{
   if (length(vwd.filtds) ) {
	vwd.filtd = vwd.filtds[which];
	g_signal_emit_by_name(vwd.tbox.buttons[vwd.tbox.curr],"clicked");
	vw->display_zoom(vwd);
   }
}

private define determine_result(vwd)
{
   vwd.result = vw->apply_filters(vwd);
   if (vwd.result == NULL)	% applying zero regions returns entire dataset
	vwd.result = [0 : vwd.veclen - 1];
   else
	vwd.result = where(vwd.result);

   destroy(vwd,TRUE);
}

private define validate_vector(vwd,name,vec)
{
   if (Array_Type == typeof(vec) and _is_numeric(vec)) {
	!if (vwd.veclen)
	   vwd.veclen = length(vec);
	else if (length(vec) != vwd.veclen) {
	   () = printf("Ignored Field: %s, mismatched length\n",name);
	   return;
	}
   }
   else {
	() = printf("Ignored Field: %s, not a numerical vector\n",name);
	return;
   }


   % Push vector onto the stack, and define an alias for it which may
   % be directly referenced in S-Lang arithmetic expressions (possibly
   % prefixed with underscores to avoid overwriting existing symbols)

   variable test_ns = vw_eval_ns;
   if (test_ns != "") test_ns = strcat(test_ns,"->");
   while ( orelse {is_defined(name)} {is_defined(strcat(test_ns, name))} )
	name = strcat("_",name);

   vwd.names = [ vwd.names, name ];
   vwd.vectors [ name ] = typecast(vec, Double_Type);
   vwd.vectors[name];
   () = vw->evaluate( vwd, sprintf("%s = ();", name), vw_eval_ns);
}

private define vw_new()
{
   variable name, value, vwd = @VWhereDescriptor;

   vwd.status = gtk_statusbar_new();
   vwd.vectors = Assoc_Type[Any_Type, NULL];
   vwd.names = String_Type[0];
   vwd.result = Integer_Type[0];
   vwd.filtds = Struct_Type[0];
   vwd.point_drawer = &vw->draw_points;

   if (_NARGS == 1) {

	variable d =  ();
	if (typeof(d) == Struct_Type) {

	   foreach (get_struct_field_names(d)) {

		name = ();
		if (name[0] == '_')	% ignore underscore-prefixed fields
		   continue;

		value = get_struct_field(d,name);
		validate_vector(vwd,name,value);
	   }
	}
   }
   else {

	variable args = __pop_args(_NARGS), i = 0;
	while (i < _NARGS) {
	   validate_vector(vwd, sprintf("array%d",i+1),args[i].value);
	   i++;
	}
   }

   if (orelse {vwd.names == NULL} {length(vwd.names) < 2}) 
	usage("ArrayType = vwhere(struct | 1Darray, 1Darray [,1Darray, ...])\n"+
	      		"Input must contain at least 2 numeric vectors,"+
			" all of equal length");

   return vwd;
}
% }}}

% Plot window {{{
private define plotwin_new(vwd)
{
   variable window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(window, vw->window_title);
   gtk_container_set_border_width(window, 10);
   () = g_signal_connect_swapped (window,"destroy",&destroy, vwd,
						(vwd.selector == NULL));
   variable main_vbox = gtk_vbox_new(FALSE,10);
   gtk_container_add(window, main_vbox);

   variable vbox2 = gtk_vbox_new(FALSE,10);
   gtk_box_pack_start(main_vbox,vbox2,TRUE,TRUE,0);

   vwd.tbox = vw->make_toolbox(&vw->tb_choose,vwd);
   gtk_container_add(vbox2, vwd.tbox.widget);

   vwd.panes = gtk_notebook_new();
   gtk_notebook_set_tab_pos(vwd.panes,GTK_POS_TOP);
   gtk_notebook_set_scrollable(vwd.panes, TRUE);

   gtk_box_pack_start(vbox2,vwd.panes,TRUE,TRUE,0);
   () = g_signal_connect(vwd.panes,"switch_page", &switch_plot, vwd);

   variable hbox = gtk_hbox_new(FALSE,5);
   gtk_box_pack_start(main_vbox,hbox,FALSE,FALSE,0);

   variable button = gtk_button_new_with_label("Done");
   () = g_signal_connect_swapped(button,"clicked", &determine_result, vwd);
   gtk_box_pack_start(hbox,button,FALSE,FALSE,0);
   gtk_widget_grab_focus(button);
   gtk_tooltips_set_tip(vwd.ttips,button,
	 "Return where indices by filtering plots with applied regions","");

   vwd.zoom_area = gtk_label_new("Zoom: 1.0");
   gtk_box_pack_start(hbox,vwd.zoom_area,FALSE,FALSE,25);
   vwd.coords_area = gtk_label_new("");
   gtk_box_pack_start(hbox,vwd.coords_area,TRUE,TRUE,0);

   button = gtk_button_new_with_label("Delete");
   () = g_signal_connect_swapped(button,"clicked",&delete_plot, vwd);
   gtk_box_pack_start(hbox,button,FALSE,FALSE,0);
   gtk_tooltips_set_tip(vwd.ttips,button,"Remove current plot","");

   button = gtk_button_new_with_label("Cancel");
   () = g_signal_connect_swapped(button,"clicked",&destroy, vwd,
				 	(vwd.selector == NULL));
   gtk_box_pack_start(hbox,button,FALSE,FALSE,0);
   gtk_tooltips_set_tip(vwd.ttips,button,"Return empty where indices","");

   gtk_widget_show_all(window);
   return window;
}
% }}}

% Axis Expression Window {{{

private define make_expr_menu_entry(text, menu, entry) % {{{
{
   variable item = gtk_menu_item_new_with_label(text);
   gtk_widget_show(item);
   gtk_menu_shell_append( menu, item);
   () = g_signal_connect_swapped(item,"activate", &gtk_entry_set_text,
								entry, text);
} % }}}

private define get_axis_expr(vwd, which) % {{{
{
   variable expr = vwd.expressors[which];
   return strcompress(gtk_entry_get_text(expr.expression)," \t");
} % }}}

private define get_axis_vector(vwd, axis_expr) % {{{
{
   if (axis_expr == "") {
	vw->echo_status(vwd,"blank axis expression");
	return NULL;
   }

   variable vector = vwd.vectors[axis_expr];
   if (vector != NULL)
	return vector;

   variable nargs = _stkdepth();
   () = vw->evaluate(vwd, axis_expr, vw_eval_ns);

   if ( _stkdepth() > nargs)
	vector = ();

   ERROR_BLOCK { _clear_error(); return NULL; }

   return typecast(vector, Double_Type);
} % }}}

private define get_axis_vector_and_expr(vwd, which) % {{{
{
   variable expr = get_axis_expr(vwd, which);
   variable vector = get_axis_vector(vwd, expr);

   if (orelse {typeof(vector) != Array_Type} {length(vector) != vwd.veclen}) {
	vw->echo_status(vwd,
		sprintf("%s : does not represent %d-element numeric vector",
						expr,vwd.veclen));
	return NULL, NULL;
   }

   % Store each new/unique expression in menus for ease-of-access in new plots
   if (vwd.vectors[expr] == NULL) {
	foreach(vwd.expressors) {
	   variable e = ();
	   make_expr_menu_entry(expr, e.menu, e.expression);
	}
	vwd.vectors[expr] = vector;
   }

   return vector, expr;

} % }}}

private define plot_grab_focus(c,a,nb,i) {gtk_notebook_set_current_page(nb,i);}
private variable colors = _get_predefined_colors()[[2:]];  % omit white/black
private variable color_index = -1;

private define plot_new(vwd, clear_status_line) % {{{
{
   variable x, xexpr, y, yexpr;
   (x, xexpr) = get_axis_vector_and_expr(vwd, 0);
   if (x == NULL) return;
   (y, yexpr) = get_axis_vector_and_expr(vwd, 1);
   if (y == NULL) return;

   if (clear_status_line) vw->echo_status(vwd, "");

   !if ( (@vwd.point_drawer)(vwd, x, y))
      return;

   if (vwd.plotwin == NULL) vwd.plotwin = plotwin_new(vwd);

   vwd.filtds = [ vwd.filtds, vwd.filtd];

   variable plotd = vwd.filtd.plotd;
   variable label = gtk_label_new(NULL);
   gtk_label_set_markup(label,sprintf("<b>%S</b> | <b>%S</b>",xexpr,yexpr));

   _gtk_plot_set_labels(plotd, xexpr, yexpr);

   gtk_notebook_append_page(vwd.panes,plotd.canvas,label);
   if (_gtk_version >= 20400) pop;

   gtk_widget_show_all(vwd.plotwin);
   () = g_signal_connect_after(plotd.canvas, "size_allocate",
			&plot_grab_focus, vwd.panes, length(vwd.filtds) - 1);

   gtk_widget_set_sensitive(vwd.oplotter, TRUE);
} % }}}

private define oplot(vwd) % {{{
{
   variable x = get_axis_vector(vwd, get_axis_expr(vwd, 0) );
   if (x == NULL) return;
   variable y = get_axis_vector(vwd, get_axis_expr(vwd, 1) );
   if (y == NULL) return;

   variable fd = vwd.filtd, pd = fd.plotd, fgcolor = fd.prefs.fgcolor.value;
   do {
 	color_index = (color_index + 1) mod length(colors);
	variable oplot_color = colors[color_index];
   } while (gdk_color_equal(fgcolor, oplot_color));

   variable previous_dataset = pd.dset;
   _gtk_oplot(pd, x, y, oplot_color);
   gtk_widget_show_all(pd.canvas);
   _gtk_plot_redraw(pd);
   pd.dset = previous_dataset;
} % }}}

private define create_expressor(parent, axis, vwd, focus_next) % {{{
{
   variable hbox = gtk_hbox_new(FALSE,5);
   variable axis_labels = ["X", "Y"];

   gtk_box_pack_start(hbox, gtk_label_new(axis_labels[axis]+" "),FALSE,FALSE,0);
   variable expression = gtk_entry_new();
   gtk_entry_set_width_chars(expression, 20);
   gtk_entry_set_max_length(expression, 255);
   gtk_entry_set_text(expression,vwd.names[axis]);	% default expression

   gtk_box_pack_start(hbox, expression, TRUE, TRUE, 0);
   gtk_tooltips_set_tip(vwd.ttips, expression,
	sprintf(
	"Choose a vector from the menu, or enter any valid S-Lang\n"+
	"expression which will create a vector of length %d.",vwd.veclen),"");

   variable menubar = gtk_menu_bar_new();
   variable menu = gtk_menu_item_new_with_label( "Choose" );

   variable submenu = gtk_menu_new();
   array_map(Void_Type, &make_expr_menu_entry, vwd.names, submenu,
								expression);

   () = g_signal_connect_swapped(expression, "activate",
				 	&gtk_widget_grab_focus, focus_next);
	           
   gtk_menu_item_set_submenu (menu, submenu);
   gtk_menu_shell_append (menubar, menu);

   variable button = gtk_button_new_with_label("Clear");
   () = g_signal_connect_swapped(button,"clicked",
				&gtk_entry_set_text,expression,"");

   gtk_box_pack_end(hbox,button,FALSE,FALSE,0);
   gtk_box_pack_end(hbox,menubar,FALSE,FALSE,0);
   gtk_box_pack_end(parent,hbox,FALSE,FALSE,0);

   variable expressor = struct {expression, menu};
   expressor.expression = expression;
   expressor.menu = submenu;
   return expressor;
} % }}}

private define echo_ns(vwd) % {{{
{
   variable m;
   if (vw_eval_ns == "")
	m = "The current namespace is anonymous";
   else
	m = sprintf("S-Lang Namespace: %S", vw_eval_ns);

   vw->echo_status(vwd, m);
} % }}}

private define prompter_ns(entry, event, vwd) % {{{
{
  if (event.button == 1) return FALSE;

  if (event.button == 3) {

	variable new_ns = _input_dialog("Namespace ","Change S-Lang Namespace",
	"Enter name of the namespace in which S-Lang statements will be\n"+
	"evaluated.  If the namespace does not exist it will be created.\n"+
	"Leave the field blank if you decide to not change the namespace.");

	new_ns = strtrim(new_ns);

	if (new_ns !=  "" and new_ns != vw_eval_ns) {

	   % Promote existing, non-Global vector expressions to new namespace
	   if (vw_eval_ns != "Global")  {
		foreach(vwd.vectors) using ("keys") {
		   variable name = ();
		   vwd.vectors[name];
		   eval( sprintf("variable %s = ();", name), new_ns);
		}
	   }
	   vw_eval_ns = new_ns;
	}
  }
  echo_ns(vwd);
  return TRUE;
} % }}}

private define prompter_eval(prompt, vwd) % {{{
{
   variable expr = strcompress(gtk_entry_get_text(prompt)," \t");
   if (expr == "") return;

   variable prev_stack_depth = _stkdepth();
   variable success = vw->evaluate(vwd, expr, vw_eval_ns);

   variable stack_leftovers = _stkdepth() - prev_stack_depth;
   if (stack_leftovers) {
	variable top = string(());
	if (success) 			% don't erase failure msg in statusbar
	   vw->echo_status(vwd, top);
	stack_leftovers--;
	_pop_n(stack_leftovers);
   }
   else if (success) {
	gtk_entry_set_text(prompt, "");
	vw->echo_status(vwd, sprintf("SUCCESS: %s",expr));
   }
} % }}}

private define prompter_toggle(toggle_button, vwd) % {{{
{
   variable prompter = vwd.prompter;
   if (gtk_toggle_button_get_active(toggle_button)) {
	if (prompter.activate_button == NULL) {
	   gtk_box_pack_start(prompter.parent,prompter.hbox, FALSE, FALSE, 0);
	   prompter.activate_button = toggle_button;
	}
	echo_ns(vwd);
	gtk_widget_show_all(prompter.parent);
   }
   else  {
	gtk_window_resize(vwd.selector, 1, 1);
	gtk_widget_hide(prompter.hbox);
   }
} % }}}

private define prompter_create(parent, vwd) % {{{
{
   variable p = struct { parent, hbox, prompt, activate_button };


   p.parent = parent;
   p.hbox = gtk_hbox_new(FALSE,5);
   gtk_box_pack_start(p.hbox, gtk_label_new("S-Lang>"),FALSE,FALSE,0);
   p.prompt = gtk_entry_new();
   gtk_entry_set_width_chars(p.prompt, 20);
   gtk_entry_set_max_length(p.prompt, 255);
   gtk_tooltips_set_tip(vwd.ttips, p.prompt,
	 "Type arbitrary S-Lang expression, then hit ENTER to evaluate.\n"+
	 "Mouse2 to see current namespace, Mouse3 to change it","");

   () = g_signal_connect(p.prompt, "activate", &prompter_eval, vwd);
   () = g_signal_connect(p.prompt, "button_press_event", &prompter_ns,vwd);
   gtk_box_pack_start(p.hbox, p.prompt,TRUE,TRUE, 0);

   variable clear_button = gtk_button_new_with_label("Clear");
   gtk_box_pack_end(p.hbox, clear_button, FALSE, FALSE, 0);
   () = g_signal_connect_swapped(clear_button,"clicked", &gtk_entry_set_text,
				 				p.prompt,"");
   return p;
} % }}}

% }}}

define vwhere() % {{{
{
   variable label, args = __pop_args(_NARGS);
   variable vwd = vw_new(__push_args(args));

   vwd.selector = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_container_set_border_width ( vwd.selector, 5);
   gtk_window_set_title(vwd.selector, vw->window_title);

   () = g_signal_connect_swapped (vwd.selector,"destroy",&destroy, vwd, TRUE);

   variable main_vbox = gtk_vbox_new (FALSE, 5);
   gtk_container_add (vwd.selector, main_vbox);

   if (log10(vwd.veclen) >= 6)
	label = sprintf("(vector length: %E )",vwd.veclen);
   else
	label = sprintf("(vector length: %d )",vwd.veclen);

   variable frame = gtk_frame_new(strcat("Plot Axes Expressions ",label));
   gtk_box_pack_start(main_vbox,frame,FALSE,FALSE,5);
   vwd.ttips = gtk_tooltips_new();
   gtk_label_set_selectable( gtk_frame_get_label_widget(frame), TRUE);

   variable plot_button = gtk_button_new_with_label("Plot");
   () = g_signal_connect_swapped ( plot_button, "clicked", &plot_new,vwd, 1);
   gtk_tooltips_set_tip(vwd.ttips,plot_button,
	"Plot X/Y into new pane, applying filters","");

   vwd.oplotter = gtk_button_new_with_label("OPlot");
   () = g_signal_connect_swapped ( vwd.oplotter, "clicked", &oplot, vwd);
   gtk_tooltips_set_tip(vwd.ttips,vwd.oplotter,
	"Overplot X/Y into current pane, without applying filters","");

   variable axis_vbox = gtk_vbox_new(FALSE,0);
   gtk_container_add(frame, axis_vbox);
   gtk_container_set_border_width(axis_vbox, 5);
   vwd.expressors = Struct_Type[2];
   vwd.expressors[1] = create_expressor(axis_vbox, 1, vwd, plot_button);
   vwd.expressors[0] = create_expressor(axis_vbox, 0, vwd,
						vwd.expressors[1].expression);

   vwd.prompter = prompter_create(main_vbox, vwd);

   gtk_box_pack_end(main_vbox,vwd.status,FALSE,FALSE,0);
   vwd.fprefs = vw->filter_prefs_new(vwd);

   variable hbox = gtk_hbox_new(FALSE,0);
   gtk_box_pack_end(main_vbox,hbox,FALSE,FALSE,0);

   gtk_box_pack_start(hbox,plot_button,FALSE,FALSE,0);

   variable align = gtk_alignment_new(0.6,0.5,0,0);
   gtk_widget_set_sensitive(vwd.oplotter, FALSE);
   gtk_widget_unset_flags (vwd.oplotter, GTK_CAN_FOCUS);
   gtk_container_add(align, vwd.oplotter);
   gtk_box_pack_start(hbox, align, TRUE, TRUE, 0);

   align = gtk_alignment_new(0.6,0.5,0,0);
   variable button = gtk_toggle_button_new_with_label("Prompt");
   gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
   gtk_tooltips_set_tip(vwd.ttips,button,
	 	"Show/hide interactive S-Lang prompt", "");
   () = g_signal_connect(button, "toggled", &prompter_toggle, vwd);
   gtk_container_add(align, button);
   gtk_box_pack_start(hbox, align, TRUE, TRUE, 0);

   align = gtk_alignment_new(0.4,0.5,0,0);
   button = gtk_button_new_with_label(" Help ");
   gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
   gtk_container_add(align,button);
   gtk_box_pack_start(hbox,align,TRUE,TRUE,0);
   () = g_signal_connect( button, "clicked", &vw->help);

   button = gtk_button_new_with_label("Quit");
   gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
   gtk_box_pack_end(hbox,button,FALSE,FALSE,0);
   () = g_signal_connect_swapped( button,"clicked", &destroy, vwd, TRUE);

   gtk_widget_grab_focus(plot_button);

   if (length(vwd.names) == 2)		% automatically create plot win, too,
	plot_new(vwd, 0);		% when only 2 vectors have been given

   gtk_widget_show_all (vwd.selector);
   gtk_main();

   return vwd.result;
} % }}}

#ifexists provide
provide("vwhere");
#endif
