/*  slrnface - feeble attempt at delivering X-Faces to slrn and mutt users
 *  Copyright (C) 2000, 2001, 2002  Drazen Kacar
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#ifdef __linux
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termio.h>
#else
#include <termios.h>
#endif

#define countof(x) (sizeof(x)/sizeof(x[0]))

extern void uncompface (char *);

extern int errno;

/*
 * The values in optvalues[] array are just fallbacks if appropriate
 * X resources are not found. You can set them here if you like, but
 * you should really use X resources for that. Command line parameters
 * should also work, although I don't think there's a need for them,
 * except for testing.
 */

typedef enum {
   val_int,
   val_color
} OptionType;

struct optval {
   OptionType type;
   union {
      unsigned long intval;
      char *string;
   } value;
};

struct optval optvalues[] = {
   { val_int, 0 },
   { val_int, 1 },
   { val_int, 0 },
   { val_int, 2 },
   { val_int, 0 },
   { val_color, 0 },
   { val_color, 0 },
   { val_color, 0 }
};

#define X_OFFSET_CHAR	(optvalues[0].value.intval)
#define Y_OFFSET_CHAR	(optvalues[1].value.intval)
#define X_OFFSET_PIX	(optvalues[2].value.intval)
#define Y_OFFSET_PIX	(optvalues[3].value.intval)
#define XFACE_PAD	(optvalues[4].value.intval)

/* Very unconsistent */

#define INK		(optvalues[5])
#define PAPER		(optvalues[6])
#define PADCOLOR	(optvalues[7])

XrmOptionDescRec my_options[] = {
   { "-xOffsetChar", "*slrnface.xOffsetChar", XrmoptionSepArg, NULL },
   { "-yOffsetChar", "*slrnface.yOffsetChar", XrmoptionSepArg, NULL },
   { "-xOffsetPix",  "*slrnface.xOffsetPix",  XrmoptionSepArg, NULL },
   { "-yOffsetPix",  "*slrnface.yOffsetPix",  XrmoptionSepArg, NULL },
   { "-XFacePad",    "*slrnface.XFacePad",    XrmoptionSepArg, NULL },
   { "-ink",	     "*slrnface.ink",	      XrmoptionSepArg, NULL },
   { "-paper",	     "*slrnface.paper",	      XrmoptionSepArg, NULL },
   { "-padColor",    "*slrnface.padColor",    XrmoptionSepArg, NULL }
};

/*
 * MAXRESOURCESIZE should be the size of the longest resource string in the
 * above array. It can be a bit larger, of course. If you're adding
 * resources, make sure it's not smaller. Wouldn't it be nice if preprocessor
 * was able to calculate that?
 */

#define MAXRESOURCESIZE 30

int is_mapped = 0;

Display *d = NULL;
Window win = 0;
int x_pos, y_pos;
GC gc;
Pixmap bitmap = 0;
int fifo_fd;

int buffered_face_len = -1, pixmap_stale = 1;
char last_face[8000];

struct pollfd fifo_pollfd;

/*
 * Flags for signals. We can't use bits in the same variable because bit
 * operations are not atomic.
 */

volatile sig_atomic_t got_sigterm = 0, got_sigstop = 0, got_sigcont = 0;

/* Signal handlers */

void
handle_sigterm (void)
{
   got_sigterm = 1;
}

void
handle_sigstop (void)
{
   got_sigstop = 1;
}

void
handle_sigcont (void)
{
   got_sigcont = 1;
}

void
cleanup (void)
{
   if (d)
   {
      if (bitmap)
	 XFreePixmap (d, bitmap);
      if (win)
	 XDestroyWindow (d, win);
      XCloseDisplay (d);
   }
}

void
suspend_me (void)
{
   if (is_mapped)
   {
      XUnmapWindow (d, win);
      XFlush (d);
   }
   got_sigstop = 0;
   /*
    * And we'll proceed to sleep in poll(). In case we get a command through
    * the FIFO we'll process it, so something might show up on the screen.
    * I'm not sure if this is a bug or a feature. :-)
    */
}

void
restart_me (void)
{
   if (is_mapped)
   {
      XMapWindow (d, win);
      XFlush (d);
   }
   got_sigcont = 0;
}

void
setup (int *argc, char **argv)
{
   int i, free_fifo = 0, term_fd, status;
   char *winenv, *fifo_file, *home;
   unsigned long pad_color;
   unsigned long gcmask;
   Colormap cmap;
   Window winid;
   XWindowAttributes winattrs;
   Cursor cursor_move;
   XGCValues gcvals;
   struct winsize winsz;
   struct stat inode;
   XClassHint terminal;

   /* See if we can connect to the display */
   if (!(d = XOpenDisplay (NULL)))
      exit (1);

   /* Then close because we'll fork. We'll reopen it in the child. Hopefully
      it won't run away. */
   XCloseDisplay (d);

   /* Terminal Window ID. We must have this. */
   if (!(winenv = getenv ("WINDOWID")) || !(winid = strtol (winenv, NULL, 0)))
      exit (2);

   /* Stupid method for determining controlling terminal. It should be
      one of those. But it doesn't have to. FIXME */
   for (i = 0; i < 3; ++i)
      if (isatty (term_fd = i))
	 break;
   if (i == 3)
      exit (3);
   ioctl (term_fd, TIOCGWINSZ, &winsz);

   /* We could work without this, but we're just picky. */
   if (!winsz.ws_row || !winsz.ws_col)
      exit (4);

   /* See if we got FIFO name as the argument. */
   fifo_file = NULL;
   for (i = 1; i < *argc && argv[i]; i += 2)
   {
      if (*argv[i] != '-')
      {
	 fifo_file = argv[i];
	 break;
      }
   }
   if (!fifo_file)
   {
      /* FIFO not in argument; construct the name */
      struct utsname u;
      int maxlen;

      if (!(home = getenv ("HOME")))
      {
	 struct passwd *pw;

	 pw = getpwuid (geteuid ());
	 if (!pw || !(home = pw->pw_dir))
	    exit (5);
      }
      if (uname (&u) < 0)
	 exit (5);
      
      maxlen = strlen (home) + sizeof ("/.slrnfaces/") + strlen (u.nodename)
	       + 30;
      fifo_file = malloc (maxlen);
      if (!fifo_file)
	 exit (5);
      free_fifo = 1;
      if (snprintf(fifo_file, maxlen, "%s/.slrnfaces/%s.%ld", home,
	           u.nodename, (long) getppid ()) < 0)
	 exit (5);
   }

   /*
    * The FIFO should have been created by the parent. If it doesn't exist
    * we'll create it here, but it's a bad omen. If the file exists, but
    * is not a FIFO, we need to return an error now, because we won't have
    * the chance later. Parent could block waiting for us or waiting on a
    * FIFO without a reader. We can't allow that. This whole show would have
    * been unnecessary if slang could open a pipe. A simple, unnamed,
    * unidirectional pipe. Is that too much to ask?
    */

   /* Open non-blocking FIFO now and remove that flag after the fork. */
   if (stat (fifo_file, &inode))
      if (errno == ENOENT)
      {
	 if (mkfifo (fifo_file, 0600))
	    exit (5);
      }
      else
	 exit (5);
   else
      if (!S_ISFIFO (inode.st_mode))
	 exit (5);
   if ((fifo_fd = open (fifo_file, O_RDONLY | O_NONBLOCK)) < 0)
      exit (5);
   if (free_fifo)
      free (fifo_file);

   switch (fork ())
   {
      case -1: exit (6);
      case  0: break;	 /* Child. Just continue. */
      default: exit (0); /* Parent. Return success and make its parent happy. */
   }

   fcntl (fifo_fd, F_SETFL, 0);

   /*
    * Fill this here. The structure will be used for the normal processing
    * as well, which is why it's a global thing.
    */

   fifo_pollfd.fd = fifo_fd;
   fifo_pollfd.events = POLLIN;

   /* Sync with the parent. */

   do {
      status = poll (&fifo_pollfd, 1, -1);
   } while (status == -1 && errno == EINTR);

   if (!(d = XOpenDisplay (NULL)))
      exit (1);

   /* Get colormap, preferably the one used by the parent terminal window. */

   if (XGetWindowAttributes (d, winid, &winattrs) == Success)
      cmap = winattrs.colormap;
   else
      cmap = DefaultColormap (d, DefaultScreen (d));

   /* X Resources stuff. Suggested by a user's demented mind. */

   if (XGetClassHint (d, winid, &terminal))
   {
      XtAppContext app_context;
      XrmDatabase res;
      int name_len, class_len, i;
      char *name_str, *class_str;
      char *name_start, *class_start;
      char *type_str;
      XrmValue value;

      XtToolkitInitialize ();
      app_context = XtCreateApplicationContext ();
      XtDisplayInitialize (app_context, d, terminal.res_name,
	    		   terminal.res_class, my_options,
			   countof (my_options), argc, argv);
      res = XtScreenDatabase (DefaultScreenOfDisplay (d));
      name_len = strlen (terminal.res_name);
      name_str = malloc (name_len + MAXRESOURCESIZE);
      memcpy (name_str, terminal.res_name, name_len);
      name_str[name_len] = '.';
      name_start = name_str + name_len + 1;
      class_len = strlen (terminal.res_class);
      class_str = malloc (class_len + MAXRESOURCESIZE);
      memcpy (class_str, terminal.res_class, class_len);
      class_str[class_len] = '.';
      class_start = class_str + class_len + 1;

      for (i = 0; i < countof (my_options); ++i)
      {
	 strcpy (name_start, my_options[i].specifier + 1);
	 strcpy (class_start, my_options[i].specifier + 1);

	 if(XrmGetResource(res, name_str, class_str, &type_str, &value) == True)
	 {
	    /* We don't have generic strings yet, so this will do. */

	    char *resource = malloc (value.size + 1);

	    memcpy (resource, value.addr, value.size);
	    resource[value.size] = 0;

	    if (optvalues[i].type == val_int)
	       optvalues[i].value.intval = strtol (resource, NULL, 0);
	    else	/* val_color */
	    {
	       XColor col;

	       if (XParseColor (d, cmap, resource, &col)
		   && XAllocColor (d, cmap, &col))
	       {
		  optvalues[i].value.intval = col.pixel;
		  /*
		   * Change type to val_int, so we know the value is set here.
		   */
		  optvalues[i].type = val_int;
	       }
	    }

	    free (resource);
	 }
      }

      free (name_str);
      free (class_str);
      XFree (terminal.res_name);
      XFree (terminal.res_class);
      XrmDestroyDatabase (res);
   }

   /*
    * The problem with terminals is that they might have scrollbars on
    * the unknown side. If it's on the right, you'll have to use X_OFFSET_PIX
    * to place the window at the correct position.
    */

   if (!winsz.ws_xpixel || !winsz.ws_ypixel)	/* Bastards... */
   {
      Window root_return;
      int x_return, y_return;
      unsigned width_return, height_return;
      unsigned border_width_return;
      unsigned depth_return;

      XGetGeometry (d, winid, &root_return, &x_return, &y_return,
	    	    &width_return, &height_return, &border_width_return,
		    &depth_return);

      /*
       * First try to substract scrollbar and hope the rest are
       * just characters on the screen.
       */
      width_return -= X_OFFSET_PIX;

      x_pos = width_return - X_OFFSET_CHAR * (width_return / winsz.ws_col);
      y_pos = Y_OFFSET_CHAR * (height_return / winsz.ws_row) + Y_OFFSET_PIX;
   }
   else
   {
      x_pos = winsz.ws_xpixel - X_OFFSET_CHAR * (winsz.ws_xpixel / winsz.ws_col)
      			      - X_OFFSET_PIX;
      y_pos = Y_OFFSET_CHAR * (winsz.ws_ypixel / winsz.ws_row) + Y_OFFSET_PIX;
   }
   x_pos -= 48 + XFACE_PAD;	/* X-Face window width. */

   /* Set up background color and create window. */

   if (PADCOLOR.type == val_color)
      pad_color = BlackPixelOfScreen (DefaultScreenOfDisplay (d));
   else
      pad_color = PADCOLOR.value.intval;

   win = XCreateSimpleWindow (d, winid, x_pos, y_pos, 48 + XFACE_PAD,
	 		      48, 0, 0, pad_color);

   /* Then set up colors in the GC. */

   if (INK.type == val_color)
      gcvals.foreground = BlackPixelOfScreen (DefaultScreenOfDisplay (d));
   else
      gcvals.foreground = INK.value.intval;

   if (PAPER.type == val_color)
      gcvals.background = WhitePixelOfScreen (DefaultScreenOfDisplay (d));
   else
      gcvals.background = PAPER.value.intval;

   gcvals.graphics_exposures = False;
   gcmask = GCForeground | GCBackground | GCGraphicsExposures;

   gc = XCreateGC(d, win, gcmask, &gcvals);

   /* And the last piece of crap. */

   cursor_move = XCreateFontCursor (d, XC_fleur);
   XGrabButton (d, Button1, AnyModifier, win, False,
	        ButtonPressMask | ButtonReleaseMask | Button1MotionMask,
	        GrabModeSync, GrabModeAsync, win, cursor_move);
   XSelectInput (d, win, ExposureMask | StructureNotifyMask);

   /* We are done. Amazing. */
}

typedef enum {
   xface_eof,		/* EOF or error on FIFO */
   xface_noop,		/* Do nothing */
   xface_clear,		/* Unmap window */
   xface_show,		/* Map window */
   xface_display	/* Map window and paint X-Face */
} XFaceState;

/* Read data from FIFO and return state. Reimplementing stdio is always fun. */

XFaceState
read_fifo (void)
{
   int n, status;
   size_t toread;
   int state, skip_junk = 0;
   char *command, *bufstart, *bufend, *bufp;
   char buf[8000];

   do {
begin:
      toread = sizeof(buf);
      bufstart = command = buf;

readmore:
      do {
	 n = read(fifo_fd, bufstart, toread);
      } while(n < 0 && errno == EINTR);

      if(n <= 0)
	 return xface_eof;

      bufend = bufstart + n;
      bufp = bufstart;
      state = xface_noop;
      while(bufp < bufend)
      {
	 /* Find the end of the command. */
	 while(bufp < bufend && *bufp != '\n')
	    ++bufp;
	 if(*bufp != '\n')
	 {
	    /* Partial command in the buffer. Try to read some more. */
	    toread = sizeof(buf) - (bufend - buf);
	    if (toread)
	    {
	       bufstart = bufend;
	       goto readmore;
	    }
	    if(command == buf)
	    {
	       /*
		* The command doesn't fit in the buffer. It must be
		* some junk, so discard it and resync on newline.
		*/
	       skip_junk = 1;
	       goto begin;
	    }
	    /*
	     * Copy what we have to the beginning of the buffer and read
	     * some more.
	     */
	    memmove(buf, command, bufend - command);
	    bufstart = buf + (bufend - command);
	    toread = sizeof(buf) - (bufend - command) + 1;
	    goto readmore;
	 }

	 /*
	  * See if we were supposed to skip the junk. This can happen
	  * only with the first chunk in the buffer.
	  */

	 if(skip_junk)
	 {
	    skip_junk = 0;
	    if (bufend == bufp + 1)
	       goto begin;

	    /* The next command needs our attention. */
	    command = ++bufp;
	    continue;
	 }

	 if(!memcmp(command, "clear\n", sizeof "clear"))
	    state = xface_clear;
	 else if(!memcmp(command, "show\n", sizeof "show"))
	    state = xface_show;
	 else if(!memcmp(command, "xface ", sizeof "xface"))
	 {
	    int face_len;
	    char *face_start;

	    face_start = command + sizeof "xface";
	    face_len = bufp - face_start;
	    if (face_len == buffered_face_len
		&& !memcmp(face_start, last_face, face_len))
	       if (pixmap_stale)
		  state = xface_display;
	       else
		  state = xface_show;
	    else
	    {
	       memcpy(last_face, face_start, face_len);
	       pixmap_stale = 1;
	       buffered_face_len = face_len;
	       state = xface_display;
	    }
	 }

	 /* Otherwise it's unrecognized command. Just skip it. */

	 command = ++bufp;
      }

      /*
       * Wait 10 milliseconds in case the parent has something else to send.
       * The user is not likely to notice this pause and we might avoid
       * certain amount of processing and roundtrips.
       */
      do {
	 status = poll (&fifo_pollfd, 1, 10);
      } while (status == -1 && errno == EINTR);
   } while (status > 0);

   return state;
}

char ls[] = { 0, 8, 4, 12, 2, 10, 6, 14, 1, 9, 5, 13, 3, 11, 7, 15 };

int
process_command (void)
{
   int i, n;
   unsigned char *datap;
   unsigned char data[288];
   char uncompfacebuf[15000];

   switch (read_fifo ())
   {
      case xface_eof:

	 return 1;

      case xface_clear:

	 if (is_mapped)
	 {
	    XUnmapWindow (d, win);
	    is_mapped = 0;
	    XFlush (d);
	 }
	 break;

      case xface_show:

	 if (!is_mapped)
	 {
	    XMapWindow (d, win);
	    is_mapped = 1;
	    XFlush (d);
	 }
	 break;

      case xface_display:

	 /* Try to prevent buffer overflow. uncompface() sucks. */
	 if (buffered_face_len > 2000)
	    break;

	 memcpy (uncompfacebuf, last_face, buffered_face_len);
	 uncompfacebuf[buffered_face_len] = 0;
	 pixmap_stale = 0;
	 uncompface (uncompfacebuf);

	 /* Are we alive? No buffer overruns? Fuck. */

	 for (i = 0, n = 0, datap = data; i < 48; ++i)
	 {
	    int i1, i2, i3, len;

	    sscanf (uncompfacebuf + n, "%i,%i,%i,%n", &i1, &i2, &i3, &len);
	    n += len;

	    /* Who invented that all-bits-are-reversed format? */

	    *datap++ = ls[i1 >> 12 & 15] | ls[i1 >> 8 & 15] << 4;
	    *datap++ = ls[i1 >> 4 & 15]  | ls[i1 & 15] << 4;
	    *datap++ = ls[i2 >> 12 & 15] | ls[i2 >> 8 & 15] << 4;
	    *datap++ = ls[i2 >> 4 & 15]  | ls[i2 & 15] << 4;
	    *datap++ = ls[i3 >> 12 & 15] | ls[i3 >> 8 & 15] << 4;
	    *datap++ = ls[i3 >> 4 & 15]  | ls[i3 & 15] << 4;
	 }

	 if (bitmap)
	    XFreePixmap (d, bitmap);
	 bitmap = XCreateBitmapFromData (d, win, (char *) data, 48, 48);
	 if (is_mapped)
	    XCopyPlane(d, bitmap, win, gc, 0, 0, 48, 48, XFACE_PAD, 0, 1);
	 else
	    XMapWindow (d, win); /* It will be filled on expose. */
	 XFlush (d);
   }

   return 0;
}

void
process_xevent(void)
{
   static int x_pointer, y_pointer;
   static int grab = AnyButton;
   int x, y;
   XEvent event;

   while (XPending(d))
   {
      XNextEvent(d, &event);

      switch(event.type)
      {
	 case MapNotify:
	 case Expose:

	    is_mapped = 1;
	    XCopyPlane(d, bitmap, win, gc, 0, 0, 48, 48, XFACE_PAD, 0, 1);
	    XFlush(d);
	    break;

	 case ButtonPress:

	    switch(((XButtonEvent *)&event)->button)
	    {
	       case Button1:
		  if(grab != AnyButton)
		  {
		     XAllowEvents(d, SyncPointer, CurrentTime);
		     break;
		  }
		  grab = Button1;
		  x_pointer = ((XButtonEvent *)&event)->x_root;
		  y_pointer = ((XButtonEvent *)&event)->y_root;
		  XAllowEvents(d, SyncPointer, CurrentTime);
		  break;
	    }
	    break;

	 case ButtonRelease:

	    if(grab == ((XButtonEvent *)&event)->button)
	       grab = AnyButton;
	    XAllowEvents(d, AsyncPointer, CurrentTime);
	    break;

	 case MotionNotify:

	    x = ((XMotionEvent *)&event)->x_root;
	    y = ((XMotionEvent *)&event)->y_root;
	    x_pos += x - x_pointer;
	    y_pos += y - y_pointer;
	    x_pointer = x;
	    y_pointer = y;
	    XMoveWindow(d, win, x_pos, y_pos);
	    break;
      }
   }
}

int
main(int argc, char **argv)
{
   struct pollfd p[2];
   struct sigaction sigact;

   /*
    * We don't need setlocale() call, because nothing here depends on the
    * locale. The only effect it might have is loading and initializing
    * one or more shared libraries. We don't need that.
    *
    * setlocale (LC_ALL, "");
    *
    */

   setup (&argc, argv);	/* child returns, FIFO set and ready, window created */

   /*
    * We'll ignore Ctrl-C because our action should depend on parent's
    * action. If the parent decides to exit it will close the pipe and
    * then we'll exit as well. If it doesn't exit, we are not supposed
    * to exit either. So we need to ignore this signal.
    */

   signal (SIGINT, SIG_IGN);

   /* But we need to handle these. */

   sigemptyset (&sigact.sa_mask);
   sigact.sa_flags = SA_RESTART;	/* Not that it matters. */

   sigact.sa_handler = handle_sigterm;
   sigaction (SIGTERM, &sigact, NULL);

   sigact.sa_handler = handle_sigstop;
   sigaction (SIGTSTP, &sigact, NULL);

   sigact.sa_handler = handle_sigcont;
   sigaction (SIGCONT, &sigact, NULL);

   /* Set up poll struct. */

   p[0].fd = ConnectionNumber (d);
   p[1].fd = fifo_fd;
   p[0].events = p[1].events = POLLIN;

   while (1)
   {
      int status;

      status = poll (p, 2, -1);

      if (status < 0)
      {
	 /* A signal was caught, most likely. */

	 if (got_sigterm)
	    break;
	 if (got_sigstop)
	    suspend_me ();
	 if (got_sigcont)
	    restart_me ();
      }
      else
      {
	 if ((p[0].revents | p[1].revents) & (POLLERR | POLLHUP | POLLNVAL))
	    break;
	 if (p[0].revents)
	    process_xevent ();
	 if (p[1].revents)
	    if (process_command ())
	       break;
      }
   }

   cleanup ();

   return 0;
}
