/*
 * raw2dv.cc -- Saves Raw DV streams in various formats
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <iostream>
#include <string>
#include <cmath>
using std::string;
using std::cerr;
using std::cin;
using std::endl;

#include <RawDVFileInput.h>
#include <Threader.h>

#include <Diagnostics.h>
#include <filehandler.h>
#include <time.h>
#include <pthread.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
 
typedef enum 
{
	raw2dv_idle,
	raw2dv_saving,
	raw2dv_stopping,
	raw2dv_shutdown
}
raw2dv_state;

typedef enum
{
	raw2dv_pipe_feed,
	raw2dv_pipe_starved
}
raw2dv_pipe_state;

class Raw2DV : public Threader
{
	private:
		pthread_t thread;
		string filename;
		int format;
		long maxfilesize;
		bool auto_split;
		bool time_stamp;
		int every_nth;
		long count;
		bool pass_through;
		string pass_through_pipe;
		string command_pipe;
		raw2dv_state state;
		raw2dv_pipe_state pipe_state;
		int pipe_fd;

	public:
		Raw2DV( ) : 
			filename( "" ), 
			format( 0 ), 
			maxfilesize( 1024 * 1024 * 1024 ),
			auto_split( false ),
			time_stamp( false ),
			every_nth( 1 ),
			count( 0 ),
			pass_through( false ),
			pass_through_pipe( "" ),
			command_pipe( "" ),
			state( raw2dv_saving ),
			pipe_state( raw2dv_pipe_starved ),
			pipe_fd( -1 )
		{
		}

		virtual ~Raw2DV( )
		{
		}

		virtual string LogId( )
		{
			return "Raw2DV";
		}

		void SetFileName( string _filename )
		{
			filename = _filename;
		}

		void SetFormat( string _format )
		{
			if ( _format == "raw" )
				format = 0;
			else if ( _format == "avi1" )
				format = 1;
			else if ( _format == "avi2" )
				format = 2;
			else if ( _format == "avi1dml" )
				format = 3;
			else if ( _format == "avi2dml" )
				format = 4;
#ifdef HAVE_LIBQUICKTIME
			else if ( _format == "mov" )
				format = 5;
#endif
			else
				throw "Unrecognised format.";
		}

		void SetMaxFileSize( long _maxfilesize )
		{
			maxfilesize = _maxfilesize;
		}

		void SetAutoSplit( bool _auto_split )
		{
			auto_split = _auto_split;
		}

		void SetTimeStamp( bool _time_stamp )
		{
			time_stamp = _time_stamp;
		}

		void SetEveryNth( int _every_nth )
		{
			every_nth = _every_nth;
		}

		bool IsRunnable( )
		{
			return filename != "";
		}

		void SetCount( long _count )
		{
			count = _count;
		}

		void SetPassThrough( bool _pass_through )
		{
			pass_through = _pass_through;
		}

		void SetPassThroughPipe( string _pass_through_pipe )
		{
			pass_through_pipe = _pass_through_pipe;
		}
		
		void SetCommandPipe( string _command_pipe )
		{
			command_pipe = _command_pipe;
			state = raw2dv_idle;
		}

		bool TimeChanged( TimeCode &lastTimeCode, TimeCode &currentTimeCode )
		{
			int lastSeconds = lastTimeCode.hour * 60 * 60 + lastTimeCode.min * 60 + lastTimeCode.sec;
			int currentSeconds = currentTimeCode.hour * 60 * 60 + currentTimeCode.min * 60 + currentTimeCode.sec;
			lastTimeCode = currentTimeCode;
			return abs( currentSeconds - lastSeconds ) > 1;
		}

	protected:
		void Thread( )
		{
			bool first = true;
			bool running = true;
			int frames = 0;
			TimeCode lastTimeCode;
			TimeCode currentTimeCode;

			FileHandler *handler = NULL;
			
			switch( format )
			{
				case 0:
					handler = new RawHandler( );
					break;
				case 1:
					handler = new AVIHandler( AVI_DV1_FORMAT );
					( ( AVIHandler *)handler )->SetOpenDML( false );
					break;
				case 2:
					handler = new AVIHandler( AVI_DV2_FORMAT );
					( ( AVIHandler *)handler )->SetOpenDML( false );
					break;
				case 3:
					handler = new AVIHandler( AVI_DV1_FORMAT );
					( ( AVIHandler *)handler )->SetOpenDML( true );
					break;
				case 4:
					handler = new AVIHandler( AVI_DV2_FORMAT );
					( ( AVIHandler *)handler )->SetOpenDML( true );
					break;
#ifdef HAVE_LIBQUICKTIME
				case 5:
					handler = new QtHandler( );
					break;
#endif
				default:
					throw "Unknown file handler.";
					break;
			}

			handler->SetBaseName( filename );
			handler->SetMaxFileSize( maxfilesize );
			handler->SetTimeStamp( time_stamp );
			handler->SetEveryNthFrame( every_nth );

			RawDVFileInput input;
			input.SetPumpSize( 25 );
			input.SetFile( stdin );

			input.ThreadStart( );

			if ( command_pipe != "" )
				pthread_create( &thread, NULL, CommandThreadStarter, this );
			
			while( running && ThreadIsRunning( ) && state != raw2dv_shutdown )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );

					if ( first )
					{
						handler->SetSampleFrame( frame );
						frame.GetTimeCode( lastTimeCode );
						first = false;
					}

					if ( state == raw2dv_stopping )
					{
						handler->Close( );
						state = raw2dv_idle;
					}
					
					if ( state == raw2dv_saving )
					{
						if ( auto_split )
						{
							frame.GetTimeCode( currentTimeCode );
							if ( TimeChanged( lastTimeCode, currentTimeCode ) )	
								handler->Close( );
						}
						running = handler->WriteFrame( frame );
					}
					
					if ( pass_through )
						fwrite( frame.data, frame.GetFrameSize( ), 1, stdout );
					
					if ( pipe_state == raw2dv_pipe_feed )
					{
						write( pipe_fd, frame.data, frame.GetFrameSize( ) );
					}
					else if ( pipe_fd != -1 )
					{
						close( pipe_fd );
						pipe_fd = -1;
					}
					
					input.QueueInputFrame( );

					if ( count != 0 && ++ frames >= count )
						running = false;
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}

			handler->Close( );
			delete handler;

			input.ThreadStop( );
			
			if ( command_pipe != "" )
				pthread_join( thread, NULL );			
		}
		
		static void *CommandThreadStarter( void *arg )
		{
			Raw2DV *raw = (Raw2DV *)arg;
			raw->CommandThread( );
			return NULL;
		}
		
		int ReadPipe( int fd, char *command, int length )
		{
			int n = 1;
			struct timeval tv;
			fd_set rfds;
		
			command[ 0 ] = '\0';
			
			FD_ZERO( &rfds );
			FD_SET( fd, &rfds );
			
			tv.tv_sec = 1;
			tv.tv_usec = 0;
			n = select( fd + 1, &rfds, NULL, NULL, &tv );
			if ( n > 0 ) 
			{
				n = read( fd, command, length );
				if ( n >= 0 )
					command[ n ] = '\0';
				if ( n == 0 )
					n = -1;
				return n;
			}
			
			return 0;
		}

		char *chomp( char *string )
		{
			if ( strchr( string, '\n' ) )
				*( strchr( string, '\n' ) ) = '\0';
			if ( strchr( string, '\r' ) )
				*( strchr( string, '\r' ) ) = '\0';
			return string;
		}
		
		void CommandThread( )
		{
			while( ThreadIsRunning( ) && state != raw2dv_shutdown )
			{
				bool connected = true;
				int fd = open( command_pipe.c_str( ), O_RDONLY | O_NONBLOCK );
				while ( ThreadIsRunning( ) && connected && state != raw2dv_shutdown )
				{
					char command[ 1024 ];
					int length;
					if ( ( length = ReadPipe( fd, command, 1024 ) ) != -1 )
					{
						chomp( command );
						if ( !strcmp( command, "start" ) )
						{
							cerr << "Start capture request received" << endl;
							state = raw2dv_saving;
						}
						else if ( !strcmp( command, "stop" ) )
						{
							cerr << "Stop capture request received" << endl;
							state = raw2dv_stopping;
						}
						else  if ( !strcmp( command, "shutdown" ) )
						{
							cerr << "Shutdown request received" << endl;
							state = raw2dv_shutdown;
						}
						else if ( !strcmp( command, "feed" ) )
						{
							cerr << "Feed request received" << endl;
							if ( pass_through_pipe != "" && pipe_fd == -1 ) 
								pipe_fd = open( pass_through_pipe.c_str( ), O_WRONLY );
							if ( pipe_fd != -1 )
								pipe_state = raw2dv_pipe_feed;
						}
						else if ( !strcmp( command, "starve" ) )
						{
							cerr << "Starve request received" << endl;
							pipe_state = raw2dv_pipe_starved;
						}
						else if ( strcmp( command, "" ) )
						{
							cerr << ">>  Unknown command received: " << command << endl;
						}
					}
					else
					{
						connected = false;
					}
				}
				close( fd );
			}
		}
};

static void Usage( )
{
	cerr << "Usage: raw2dv [ options ] name" << endl;
	cerr << "Where: options are" << endl;
	cerr << "       -f type   - DV wrapping type" << endl;
	cerr << "          type is raw (default), avi1, avi2, avi1dml, avi2dml, mov" << endl;
	cerr << "       -e number - write every number frame (default: 1)" << endl;
	cerr << "       -a        - autosplit (default: off)" << endl;
	cerr << "       -t        - timestamp in file name (default: off)" << endl;
	cerr << "       -m max    - max file size in bytes (default: 1gig)" << endl;
	cerr << "       -c count  - number of frames to capture (default: unlimited)" << endl;
	cerr << "       -p        - passthrough mode (stdin is echoed on stdout)" << endl;
	cerr << "       -P pipe   - passthrough to pipe while feeding" << endl;
	cerr << "       -x pipe   - pipe for receiving commands" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	Raw2DV output;

	if ( isatty( fileno( stdin ) ) )
	{
		cerr << "Input must must be obtained from a pipe or a file." << endl;
		Usage( );
	}

	Diagnostics::SetApp( "raw2dv" );

	try
	{
		signal( SIGPIPE, SIG_IGN );
		
		for ( int i = 1; i < argc; i ++ )
		{
			if ( !strcmp( argv[ i ], "-a" ) )
				output.SetAutoSplit( true );
			else if ( !strcmp( argv[ i ], "-t" ) )
				output.SetTimeStamp( true );
			else if ( !strcmp( argv[ i ], "-e" ) )
				output.SetEveryNth( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-c" ) )
				output.SetCount( atol( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-m" ) )
				output.SetMaxFileSize( atol( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-f" ) )
				output.SetFormat( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-p" ) )
				output.SetPassThrough( true );
			else if ( !strcmp( argv[ i ], "-P" ) )
				output.SetPassThroughPipe( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-x" ) )
				output.SetCommandPipe( string( argv[ ++ i ] ) );
			else if ( strcmp( argv[ i ], "--help" ) )
				output.SetFileName( string( argv[ i ] ) );
			else if ( !strcmp( argv[ i ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ i ] ) );
			else
				Usage( );
		}

		if ( output.IsRunnable( ) )
			output.ThreadExecute( );
		else
			cerr << "No file name specified - aborting" << endl;
	}
	catch ( string exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}
	catch ( char *exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}

	return 0;
}
