/*
 * Copyright (c) 1997-2002 Motonori Nakamura <motonori@media.kyoto-u.ac.jp>
 * Copyright (c) 1997-2002 WIDE Project
 * Copyright (c) 1997-2002 Kyoto University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 4. Neither the name of the Project, the University nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char *_id_ = "$Id: util.c,v 1.19 2002/01/20 14:19:39 motonori Exp $";
#endif

# include "common.h"
# include "extern.h"

char *
newstr(str)
char *str;
{
	char *ptr;

	ptr = (char *)MALLOC(strlen(str) + 1);
	if (ptr == NULL) {
		log(LOG_NOTICE, "out of memory (newstr)");
		return NULL;
	}
	strcpy(ptr, str);
	return ptr;
}

void
strlower(str)
char *str;
{
	while (*str != '\0') {
		if (isupper(*str))
			*str = tolower(*str);
		str++;
	}
}

static void
tv_sub(a, b, c)
struct timeval *a, *b, *c;
{
	c->tv_sec = a->tv_sec - b->tv_sec; 
	c->tv_usec = a->tv_usec - b->tv_usec; 
	if (c->tv_usec < 0) {
		c->tv_sec --;
		c->tv_usec += 1000000;
	}
	if (c->tv_sec < 0) {
		/* XXX */
		c->tv_sec = c->tv_usec = 0;
	}
}

void
resource_usage(buf)
char *buf;
{
	static firsttime=1;
	static struct rusage p;
	static struct timeval pt;
	struct rusage c, d;
	struct timeval ct, dt;

	if (!(cnf.debug & DEBUG_RESUSE))
		return;

	if (firsttime) {
		firsttime = 0;
		bzero(&p, sizeof(p));
		gettimeofday(&pt, NULL);
		return;
	}
	getrusage(0, &c);
	gettimeofday(&ct, NULL); /* XXX */
	tv_sub(&c.ru_utime, &p.ru_utime, &d.ru_utime);
	tv_sub(&c.ru_stime, &p.ru_stime, &d.ru_stime);
	tv_sub(&ct, &pt, &dt);
	d.ru_maxrss = c.ru_maxrss - p.ru_maxrss;
	d.ru_nivcsw = c.ru_nivcsw - p.ru_nivcsw;
	log(LOG_INFO, "time=%d.%03d+%d.%03d/%d.%03d rss=%ld nivcsw=%ld (%s)",
		d.ru_utime.tv_sec, d.ru_utime.tv_usec/1000,
		d.ru_stime.tv_sec, d.ru_stime.tv_usec/1000,
		dt.tv_sec, dt.tv_usec/1000,
		d.ru_maxrss, d.ru_nivcsw, buf);
	bcopy(&c, &p, sizeof(c));
	bcopy(&ct, &pt, sizeof(ct));
}

#define HASHSIZE	2003

static int
hash_func(name)
char *name;
{
	char *p = name;
	int val = 0;
	int c;

	while (*p != '\0')
	{
		if (isascii(*p) && isupper(*p))
			c = tolower(*p) & 0xff;
		else
			c = *p & 0xff;
		val = ((val << 1) ^ c) % HASHSIZE;
		p++;
	}

	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "hash_func(%s) = %d", name, val);

	return val;
}

static struct dns_stat *hash_query[HASHSIZE];

struct dns_stat *
hash_query_lookup(name, ptr)
char *name;
struct dns_stat ***ptr;
{
	struct dns_stat *qp;

	if (ptr != NULL)
	{
		*ptr = &hash_query[hash_func(name)];
		qp = **ptr;
	}
	else
		qp = hash_query[hash_func(name)];
	while (qp != NULL)
        {
                if (strcasecmp(qp->name, name) == 0)
		{
			if (cnf.debug & DEBUG_HASH)
			log(LOG_DEBUG, "query %s found in hash table", name);
                        return qp;
		}
                qp = qp->next;
        }
	return NULL;
}

static struct domain *hash_domain[HASHSIZE];

struct domain *
hash_domain_lookup(name, ptr)
char *name;
struct domain ***ptr;
{
	struct domain *domp;

	if (ptr != NULL)
	{
		*ptr = &hash_domain[hash_func(name)];
		domp = **ptr;
	}
	else
		domp = hash_domain[hash_func(name)];
	while (domp != NULL)
        {
                if (strcasecmp(domp->name, name) == 0)
		{
			if (cnf.debug & DEBUG_HASH)
			log(LOG_DEBUG, "domain %s found in hash table", name);
                        return domp;
		}
                domp = domp->hash;
        }
	return NULL;
}

#if 0
void
hash_domain_enter(name, newdomp)
char *name;
struct domain *newdomp;
{
	int idx = hash_func(name);

	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "enter domain %s into hash table", name);

	newdomp->hash = hash_domain[idx];
	hash_domain[idx] = newdomp;
}
#endif

static struct domain_rcpts *hash_domain_rcpts[HASHSIZE];

int
hash_domain_rcpts_lookup(name)
char *name;
{
	struct domain_rcpts *dom_r_p;
	int idx = hash_func(name);

	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "look up domain(%s) index(%d)", name, idx);
	dom_r_p = hash_domain_rcpts[idx];
	while (dom_r_p != NULL)
	{
		if (strcasecmp(dom_r_p->name, name) == 0)
		{
			if (cnf.debug & DEBUG_HASH)
			log(LOG_DEBUG, "hit domain(%s) rcpts(%d) in hash table",
				name, dom_r_p->rcpts);
			return dom_r_p->rcpts;
		}
		dom_r_p = dom_r_p->hash;
	}
	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "domain(%s) not found in hash table, return %d",
		name, cnf.rcpts_trans);
	return cnf.rcpts_trans;
}

void
hash_domain_rcpts_enter(name, newdom_r_p)
char *name;
struct domain_rcpts *newdom_r_p;
{
	int idx = hash_func(name);

	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "enter domain(%s) index(%d) rcpts(%d) into hash table",
		name, idx, newdom_r_p->rcpts);

	newdom_r_p->hash = hash_domain_rcpts[idx];
	hash_domain_rcpts[idx] = newdom_r_p;
}

int
isnumeric(p)
char *p;
{
	while(*p != '\0')
	{
		if(!isdigit(*p))
			return 0;
		p++;
	}
	return 1;
}

#define DEFAULTRCPTS  (char *)"DEFAULT"

int
rcpts_def_load()
{
	FILE *fp;
	char buf[MAXLINE], *p;
	int n;
	struct domain_rcpts *dom_r_p;
	char *strp, *argp;

	/* open cnf.rcpts_def */
	fp = fopen(cnf.rcpts_def, "r");
	if (fp == NULL)
	{
		log(LOG_NOTICE, "file %s not found", cnf.rcpts_def);
		return -1;
	}
	/* read cnf.rcpts_def */
	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
		n = strlen(buf);
		if (n > 0)
			n--;
		if (buf[n] == '\n')
			buf[n--] = '\0';
		if ((p = strchr(buf, '#')) != NULL)
		{
			*p-- = '\0';
			while (buf < p && isspace(*p))
				*p-- = '\0';
		}
		if (buf[0] == '\0' /* || buf[0] == '#' */)
			continue;

		/* get domain name and # of recipients */
		n = strlen(buf);
		p = buf;
		while(p < buf+n && ! isspace(*p))
			*p++;
		if(p == buf+n)
			continue;
		else
			*p = '\0';
		strp = buf;		/* domain name */
		*p++;
		while(p < buf+n && isspace(*p))
			*p++;
		if(p == buf+n)
			continue;
		argp = p;		/* # of recipients */
		while(p < buf+n && ! isspace(*p))
			*p++;
		if(p < buf+n)
			*p = '\0';

		/* set domain name and # of recipients */
		if(strp != NULL && argp != NULL && isnumeric(argp))
		{
			dom_r_p = (struct domain_rcpts*) MALLOC(sizeof(struct domain_rcpts));
			if (dom_r_p == NULL)
			{
				log(LOG_NOTICE, "out of memory (domain_rcpts)");
				fclose(fp);
				return -1;
			}
			bzero(dom_r_p, sizeof(struct domain_rcpts));
			dom_r_p->name  = newstr(strp);
			if(dom_r_p->name == NULL)
			{
				log(LOG_NOTICE, "out of memory (dom_r_p->name)");
				fclose(fp);
				return -1;
			}
			dom_r_p->rcpts = atoi(argp);
			hash_domain_rcpts_enter(dom_r_p->name, dom_r_p);
		}
	}
	fclose(fp);
	cnf.rcpts_trans = hash_domain_rcpts_lookup(DEFAULTRCPTS);
	if (cnf.debug & DEBUG_TRANS)
	log(LOG_DEBUG, "set cnf.rcpts_trans = %d", cnf.rcpts_trans);
	return 0;
}

static struct host *hash_host[HASHSIZE];

struct host *
hash_host_lookup(name, ptr)
char *name;
struct host ***ptr;
{
	struct host *hostp;

	if (ptr != NULL)
	{
		*ptr = &hash_host[hash_func(name)];
		hostp = **ptr;
	}
	else
		hostp = hash_host[hash_func(name)];
	while (hostp != NULL)
        {
                if (strcasecmp(hostp->name, name) == 0)
		{
			if (cnf.debug & DEBUG_HASH)
			log(LOG_DEBUG, "host %s found in hash table", name);
                        return hostp;
		}
                hostp = hostp->hash;
        }
	return NULL;
}

#if 0
void
hash_host_enter(name, newhostp)
char *name;
struct host *newhostp;
{
	int idx = hash_func(name);

	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "enter host %s into hash table", name);

	newhostp->hash = hash_host[idx];
	hash_host[idx] = newhostp;
}
#endif

static struct host_map *hash_map[HASHSIZE];

struct host_map *
hash_map_lookup(name, ptr)
char *name;
struct host_map ***ptr;
{
	struct host_map *mapp;

	if (ptr != NULL)
	{
		*ptr = &hash_map[hash_func(name)];
		mapp = **ptr;
	}
	else
		mapp = hash_map[hash_func(name)];
	while (mapp != NULL)
        {
                if (strcasecmp(mapp->name, name) == 0)
		{
			if (cnf.debug & DEBUG_HASH)
			log(LOG_DEBUG, "map %s found in hash table", name);
                        return mapp;
		}
                mapp = mapp->hash;
        }
	return NULL;
}

void
hash_map_enter(name, newmapp)
char *name;
struct host_map *newmapp;
{
	int idx = hash_func(name);

	if (cnf.debug & DEBUG_HASH)
	log(LOG_DEBUG, "enter map %s into hash table", name);

	newmapp->hash = hash_map[idx];
	hash_map[idx] = newmapp;
}

static struct host_map *rmap_head;

int
host_map_load()
{
	FILE *fp;
	char buf[MAXLINE], *p, *q;
	int n, cols;
	struct host_map *mapp;
	char *strp, **argp;

	fp = fopen(cnf.map, "r");
	if (fp == NULL)
	{
		log(LOG_NOTICE, "map %s not found", cnf.map);
		return -1;	/* should be TEMPFAIL */
	}
	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
		n = strlen(buf);
		if (n >= sizeof(buf))
			buf[--n] = '\0';
		if (buf[n-1] == '\n')
			buf[--n] = '\0';
		if ((p = strchr(buf, '#')) != NULL)
		{
			*p-- = '\0';
			while (buf < p && isspace(*p))
				*p-- = '\0';
			n = strlen(buf);
		}
		if (buf[0] == '\0' /* || buf[0] == '#' */)
			continue;
		/* count number of columns */
		p = buf;
		cols = 2;	/* adjustment for NULL termination */
		while ((q = strchr(p, ':')) != NULL)
		{
			if (q != p)
				cols++;
			p = q + 1;
		}

		mapp = (struct host_map*) MALLOC(sizeof(struct host_map)
			+ n + sizeof(char *) * cols);
		if (mapp == NULL)
		{
			fclose(fp);
			return 0;
		}
		argp = (char **)(mapp + 1);
		strp = (char *)(argp + cols);
		mapp->next = NULL;
		mapp->name = strp;
		mapp->arg = argp;

		p = buf;
		/* skip heading spaces */
		while (*p != '\0')
		{
			if (!isspace(*p))
				break;
			p++;
		}
		q = p;	/* top of key name */
		while (*p != '\0')
		{
			if (isspace(*p))
				break;
			p++;
		}
		if (q == p || *p == '\0')
		{
			/* no argument, ignore this line */
			free(mapp);
			continue;
		}
		*p++ = '\0';
		strcpy(strp, q);
		strp += strlen(strp) + 1;

		/* skip separator spaces */
		while (*p != '\0')
		{
			if (!isspace(*p))
				break;
			p++;
		}

		/* parse "arg" portion */
		while ((q = strchr(p, ':')) != NULL)
		{
			if (q == p)
			{
				p++;
				continue;
			}
			*q++ = '\0';
			strcpy(strp, p);
			*argp++ = strp;
			strp += strlen(strp) + 1;
			p = q;
		}
		/* final "arg" portion */
		if (*p != '\0')
		{
			strcpy(strp, p);
			*argp++ = strp;
		}
		*argp = NULL;

		if (rmap_head != NULL)
			mapp->next = rmap_head;
		rmap_head = mapp;
		hash_map_enter(mapp->name, mapp);

		if (cnf.debug & DEBUG_MAP)
		{
			char tmpbuf[MAXLINE];

			argp = mapp->arg;
			while (*argp != NULL)
			{
				if (argp == mapp->arg)
					strncpy (tmpbuf, *argp, sizeof(tmpbuf));
				else
				{
					strncat (tmpbuf, ":", sizeof(tmpbuf));
					strncat (tmpbuf, *argp, sizeof(tmpbuf));
				}
				argp++;
			}
			log(LOG_DEBUG, "reading map: domain = %s, map = %s",
				mapp->name, tmpbuf);
		}
	}
	fclose(fp);
	return 0;
}

char **
host_map_lookup(domain)
char *domain;
{
	static char *default_map[] = {"MX", NULL};
	static char *root = ".";
	struct host_map *mapp;
	char *p;

	if (rmap_head == NULL)
		return default_map;

	p = domain;
	while (p != NULL)
	{
		if (cnf.debug & DEBUG_MAP)
		log(LOG_DEBUG, "searching host map for %s: %s", domain, p);

		if ((mapp = hash_map_lookup(p, NULL)) != NULL)
		{
			char **mp = mapp->arg;
			int found = 0;

			/* is there an aliasing? */
			if (*p == '.')
			{

				while (*mp != NULL)
				{
					if (!found && **mp == '=')
						found = 1;
					mp++;
				}
			}

			if (found)
			{
				/* yes! do aliasing */
				char **new_arg, **new_mp;

				new_arg = (char **)MALLOC(sizeof(char *)
					* (mp - mapp->arg + 1));
				new_mp = new_arg;
				mp = mapp->arg;

				while (*mp != NULL)
				{
					if (**mp == '=')
					{
						/* replace '=' with host part */

						*new_mp = (char *)
							MALLOC((p - domain)
							+ strlen(*mp));
						strncpy(*new_mp, domain,
							p - domain);
						strcat(*new_mp + (p - domain),
							*mp + 1);
					}
					else
						*new_mp = *mp;
					mp++;
					new_mp++;
				}
				*new_mp = NULL;
				return new_arg;
			}
			return mapp->arg;
		}
		if (p == root)
			break;
		p = strchr(p+1, '.');
		if (p == NULL)
			p = root;
	}
	return default_map;
}
