-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with CommandLineData;
with ErrorHandler;
with LexTokenManager;
with SP_Symbols;

use type LexTokenManager.Str_Comp_Result;

package body Casing is

   --  Return TRUE if the casing is revevant for the token Lex_Str.
   function Is_Casing_Activated (Lex_Str : in LexTokenManager.Lex_String) return Boolean
   --# global in CommandLineData.Content;
   --#        in LexTokenManager.State;
   is
   begin
      return (LexTokenManager.Is_Standard_Token (Lex_Str => Lex_Str) and then CommandLineData.Content.Casing_Standard)
        or else ((not LexTokenManager.Is_Standard_Token (Lex_Str => Lex_Str))
                 and then CommandLineData.Content.Casing_Identifier);
   end Is_Casing_Activated;

   -----------------------------------------------------------------------------------------

   procedure Check_String_Casing
     (Str      : in E_Strings.T;
      Lex_Str  : in LexTokenManager.Lex_String;
      Position : in LexTokenManager.Token_Position) is
   begin
      if Is_Casing_Activated (Lex_Str => Lex_Str)
        and then not E_Strings.Eq_CS_String (E_Str1 => Str,
                                             E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => Lex_Str))
      then
         --  The casing is not the same and the casing switch has been
         --  set -> raise the warning.
         ErrorHandler.Semantic_Warning (Err_Num  => 380,
                                        Position => Position,
                                        Id_Str   => Lex_Str);
      end if;
   end Check_String_Casing;

   -----------------------------------------------------------------------------------------

   procedure Check_Casing
     (Lex_Str1 : in LexTokenManager.Lex_String;
      Lex_Str2 : in LexTokenManager.Lex_String;
      Position : in LexTokenManager.Token_Position) is
   begin
      if Is_Casing_Activated (Lex_Str => Lex_Str1)
        and then LexTokenManager.Lex_String_Case_Sensitive_Compare (Lex_Str1 => Lex_Str1,
                                                                    Lex_Str2 => Lex_Str2) /=
        LexTokenManager.Str_Eq then
         --  The casing is not the same and the casing switch has been
         --  set -> raise the warning.
         ErrorHandler.Semantic_Warning (Err_Num  => 380,
                                        Position => Position,
                                        Id_Str   => Lex_Str1);
      end if;
   end Check_Casing;

   -----------------------------------------------------------------------------------------

   procedure Check_Node_Casing (Top_Node : in STree.SyntaxNode) is
      It : STree.Iterator;
   begin
      --  Iterate on all the identifiers in the tree with the top node
      --  TOP_NODE.
      It := STree.Find_First_Node (Node_Kind    => SP_Symbols.identifier,
                                   From_Root    => Top_Node,
                                   In_Direction => STree.Down);
      while It /= STree.NullIterator loop
         --  Check if the casing is correct between the reference
         --  string LEX_STR1 and the specific usage of the string
         --  LEX_STR2.
         Check_Casing
           (Lex_Str1 => STree.Node_Lex_String (Node => STree.Get_Node (It => It)),
            Lex_Str2 => STree.Node_Token_String (Node => STree.Get_Node (It => It)),
            Position => STree.Node_Position (Node => STree.Get_Node (It => It)));
         It := STree.NextNode (It => It);
      end loop;
   end Check_Node_Casing;

end Casing;
