-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Maths;
with FileSystem;
with E_Strings;

separate (Dictionary)
procedure Write (File_Name : in     E_Strings.T;
                 Status    :    out SPARK_IO.File_Status) is

   File            : SPARK_IO.File_Type := SPARK_IO.Null_File;
   File_Status     : SPARK_IO.File_Status;
   Local_File_Name : E_Strings.T;

   --------------------------------------------------------------------------------

   procedure Write_Library_Units (File : in SPARK_IO.File_Type)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File,
   --#                                LexTokenManager.State;
   is

      Library_Units : Iterator;

      procedure Write_Generic_Formal_Parameters (File        : in SPARK_IO.File_Type;
                                                 The_Generic : in Symbol)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                LexTokenManager.State,
      --#                                The_Generic;
      is

         Parameter_It : Iterator;
         Number       : Positive;

         --------------------------------------------------------------------------------

         procedure Write_Generic_Formal_Parameter
           (File        : in SPARK_IO.File_Type;
            The_Generic : in Symbol;
            Number      : in Positive;
            Parameter   : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Number,
         --#                                Parameter,
         --#                                The_Generic;
         is

            procedure Write_Generic_Type (File      : in SPARK_IO.File_Type;
                                          Parameter : in Symbol)

            --# global in     Dict;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                Parameter;
            is
               The_Type : Symbol;
            begin
               The_Type := RawDict.GetGenericParameterTypeOrObject (Parameter);
               case RawDict.GetTypeDiscriminant (The_Type) is
                  when GenericLimitedType =>
                     WriteString (File, "generic limited ");
                  when GenericPrivateType =>
                     WriteString (File, "generic private ");
                  when GenericOrderedDiscreteType =>
                     WriteString (File, "generic dicrete (ordered) ");
                  when GenericUnorderedDiscreteType =>
                     WriteString (File, "generic discrete ");
                  when GenericIntegerType =>
                     WriteString (File, "generic integer ");
                  when GenericModularType =>
                     WriteString (File, "generic modular ");
                  when GenericFloatingPointType =>
                     WriteString (File, "generic floating point ");
                  when GenericFixedPointType =>
                     WriteString (File, "generic fixed point ");
                  when GenericArrayType =>
                     WriteString (File, "generic array ");
                  when others =>
                     WriteString (File, "invalid generic type ");
               end case;
            end Write_Generic_Type;

         begin -- Write_Generic_Formal_Parameter
            WriteString (File, "generic formal parameter # ");
            WriteInteger (File, Number);
            WriteString (File, " of ");
            WriteName (File, The_Generic);
            WriteString (File, " is ");
            Write_Simple_Name (File => File,
                               Item => Parameter);
            WriteString (File, " which is ");
            case RawDict.GetGenericParameterKind (Parameter) is
               when GenericTypeParameter =>
                  WriteString (File, "a ");
                  Write_Generic_Type (File      => File,
                                      Parameter => Parameter);
                  WriteString (File, "type parameter");
               when GenericObjectParameter =>
                  WriteString (File, "a generic object parameter of type ");
                  WriteName (File, GetType (RawDict.GetGenericParameterTypeOrObject (Parameter)));
            end case;
            WriteLine (File, " ;");
         end Write_Generic_Formal_Parameter;

         --------------------------------------------------------------------------------

      begin
         Parameter_It := FirstGenericFormalParameter (The_Generic);
         Number       := 1;
         loop
            exit when IsNullIterator (Parameter_It);
            Write_Generic_Formal_Parameter
              (File        => File,
               The_Generic => The_Generic,
               Number      => Number,
               Parameter   => CurrentSymbol (Parameter_It));
            Parameter_It := NextSymbol (Parameter_It);
            Number       := Number + 1;
         end loop;
      end Write_Generic_Formal_Parameters;

      --------------------------------------------------------------------------------

      function First_Library_Unit return Iterator
      --# global in Dict;
      is
         Declaration   : Symbol;
         Item          : Symbol;
         Found         : Boolean;
         Library_Units : Iterator;
      begin

         Declaration := RawDict.GetPackageFirstLocalDeclaration (GetPredefinedPackageStandard);

         loop
            if Declaration = NullSymbol then
               Library_Units := NullIterator;
               exit;
            end if;
            Item := RawDict.GetDeclarationItem (Declaration);
            case RawDict.GetSymbolDiscriminant (Item) is
               when PackageSymbol =>
                  Found := Declaration /= RawDict.GetPackageBody (Item);
               when others =>
                  Found := Declaration /= RawDict.GetSubprogramBody (Item);
            end case;
            if Found then
               Library_Units := Iterator'(LibraryUnitIterator, IsAbstract, Item, Declaration);
               exit;
            end if;
            Declaration := RawDict.GetNextDeclaration (Declaration);
         end loop;

         return Library_Units;

      end First_Library_Unit;

      --------------------------------------------------------------------------------

      procedure Write_Library_Unit (File : in SPARK_IO.File_Type;
                                    Unit : in Symbol)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                LexTokenManager.State,
      --#                                Unit;
      is
         --# hide Write_Library_Unit;

         --------------------------------------------------------------------------------

         procedure Write_With_References (File  : in SPARK_IO.File_Type;
                                          Scope : in Scopes)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Scope;
         is

            With_Reference : Iterator;

            --------------------------------------------------------------------------------

            function First_Withed_Package (Scope : Scopes) return Iterator
            --# global in Dict;
            is
               Context_Clause  : Symbol;
               Region          : Symbol;
               Withed_Packages : Iterator;
            begin

               case Scope.TypeOfScope is
                  when Visible =>
                     Context_Clause := RawDict.GetPackageVisibleWithClauses (GetRegion (Scope));
                  when others =>
                     Region := GetRegion (Scope);
                     case RawDict.GetSymbolDiscriminant (Region) is
                        when PackageSymbol =>
                           Context_Clause := RawDict.GetPackageLocalWithClauses (Region);
                        when others =>
                           Context_Clause := RawDict.GetSubprogramWithClauses (Region);
                     end case;
               end case;

               if Context_Clause = NullSymbol then
                  Withed_Packages := NullIterator;
               else
                  Withed_Packages :=
                    Iterator'(WithedPackageIterator, IsAbstract, RawDict.GetContextClausePackage (Context_Clause), Context_Clause);
               end if;

               return Withed_Packages;

            end First_Withed_Package;

            --------------------------------------------------------------------------------

            procedure Write_With_Reference (File        : in SPARK_IO.File_Type;
                                            The_Package : in Symbol;
                                            Scope       : in Scopes)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Scope,
            --#                                The_Package;
            is
            begin
               WriteString (File, "with reference to ");
               Write_Simple_Name (File => File,
                                  Item => The_Package);
               WriteString (File, " in ");
               WriteScope (File, Scope);
               WriteLine (File, " ;");
            end Write_With_Reference;

            --------------------------------------------------------------------------------

         begin
            With_Reference := First_Withed_Package (Scope => Scope);
            loop
               exit when IsNullIterator (With_Reference);
               Write_With_Reference (File        => File,
                                     The_Package => CurrentSymbol (With_Reference),
                                     Scope       => Scope);
               With_Reference := NextSymbol (With_Reference);
            end loop;
         end Write_With_References;

         --------------------------------------------------------------------------------

         procedure Write_Inherits_References (File             : in SPARK_IO.File_Type;
                                              Compilation_Unit : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Compilation_Unit,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State;
         is

            Inherits_Reference : Iterator;

            --------------------------------------------------------------------------------

            function First_Inherited_Package (Compilation_Unit : Symbol) return Iterator
            --# global in Dict;
            is
               Context_Clause     : Symbol;
               Inherited_Packages : Iterator;
            begin

               if IsPackage (Compilation_Unit) then
                  Context_Clause := RawDict.GetPackageInheritClauses (Compilation_Unit);
               elsif IsMainProgram (Compilation_Unit) then
                  Context_Clause := Dict.Main.InheritClauses;
               else
                  Context_Clause := NullSymbol;
               end if;

               if Context_Clause = NullSymbol then
                  Inherited_Packages := NullIterator;
               else
                  Inherited_Packages :=
                    Iterator'
                    (InheritedPackageIterator,
                     IsAbstract,
                     RawDict.GetContextClausePackage (Context_Clause),
                     Context_Clause);
               end if;

               return Inherited_Packages;

            end First_Inherited_Package;

            --------------------------------------------------------------------------------

            procedure Write_Inherits_Reference
              (File             : in SPARK_IO.File_Type;
               The_Package      : in Symbol;
               Compilation_Unit : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Compilation_Unit,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Package;
            is
            begin
               WriteString (File, "inherits reference to ");
               Write_Simple_Name (File => File,
                                  Item => The_Package);
               WriteString (File, " in ");
               WriteName (File, Compilation_Unit);
               WriteLine (File, " ;");
            end Write_Inherits_Reference;

            --------------------------------------------------------------------------------

         begin
            Inherits_Reference := First_Inherited_Package (Compilation_Unit => Compilation_Unit);
            loop
               exit when IsNullIterator (Inherits_Reference);
               Write_Inherits_Reference
                 (File             => File,
                  The_Package      => CurrentSymbol (Inherits_Reference),
                  Compilation_Unit => Compilation_Unit);
               Inherits_Reference := NextSymbol (Inherits_Reference);
            end loop;
         end Write_Inherits_References;

         --------------------------------------------------------------------------------

         procedure Write_Package_Info (File        : in SPARK_IO.File_Type;
                                       The_Package : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                The_Package;
         is
            Scope : Scopes;
         begin
            if Is_Generic_Package (The_Symbol => The_Package) then
               WriteString (File, "generic ");
            end if;

            if RawDict.GetPackageParent (The_Package) /= NullSymbol then
               WriteString (File, "child ");
            end if;

            WriteString (File, "package named ");
            Write_Simple_Name (File => File,
                               Item => The_Package);
            Scope := GetScope (The_Package);
            if GetRegion (Scope) /= GetPredefinedPackageStandard then
               WriteString (File, " declared in ");
               WriteScope (File, Scope);
            end if;
            WriteLine (File, " ;");
         end Write_Package_Info;

         --------------------------------------------------------------------------------

         procedure Write_Subprogram (File       : in SPARK_IO.File_Type;
                                     Subprogram : in Symbol);

         --------------------------------------------------------------------------------

         procedure Write_Package (File        : in SPARK_IO.File_Type;
                                  The_Package : in Symbol);

         --------------------------------------------------------------------------------

         procedure Write_Global_Variables (Abstraction : in Abstractions;
                                           File        : in SPARK_IO.File_Type;
                                           Subprogram  : in Symbol);

         --------------------------------------------------------------------------------

         procedure Write_Dependency_Clauses
           (Abstraction   : in Abstractions;
            File          : in SPARK_IO.File_Type;
            The_Procedure : in Symbol);

         --------------------------------------------------------------------------------

         function First_Declarative_Item (Scope : Scopes) return Iterator
         --# global in Dict;
         is
            Declaration       : Symbol;
            Region            : Symbol;
            Item              : Symbol;
            Found             : Boolean;
            Declarative_Items : Iterator;
         begin

            case Scope.TypeOfScope is
               when Visible =>
                  Declaration := RawDict.GetPackageFirstVisibleDeclaration (GetRegion (Scope));
               when Local =>
                  Region := GetRegion (Scope);
                  case RawDict.GetSymbolDiscriminant (Region) is
                     when PackageSymbol =>
                        Declaration := RawDict.GetPackageFirstLocalDeclaration (Region);
                     when SubprogramSymbol =>
                        Declaration := RawDict.GetSubprogramFirstDeclaration (Region);
                     when others =>
                        Declaration := NullSymbol;
                  end case;
               when Privat =>
                  Declaration := RawDict.GetPackageFirstPrivateDeclaration (GetRegion (Scope));
            end case;

            loop
               exit when Declaration = NullSymbol;
               Item := RawDict.GetDeclarationItem (Declaration);
               case RawDict.GetSymbolDiscriminant (Item) is
                  when PackageSymbol =>
                     Found := Declaration /= RawDict.GetPackageBody (Item);
                  when SubprogramSymbol =>
                     Found := Declaration /= RawDict.GetSubprogramBody (Item);
                  when others =>
                     Found := True;
               end case;
               exit when Found;
               Declaration := RawDict.GetNextDeclaration (Declaration);
            end loop;

            if Declaration = NullSymbol then
               Declarative_Items := NullIterator;
            else
               Declarative_Items := Iterator'(DeclarativeItemIterator, IsAbstract, Item, Declaration);
            end if;

            return Declarative_Items;

         end First_Declarative_Item;

         --------------------------------------------------------------------------------

         procedure Write_Declarative_Items (File  : in SPARK_IO.File_Type;
                                            Scope : in Scopes)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Scope;
         is

            Declarative_Items : Iterator;
            Declarative_Item  : Symbol;

            procedure Write_Lex_String (File    : in SPARK_IO.File_Type;
                                        Lex_Str : in LexTokenManager.Lex_String)
            --# global in     LexTokenManager.LexStringTable;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                File,
            --#                                LexTokenManager.LexStringTable,
            --#                                Lex_Str;
            is
            begin
               E_Strings.Put_String (File  => File,
                                     E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Lex_Str));
            end Write_Lex_String;

            --------------------------------------------------------------------------------

            procedure WriteStaticValue
              (File     : in SPARK_IO.File_Type;
               Value    : in LexTokenManager.Lex_String;
               TypeMark : in Symbol)
            --# global in     LexTokenManager.State;
            --#        in out Dict;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                TypeMark,
            --#                                Value;
            is
               RootType : Symbol;
            begin
               RootType := GetRootType (TypeMark);

               if IsUnknownTypeMark (TypeMark) then
                  WriteString (File, " unknown ");
               elsif TypeIsNumeric (TypeMark) or else TypeIsCharacter (TypeMark) then
                  E_Strings.Put_String (File  => File,
                                        E_Str => Maths.ValueToString (Maths.ValueRep (Value)));
               elsif TypeIsBoolean (TypeMark) or else IsPredefinedStringType (RootType) then
                  Write_Lex_String (File    => File,
                                    Lex_Str => Value);
               elsif TypeIsEnumeration (TypeMark) then
                  WriteName (File, GetEnumerationLiteral (TypeMark, Value));
               else
                  WriteString (File, " unknown ");
               end if;
            end WriteStaticValue;

            --------------------------------------------------------------------------------

            procedure WriteType (File     : in SPARK_IO.File_Type;
                                 The_Type : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Type;
            is

               --------------------------------------------------------------------------------

               procedure WriteTypeInfo (File     : in SPARK_IO.File_Type;
                                        The_Type : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                The_Type;
               is

                  --------------------------------------------------------------------------------

                  procedure WriteDiscriminant (File     : in SPARK_IO.File_Type;
                                               The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                The_Type;
                  is
                  begin
                     case RawDict.GetTypeDiscriminant (The_Type) is
                        when UnknownType =>
                           WriteString (File, "unknown");
                        when EnumerationType =>
                           WriteString (File, "enumeration");
                        when IntegerType =>
                           WriteString (File, "integer");
                        when ModularType =>
                           WriteString (File, "modular");
                        when FloatingPointType =>
                           WriteString (File, "floating point");
                        when FixedPointType =>
                           WriteString (File, "fixed point");
                        when ArrayType =>
                           WriteString (File, "array");
                        when RecordType =>
                           WriteString (File, "record");
                        when AbstractProofType =>
                           WriteString (File, "abstract ");
                        when ProtectedType =>
                           WriteString (File, "protected");
                        when TaskType =>
                           WriteString (File, "task");
                        when AccessType =>
                           WriteString (File, "access ");
                           WriteName (File, RawDict.GetTypeAccesses (The_Type));
                        when GenericLimitedType =>
                           WriteString (File, "generic limited ");
                        when GenericPrivateType =>
                           WriteString (File, "generic private ");
                        when GenericOrderedDiscreteType =>
                           WriteString (File, "generic dicrete (ordered) ");
                        when GenericUnorderedDiscreteType =>
                           WriteString (File, "generic discrete ");
                        when GenericIntegerType =>
                           WriteString (File, "generic integer ");
                        when GenericModularType =>
                           WriteString (File, "generic modular ");
                        when GenericFloatingPointType =>
                           WriteString (File, "generic floating point ");
                        when GenericFixedPointType =>
                           WriteString (File, "generic fixed point ");
                        when GenericArrayType =>
                           WriteString (File, "generic array ");
                     end case;
                  end WriteDiscriminant;

                  -------------------------------------------------------------------------

                  procedure WriteBound
                    (File     : in SPARK_IO.File_Type;
                     Bound    : in LexTokenManager.Lex_String;
                     The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Bound,
                  --#                                Dict,
                  --#                                File,
                  --#                                The_Type;
                  is
                  begin
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => Bound,
                        Lex_Str2 => LexTokenManager.Null_String) =
                       LexTokenManager.Str_Eq then
                        SPARK_IO.Put_String (File, "unknown", 7);
                     else
                        WriteStaticValue (File, Bound, The_Type);
                     end if;
                  end WriteBound;

                  --------------------------------------------------------------------------------

                  function TypeIsLimitedPrivate (The_Type : Symbol) return Boolean
                  --# global in Dict;
                  is
                  begin
                     return RawDict.GetTypeLimited (The_Type) /= Never;
                  end TypeIsLimitedPrivate;

                  --------------------------------------------------------------------------------

                  procedure WriteProtectedRefinement (File     : in SPARK_IO.File_Type;
                                                      The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                The_Type;
                  is
                     It : Iterator;
                  begin
                     WriteString (File, "The implicit state variable of type ");
                     Write_Simple_Name (File => File,
                                        Item => The_Type);
                     WriteLine (File, " has the following refinement constituents:");
                     It := FirstConstituent (GetProtectedTypeOwnVariable (The_Type));
                     while not IsNullIterator (It) loop
                        WriteString (File, "      ");
                        Write_Simple_Name (File => File,
                                           Item => CurrentSymbol (It));
                        WriteLine (File, ",");
                        It := NextSymbol (It);
                     end loop;
                     WriteLine (File, ";");
                  end WriteProtectedRefinement;

                  --------------------------------------------------------------------------------

                  procedure WriteTypeDiscriminants (File     : in SPARK_IO.File_Type;
                                                    The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                The_Type;
                  is
                     It : Iterator;
                  begin
                     WriteString (File, "Type ");
                     Write_Simple_Name (File => File,
                                        Item => The_Type);
                     WriteLine (File, " has the following known discriminants:");
                     It := FirstKnownDiscriminant (The_Type);
                     if IsNullIterator (It) then
                        WriteLine (File, "None;");
                     else
                        while not IsNullIterator (It) loop
                           WriteString (File, "      ");
                           Write_Simple_Name (File => File,
                                              Item => CurrentSymbol (It));
                           WriteString (File, " which is of type ");
                           WriteName (File, GetType (CurrentSymbol (It)));
                           if SetsPriority (CurrentSymbol (It)) then
                              WriteString (File, " and is used to set the priority");
                           end if;
                           WriteLine (File, ";");
                           It := NextSymbol (It);
                        end loop;
                     end if;
                  end WriteTypeDiscriminants;

                  --------------------------------------------------------------------------------

                  procedure WriteDiscriminantConstraint (File     : in SPARK_IO.File_Type;
                                                         The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                The_Type;
                  is
                     ConstraintIt : Iterator;
                     KnownIt      : Iterator;

                     procedure WriteConstraintValue (File          : in SPARK_IO.File_Type;
                                                     TheConstraint : in Symbol)
                     --# global in     Dict;
                     --#        in     LexTokenManager.State;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Dict,
                     --#                                File,
                     --#                                LexTokenManager.State,
                     --#                                TheConstraint;
                     is
                        Val : LexTokenManager.Lex_String;
                     begin
                        Val := RawDict.GetDiscriminantConstraintStaticValue (TheConstraint);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => Val,
                           Lex_Str2 => LexTokenManager.Null_String) /=
                          LexTokenManager.Str_Eq then
                           WriteStaticValue (File, Val, GetPredefinedIntegerType);
                        elsif RawDict.GetDiscriminantConstraintAccessedObject (TheConstraint) /= NullSymbol then
                           -- if no static value then must be access to PO
                           WriteName (File, RawDict.GetDiscriminantConstraintAccessedObject (TheConstraint));
                        end if;
                     end WriteConstraintValue;

                  begin -- WriteDiscriminantConstraint
                     WriteString (File, "Type ");
                     Write_Simple_Name (File => File,
                                        Item => The_Type);
                     WriteLine (File, " has the following discriminant constraints:");
                     ConstraintIt := FirstDiscriminantConstraint (The_Type);
                     if IsNullIterator (ConstraintIt) then
                        WriteLine (File, "None;");
                     else
                        KnownIt := FirstKnownDiscriminant (GetParentType (The_Type));
                        while not IsNullIterator (ConstraintIt) and then not IsNullIterator (KnownIt) loop
                           WriteString (File, "      ");
                           Write_Simple_Name (File => File,
                                              Item => CurrentSymbol (KnownIt));
                           WriteString (File, " => ");
                           WriteConstraintValue (File, CurrentSymbol (ConstraintIt));
                           if SetsPriority (CurrentSymbol (KnownIt)) then
                              WriteString (File, " which is used to set the priority");
                           end if;
                           WriteLine (File, ";");
                           KnownIt      := NextSymbol (KnownIt);
                           ConstraintIt := NextSymbol (ConstraintIt);
                        end loop;
                     end if;
                  end WriteDiscriminantConstraint;

                  --------------------------------------------------------------------------------

                  procedure WriteTypePriority (File     : in SPARK_IO.File_Type;
                                               The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                The_Type;
                  is
                     Val : LexTokenManager.Lex_String;
                  begin
                     if GetTypeHasPragma (The_Type, Priority) then
                        WriteString (File, "Type ");
                        Write_Simple_Name (File => File,
                                           Item => The_Type);
                        WriteString (File, " has pragma Priority");
                        Val := GetTypePragmaValue (The_Type, Priority);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => Val,
                           Lex_Str2 => LexTokenManager.Null_String) =
                          LexTokenManager.Str_Eq then
                           WriteLine (File, " of unknown value;");
                        else
                           WriteString (File, " of value ");
                           WriteStaticValue (File, Val, GetPredefinedIntegerType);
                           WriteLine (File, ";");
                        end if;

                     elsif GetTypeHasPragma (The_Type, InterruptPriority) then
                        WriteString (File, "Type ");
                        Write_Simple_Name (File => File,
                                           Item => The_Type);
                        WriteString (File, " has pragma Interrupt_Priority");
                        Val := GetTypePragmaValue (The_Type, InterruptPriority);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => Val,
                           Lex_Str2 => LexTokenManager.Null_String) =
                          LexTokenManager.Str_Eq then
                           WriteLine (File, " of unknown value;");
                        else
                           WriteString (File, " of value ");
                           WriteStaticValue (File, Val, GetPredefinedIntegerType);
                           WriteLine (File, ";");
                        end if;
                     end if;
                  end WriteTypePriority;

                  --------------------------------------------------------------------------------
                  procedure WritePriority (File     : in SPARK_IO.File_Type;
                                           The_Type : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                The_Type;
                  is
                  begin
                     WriteString (File, "Type ");
                     WriteName (File, The_Type);
                     WriteString (File, " has a priority of ");
                     WriteStaticValue (File, GetTypePriority (The_Type), GetPredefinedIntegerType);
                     WriteLine (File, ";");
                  end WritePriority;

                  --------------------------------------------------------------------------------

                  procedure WriteSuspendsList (File    : in SPARK_IO.File_Type;
                                               TheTask : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheTask;
                  is
                     It : Iterator;
                  begin
                     WriteString (File, "Type ");
                     Write_Simple_Name (File => File,
                                        Item => TheTask);
                     WriteLine (File, " has the following PO or SO names in its Suspend list");
                     It := FirstSuspendsListItem (TheTask);
                     if IsNullIterator (It) then
                        WriteLine (File, "None;");
                     else
                        while not IsNullIterator (It) loop
                           WriteString (File, "      ");
                           WriteName (File, CurrentSymbol (It));
                           WriteLine (File, ";");
                           It := NextSymbol (It);
                        end loop;
                     end if;
                  end WriteSuspendsList;

                  --------------------------------------------------------------------------------

               begin
                  WriteString (File, "type named ");
                  Write_Simple_Name (File => File,
                                     Item => The_Type);
                  WriteString (File, " is ");

                  if GetRootOfExtendedType (The_Type) /= NullSymbol then
                     WriteString (File, "an extension of type ");
                     WriteName (File, GetRootOfExtendedType (The_Type));
                     WriteString (File, " and is ");
                  end if;

                  if TypeIsAbstract (The_Type) then
                     WriteString (File, "abstract ");
                  end if;
                  if TypeIsTagged (The_Type) then
                     WriteString (File, "tagged ");
                  end if;
                  if TypeIsPrivate (The_Type) then
                     if TypeIsLimitedPrivate (The_Type) then
                        WriteString (File, "limited ");
                     end if;
                     WriteString (File, "private ");
                  end if;
                  if IsUnconstrainedArrayType (The_Type) then
                     WriteString (File, "unconstrained ");
                  end if;
                  if IsStatic (The_Type, GetScope (The_Type)) then
                     WriteString (File, "static ");
                  end if;
                  WriteDiscriminant (File, The_Type);
                  if IsProofType (The_Type) then
                     WriteString (File, "proof ");
                  end if;
                  WriteSpace (File);
                  if IsType (The_Type) then
                     WriteString (File, "type");
                  else
                     WriteString (File, "subtype of ");
                     WriteName (File, GetParentType (The_Type));
                  end if;
                  if TypeIsScalar (The_Type) then
                     WriteString (File, " range ");
                     WriteBound (File, RawDict.GetTypeLower (The_Type), The_Type);
                     WriteString (File, " .. ");
                     WriteBound (File, RawDict.GetTypeUpper (The_Type), The_Type);
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => RawDict.GetTypeErrorBound (The_Type),
                        Lex_Str2 => LexTokenManager.Null_String) /=
                       LexTokenManager.Str_Eq then
                        if TypeIsFloatingPoint (The_Type) then
                           WriteString (File, " digits ");
                        else
                           WriteString (File, " delta ");
                        end if;
                        WriteStaticValue (File, RawDict.GetTypeErrorBound (The_Type), The_Type);
                     end if;
                  elsif TypeIsArray (The_Type) then
                     WriteString (File, " of ");
                     WriteName (File, GetArrayComponent (The_Type));
                  elsif IsType (The_Type)
                    and then TypeIsProtected (The_Type)
                    and then GetProtectedTypeElementsHidden (The_Type) then
                     WriteString (File, " with hidden elements ");
                  end if;
                  if TypeIsAccess (The_Type) then
                     WriteString (File, " implicitly");
                  end if;
                  WriteString (File, " declared in ");
                  WriteScope (File, GetScope (The_Type));
                  WriteLine (File, " ;");
                  if TypeIsProtected (The_Type) then
                     if IsType (The_Type) then
                        WriteTypeDiscriminants (File, The_Type);
                        WriteProtectedRefinement (File, The_Type);
                        WriteTypePriority (File, The_Type);
                     else -- subtype
                        WriteDiscriminantConstraint (File, The_Type);
                     end if;
                     WritePriority (File, The_Type);
                  end if;
                  if TypeIsTask (The_Type) then
                     if IsType (The_Type) then
                        WriteTypeDiscriminants (File, The_Type);
                        Write_Global_Variables (Abstraction => IsAbstract,
                                                File        => File,
                                                Subprogram  => The_Type);
                        Write_Dependency_Clauses (Abstraction   => IsAbstract,
                                                  File          => File,
                                                  The_Procedure => The_Type);
                        WriteSuspendsList (File, The_Type);
                        WriteTypePriority (File, The_Type);
                        if not RawDict.GetTaskTypeSignatureIsWellformed (IsAbstract, The_Type) then
                           WriteLine (File, "Task type signature contains errors");
                        end if;
                     else --subtype
                        WriteDiscriminantConstraint (File, The_Type);
                     end if;
                     WritePriority (File, The_Type);
                  end if;
               end WriteTypeInfo;

               --------------------------------------------------------------------------------

               procedure WriteEnumerationLiterals (File            : in SPARK_IO.File_Type;
                                                   EnumerationType : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                EnumerationType,
               --#                                File,
               --#                                LexTokenManager.State;
               is

                  Literal : Iterator;
                  Value   : Natural;

                  --------------------------------------------------------------------------------

                  procedure WriteEnumerationLiteral
                    (File               : in SPARK_IO.File_Type;
                     EnumerationType    : in Symbol;
                     EnumerationValue   : in Natural;
                     EnumerationLiteral : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                EnumerationLiteral,
                  --#                                EnumerationType,
                  --#                                EnumerationValue,
                  --#                                File,
                  --#                                LexTokenManager.State;
                  is
                  begin
                     WriteString (File, "enumeration literal # ");
                     WriteInteger (File, EnumerationValue);
                     WriteString (File, " of ");
                     WriteName (File, EnumerationType);
                     WriteString (File, " is ");
                     Write_Simple_Name (File => File,
                                        Item => EnumerationLiteral);
                     WriteLine (File, " ;");
                  end WriteEnumerationLiteral;

                  --------------------------------------------------------------------------------

               begin
                  Literal := FirstEnumerationLiteral (EnumerationType);
                  Value   := 0;
                  loop
                     exit when IsNullIterator (Literal);
                     WriteEnumerationLiteral (File, EnumerationType, Value, CurrentSymbol (Literal));
                     Literal := NextSymbol (Literal);
                     Value   := Value + 1;
                  end loop;
               end WriteEnumerationLiterals;

               --------------------------------------------------------------------------------

               procedure WriteRecordComponents (File       : in SPARK_IO.File_Type;
                                                RecordType : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                RecordType;
               is

                  Component : Iterator;
                  Number    : Positive;

                  --------------------------------------------------------------------------------

                  procedure WriteRecordComponent
                    (File            : in SPARK_IO.File_Type;
                     RecordType      : in Symbol;
                     Number          : in Positive;
                     RecordComponent : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                Number,
                  --#                                RecordComponent,
                  --#                                RecordType;
                  is
                  begin
                     WriteString (File, "record component # ");
                     WriteInteger (File, Number);
                     WriteString (File, " of ");
                     WriteName (File, RecordType);
                     WriteString (File, " is ");
                     Write_Simple_Name (File => File,
                                        Item => RecordComponent);
                     WriteString (File, " of type ");
                     WriteName (File, GetType (RecordComponent));
                     WriteLine (File, " ;");
                  end WriteRecordComponent;

                  --------------------------------------------------------------------------------

               begin
                  Component := FirstRecordComponent (RecordType);
                  Number    := 1;
                  loop
                     exit when IsNullIterator (Component);
                     WriteRecordComponent (File, RecordType, Number, CurrentSymbol (Component));
                     Component := NextSymbol (Component);
                     Number    := Number + 1;
                  end loop;
               end WriteRecordComponents;

               --------------------------------------------------------------------------------

               procedure WriteArrayIndices (File      : in SPARK_IO.File_Type;
                                            ArrayType : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                ArrayType,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State;
               is

                  ArrayIndex : Iterator;
                  Dimension  : Positive;

                  --------------------------------------------------------------------------------

                  procedure WriteArrayIndex
                    (File      : in SPARK_IO.File_Type;
                     ArrayType : in Symbol;
                     Dimension : in Positive;
                     IndexType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                ArrayType,
                  --#                                Dict,
                  --#                                Dimension,
                  --#                                File,
                  --#                                IndexType,
                  --#                                LexTokenManager.State;
                  is
                  begin
                     WriteString (File, "index # ");
                     WriteInteger (File, Dimension);
                     WriteString (File, " of ");
                     WriteName (File, ArrayType);
                     WriteString (File, " is ");
                     WriteName (File, IndexType);
                     WriteLine (File, " ;");
                  end WriteArrayIndex;

                  --------------------------------------------------------------------------------

               begin
                  ArrayIndex := FirstArrayIndex (ArrayType);
                  Dimension  := 1;
                  loop
                     exit when IsNullIterator (ArrayIndex);
                     WriteArrayIndex (File, ArrayType, Dimension, CurrentSymbol (ArrayIndex));
                     ArrayIndex := NextSymbol (ArrayIndex);
                     Dimension  := Dimension + 1;
                  end loop;
               end WriteArrayIndices;

               --------------------------------------------------------------------------------

               procedure WriteProtectedDeclarations (File     : in SPARK_IO.File_Type;
                                                     The_Type : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                The_Type;
               is
                  It : Iterator;

                  procedure WriteProtectedElement (File       : in SPARK_IO.File_Type;
                                                   The_Type   : in Symbol;
                                                   TheElement : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheElement,
                  --#                                The_Type;
                  is
                  begin
                     WriteString (File, "protected element named ");
                     Write_Simple_Name (File => File,
                                        Item => TheElement);
                     WriteString (File, "  of ");
                     WriteName (File, GetType (TheElement));
                     WriteString (File, " declared in ");
                     WriteName (File, The_Type);
                     WriteLine (File, " ;");
                  end WriteProtectedElement;

               begin -- WriteProtectedDeclarations
                  It := FirstVisibleSubprogram (The_Type);
                  while not IsNullIterator (It) loop
                     Write_Subprogram (File       => File,
                                       Subprogram => CurrentSymbol (It));
                     It := NextSymbol (It);
                  end loop;
                  It := FirstProtectedElement (The_Type);
                  while not IsNullIterator (It) loop
                     WriteProtectedElement (File, The_Type, CurrentSymbol (It));
                     It := NextSymbol (It);
                  end loop;
               end WriteProtectedDeclarations;

               --------------------------------------------------------------------------------

            begin
               WriteTypeInfo (File, The_Type);
               if TypeIsEnumeration (The_Type) and then IsType (The_Type) then
                  WriteEnumerationLiterals (File, The_Type);
               elsif TypeIsRecord (The_Type) then
                  WriteRecordComponents (File, The_Type);
               elsif TypeIsArray (The_Type) then
                  WriteArrayIndices (File, The_Type);
               elsif IsType (The_Type) and then TypeIsProtected (The_Type) then
                  WriteProtectedDeclarations (File, The_Type);
               end if;
            end WriteType;

            --------------------------------------------------------------------------------

            procedure WriteVariable (File     : in SPARK_IO.File_Type;
                                     Variable : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Variable;
            is
            begin
               WriteString (File, "variable named ");
               Write_Simple_Name (File => File,
                                  Item => Variable);
               WriteString (File, " is");
               if VariableIsInitialized (Variable) then
                  WriteString (File, " initialized");
               end if;
               if VariableIsAliased (Variable) then
                  WriteString (File, " aliased");
               end if;
               WriteString (File, " variable of ");
               WriteName (File, GetType (Variable));
               WriteString (File, " declared in ");
               WriteScope (File, GetScope (Variable));
               if VariableHasPragmaImport (Variable) then
                  WriteString (File, " and completed by a pragma Import");
               end if;
               WriteLine (File, " ;");
            end WriteVariable;

            --------------------------------------------------------------------------------

            procedure WriteConstant (File        : in SPARK_IO.File_Type;
                                     TheConstant : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                TheConstant;
            is
            begin
               WriteString (File, "constant named ");
               Write_Simple_Name (File => File,
                                  Item => TheConstant);
               WriteString (File, " is ");
               if ConstantIsDeferred (TheConstant) then
                  WriteString (File, "deferred ");
               end if;
               if IsStatic (TheConstant, GetScope (TheConstant)) then
                  WriteString (File, "static ");
               end if;
               if IsProofConstant (TheConstant) then
                  WriteString (File, "proof ");
               end if;
               WriteString (File, "constant of ");
               WriteName (File, GetType (TheConstant));
               if LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => GetValue (TheConstant),
                  Lex_Str2 => LexTokenManager.Null_String) /=
                 LexTokenManager.Str_Eq then
                  WriteString (File, " value ");
                  WriteStaticValue (File, GetValue (TheConstant), GetType (TheConstant));
               end if;
               WriteString (File, " declared in ");
               WriteScope (File, GetScope (TheConstant));
               WriteLine (File, " ;");
            end WriteConstant;

            --------------------------------------------------------------------------------

            procedure Write_Declarative_Item (File : in SPARK_IO.File_Type;
                                              Item : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                Item,
            --#                                LexTokenManager.State;
            is
            begin
               case RawDict.GetSymbolDiscriminant (Item) is
                  when TypeSymbol =>
                     WriteType (File, Item);
                  when VariableSymbol =>
                     WriteVariable (File, Item);
                  when ConstantSymbol =>
                     WriteConstant (File, Item);
                  when SubprogramSymbol =>
                     Write_Subprogram (File       => File,
                                       Subprogram => Item);
                  when others =>
                     Write_Package (File        => File,
                                    The_Package => Item);
               end case;
            end Write_Declarative_Item;

            --------------------------------------------------------------------------------

         begin

            Declarative_Items := First_Declarative_Item (Scope => Scope);

            loop

               exit when IsNullIterator (Declarative_Items);

               Declarative_Item := CurrentSymbol (Declarative_Items);

               if IsType (Declarative_Item) and then TypeIsPrivate (Declarative_Item) and then IsLocalOrPrivateScope (Scope) then
                  null;
               elsif IsLocalOrPrivateScope (Scope) and then IsDeferredConstant (Declarative_Item) then
                  null;
               else
                  Write_Declarative_Item (File => File,
                                          Item => Declarative_Item);
               end if;

               Declarative_Items := NextSymbol (Declarative_Items);

            end loop;

         end Write_Declarative_Items;

         --------------------------------------------------------------------------------

         procedure WriteLoops (File             : in SPARK_IO.File_Type;
                               Compilation_Unit : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Compilation_Unit,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State;
         is

            TheLoop : Iterator;

            --------------------------------------------------------------------------------

            procedure WriteLoop (File    : in SPARK_IO.File_Type;
                                 TheLoop : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                TheLoop;
            is

               --------------------------------------------------------------------------------

               procedure WriteLoopInfo (File    : in SPARK_IO.File_Type;
                                        TheLoop : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TheLoop;
               is
               begin
                  WriteString (File, "loop named ");
                  Write_Simple_Name (File => File,
                                     Item => TheLoop);
                  WriteString (File, " declared in ");
                  WriteScope (File, GetScope (TheLoop));
                  WriteLine (File, " ;");
               end WriteLoopInfo;

               --------------------------------------------------------------------------------

               procedure WriteLoopParameter (File          : in SPARK_IO.File_Type;
                                             LoopParameter : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                LoopParameter;
               is
               begin
                  WriteString (File, "loop parameter of ");
                  WriteName (File, GetRegion (GetScope (LoopParameter)));
                  WriteString (File, " named ");
                  Write_Simple_Name (File => File,
                                     Item => LoopParameter);
                  WriteString (File, " is of type ");
                  WriteName (File, GetType (LoopParameter));
                  WriteLine (File, " ;");
               end WriteLoopParameter;

               --------------------------------------------------------------------------------

            begin
               WriteLoopInfo (File, TheLoop);
               if IsForLoop (TheLoop) then
                  WriteLoopParameter (File, GetLoopParameter (TheLoop));
               end if;
            end WriteLoop;

            --------------------------------------------------------------------------------

         begin
            TheLoop := FirstLoop (Compilation_Unit);
            loop
               exit when IsNullIterator (TheLoop);
               WriteLoop (File, CurrentSymbol (TheLoop));
               TheLoop := NextSymbol (TheLoop);
            end loop;
         end WriteLoops;

         --------------------------------------------------------------------------------

         procedure Write_Global_Variables (Abstraction : in Abstractions;
                                           File        : in SPARK_IO.File_Type;
                                           Subprogram  : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Abstraction,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Subprogram;
         is

            GlobalVariable : Iterator;

            --------------------------------------------------------------------------------

            procedure WriteGlobalVariable
              (Abstraction    : in Abstractions;
               File           : in SPARK_IO.File_Type;
               Subprogram     : in Symbol;
               GlobalVariable : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Abstraction,
            --#                                Dict,
            --#                                File,
            --#                                GlobalVariable,
            --#                                LexTokenManager.State,
            --#                                Subprogram;
            is
            begin
               if Abstraction = IsRefined then
                  WriteString (File, "refined ");
               end if;
               WriteString (File, "global variable named ");
               WriteName (File, GlobalVariable);

               WriteString (File, " of mode ");
               case GetGlobalMode (Abstraction, Subprogram, GlobalVariable) is
                  when DefaultMode =>
                     WriteString (File, "default");
                  when InMode =>
                     WriteString (File, "in");
                  when OutMode =>
                     WriteString (File, "out");
                  when InOutMode =>
                     WriteString (File, "in out");
                  when InvalidMode =>
                     WriteString (File, "invalid");
               end case;
               WriteString (File, " is referenced by ");
               WriteName (File, Subprogram);
               WriteLine (File, " ;");
            end WriteGlobalVariable;

            --------------------------------------------------------------------------------

         begin
            GlobalVariable := FirstGlobalVariable (Abstraction, Subprogram);
            loop
               exit when IsNullIterator (GlobalVariable);
               WriteGlobalVariable (Abstraction, File, Subprogram, CurrentSymbol (GlobalVariable));
               GlobalVariable := NextSymbol (GlobalVariable);
            end loop;
         end Write_Global_Variables;

         --------------------------------------------------------------------------------

         procedure Write_Dependency_Clauses
           (Abstraction   : in Abstractions;
            File          : in SPARK_IO.File_Type;
            The_Procedure : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Abstraction,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                The_Procedure;
         is

            Export : Iterator;

            --------------------------------------------------------------------------------

            procedure WriteDependencies
              (Abstraction   : in Abstractions;
               File          : in SPARK_IO.File_Type;
               The_Procedure : in Symbol;
               Export        : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Abstraction,
            --#                                Dict,
            --#                                Export,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Procedure;
            is

               Dependency : Iterator;

               --------------------------------------------------------------------------------

               procedure WriteDependency
                 (Abstraction   : in Abstractions;
                  File          : in SPARK_IO.File_Type;
                  The_Procedure : in Symbol;
                  Export        : in Symbol;
                  Import        : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Abstraction,
               --#                                Dict,
               --#                                Export,
               --#                                File,
               --#                                Import,
               --#                                LexTokenManager.State,
               --#                                The_Procedure;
               is
               begin
                  if Abstraction = IsRefined then
                     WriteString (File, "refined ");
                  end if;
                  if GetOwnVariableOrConstituentMode (Export) = InMode then
                     WriteString (File, "implicit ");
                  end if;
                  WriteString (File, "export named ");
                  WriteName (File, Export);
                  WriteString (File, " is ");
                  if GetOwnVariableOrConstituentMode (Import) = OutMode then
                     WriteString (File, "implicitly ");
                  end if;
                  WriteString (File, "derived from ");
                  if Import = NullSymbol then
                     WriteString (File, "nothing");
                  else
                     WriteName (File, Import);
                  end if;
                  WriteString (File, " in ");
                  WriteName (File, The_Procedure);
                  WriteLine (File, " ;");
               end WriteDependency;

               --------------------------------------------------------------------------------

            begin
               Dependency := FirstDependency (Abstraction, The_Procedure, Export);
               if IsNullIterator (Dependency) then
                  WriteDependency (Abstraction, File, The_Procedure, Export, NullSymbol);
               else
                  loop
                     exit when IsNullIterator (Dependency);
                     WriteDependency (Abstraction, File, The_Procedure, Export, CurrentSymbol (Dependency));
                     Dependency := NextSymbol (Dependency);
                  end loop;
               end if;
            end WriteDependencies;

            --------------------------------------------------------------------------------

         begin
            Export := FirstExport (Abstraction, The_Procedure);
            loop
               exit when IsNullIterator (Export);
               WriteDependencies (Abstraction, File, The_Procedure, CurrentSymbol (Export));
               Export := NextSymbol (Export);
            end loop;
         end Write_Dependency_Clauses;

         --------------------------------------------------------------------------------

         procedure Write_Package (File        : in SPARK_IO.File_Type;
                                  The_Package : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                The_Package;
         is

            --------------------------------------------------------------------------------

            procedure WriteOwnVariables (File        : in SPARK_IO.File_Type;
                                         The_Package : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Package;
            is

               OwnVariables : Iterator;
               OwnVariable  : Symbol;

               --------------------------------------------------------------------------------

               procedure WriteOwnVariable (File        : in SPARK_IO.File_Type;
                                           The_Package : in Symbol;
                                           Variable    : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                The_Package,
               --#                                Variable;
               is
               begin
                  case GetOwnVariableMode (Variable) is
                     when DefaultMode =>
                        WriteString (File, "default");
                     when InMode =>
                        WriteString (File, "in");
                     when OutMode =>
                        WriteString (File, "out");
                     when InOutMode =>
                        WriteString (File, "in out");
                     when InvalidMode =>
                        WriteString (File, "invalid");
                  end case;
                  WriteString (File, " mode ");
                  if GetOwnVariableProtected (Variable) then
                     WriteString (File, "protected ");
                  end if;
                  WriteString (File, "own variable ");
                  Write_Simple_Name (File => File,
                                     Item => Variable);
                  if RawDict.GetOwnVariableTyped (RawDict.GetVariableOwnVariable (Variable)) then
                     WriteString (File, " of type ");
                     WriteName (File, GetType (Variable));
                  end if;
                  WriteString (File, " is owned ");
                  if OwnVariableIsInitialized (Variable) then
                     WriteString (File, "and initialized ");
                  end if;
                  WriteString (File, "by ");
                  WriteName (File, The_Package);
                  WriteLine (File, " ;");
               end WriteOwnVariable;

               --------------------------------------------------------------------------------

            begin
               OwnVariables := FirstOwnVariable (The_Package);
               loop
                  exit when IsNullIterator (OwnVariables);
                  OwnVariable := CurrentSymbol (OwnVariables);
                  WriteOwnVariable (File, The_Package, OwnVariable);
                  OwnVariables := NextSymbol (OwnVariables);
               end loop;
            end WriteOwnVariables;

            --------------------------------------------------------------------------------

            procedure WriteEmbeddedPackages (File        : in SPARK_IO.File_Type;
                                             The_Package : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Package;
            is
               Packages : Iterator;
            begin
               Packages := FirstEmbeddedPackage (The_Package);
               loop
                  exit when IsNullIterator (Packages);
                  Write_Package_Info (File        => File,
                                      The_Package => CurrentSymbol (Packages));
                  Packages := NextSymbol (Packages);
               end loop;
            end WriteEmbeddedPackages;

            --------------------------------------------------------------------------------
            --1606 added by JEB
            procedure WriteOwnedPackages (File        : in SPARK_IO.File_Type;
                                          The_Package : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Package;
            is
               Packages : Iterator;
            begin
               Packages := FirstOwnedPackage (The_Package);
               loop
                  exit when IsNullIterator (Packages);
                  Write_Package_Info (File        => File,
                                      The_Package => CurrentSymbol (Packages));
                  Write_Package (File        => File,
                                 The_Package => CurrentSymbol (Packages));
                  Packages := NextSymbol (Packages);
               end loop;
            end WriteOwnedPackages;

            --------------------------------------------------------------------------------

            procedure WriteAbstractOwnVariables (File        : in SPARK_IO.File_Type;
                                                 The_Package : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                The_Package;
            is

               AbstractOwnVariables : Iterator;

               --------------------------------------------------------------------------------

               procedure WriteAbstractOwnVariable (File    : in SPARK_IO.File_Type;
                                                   Subject : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Subject;
               is

                  Constituents : Iterator;
                  Constituent  : Symbol;

                  --------------------------------------------------------------------------------

                  procedure WriteConstituent (File        : in SPARK_IO.File_Type;
                                              Subject     : in Symbol;
                                              Constituent : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Constituent,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                Subject;
                  is
                  begin
                     case GetConstituentMode (Constituent) is
                        when DefaultMode =>
                           WriteString (File, "default");
                        when InMode =>
                           WriteString (File, "in");
                        when OutMode =>
                           WriteString (File, "out");
                        when InOutMode =>
                           WriteString (File, "in out");
                        when InvalidMode =>
                           WriteString (File, "invalid");
                     end case;
                     WriteString (File, " mode ");
                     WriteString (File, "constituent of ");
                     WriteName (File, Subject);
                     WriteString (File, " is ");
                     Write_Simple_Name (File => File,
                                        Item => Constituent);
                     if GetOwner (Subject) /= GetOwner (Constituent) then
                        WriteString (File, " declared in ");
                        WriteName (File, GetOwner (Constituent));
                     end if;
                     WriteLine (File, " ;");
                  end WriteConstituent;

                  --------------------------------------------------------------------------------

               begin
                  Constituents := FirstConstituent (Subject);
                  loop
                     exit when IsNullIterator (Constituents);
                     Constituent := CurrentSymbol (Constituents);
                     WriteConstituent (File, Subject, Constituent);
                     Constituents := NextSymbol (Constituents);
                  end loop;
               end WriteAbstractOwnVariable;

               --------------------------------------------------------------------------------

            begin
               AbstractOwnVariables := FirstAbstractOwnVariable (The_Package);
               loop
                  exit when IsNullIterator (AbstractOwnVariables);
                  WriteAbstractOwnVariable (File, CurrentSymbol (AbstractOwnVariables));
                  AbstractOwnVariables := NextSymbol (AbstractOwnVariables);
               end loop;
            end WriteAbstractOwnVariables;

            --------------------------------------------------------------------------------

            procedure Write_Subprogram_Bodies (File  : in SPARK_IO.File_Type;
                                               Scope : in Scopes)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Scope;
            is
               Declarative_Items : Iterator;
               Declarative_Item  : Symbol;
            begin

               Declarative_Items := First_Declarative_Item (Scope => Scope);

               loop
                  exit when IsNullIterator (Declarative_Items);
                  Declarative_Item := CurrentSymbol (Declarative_Items);
                  if IsSubprogram (Declarative_Item) then
                     Write_Global_Variables (Abstraction => IsRefined,
                                             File        => File,
                                             Subprogram  => Declarative_Item);
                     if IsProcedure (Declarative_Item) then
                        Write_Dependency_Clauses (Abstraction   => IsRefined,
                                                  File          => File,
                                                  The_Procedure => Declarative_Item);
                     end if;
                  end if;

                  Declarative_Items := NextSymbol (Declarative_Items);

               end loop;

            end Write_Subprogram_Bodies;

            --------------------------------------------------------------------------------

         begin
            Write_With_References (File  => File,
                                   Scope => VisibleScope (The_Package));
            Write_With_References (File  => File,
                                   Scope => LocalScope (The_Package));
            Write_Inherits_References (File             => File,
                                       Compilation_Unit => The_Package);
            WriteOwnVariables (File, The_Package);
            if Is_Generic_Package (The_Symbol => The_Package) then
               Write_Generic_Formal_Parameters (File        => File,
                                                The_Generic => The_Package);
            end if;
            WriteEmbeddedPackages (File, The_Package);
            WriteAbstractOwnVariables (File, The_Package);
            Write_Declarative_Items (File  => File,
                                     Scope => VisibleScope (The_Package));
            Write_Declarative_Items (File  => File,
                                     Scope => PrivateScope (The_Package));
            Write_Declarative_Items (File  => File,
                                     Scope => LocalScope (The_Package));
            Write_Subprogram_Bodies (File  => File,
                                     Scope => VisibleScope (The_Package));
            Write_Subprogram_Bodies (File  => File,
                                     Scope => LocalScope (The_Package));
            WriteLoops (File, The_Package);
            WriteOwnedPackages (File, The_Package);
         end Write_Package;

         --------------------------------------------------------------------------------

         procedure WriteLibraryPackage (File        : in SPARK_IO.File_Type;
                                        The_Package : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                The_Package;
         is
         begin
            if GetRootPackage (The_Package) /= GetPredefinedPackageAda then
               Write_Package_Info (File        => File,
                                   The_Package => The_Package);
               Write_Package (File        => File,
                              The_Package => The_Package);
            end if;
         end WriteLibraryPackage;

         --------------------------------------------------------------------------------

         procedure Write_Subprogram (File       : in SPARK_IO.File_Type;
                                     Subprogram : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Subprogram;
         is

            --------------------------------------------------------------------------------

            procedure Write_Subprogram_Info (File       : in SPARK_IO.File_Type;
                                             Subprogram : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Subprogram;
            is
            begin
               WriteString (File, "subprogram named ");
               Write_Simple_Name (File => File,
                                  Item => Subprogram);
               WriteString (File, " is ");
               if Is_Generic_Subprogram (The_Symbol => Subprogram) then
                  WriteString (File, "generic ");
               end if;
               if IsEntry (Subprogram) then
                  WriteString (File, "entry");
               elsif IsProcedure (Subprogram) then
                  WriteString (File, "procedure");
               else
                  if IsProofFunction (Subprogram) then
                     WriteString (File, "proof ");
                  end if;
                  WriteString (File, "function of ");
                  WriteName (File, GetType (Subprogram));
               end if;
               -- don't try and print declaration region for library-level units
               if not (IsMainProgram (Subprogram)
                         or else (Subprogram = GetThePartition)
                         or else Is_Generic_Subprogram (The_Symbol => Subprogram)) then
                  WriteString (File, " declared in ");
                  WriteScope (File, GetScope (Subprogram));
               end if;
               if RawDict.GetSubprogramInstantiationOf (Subprogram) /= NullSymbol then
                  WriteString (File, " and which is an instantiation of ");
                  WriteName (File, RawDict.GetSubprogramInstantiationOf (Subprogram));
               end if;
               if IsInterruptHandler (Subprogram) then
                  WriteString (File, " and is an interrupt handler");
               end if;
               if IsThePartition (Subprogram) then
                  WriteString (File, " and is the partition table of the whole program");
               end if;
               WriteLine (File, " ;");
            end Write_Subprogram_Info;

            --------------------------------------------------------------------------------

            procedure Write_Subprogram_Parameters (File       : in SPARK_IO.File_Type;
                                                   Subprogram : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Subprogram;
            is

               Parameter : Iterator;
               Number    : Positive;

               --------------------------------------------------------------------------------

               procedure Write_Subprogram_Parameter
                 (File       : in SPARK_IO.File_Type;
                  Subprogram : in Symbol;
                  Number     : in Positive;
                  Parameter  : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Number,
               --#                                Parameter,
               --#                                Subprogram;
               is
               begin
                  WriteString (File, "subprogram parameter # ");
                  WriteInteger (File, Number);
                  WriteString (File, " of ");
                  WriteName (File, Subprogram);
                  WriteString (File, " is ");
                  Write_Simple_Name (File => File,
                                     Item => Parameter);
                  WriteString (File, " which is ");
                  case GetSubprogramParameterMode (Parameter) is
                     when DefaultMode =>
                        WriteString (File, "default");
                     when InMode =>
                        WriteString (File, "in");
                     when OutMode =>
                        WriteString (File, "out");
                     when InOutMode =>
                        WriteString (File, "in out");
                     when InvalidMode =>
                        WriteString (File, "invalid");
                  end case;
                  WriteString (File, " parameter of type ");
                  WriteName (File, GetType (Parameter));
                  WriteLine (File, " ;");
               end Write_Subprogram_Parameter;

               --------------------------------------------------------------------------------

            begin
               Parameter := FirstSubprogramParameter (Subprogram);
               Number    := 1;
               loop
                  exit when IsNullIterator (Parameter);
                  Write_Subprogram_Parameter
                    (File       => File,
                     Subprogram => Subprogram,
                     Number     => Number,
                     Parameter  => CurrentSymbol (Parameter));
                  Parameter := NextSymbol (Parameter);
                  Number    := Number + 1;
               end loop;
            end Write_Subprogram_Parameters;

            --------------------------------------------------------------------------------

         begin
            Write_Subprogram_Info (File       => File,
                                   Subprogram => Subprogram);
            Write_With_References (File  => File,
                                   Scope => LocalScope (Subprogram));
            Write_Inherits_References (File             => File,
                                       Compilation_Unit => Subprogram);
            Write_Subprogram_Parameters (File       => File,
                                         Subprogram => Subprogram);
            Write_Global_Variables (Abstraction => IsAbstract,
                                    File        => File,
                                    Subprogram  => Subprogram);
            if IsProcedure (Subprogram) then
               Write_Dependency_Clauses (Abstraction   => IsAbstract,
                                         File          => File,
                                         The_Procedure => Subprogram);
            end if;
            if Is_Generic_Subprogram (The_Symbol => Subprogram) then
               Write_Generic_Formal_Parameters (File        => File,
                                                The_Generic => Subprogram);
            end if;
            Write_Declarative_Items (File  => File,
                                     Scope => LocalScope (Subprogram));
            WriteLoops (File, Subprogram);
         end Write_Subprogram;

         --------------------------------------------------------------------------------

      begin
         if IsPackage (Unit) then
            WriteLibraryPackage (File, Unit);
         else
            Write_Subprogram (File       => File,
                              Subprogram => Unit);
         end if;
      end Write_Library_Unit; -- Hidden body expected

      --------------------------------------------------------------------------------

   begin
      Library_Units := First_Library_Unit;
      loop
         exit when IsNullIterator (Library_Units);
         Write_Library_Unit (File => File,
                             Unit => CurrentSymbol (Library_Units));
         Library_Units := NextSymbol (Library_Units);
      end loop;
      -- If we are in Ravencar mode, then a pseudo procedure called main_program will have been
      -- created as a place to store partition-wide flow anno data.  We print these details out now.
      if GetThePartition /= NullSymbol then
         Write_Library_Unit (File => File,
                             Unit => GetThePartition);
      end if;
   end Write_Library_Units;

   --------------------------------------------------------------------------------

   procedure AppendViewerSpecificInfo (File : in SPARK_IO.File_Type)
   --# global in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from * &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File;
   is
      MaxLineLength : constant Positive := 512;
      subtype LineIndex is Positive range 1 .. MaxLineLength;
      subtype LineCount is Natural range 1 .. MaxLineLength;
      subtype Lines is String (LineIndex);
      Line          : Lines;
      Length        : LineCount;
      File_Status   : SPARK_IO.File_Status;
      TemporaryFile : SPARK_IO.File_Type;
   begin
      TemporaryFile := Dict.TemporaryFile;
      SPARK_IO.Reset (TemporaryFile, SPARK_IO.In_File, File_Status);
      if File_Status = SPARK_IO.Ok then
         loop
            exit when SPARK_IO.End_Of_File (TemporaryFile);
            SPARK_IO.Get_Line (TemporaryFile, Line, Length);
            SPARK_IO.Put_Line (File, Line, Length);
         end loop;
      end if;
      Dict.TemporaryFile := TemporaryFile;
   end AppendViewerSpecificInfo;

begin

   Local_File_Name := FileSystem.Case_Of_Files_For_Create (E_Str => File_Name);
   E_Strings.Create (File         => File,
                     Name_Of_File => Local_File_Name,
                     Form_Of_File => "",
                     Status       => File_Status);

   if File_Status = SPARK_IO.Ok then
      -- File := SPARK_IO.Standard_Output; -- useful debug statement
      Write_Library_Units (File => File);
      AppendViewerSpecificInfo (File);
      --# accept Flow, 10, File, "Expected ineffective assignment";
      SPARK_IO.Close (File, File_Status);
      --# end accept;
   end if;

   Status := File_Status;

end Write;
