-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--------------------------------------------------------------------------------
--  SLI
--
--  Purpose :
--
--  The goal of this work is to provide for SPARK a way to output the
--  SPARK Ada structure. This structure will be used to navigate
--  through the SPARK Ada source code using IDE like GPS, GNATbench or
--  Emacs. A similar output already exists for GNAT. The compiler
--  writes structure information about the Ada source code in an ALI
--  file. This file is later used to navigate into the Ada source code
--  with GPS, GNATbench or Emacs.
--
--  SPARK needs at least to provide the same level of navigation
--  facilities as we get from the GNAT Ada compiler. Related with the
--  SPARK Ada language (mainly the SPARK annotation), the following
--  functionalities should be provided:
--
--  Jump between the state (ASM) and the refinement of the state in
--  the body
--
--  Jump from the usage of a state to the definition of the state
--
--  Jump from the usage of a proof function to the declaration of this
--  proof function
--
--  Jump between the global/derives annotation to the global/derives
--  refined annotation in the body
--
--  Jump from a variable inside a SPARK annotation to the definition
--  of this variable in the Ada source code or to the definition of
--  the state in the SPARK annotation. The user should have the choice
--  between the 2 possibilities.
--
--  The SLI file should contain only symbols about one compilation
--  unit (the body and the spec but not the separates). Currently,
--  when the Examiner is analysing a body, all the separates
--  related to this body are not analysed at all. This makes a
--  difference in the definition of a compilation unit between the
--  GNAT compiler and the Examiner. The consequence is that, for
--  a package containing separates, we will get one ALI file but many
--  SLI files (one SLI file for the main body and one SLI file for
--  each separate). This needs to be supported by the IDE.
--
--  The format really hasn't changed from the ALI format. The SLI
--  reader needs a first V line, and dummy P and R lines which can be
--  left empty. It then needs a valid U line referencing the main file
--  being considered, and then the regular cross-reference format as
--  documented in Lib.XRef file, except that checksums and timestamps
--  for 'D' lines can be omitted.
--
--  For Ada source code navigation, the only relevant section are the
--  X section and the D section because the X section refers to the D
--  section. The format of the cross-reference section of the SLI file
--  is completely described in the package Lib.XRef of the GNAT
--  compiler.
--
--  Cross-references for SPARK annotations are similar to those
--  contained in the X section of the ALI file. For the SPARK
--  annotations functionalities, the cross-reference format used in
--  the ALI file can be reused.
--
--  The Generate_* procedures are called during the sematic
--  analysis. These procedures store all the cross-references in a
--  data structure. Increment_Nb_Separates is also called during the
--  sematic analysis to count the number of bodies declared as
--  separate in the compilation unit. After the semantic analysis of
--  the compilation unit, the SLI file is generated using the
--  following sequence : Create_File to create the SLI file, Header to
--  write the begining of the file, Dump_Xref to dump the data
--  structure containing all the cross-references, Close_File to close
--  the SLI file.
--------------------------------------------------------------------------------

with ContextManager;
with Dictionary;
with LexTokenManager;
with STree;

use type ContextManager.UnitDescriptors;
use type ContextManager.FileDescriptors;
use type ContextManager.UnitTypes;
use type Dictionary.Modes;
use type Dictionary.Symbol;
use type Dictionary.SLI_Type;
use type LexTokenManager.Token_Position;
use type STree.Iterator;
use type STree.SyntaxNode;

--# inherit CommandLineData,
--#         ContextManager,
--#         ContextManager.Ops,
--#         Dictionary,
--#         ErrorHandler,
--#         ExaminerConstants,
--#         E_Strings,
--#         FileSystem,
--#         IndexManager,
--#         LexTokenLists,
--#         LexTokenManager,
--#         SPARK_IO,
--#         SP_Symbols,
--#         STree,
--#         SystemErrors,
--#         Version,
--#         XMLReport;

package SLI
--# own State;
--# initializes State;
is

   --  Create the SLI file associated with the compilation unit
   --  (File_Descriptor).
   procedure Create_File (File_Descriptor : in ContextManager.FileDescriptors);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys,
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                ContextManager.Ops.File_Heap,
   --#                                File_Descriptor,
   --#                                LexTokenManager.State;

   --  Close the SLI file.
   procedure Close_File;
   --# global in out State;
   --# derives State from *;

   --  Write the header of the SLI file associated with the
   --  compilation unit (File_Descriptor).
   procedure Header (File_Descriptor : in ContextManager.FileDescriptors);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ContextManager.Ops.Unit_Heap;
   --#        in out ErrorHandler.Error_Context;
   --#        in out IndexManager.State;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --#        in out XMLReport.State;
   --# derives ContextManager.Ops.Unit_Heap from * &
   --#         ErrorHandler.Error_Context,
   --#         IndexManager.State,
   --#         LexTokenManager.State,
   --#         SPARK_IO.File_Sys            from CommandLineData.Content,
   --#                                           ContextManager.Ops.Unit_Heap,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.Error_Context,
   --#                                           File_Descriptor,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           SPARK_IO.File_Sys,
   --#                                           STree.Table &
   --#         State                        from *,
   --#                                           CommandLineData.Content,
   --#                                           ContextManager.Ops.File_Heap,
   --#                                           ContextManager.Ops.Unit_Heap,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.Error_Context,
   --#                                           File_Descriptor,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           SPARK_IO.File_Sys,
   --#                                           STree.Table,
   --#                                           XMLReport.State &
   --#         XMLReport.State              from *,
   --#                                           CommandLineData.Content;

   --  Write all the cross-references for the compilation unit
   --  associated with the file descriptor (File_Descriptor) and the
   --  closure in the SLI file.
   procedure Dump_Xref (File_Descriptor : in ContextManager.FileDescriptors);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         Dictionary.Dict,
   --#                                         File_Descriptor,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         File_Descriptor,
   --#                                         LexTokenManager.State,
   --#                                         State,
   --#                                         STree.Table &
   --#         State                      from *,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         Dictionary.Dict,
   --#                                         File_Descriptor,
   --#                                         LexTokenManager.State;

   --  Try to find the symbol declaration associated with the name
   --  (Lex_Str) at position (Pos) in the dictionary giving a prefix
   --  (Prefix), a procedure/function name (Subprog_Sym) and a scope
   --  (Scope). Prefix contains the symbol declaration if found.
   procedure Look_Up
     (Prefix            : in out Dictionary.Symbol;
      Scope             : in     Dictionary.Scopes;
      Subprog_Sym       : in     Dictionary.Symbol;
      Lex_Str           : in     LexTokenManager.Lex_String;
      Pos               : in     LexTokenManager.Token_Position;
      Full_Package_Name : in     Boolean);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Prefix            from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                Full_Package_Name,
   --#                                LexTokenManager.State,
   --#                                Lex_Str,
   --#                                Scope,
   --#                                Subprog_Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                Full_Package_Name,
   --#                                LexTokenManager.State,
   --#                                Lex_Str,
   --#                                Pos,
   --#                                Prefix,
   --#                                Scope,
   --#                                Subprog_Sym;

   --  Add cross-references for the inherit annotation (Parse_Tree)
   --  giving a scope (Scope). The inherit annotation is located in
   --  the compilation unit (Comp_Unit).
   procedure Generate_Xref_Inherit
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the property list annotation
   --  (Parse_Tree) giving a scope (Scope). The property list
   --  annotation is located in the compilation unit (Comp_Unit).
   procedure Generate_Xref_Interrupt
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the property list annotation
   --  (Parse_Tree) giving a scope (Scope). The property list
   --  annotation is located in the compilation unit (Comp_Unit).
   procedure Generate_Xref_Suspends_Protects
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the own annotation (Parse_Tree) giving
   --  a scope (Scope). The own annotation is located in the
   --  compilation unit (Comp_Unit).
   procedure Generate_Xref_Own
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the refinement annotation (Parse_Tree)
   --  giving a scope (Scope). The refinement annotation is located in
   --  the compilation unit (Comp_Unit).
   procedure Generate_Xref_Refinement
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the initializes annotation
   --  (Parse_Tree) giving a scope (Scope). The initializes annotation
   --  is located in the compilation unit (Comp_Unit).
   procedure Generate_Xref_Initializes
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the global annotation (Parse_Tree)
   --  giving a procedure/function name (Subprog_Sym) and a scope
   --  (Scope). The global annotation is located in the compilation
   --  unit (Comp_Unit).
   procedure Generate_Xref_Global
     (Comp_Unit   : in ContextManager.UnitDescriptors;
      Parse_Tree  : in STree.SyntaxNode;
      Scope       : in Dictionary.Scopes;
      Subprog_Sym : in Dictionary.Symbol);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table,
   --#                                Subprog_Sym;

   --  Add cross-references for the derives annotation (Parse_Tree)
   --  giving a procedure/function name (Subprog_Sym) and a scope
   --  (Scope). The derives annotation is located in the compilation
   --  unit (Comp_Unit).
   procedure Generate_Xref_Derives
     (Comp_Unit   : in ContextManager.UnitDescriptors;
      Parse_Tree  : in STree.SyntaxNode;
      Scope       : in Dictionary.Scopes;
      Subprog_Sym : in Dictionary.Symbol);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table,
   --#                                Subprog_Sym;

   --  Add cross-references for the justification annotation
   --  (Parse_Tree) giving a scope (Scope). The justification
   --  annotation is located in the compilation unit (Comp_Unit).
   procedure Generate_Xref_Justification
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the formal parameters of a proof
   --  function annotation (Parse_Tree) giving a procedure/function
   --  name (Subprog_Sym) and a scope (Scope). The derives annotation
   --  is located in the compilation unit (Comp_Unit).
   procedure Generate_Xref_Proof_Function
     (Comp_Unit   : in ContextManager.UnitDescriptors;
      Parse_Tree  : in STree.SyntaxNode;
      Scope       : in Dictionary.Scopes;
      Subprog_Sym : in Dictionary.Symbol);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table,
   --#                                Subprog_Sym;

   --  Add cross-references for the object assertion annotation
   --  (Parse_Tree) giving a scope (Scope). The justification
   --  annotation is located in the compilation unit (Comp_Unit).
   procedure Generate_Xref_Object_Assertion
     (Comp_Unit  : in ContextManager.UnitDescriptors;
      Parse_Tree : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                State,
   --#                                STree.Table &
   --#         State             from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                Scope,
   --#                                STree.Table;

   --  Add cross-references for the symbol (Symbol) in a predicate
   --  annotation (precondition, postcondition, return, assert and
   --  check). The predicate annotation is located in the compilation
   --  unit (Comp_Unit). The identifier of the symbol (Symbol) is in
   --  the subtree (Parse_Tree).
   procedure Generate_Xref_Symbol
     (Comp_Unit      : in ContextManager.UnitDescriptors;
      Parse_Tree     : in STree.SyntaxNode;
      Symbol         : in Dictionary.Symbol;
      Is_Declaration : in Boolean);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.File_Heap,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                Is_Declaration,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                State,
   --#                                STree.Table,
   --#                                Symbol &
   --#         State             from *,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                Dictionary.Dict,
   --#                                Is_Declaration,
   --#                                Parse_Tree,
   --#                                STree.Table,
   --#                                Symbol;

   --  Cleanup all the cross-references tables.
   procedure Cleanup;
   --# global in out State;
   --# derives State from *;

   --  Increment by the number of separates found in the subtree
   --  (Parse_Tree) of the compilation unit (Comp_Unit).
   procedure Increment_Nb_Separates (Comp_Unit  : in ContextManager.UnitDescriptors;
                                     Parse_Tree : in STree.SyntaxNode);
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out SPARK_IO.File_Sys;
   --#        in out State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                ContextManager.Ops.Unit_Heap,
   --#                                LexTokenManager.State,
   --#                                Parse_Tree,
   --#                                STree.Table &
   --#         State             from *,
   --#                                Comp_Unit,
   --#                                Parse_Tree,
   --#                                STree.Table;

end SLI;
