/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2004, 2005  Clifford Wolf <clifford@clifford.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  mod_environ.c: A module for accessing environment variables
 */

/**
 * A module for accessing environment variables
 */

#include <stdlib.h>
#include "spl.h"
#include "compat.h"

/**
 * This namespace contains all environment variables. It is possible to create,
 * modify and remove entries:
 *
 *	environ.PATH ~= ":${environ.HOME}/bin";
 *
 */
// namespace environ;

extern void SPL_ABI(spl_mod_environ_init)(struct spl_vm *vm, struct spl_module *mod, int restore);
extern void SPL_ABI(spl_mod_environ_done)(struct spl_vm *vm, struct spl_module *mod);

static void handler_environ(struct spl_task *task UNUSED, struct spl_vm *vm UNUSED,
		struct spl_node *node UNUSED, struct spl_hnode_args *args, void *data UNUSED)
{
	if (args->action == SPL_HNODE_ACTION_LOOKUP)
	{
		const char *key = args->key;
		char *env_name = spl_hash_decode(key);
		char *value = getenv(env_name);

		if (value)
			args->value = SPL_NEW_STRING_DUP(value);

		free(env_name);
		return;
	}

	if (args->action == SPL_HNODE_ACTION_CREATE)
	{
		const char *key = args->key;
		char *env_name = spl_hash_decode(key);
#ifdef USEIRIXAPI
		char *env;
		my_asprintf(&env, "%s=%s", env_name, spl_get_string(args->value));
		putenv(env);
#else
		setenv(env_name, spl_get_string(args->value), 1);
#endif
		free(env_name);
		return;
	}

	if (args->action == SPL_HNODE_ACTION_DELETE)
	{
		const char *key = args->key;
		char *env_name = spl_hash_decode(key);
#ifdef USEIRIXAPI
		putenv(env_name);
#else
		unsetenv(env_name);
		free(env_name);
#endif
		return;
	}
}

void SPL_ABI(spl_mod_environ_init)(struct spl_vm *vm, struct spl_module *mod, int restore)
{
	spl_hnode_reg(vm, "environ", handler_environ, 0);

	if ( !restore )
		spl_hnode(vm, vm->root, "environ", "environ", mod);
}

void SPL_ABI(spl_mod_environ_done)(struct spl_vm *vm UNUSED, struct spl_module *mod UNUSED)
{
	return;
}

