// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  parse.h
  Header file for the starconvert utility; contains lots of necessary structs.
*/

#ifndef _PARSE_H
#define _PARSE_H

#include <fstream>
#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <cmath>
#include <vector>
#include <iostream>
using namespace std;

#include "../classes/stringops.h"
#include "../classes/specclass.h"
#include "../classes/mathops.h"

// Lots of typedef'd structs in which to store the results of parsing the
//  configuration file

struct comments {
  int comments_start, comments_len;
};

const unsigned int NUM_COORD_OPTIONS = 7;
#ifndef _CONVERT_H
static const char * coordoptions[NUM_COORD_OPTIONS][2] =
{ { "ra-hours", "long-deg" },
  { "ra-min",   "long-min" },
  { "ra-sec",   "long-sec" },
  { "dec-sign", "lat-sign" },
  { "dec-deg",  "lat-deg"  },
  { "dec-min",  "lat-min"  },
  { "dec-sec",  "lat-sec"  }
};
#endif
// The above #ifndef prevents coordoptions from being unnecessarily defined
//  in the convert.cc file.  (I'm one of those people who hate to see
//  compiler warnings.)

struct coordinates {
  bool isCelestial;
  int coord_start[NUM_COORD_OPTIONS], coord_len[NUM_COORD_OPTIONS];
};

const unsigned int NUM_DISTANCE_UNITS = 5;
#ifndef _CONVERT_H
static const char * distanceunits[NUM_DISTANCE_UNITS] =
{ "milliarcsec", "arcsec", "pc", "ly", "specclass" };
#endif
enum eunittype { MILLIARCSEC, ARCSEC, PC, LY, SPECCLASS };
// The above char *[] and enum must be kept in sync!
//  The #ifndef above prevents distanceunits from being unnecessarily defined
//  in the convert.cc file.

struct stardistance {
  eunittype type;
  int start, len;
  int errorstart, errorlen;
  double minparallax, maxerror;
};

enum emagtype { ABSOLUTE, VISUAL };

struct magnitude {
  emagtype type;
  int start, len;
};

struct characteristics {
  vector<int> specclass_start;
  vector<int> specclass_len;
  vector<stardistance> distarray;
  vector<magnitude> magarray;

  characteristics() : 
    specclass_start(), specclass_len(), distarray(), magarray() { }
};

struct systems {
  int comp_start, comp_len;
  bool isSeparationCommented;
  int sep_start, sep_len;
  StringList sep_prefixes;

  systems() : sep_prefixes() { }
};

const unsigned int NUM_NAME_TYPES = 5;
#ifndef _CONVERT_H
static const char * nametypes[NUM_NAME_TYPES] =
{ "bayer", "flamsteed", "const-specific", "dm", "other" };
#endif
enum enametype { BAYER, FLAMSTEED, CONSTSPECIFIC, DM, OTHER };
// The above char *[] and enum must be kept in sync!
//  The #ifndef above prevents nametypes from being unnecessarily defined
//  in the convert.cc file.

struct name {
  enametype type;
  bool isNameCommented;
  int name_start, name_len;
  StringList name_prefixes;

  name() : name_prefixes() { }
};

struct namedata {
  vector<name> names;
  bool isSubstCaseSensitive;
  StringList subst1, subst2;
 
  namedata() : names(), subst1(), subst2() { }
};

struct parsedata {
  comments Comments;
  coordinates Coord;
  characteristics Charact;
  systems Systems;
  namedata Names;

  parsedata() : Comments(), Coord(), Charact(), Systems(), Names() { }
};

enum parsetype { NONE, COORDS, CHARACT, SYSTEMS, NAMES, SUBST };

// External function declarations for the parsing file

extern void parse_config_file(istream &, parsedata *);
extern double sigdigits(double, unsigned int);
extern double findabsmag(const char *);

#endif // #ifndef _PARSE_H
