// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  filedialogs.cc
  This file contains code for the dialog boxes in the StarPlot File menu,
  and also code for reading to / writing from the .starplotrc settings file.
*/

#include <gtk/gtk.h>
#include "starplot.h"

bool read_params(const char *filename);
bool write_params(const char *filename, const char *header);
void my_gtk_file_select (void (*)(GtkWidget *, GtkFileSelection *),
			 const char *);

void starplot_quit()
{
  delete viewer;
  delete stararray;
  write_rc_file();
  gtk_main_quit();
  return;
}


/* File selection dialogues */
/* These cover selection of data / parameter files to load / save, and also
 *  the selection of the help browser program. */

/* This opens a new file */
void file_ok_sel( GtkWidget        *w,
		  GtkFileSelection *fs )
{
  starrules.ChartFileNames = StringList((char *)gtk_file_selection_get_filename
					(GTK_FILE_SELECTION (fs)));
  redraw_all(FILE_CHANGE);
}

/* This merges the selected file with the other(s) already open */
void merge_ok_sel( GtkWidget        *w,
		   GtkFileSelection *fs )
{
  const char *filename = 
    (char *)gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));
  				
  // make sure the file we want to merge isn't already open
  //  (not foolproof due to the possibility of symlinks, etc., but better
  //  than nothing)
  for (unsigned int i = 0; i < starrules.ChartFileNames.size(); i++)
    if (strcmp(starrules.ChartFileNames[i], filename) == 0) return;

  starrules.ChartFileNames.append(filename);
  redraw_all(FILE_CHANGE);
}

/* This opens a new set of chart parameters */
void param_open_sel( GtkWidget        *w,
		     GtkFileSelection *fs )
{
  const char *filename =
    (char *)gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));
  if (read_params(filename))
    redraw_all(FILE_CHANGE);
}

/* This saves the current chart parameters to a file */
void param_save_sel( GtkWidget        *w,
		     GtkFileSelection *fs )
{
  const char *filename =
    (char *)gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));
  write_params(filename, "# StarPlot Parameter File\n");
}


/* This selects a help file browser */
void help_ok_sel( GtkWidget        *w,
		  GtkFileSelection *fs )
{
  const char *filename =
    (char *)gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));

  delete [] help_browser;
  help_browser = new char [strlen(filename) + 1];
  strcpy(help_browser, filename);
}

/* This selects a help file browser and also opens the help documentation */
void help_open_sel( GtkWidget        *w,
		    GtkFileSelection *fs )
{
  help_ok_sel(w, fs);
  helpdocs();
}


/* We need several functions for the menu generator, plus one for the case
 *  where a help browser must be selected before opening help docs,
 *  but they all call the same interface, just choose the correct file
 *  selection function */
void file_open()
{ my_gtk_file_select(file_ok_sel, "StarPlot: Select data file to open"); }
void file_merge()  
{ my_gtk_file_select(merge_ok_sel, "StarPlot: Select data file to merge"); }
void params_open()
{ my_gtk_file_select(param_open_sel,"StarPlot: Select parameter file to use");}
void params_save()
{ my_gtk_file_select(param_save_sel, "StarPlot: Save chart parameters as:"); }
void help_select() 
{ my_gtk_file_select(help_ok_sel, "StarPlot: Select help browser program"); }
void help_select_and_open()
{ my_gtk_file_select(help_open_sel, "StarPlot: Select help browser program"); }


/* The interface which creates the dialog */
void my_gtk_file_select 
(void (*select_function)(GtkWidget *, GtkFileSelection *), const char *title)
{ 
  GtkWidget *filew;

  filew = gtk_file_selection_new (title);
  gtk_window_set_modal (GTK_WINDOW (filew), TRUE);
  gtk_signal_connect (GTK_OBJECT (filew), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroy), 
		      GTK_WIDGET (filew));

  /* Connect the ok_button to the appropriate file selection function */
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filew)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC (select_function), filew);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION
					 (filew)->ok_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filew));

  /* Connect the cancel_button to destroy the widget */
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION
					 (filew)->cancel_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filew));

  /* Set the default filename */
  if (select_function == help_ok_sel || select_function == help_open_sel) {
    if (!help_browser)
      gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew),
				       "/usr/bin/");
    else
      gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew),
				       help_browser);
  }
  else if (select_function == param_open_sel
	   || select_function == param_save_sel)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew), "~/");
  else
    gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew), DATADIR "/");

  gtk_widget_show(filew);
}


/* Saving as PNG is not yet implemented: */
void file_save() { }


// read_params() : Read a StarPlot parameters file.  Note that this function
//  will have problems if any of the filenames stored in ~/.starplotrc contain
//  trailing spaces or any of these characters:  # = ( ) [ ] { }
//  I think we can live with this limitation.

bool read_params(const char *filename)
{
  ifstream paramstream;
  paramstream.open(filename);
  if (!paramstream.good()) return false;

  int files = 0;
  char line[1000];
  while (paramstream.getline(line, 1000)) {
    // delete comments and munch all kinds of parentheses (which are used
    //  only to make the RC file entries look nicer)
    for (unsigned int i = 0; i < strlen(line); i++) {
      if (line[i] == '#') line[i] = 0;
      if (line[i] == '(' || line[i] == ')' || line[i] == '[' || line[i] == ']'
	  || line[i] == '{' || line[i] == '}')
	line[i] = ' ';
    }

    StringList assign = StringList(line, '=');
    if (assign.size() < 2) continue;
    assign.stripspace();

    if (strcmp("DataFile", assign[0]) == 0) {
      if (files == 0) starrules.ChartFileNames = StringList(assign[1]);
      else            starrules.ChartFileNames.append(assign[1]);
      files++;
    }
    else if (strcmp("Location", assign[0]) == 0) {
      StringList coords = StringList(assign[1], ',');
      if (coords.size() != 3) continue;
      coords.stripspace();
      starrules.ChartLocation = Vector(myatof(coords[0]), myatof(coords[1]),
				       myatof(coords[2]));
    }
    else if (strcmp("Orientation", assign[0]) == 0) {
      StringList orient = StringList(assign[1], ',');
      if (orient.size() != 2) continue;
      orient.stripspace();
      starrules.ChartOrientation = SolidAngle(myatof(orient[0]),
					      myatof(orient[1]));
    }
    else if (strcmp("Radius", assign[0]) == 0)
      starrules.ChartRadius = myatof(assign[1]);
    else if (strcmp("MagLimits", assign[0]) == 0) {
      StringList maglim = StringList(assign[1], ',');
      if (maglim.size() != 2) continue;
      maglim.stripspace();
      starrules.ChartBrightestMagnitude = myatof(maglim[0]);
      starrules.ChartDimmestMagnitude = myatof(maglim[1]);
    }
    else if (strcmp("HRMagLimits", assign[0]) == 0) {
      StringList maglim = StringList(assign[1], ',');
      if (maglim.size() != 2) continue;
      maglim.stripspace();
      hr_brightmag = myatof(maglim[0]);
      hr_dimmag = myatof(maglim[1]);
    }
    else if (strcmp("AutosetDimMag", assign[0]) == 0)
      starrules.ChartAutosetDimmestMagnitude = (bool)myatoi(assign[1]);
    else if (strcmp("SpecClassDisplay", assign[0]) == 0) {
      StringList classes = StringList(assign[1], ',');
      if (classes.size() != 10) continue;
      classes.stripspace();
      for (unsigned int i = 0; i < 10; i++)
	starrules.StarClasses[i] = (bool)myatoi(classes[i]);
    }
    else if (strcmp("CelestialCoords", assign[0]) == 0)
      set_coord_item((bool)myatoi(assign[1]), false);
    else if (strcmp("StarDiameters", assign[0]) == 0)
      set_diameter_item((Diametertype)myatoi(assign[1]), false);
    else if (strcmp("StarLabels", assign[0]) == 0)
      set_label_item((Labeltype)myatoi(assign[1]), false);
    else if (strcmp("StarBars", assign[0]) == 0)
      set_toggle_item(&starrules.StarBars, (bool)myatoi(assign[1]),
		      TOGGLE_BARS, false);
    else if (strcmp("ChartGrid", assign[0]) == 0)
      set_toggle_item(&starrules.ChartGrid, (bool)myatoi(assign[1]),
		      TOGGLE_GRID, false);
    else if (strcmp("ChartLegend", assign[0]) == 0)
      set_toggle_item(&starrules.ChartLegend, (bool)myatoi(assign[1]),
		      TOGGLE_LEGEND, false);
    else if (strcmp("HelpBrowser", assign[0]) == 0) {
      delete [] help_browser;
      help_browser = new char [strlen(assign[1]) + 1];
      strcpy(help_browser, assign[1]);
    }
  }

  return true;
}


/* write_params() : Save the current StarPlot parameters to a file. */

bool write_params(const char *filename, const char *header)
{
  ofstream paramstream;
  paramstream.open(filename);
  if (!paramstream.good()) return false;

  paramstream << header;
  paramstream << "# If you edit this file, note that distances are in "
	      << "light-years" << endl << "# and angles are in RADIANS."
	      << endl << endl;

  for (unsigned int i = 0; i < starrules.ChartFileNames.size(); i++)
    paramstream << "DataFile = " << starrules.ChartFileNames[i] << endl;

  paramstream << "Location = (" << starrules.ChartLocation.getX() << ","
	      << starrules.ChartLocation.getY() << ","
	      << starrules.ChartLocation.getZ() << ")" 
	      << " # (X,Y,Z) of chart center, in light-years" << endl;

  paramstream << "Orientation = (" << starrules.ChartOrientation.getPhi()
	      << "," << starrules.ChartOrientation.getTheta() << ")" 
	      << " # (RA,Dec) of orientation vector, in radians" << endl;

  paramstream << "Radius = " << starrules.ChartRadius << endl;
  paramstream << "MagLimits = [" << starrules.ChartBrightestMagnitude << ","
	      << starrules.ChartDimmestMagnitude << "]" << endl;
  paramstream << "HRMagLimits = [" << hr_brightmag << "," << hr_dimmag << "]"
	      << endl;
  paramstream << "AutosetDimMag = " << starrules.ChartAutosetDimmestMagnitude
	      << endl;

  paramstream << "SpecClassDisplay = {";
  for (unsigned int i = 0; i < 9; i ++)
    paramstream << starrules.StarClasses[i] << ",";
  paramstream << starrules.StarClasses[9] << "}"
	      << " # {O,B,A,F,G,K,M,D,unclass,non-stell}" << endl;

  paramstream << "CelestialCoords = " << starrules.CelestialCoords << endl;
  paramstream << "StarDiameters = " << starrules.StarDiameters << endl;
  paramstream << "StarLabels = " << starrules.StarLabels << endl;
  paramstream << "StarBars = " << starrules.StarBars << endl;
  paramstream << "ChartGrid = " << starrules.ChartGrid << endl;
  paramstream << "ChartLegend = " << starrules.ChartLegend << endl;

  if (help_browser)
    paramstream << "HelpBrowser = " << help_browser << endl;

  return true;
}



// read_rc_file() : Called when the program begins.  

bool read_rc_file()
{
  const char *homedir = getenv("HOME");
  char * rcfile = 
    new char [strlen(homedir) + strlen("/.starplotrc") + 1];

  strcpy(rcfile, homedir);
  strcat(rcfile, "/.starplotrc");
  return read_params(rcfile);
}


// write_rc_file() : Called as the program is exiting.

bool write_rc_file()
{
  const char *homedir = getenv("HOME");
  char * rcfile =
    new char [strlen(homedir) + strlen("/.starplotrc") + 1];
  char * header = "# StarPlot user settings file\n"
    "# Read by StarPlot when starting and rewritten automatically on "
    "program exit.\n";

  strcpy(rcfile, homedir);
  strcat(rcfile, "/.starplotrc");
  return write_params(rcfile, header);
}
