// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  infodialogs.cc
  This file contains code for the dialog boxes in the StarPlot Stars menu,
  and for the informational popup window.
*/

#include <gtk/gtk.h>
#include "starplot.h"


/* This function displays a popup window of information about the star: */

void my_gtk_star_popup(Star &s)
{
  GtkWidget *popup, *OK, *topbox, *label[6];
  StringList starinfo = s.GetInfo(starrules);
  char Name[25], RA[25], Dec[25], Distance[25], Class[25], Mag[25];

  if (starrules.StarLabels == NUMBER_LABEL)
    snprintf(Name, 24, "%s. %s", starinfo[0], starinfo[1]);
  else
    snprintf(Name, 24, "%s", starinfo[1]);
  if (starrules.CelestialCoords) {
    snprintf(RA, 24, "RA: %s", starinfo[2]);
    snprintf(Dec, 24, "Dec: %s", starinfo[3]);
  }
  else {
    snprintf(RA, 24, "Long: %s", starinfo[2]);
    snprintf(Dec, 24, "Lat: %s", starinfo[3]);
  }
  snprintf(Distance, 24, "Dist: %s LY", starinfo[4]);
  snprintf(Class, 24, "Class: %s", starinfo[5]);
  snprintf(Mag, 24, "Abs. Magnitude: %s", starinfo[6]);
  
  popup = gtk_dialog_new ();
  gtk_window_set_policy (GTK_WINDOW (popup), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (popup), "StarPlot");
  gtk_window_set_position (GTK_WINDOW (popup), GTK_WIN_POS_MOUSE);
  
  topbox = gtk_vbox_new (TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (topbox), 10);
  gtk_widget_show (topbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (popup)->vbox), topbox,
		      TRUE, TRUE, 0);
  
  label[0] = gtk_label_new (Name);
  label[1] = gtk_label_new (RA);
  label[2] = gtk_label_new (Dec);
  label[3] = gtk_label_new (Distance);
  label[4] = gtk_label_new (Class);
  label[5] = gtk_label_new (Mag);

  for (unsigned int i = 0; i < 6; i++) {
    gtk_box_pack_start (GTK_BOX (topbox), label[i], TRUE, TRUE, 0);
    gtk_misc_set_alignment(GTK_MISC (label[i]), (float)0.0, (float)0.0);
    gtk_widget_show (label[i]);
  }

  my_gtk_button_bar (&OK, NULL, NULL, GTK_DIALOG (popup)->action_area);
  my_gtk_buttons_connect_destroy (OK, NULL, NULL, popup);
  
  gtk_window_set_focus (GTK_WINDOW (popup), OK);
  gtk_widget_show (popup);
}


/* Here is a function which sets up the column list for starinfo() and
 *  starsearch().  "box" is the box into which the column list should
 *  be packed. */

void my_gtk_list_setup (GtkWidget *box, GtkWidget **clistptr,
			bool searchclist)
{
  GtkWidget *scrolled_window;
  GtkWidget *clist;
  gchar *titles[7];

  // set column titles for the information
  for (unsigned int i = 0; i < 7; i++)
    titles[i] = new gchar[25];

  unsigned int i = 0;
  strcpy(titles[i++], "Star");
  if (searchclist)
    strcpy(titles[i++], "Also known as");
  if (starrules.CelestialCoords) {
    strcpy(titles[i++], "Right Ascension");
    strcpy(titles[i++], "Declination");
  }
  else {
    strcpy(titles[i++], "Gal. Longitude");
    strcpy(titles[i++], "Gal. Latitude");
  }
  strcpy(titles[i++], "Distance");
  strcpy(titles[i++], "Class");
  strcpy(titles[i], "Mag.");

  // set up a viewport with scroll bars
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_box_pack_start (GTK_BOX (box), scrolled_window, TRUE, TRUE, 0);
  gtk_widget_show (scrolled_window);

  // create a column list
  *clistptr = gtk_clist_new_with_titles (searchclist ? 7 : 6, titles);
  clist = *clistptr;
  gtk_widget_set_usize (GTK_WIDGET (clist), 565, 300);
  gtk_container_add (GTK_CONTAINER (scrolled_window), clist);
  gtk_widget_show (clist);
  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_OUT);

  i = 0;
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 120);
  if (searchclist)
    gtk_clist_set_column_width (GTK_CLIST (clist), i++, 120);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 100);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 100);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 80);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 65);
  gtk_clist_set_column_width (GTK_CLIST (clist), i, 40);
  gtk_clist_set_column_justification (GTK_CLIST (clist), i - 2,
				      GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_justification (GTK_CLIST (clist), i, GTK_JUSTIFY_RIGHT);

  return;
}


/* Callback to destroy info window dialog */

GtkWidget *infowindow = NULL;
GtkWidget *info_clist = NULL;

void info_quit()
{
  gtk_widget_destroy(infowindow);
  infowindow = NULL;
  info_clist = NULL;
  return;
}

/* Function to update the chart information when necessary */

void update_info(GtkWidget *clist)
{
  gchar *information[6];

  if (!clist) return;
  gtk_clist_clear (GTK_CLIST (clist));
  gtk_clist_freeze (GTK_CLIST (clist));

  for (unsigned int i = 0; i < 6; i++)
    information[i] = new gchar[25];
  
  for (unsigned int i = 0; i < stararray->size(); i++) {
    StringList infolist = (*stararray)[i].GetInfo(starrules);
    if (starrules.StarLabels == NUMBER_LABEL)
      snprintf(information[0], 24, "%s. %s", infolist[0], infolist[1]);
    else
      snprintf(information[0], 24, "%s", infolist[1]);
    for (unsigned int j = 1; j < 6; j++)
      snprintf(information[j], 24, "%s", infolist[j + 1]);
    strcat(information[3], " LY ");
    gtk_clist_append (GTK_CLIST (clist), information);
  }
  for (unsigned int i = 0; i < 6; i++)
    delete [] information[i];

  gtk_clist_thaw (GTK_CLIST (clist));
  return;
}  

/* This function displays information about the stars shown on the chart. */

void starinfo()
{
  GtkWidget *vbox, *OK;

  // If the info window is already open, do nothing but put it in front
  if (infowindow) {
    gdk_window_raise(infowindow->window);
    return;
  }

  // make a window for it
  infowindow = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_title (GTK_WINDOW (infowindow), "StarPlot: Chart Data");
  gtk_signal_connect (GTK_OBJECT (infowindow), "destroy", 
		      GTK_SIGNAL_FUNC (info_quit), NULL);

  // make a vbox
  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (infowindow), vbox);
  gtk_widget_show (vbox);

  // set up the column list and put star information into it
  my_gtk_list_setup (vbox, &info_clist, 
		     false /*chart data, not search results*/);
  update_info(info_clist);
  
  // the button to make the window go away
  my_gtk_button_bar (&OK, NULL, NULL, vbox);
  my_gtk_buttons_connect_destroy (OK, NULL, NULL, infowindow);

  gtk_window_set_focus (GTK_WINDOW (infowindow), OK);
  gtk_widget_show (infowindow);
  return;
}


/* These functions allow the user to look up star names containing a string.
 *  (admittedly, doing a `grep' of the star data files would work just as
 *  well...) */

struct sSearchData {
  GtkWidget *casesensitive;
  GtkWidget *entrybox;
  GtkWidget *sclist;
};

void search_callback(GtkWidget *widget, gpointer search_data)
{
  sSearchData data = * (sSearchData *)search_data;
  GtkWidget *sclist = data.sclist;
  char *searchstring = gtk_entry_get_text (GTK_ENTRY (data.entrybox));
  bool casesensitive = 
    (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data.casesensitive))
     == TRUE);
  StarArray searchresults;
  gchar *information[7];

  if (isempty(searchstring)) return;

  gtk_clist_clear (GTK_CLIST (sclist));
  searchresults.Search(searchstring, starrules.ChartFileNames,
		       starrules, casesensitive);

  // error checking
  if (searchresults.size() >= MAX_STARS) {
    gchar *toomany1[7] = {"** Showing only first", "", "", "", "", "", ""};
    gchar *toomany2[7];
    toomany2[0] = new gchar[20];
    snprintf(toomany2[0], 20, "** %d results", MAX_STARS);
    for (unsigned int i = 1; i < 7; i++) {
      toomany2[i] = new gchar[1];
      toomany2[i][0] = 0;
    }
    gtk_clist_append (GTK_CLIST (sclist), toomany1);
    gtk_clist_append (GTK_CLIST (sclist), toomany2);
    for (unsigned int i = 0; i < 7; i++) delete [] toomany2[i];
  }
  else if (!searchresults.size()) {
    gchar *noresults[7] = { "Sorry, nothing found", "", "", "", "", "", "" };
    gtk_clist_append (GTK_CLIST (sclist), noresults);
    return;
  }

  for (unsigned int i = 0; i < 7; i++)
    information[i] = new gchar[25];

  for (unsigned int i = 0; i < searchresults.size(); i++) {
    StringList infolist = searchresults[i].GetInfo(starrules);

    // where is the star name which matches the search string in the
    //  list of star characteristics?
    unsigned int nameplace = myatoi(infolist[0]);
    unsigned int infolistplace = nameplace ? nameplace + 9 : 1;
    snprintf(information[0], 24, "%s", infolist[infolistplace]);

    // if this is not the best-known star name, put that in the "aka" column;
    //  else put the second-best-known star name in that column
    snprintf(information[1], 24, "%s", nameplace ? infolist[1] : infolist[10]);

    for (unsigned int j = 2; j < 7; j++)
      snprintf(information[j], 24, "%s", infolist[j]);
    strcat(information[4], " LY ");
    gtk_clist_append (GTK_CLIST (sclist), information);
  }
  for (unsigned int i = 0; i < 7; i++)
    delete [] information[i];

  return;
}

/* Callback to destroy search window dialog */

GtkWidget *searchwindow = NULL;

void search_quit()
{
  gtk_widget_destroy(searchwindow);
  searchwindow = NULL;
  return;
}

/* create the search window dialog */

void starsearch()
{
  GtkWidget *vbox, *OK, *sclist;
  GtkWidget *hbox, *casesensitive, *entrybox, *searchbutton;
  static sSearchData search_data;
   
  // If the search window is already open, do nothing but put it in front
  if (searchwindow) {
    gdk_window_raise(searchwindow->window);
    return;
  }

  // make a window for it
  searchwindow = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_title (GTK_WINDOW (searchwindow), "StarPlot: Search");
  gtk_signal_connect (GTK_OBJECT (searchwindow), "destroy", 
		      GTK_SIGNAL_FUNC (search_quit), NULL);
  
  // make a vbox
  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (searchwindow), vbox);
  gtk_widget_show (vbox);

  // set up the search input field in an hbox
  casesensitive = gtk_check_button_new_with_label ("Case sensitive");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (casesensitive), FALSE);
  gtk_widget_show (casesensitive);

  entrybox = gtk_entry_new();
  gtk_widget_show (entrybox);

  searchbutton = gtk_button_new_with_label ("Search");
  gtk_widget_set_usize (searchbutton, BUTTONWIDTH, BUTTONHEIGHT);
  gtk_widget_show (searchbutton);

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (hbox), casesensitive, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), entrybox, TRUE, TRUE, 0); 
  gtk_box_pack_start (GTK_BOX (hbox), searchbutton, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
  
  // set up the column list
  my_gtk_list_setup (vbox, &sclist, true /* a search clist */);
  
  // set up the search button
  search_data.casesensitive = casesensitive;
  search_data.entrybox = entrybox;
  search_data.sclist = sclist;
  gtk_signal_connect (GTK_OBJECT (searchbutton), "clicked", 
		      GTK_SIGNAL_FUNC (search_callback),
		      &search_data);
  gtk_signal_connect (GTK_OBJECT (entrybox), "activate", 
		      GTK_SIGNAL_FUNC (search_callback),
		      &search_data);
  
  // the button to make the window go away
  my_gtk_button_bar (&OK, NULL, NULL, vbox);
  my_gtk_buttons_connect_destroy (OK, NULL, NULL, searchwindow);

  gtk_window_set_focus (GTK_WINDOW (searchwindow), OK);
  gtk_widget_show (searchwindow);
  return;

}
