/* $Id: stdata.c,v 201.0 2002/02/14 09:29:14 sgifford Exp $ */

#include "stdata.h"
#include "phonebook.h"
#include "startalk.h"
#include "stdebug.h"

#include <string.h>
#include <unistd.h>
#include <stdio.h>

struct startalk_phonebook_entry *
startalk_get_phonebook_entry(int serialfd, int num)
{
  struct startalk_phonebook_entry *pbe;
  
  char cmdbuf[5] = { 0xcf, 0x01, 0x01, 0x00, 0x00 };
  unsigned char buf[8192];
  char phonebuf[33];
  char *phonepos;
  int msgsize;
  int whichbyte;
  int numtype;
  int i,j;
  int digit;
  
  pbe = startalk_new_phonebook_entry();
  if (!pbe)
    return (void *) reterr(0,"Error allocating new phonebook entry");

  cmdbuf[4]=num;
  
  if (startalk_writemsg(serialfd, cmdbuf, 5) != 0)
    return (void *) reterr(0,"writing phonebook entry");

  if (startalk_readmsg(serialfd, buf, 8191, &msgsize) != 0)
    return (void *) reterr(0,"getting response from phonebook entry");

  /* OK, now actually dump it. */
  if ( (buf[6] == 0) && (buf[32] == 0) && (buf[33] == 0) )
  {
    /* Empty. */
    return NULL;
  }

  pbe->pos = num;
  strncpy(pbe->name,&buf[6],MAX_NAMELEN);
  pbe->name[MAX_NAMELEN]=0;
  strncpy(pbe->company,&buf[18],MAX_COMPANYLEN);
  pbe->company[MAX_COMPANYLEN]=0;
  
  for (i=0;i<4;i++)
  {
    whichbyte = 32 + i / 2;
    if (i%2)
      numtype = buf[whichbyte] % 16;
    else
      numtype = buf[whichbyte] / 16;
    if (numtype == 0)
      continue;
    
    j=34+16*i;

    phonepos = phonebuf;
    while(1)
    {
      digit = buf[j] / 16;
      if (digit)
      {
        if ( digit == 10 )
          *phonepos = '0';
        else
          *phonepos = '0'+digit;
      }
      else
        break;
      phonepos++;

      digit = buf[j] % 16;
      if (digit)
      {
        if ( digit == 10 )
          *phonepos = '0';
        else
          *phonepos = '0'+digit;
      }
      else
        break;
      phonepos++;

      j++;
      if (j >= (34+16*(i+1)))
        break;
    }
    *phonepos = 0;
    if (!startalk_add_phone_to_entry(pbe, numtype, phonebuf))
      warn("adding phone number to entry");
  }
    
  return pbe;
}

void
startalk_encode_name(char *wbuf, char *rbuf, int len, char pad)
{
  int i;
  
  for (i=0;i<len;i++)
  {
    if (*rbuf)
    {
      *wbuf = *rbuf;
      rbuf++;
    }
    else
    {
      *wbuf = pad;
    }
    wbuf++;
  }
}

void
startalk_encode_phone(unsigned char *wbuf, unsigned char *rbuf, int len)
{
  int i;
  unsigned char byte;
  
  for (i=0;i<len*2;i++)
  {
    if (*rbuf)
    {
      if (*rbuf == '0')
        byte = 0x0a;
      else if ( (*rbuf >= '1') && (*rbuf <= '9') )
        byte = *rbuf - '0';
      else
        byte = 0;
      rbuf++;
    }
    else
      byte = 0;
    
    if ( (i % 2) == 0)
    {
      *wbuf = byte << 4;
    }
    else
    {
      *wbuf |= byte;
      wbuf++;
    }
  }
}

int
startalk_put_phonebook_entry(int serialfd, int num, struct startalk_phonebook_entry *pbe)
{
  unsigned char sendbuf[106];
  unsigned char inbuf[256];
  int byte;
  int i;
  
  /* Check if this is an empty phonebook entry.  If so, we'll
   * delete it.
   */
  if ( (pbe->name[0] == '\0') && (pbe->company[0] == '\0') &&
       (pbe->numphones == 0))
    return startalk_remove_phonebook_entry(serialfd, num);
  
  /* OK, let's build this request packet. */
  /* Header */
  /* Bytes 0-3 */
  sendbuf[0] = 0xcf;
  sendbuf[1] = 0x01;
  sendbuf[2] = 0x02;
  sendbuf[3] = 0x00;

  /* Position in phonebook */
  /* Byte 4 */
  sendbuf[4] = num;
  
  /* Next 12 characters are name */
  /* Bytes 5-16 */
  startalk_encode_name(&sendbuf[5],pbe->name,12,(pbe->company[0]?' ':'\0'));
  
  /* Next 12 are company */
  startalk_encode_name(&sendbuf[17],pbe->company,12,'\0');

  /* Bytes 29-30 */
  /* Now we have two unknown null bytes */
  sendbuf[29] = 0;
  sendbuf[30] = 0;
  
  /* Bytes 31-32 */
  /* And two bytes to indicate the types of the phone numbers */
  /* Maybe these need to be sorted, but it's much easier for now
   * not to.
   */
  for(i=0;i<4;i++)
  {
    if (i < pbe->numphones)
      byte = pbe->phone[i]->type;
    else
      byte = 0;

    if ( (i % 2) == 0)
      sendbuf[31+i/2] = byte << 4;
    else
      sendbuf[31+i/2] |= byte;
  }

  /* Bytes 33-97 */
  /* Then 16 bytes for each of the 4 phone numbers.  Each digit is
   * encoded in one nybble, so we can hold 32 digits this way.
   * The digit '0' is encoded as 0x0a, and the extra bytes are all
   * NULLs.
   */

  for (i=0;i<4;i++)
  {
    if (i < pbe->numphones)
      startalk_encode_phone(&sendbuf[33+i*16],pbe->phone[i]->num,16);
    else
      memset(&sendbuf[33+i*16], 0, 16);
  }

  if (startalk_writemsg(serialfd, sendbuf, 97) != 0)
    return errorf("writing phonebook entry");

  /* Make sure it succeeded. */
  if (startalk_readmsg(serialfd, inbuf, 255, &i) != 0)
    return errorf("getting response from writing phonebook entry");
    
  return 0;
}

int
startalk_remove_phonebook_entry(int serialfd, int num)
{
  unsigned char sendbuf[106];
  unsigned char inbuf[256];
  int i;
  
  /* OK, let's build this request packet. */
  /* Header */
  /* Bytes 0-3 */
  sendbuf[0] = 0xcf;
  sendbuf[1] = 0x01;
  sendbuf[2] = 0x02;
  sendbuf[3] = 0x00;

  /* Position in phonebook */
  /* Byte 4 */
  sendbuf[4] = num;
  memset(&sendbuf[5],0,92);
  if (startalk_writemsg(serialfd, sendbuf, 97) != 0)
    return errorf("removing phonebook entry");

  /* Make sure it succeeded. */
  if (startalk_readmsg(serialfd, inbuf, 255, &i) != 0)
    return errorf("getting response from removing phonebook entry");

  return 0;
}


int
startalk_get_greeting(int serialfd, char *greeting, int bufsiz)
{
  char buf[8192];

  if ((bufsiz != 0) && (bufsiz < 13))
    return errorf("buffer size too small (need 13 bytes)");
  
  if (startalk_writemsg(serialfd,"\xC7\x0D",2) != 0)
    return errorf("requesting greeting");
  if (startalk_readmsg(serialfd,buf,8192,NULL) != 0)
    return errorf("getting response from greeting");
  if (greeting && bufsiz)
  {
    strncpy(greeting,&buf[7],12);
    greeting[12]=0;
  }
  return 0;
}

int
startalk_dump_cmd(int serialfd, char *msgbuf, int size)
{
  char buf[8192];
  int respsize;

  printf("-----Command-----\n");
  dumpbuf(msgbuf,size);
  
  if (startalk_writemsg(serialfd,msgbuf,size) != 0)
    return errorf("sending command");
  if (startalk_readmsg(serialfd,buf,8192,&respsize) != 0)
    return errorf("reading response");

  printf("-----Response-----\n");
  dumpbuf(buf,respsize);

  return 0;
}

int
startalk_run_cmd(int serialfd, char *msgbuf, int size)
{
  char buf[8192];
  int respsize;

  if (startalk_writemsg(serialfd,msgbuf,size) != 0)
    return errorf("running command");
  if (startalk_readmsg(serialfd,buf,8192,&respsize) != 0)
    return errorf("running command");

  return 0;
}
