/*
 * gtk-list.c	-- GTK+ Listboxes
 * 
 * Copyright  2000 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 11-Oct-2000 11:21 (eg)
 * Last file update: 24-Oct-2000 17:26 (eg)
 */

#include "stklos.h"
#include "gtk-glue.h"


static SCM listbox_keyword; 	/* contains the Scheme keyword :listbox */

static SCM get_listbox(SCM lb)
{
  if (WIDGETP(lb)) {
    SCM listbox = STk_key_get(WIDGET_PLIST(lb), listbox_keyword, STk_false);
    
    if (listbox != STk_false) return listbox;
  }
  STk_error("bad listbox ~S", lb);
  return STk_void;
}


DEFINE_PRIMITIVE("%listbox", listbox, subr1, (SCM obj))
{
  GtkWidget *scrolled_window;
  SCM listbox, res;

  /* Create a scrolled window in which we will put the listbox */
  scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  // gtk_widget_set_usize(scrolled_window, 0, 0); // A VOIR
  
  /* create the listbox */
  listbox = STk_gtk_widget2scm(gtk_list_new(), STk_void);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					WIDGET_ID(listbox));
  /* The result is the scrolled window */
  res =  STk_gtk_widget2scm(scrolled_window, obj);  

  /* Keep a reference on the listbox through the scrolled_window */
  WIDGET_PLIST(res) = LIST2(listbox_keyword, listbox);

  return res;
}


void do_unselect_handler(GtkWidget *w, gpointer data)
{
  /* This handler should not be here: GTk seems to be bugged here:
   *   1. When an item is selected, the signal "select-child" is sent to the 
   *      listbox 
   *   2. When an item is unselected, the signal "unselect-child" is sent to the 
   *      listbox , BUT NOT IF IT IS UNSELECTED BY SELECTING ANOTHER ITEM (i.e.
   *      only when the item is TOGGLED).
   * The following code permits to call the signal. BTW the signal is sent 
   * TWO times when the item is toggled. But it seems to be harmless than the 
   * current behavior 
   */
  gtk_list_unselect_child(GTK_LIST(data), GTK_WIDGET(w));
}


DEFINE_PRIMITIVE("%set-list-items!", set_list_items, subr2, (SCM lb, SCM l))
{
  SCM tmp;
  GtkWidget *lst_item, *content, *listbox;
  
  ENTER_PRIMITIVE(set_list_items);

  /* First pass on the list to verify that the elements are strings or widgets */
  if (!NULLP(l) && !CONSP(l)) STk_error("bad list ~S", l);

  for (tmp = l; !NULLP(tmp); tmp = CDR(tmp)) {
    if (!STRINGP(CAR(tmp)) && !WIDGETP(CAR(tmp)))
      STk_error("bad listbox element ~S", CAR(tmp));
  }

  listbox = WIDGET_ID(get_listbox(lb));

  /* We are now sure that the list is well formed => We can delete old value */
  gtk_list_clear_items ((GtkList*) listbox, 0, -1);
  
  /* Build the new list items */
  for (tmp = l; !NULLP(tmp); tmp = CDR(tmp)) {
    if (STRINGP(CAR(tmp))) {
      lst_item = gtk_list_item_new_with_label(STRING_CHARS(CAR(tmp)));
    } else {
      content  = WIDGET_ID(CAR(tmp));
      lst_item = gtk_list_item_new();
      gtk_container_add(GTK_CONTAINER(lst_item), content);
      gtk_widget_show(content);
    }
    gtk_container_add(GTK_CONTAINER(listbox), lst_item);
    gtk_widget_show(lst_item);

    /* Correct GTk+ bug */
    gtk_signal_connect(GTK_OBJECT(lst_item), "deselect", do_unselect_handler, 
		       (gpointer) listbox);
  }
  return STk_void;
}

DEFINE_PRIMITIVE("%select-list-item", select_list_item, subr3, 
		 (SCM lb, SCM item, SCM select))
{
  int n = STk_integer_value(item);
  GtkList* listbox;

  ENTER_PRIMITIVE(select_list_item);
  if (n == LONG_MIN) STk_error("bad index ~S", item);

  listbox = (GtkList *) WIDGET_ID(get_listbox(lb));

  if (select == STk_false)
    gtk_list_unselect_item(listbox, n);
  else
    gtk_list_select_item(listbox, n);
  return STk_void;
}


/*===========================================================================*\
 * 
 * Callback associated for the signal "select-child" and "unselect-child"
 * 
\*===========================================================================*/
static void do_callback(GtkObject* w1, GtkObject *w2, gpointer data)
{
  /* The callback will be called with 2 parameters the object and the
   * index of the selected object. Here "data" is the closure to call, 
   * w1 is the object which wraps the listbox and w2 is the list item
   * selected 
   */
  int pos = gtk_list_child_position ((GtkList*) w1, (GtkWidget*) w2);
  STk_C_apply((SCM) data, 
	      2, 
	      (SCM) gtk_object_get_user_data(GTK_OBJECT(w1)), 
	      MAKE_INT(pos));
}


/* ====================================================================== */

void STk_init_gtk_list(void)
{
  /* Retain in a static the keyword :listbox to avoid its reconstruction */
  listbox_keyword = STk_makekey(":listbox");

  /* Add a new callback function for accessing special listboxes signals */
  STk_define_variable(STk_intern("%listbox-callback"),
		      STk_make_callback(do_callback),
		      STk_gtk_module);
  
  ADD_PRIMITIVE(listbox);
  ADD_PRIMITIVE(set_list_items);
  ADD_PRIMITIVE(select_list_item);
}

