/* ====================================================================
 * Copyright (c) 2006,      Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "DiffDialogCmd.h"
#include "../DiffDialog.h"
#include "../ExternProviderImpl.h"
#include "../BaseModel.h"
#include "../PostCmdResult.h"
#include "subcommander/Project.h"
#include "sublib/MessageBox.h"
#include "svn/Revision.h"
#include "util/Condition.h"

// qt
#include <qstring.h>
#include <qapplication.h>
#include <qthread.h>
#include <qlistview.h>


class DiffThread : public QThread, public CmdResultCallback
{
public:
  DiffThread( BaseModel* model, DiffDialog::DiffParams& params )
    : _model(model), _params(params), _stop(false)
  {
  }

  void run()
  {
    for( DiffDialog::DiffParams::iterator it = _params.begin(); it != _params.end(); it++ )
    {
      _model->diff( *it, this );

      _cond.reset();
      _cond.wait();

      if( _stop )
      {
        return;
      }
    }

    // \todo unused DiffParam are leaked
  }

  void stop()
  {
    _stop = true;
  }

  void handleResult( ScParam*, const sc::Error* e )
  {
    _cond.wakeAll();
  }

private:
  BaseModel*             _model;
  DiffDialog::DiffParams _params;
  sc::Condition          _cond;
  bool                   _stop;
};

///////////////////////////////////////////////////////////////////////////////

DiffDialogCmd::DiffDialogCmd( QWidget* parent, BaseModel* model,
  const Project* prj, bool enableSrc1, bool enableSrc2 )
: _parent(parent), _model(model), _dlg(NULL), _prj(prj),
  _enableSrc1(enableSrc1), _enableSrc2(enableSrc2)
{
}

DiffDialogCmd::~DiffDialogCmd()
{
  delete _dlg;
}

void DiffDialogCmd::run( const QString& src1, const QString& src2,
  const svn::Revision* first, const svn::Revision* second, bool folder )
{
  _dlg = new DiffDialog( _model, folder, new ExternProviderImpl(_model), _parent );

  connect( _dlg, SIGNAL(diff()), SLOT(diff()) );
  connect( _dlg, SIGNAL(patch()), SLOT(patch()) );
  connect( _dlg, SIGNAL(summarize()), SLOT(summarize()) );
  connect( _dlg, SIGNAL(closed()), SLOT(close()) );

  _dlg->enablePathOrUrl1(_enableSrc1);
  _dlg->enablePathOrUrl2(src1!=src2);
  _dlg->setPathOrUrl1(src1);
  _dlg->setPathOrUrl2(src2);
  _dlg->setRevision1(first);
  _dlg->setRevision2(second);

  _dlg->show();
}

void DiffDialogCmd::diff()
{
  if( _dlg->isFolder() )
  {
    DiffDialog::DiffParams params;
    _dlg->getParameters(params);
    _dlg->setEnabled(false);

    DiffThread thread(_model,params);
    thread.start();

    if( params.size() > 1 )
    {
      msgInformation( _q("subcommander:diff selection"),
        _q("press 'Ok' to stop!"), _q("&Ok") );

      thread.stop();
      thread.wait();
    }
    else
    {
      while( ! thread.wait(10) )
      {
        qApp->processEvents(500);
      }
    }

    _dlg->setEnabled(true);
    _dlg->enableButtons();
  }
  else
  {
    DiffParam*     param = _dlg->getParameters( DiffDialog::Diff );
    PostCmdResult* pcres = new PostCmdResult(_dlg);
    _dlg->setEnabled(false);

    _model->diff( param, pcres );
  }
}

void DiffDialogCmd::patch()
{
  DiffParam*     param = _dlg->getParameters( DiffDialog::Patch );
  PostCmdResult* pcres = new PostCmdResult(_dlg);
    _dlg->setEnabled(false);

  _model->diff( param, pcres );
}

void DiffDialogCmd::summarize()
{
  DiffParam*     param = _dlg->getParameters( DiffDialog::Summarize );
  PostCmdResult* pcres = new PostCmdResult(_dlg);
  _dlg->setEnabled(false);

  _model->diff( param, pcres );
}

void DiffDialogCmd::close()
{
  delete this;
}
