/*==================================================================
 * keyspan.c - Keyspan widget
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>

#include "keyspan.h"

/* # of pixels of mouse movement to cause change */
#define KEYSPAN_MOVEMENT_THRESHOLD 3

#define STRIPE_TILE_LINES 7


/* Forward declarations */

static void keyspan_class_init (KeySpanClass * klass);
static void keyspan_init (KeySpan * keyspan);
static void keyspan_destroy (GtkObject * object);
static void keyspan_realize (GtkWidget * widget);
static void keyspan_size_request (GtkWidget * widget,
  GtkRequisition * requisition);
static void keyspan_size_allocate (GtkWidget * widget,
  GtkAllocation * allocation);
static gboolean keyspan_expose (GtkWidget * widget, GdkEventExpose * event);
static gboolean keyspan_button_press (GtkWidget * widget,
				      GdkEventButton * event);
static gboolean keyspan_button_release (GtkWidget * widget,
  GdkEventButton * event);
static gboolean keyspan_motion_notify (GtkWidget * widget,
  GdkEventMotion * event);
static void keyspan_update_mouse (KeySpan * keyspan, gint x);

/* Local data */

static struct
{				/* key pixel offsets for one octave */
  guint8 selx;			/* offset to first pixel of select range */
  guint8 dispx;			/* pixel offset to first pixel of key */
  gboolean white;		/* TRUE if key is white, FALSE if black */
}
keyinfo[12] = { {6, 2, TRUE}, {11, 7, FALSE}, {15, 11, TRUE}, {20, 16, FALSE},
  {26, 20, TRUE}, {33, 29, TRUE}, {38, 34, FALSE}, {42, 38, TRUE},
  {47, 43, FALSE}, {51, 47, TRUE}, {56, 52, FALSE}, {62, 56, TRUE}
};

enum
{
  SPAN_CHANGE,			/* span change signal */
  SPAN_LAST			/* place holder for last enum */
};

static gint keyspan_signals[SPAN_LAST] = { 0 };
static GtkWidgetClass *parent_class = NULL;

static GdkPixmap *striped_bg = NULL; /* normal striped background */
static GdkGC *striped_gc = NULL; /* striped GC */


guint
keyspan_get_type (void)
{
  static guint keyspan_type = 0;

  if (!keyspan_type)
    {
      GtkTypeInfo keyspan_info = {
	"KeySpan",
	sizeof (KeySpan),
	sizeof (KeySpanClass),
	(GtkClassInitFunc) keyspan_class_init,
	(GtkObjectInitFunc) keyspan_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      keyspan_type = gtk_type_unique (gtk_widget_get_type (), &keyspan_info);
    }

  return keyspan_type;
}

static void
keyspan_class_init (KeySpanClass * class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;

  parent_class = gtk_type_class (gtk_widget_get_type ());

  keyspan_signals[SPAN_CHANGE] = gtk_signal_new ("span_change",
    GTK_RUN_FIRST, object_class->type,
    GTK_SIGNAL_OFFSET (KeySpanClass, span_change),
    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, keyspan_signals, SPAN_LAST);
  class->span_change = NULL;

  object_class->destroy = keyspan_destroy;

  widget_class->realize = keyspan_realize;
  widget_class->expose_event = keyspan_expose;
  widget_class->size_request = keyspan_size_request;
  widget_class->size_allocate = keyspan_size_allocate;
  widget_class->button_press_event = keyspan_button_press;
  widget_class->button_release_event = keyspan_button_release;
  widget_class->motion_notify_event = keyspan_motion_notify;
}

static void
keyspan_init (KeySpan * keyspan)
{
  keyspan->lokey = 0;
  keyspan->hikey = 0;
  keyspan->selected = 0;
  keyspan->mode = KEYSPAN_KEYMODE;
}

GtkWidget *
keyspan_new (void)
{
  KeySpan *keyspan;

  keyspan = gtk_type_new (keyspan_get_type ());

  return GTK_WIDGET (keyspan);
}

void
keyspan_set_mode (KeySpan * keyspan, gint mode)
{
  g_return_if_fail (keyspan != NULL);
  g_return_if_fail (IS_KEYSPAN (keyspan));
  g_return_if_fail (mode == KEYSPAN_KEYMODE || mode == KEYSPAN_VELMODE);

  keyspan->mode = mode;
}

void
keyspan_set_span (KeySpan * keyspan, guint8 lokey, guint8 hikey)
{
  g_return_if_fail (keyspan != NULL);
  g_return_if_fail (IS_KEYSPAN (keyspan));

  keyspan->lokey = lokey;
  keyspan->hikey = hikey;

  gtk_widget_draw (GTK_WIDGET (keyspan), NULL);
}

int
keyspan_pixel_to_key (KeySpan *keyspan, int pixel)
{
  int i, xofs, keynum;

  g_return_val_if_fail (keyspan != NULL, -1);

  pixel = CLAMP (pixel, 0, KEYSPAN_WIDTH - 1);

  if (keyspan->mode == KEYSPAN_KEYMODE)
    {
      xofs = pixel % 63;	/* pixel offset into keyboard octave */
      for (i = 0; i < 12; i++)	/* loop through key selection offsets */
	if (xofs <= keyinfo[i].selx) /* offset within key selection area? */
	  {
	    keynum = pixel / 63 * 12 + i; /* calculate key number */
	    break;
	  }
    }
  else				/* velocity mode */
    keynum = pixel * 127 / (KEYSPAN_WIDTH - KEYSPAN_HANDLE_WIDTH);

  return (CLAMP (keynum, 0, 127));
}

int
keyspan_key_to_pixel (KeySpan *keyspan, int keynum)
{
  int xval;

  g_return_val_if_fail (keyspan != NULL, -1);

  keynum = CLAMP (keynum, 0, 127);

  if (keyspan->mode == KEYSPAN_KEYMODE)
    xval = keynum / 12 * 63 + keyinfo [keynum % 12].dispx;
  else xval = keynum * (KEYSPAN_WIDTH - 8) / 127 + 2;

  return (xval);
}

static void
keyspan_destroy (GtkObject * object)
{
  KeySpan *keyspan;

  g_return_if_fail (object != NULL);
  g_return_if_fail (IS_KEYSPAN (object));

  keyspan = KEYSPAN (object);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
keyspan_realize (GtkWidget * widget)
{
  KeySpan *keyspan;
  GdkWindowAttr attributes;
  gint attributes_mask;
  GdkGCValues gcvals;
  int x, width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_KEYSPAN (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  keyspan = KEYSPAN (widget);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget) |
    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK |
    GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
    GDK_POINTER_MOTION_HINT_MASK;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (widget->parent->window, &attributes,
				   attributes_mask);
  gdk_window_set_user_data (widget->window, widget);

  /* create striped background */
  if (!striped_bg)
    {
      width = (STRIPE_TILE_LINES + 1) * 9;

      /* create and draw striped background pixmap */
      striped_bg = gdk_pixmap_new (widget->window, width,
				   widget->allocation.height, -1);

      gdk_draw_rectangle (striped_bg, widget->style->base_gc[GTK_STATE_NORMAL],
			  TRUE, 0, 0, width, widget->allocation.height);

      for (x = 5; x < width; x += 9)
	gdk_draw_line (striped_bg, widget->style->mid_gc[GTK_STATE_NORMAL],
		       x, 0, x, widget->allocation.height - 1);

      gcvals.fill = GDK_TILED;
      gcvals.tile = striped_bg;

      striped_gc = gdk_gc_new_with_values (widget->window, &gcvals,
					   GDK_GC_FILL | GDK_GC_TILE);
    }
}

static void
keyspan_size_request (GtkWidget * widget, GtkRequisition * requisition)
{
  requisition->width = KEYSPAN_WIDTH;
  requisition->height = KEYSPAN_HEIGHT;
}

static void
keyspan_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
  KeySpan *keyspan;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_KEYSPAN (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;
  keyspan = KEYSPAN (widget);

  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window, allocation->x, allocation->y,
	allocation->width, allocation->height);
    }
}

static gboolean
keyspan_expose (GtkWidget * widget, GdkEventExpose * event)
{
  KeySpan *keyspan;
  int lox, hix;
  GdkGC *fg_gc;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_KEYSPAN (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->count > 0)
    return FALSE;

  keyspan = KEYSPAN (widget);

  /* use striped background for unselected keyspans, regular for
     velocity spans or selected keyspans */
  if (keyspan->mode == KEYSPAN_KEYMODE
      && GTK_WIDGET_STATE (widget) == GTK_STATE_NORMAL)
    {
      fg_gc = widget->style->black_gc;

      gdk_draw_rectangle (widget->window, striped_gc, TRUE,
			  event->area.x, event->area.y,
			  event->area.width, event->area.height);
    }
  else
    {
      fg_gc = widget->style->text_gc[GTK_WIDGET_STATE (widget)];

      gtk_style_apply_default_background (widget->style, widget->window, FALSE,
					  GTK_WIDGET_STATE (widget),
					  NULL, event->area.x, event->area.y,
					  event->area.width,
					  event->area.height);
      //      bg_gc = widget->style->base_gc[GTK_WIDGET_STATE (widget)];
    }

  lox = keyspan_key_to_pixel (keyspan, keyspan->lokey);

  if (keyspan->hikey > keyspan->lokey)
    {
      hix = keyspan_key_to_pixel (keyspan, keyspan->hikey);

      gdk_draw_line (widget->window, fg_gc, lox + KEYSPAN_HANDLE_WIDTH, 7,
		     hix - 1, 7);
      gdk_draw_rectangle (widget->window, fg_gc, TRUE, hix, 2,
			  KEYSPAN_HANDLE_WIDTH, 12);
      if (keyinfo[keyspan->hikey % 12].white
	  || keyspan->mode == KEYSPAN_VELMODE)
	gtk_style_apply_default_background (widget->style, widget->window,
					    FALSE, GTK_WIDGET_STATE (widget),
					    NULL, hix + 1, 3, 3, 10);
      //gdk_draw_rectangle (widget->window, bg_gc, TRUE, hix + 1, 3, 3, 10);
    }

  gdk_draw_rectangle (widget->window, fg_gc, TRUE, lox, 2,
		      KEYSPAN_HANDLE_WIDTH, 12);
  if (keyinfo[keyspan->lokey % 12].white || keyspan->mode == KEYSPAN_VELMODE)
    gtk_style_apply_default_background (widget->style, widget->window,
					FALSE, GTK_WIDGET_STATE (widget),
					NULL, lox + 1, 3, 3, 10);

  //gdk_draw_rectangle (widget->window, bg_gc, TRUE, lox + 1, 3, 3, 10);

  return FALSE;
}

static gboolean
keyspan_button_press (GtkWidget * widget, GdkEventButton * event)
{
  KeySpan *keyspan;
  int x, lox, hix;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_KEYSPAN (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  keyspan = KEYSPAN (widget);

  if (keyspan->selected != 0) return (TRUE); /* A button is already active? */
  if (event->button != 1 && event->button != 2) return (FALSE);

  x = CLAMP (event->x, 0, KEYSPAN_WIDTH - 1);

  if (event->button == 2)	/* middle mouse button does special things */
    {
      if (keyspan->mode == KEYSPAN_KEYMODE)	/* key mode sets single key */
	{
	  keyspan->lokey = keyspan->hikey = keyspan_pixel_to_key(keyspan, x);
	  keyspan->midspecial = TRUE;
	  gtk_grab_add (widget);
	}
      else			/* velocity mode sets full range */
	{
	  keyspan->lokey = 0;
	  keyspan->hikey = 127;
	}

      gtk_widget_draw (GTK_WIDGET (keyspan), NULL);
      gtk_signal_emit (GTK_OBJECT (keyspan), keyspan_signals[SPAN_CHANGE]);
      return (TRUE);
    }

  lox = keyspan_key_to_pixel (keyspan, keyspan->lokey)
    + KEYSPAN_HANDLE_WIDTH / 2;
  hix = keyspan_key_to_pixel (keyspan, keyspan->hikey)
    + KEYSPAN_HANDLE_WIDTH / 2;

  /* cause the closest handle to be selected */
  keyspan->selected = (ABS (lox - x) <= ABS (hix - x)) ? 1 : 2;

  keyspan->movement = 0;	/* reset pixel movement amount */
  keyspan->lastxpos = event->x;

  gtk_grab_add (widget);

  return (TRUE);
}

static gboolean
keyspan_button_release (GtkWidget * widget, GdkEventButton * event)
{
  KeySpan *keyspan;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_KEYSPAN (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  keyspan = KEYSPAN (widget);

  if (event->button == 1 && keyspan->selected)
    {
      gtk_grab_remove (widget);
      keyspan->selected = 0;
      return (TRUE);
    }
  else if (event->button == 2 && keyspan->midspecial)
    {
      gtk_grab_remove (widget);
      keyspan->midspecial = FALSE;
      return (TRUE);
    }

  return (FALSE);
}

static gboolean
keyspan_motion_notify (GtkWidget * widget, GdkEventMotion * event)
{
  KeySpan *keyspan;
  GdkModifierType mods;
  int x, y;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_KEYSPAN (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  keyspan = KEYSPAN (widget);

  x = CLAMP (event->x, 0, KEYSPAN_WIDTH - 1);
  y = event->y;

  if (event->is_hint || (event->window != widget->window))
    gdk_window_get_pointer (widget->window, &x, &y, &mods);

  if (keyspan->selected)
    {
      /* movement threshold has been reached yet? */
      if (keyspan->movement >= KEYSPAN_MOVEMENT_THRESHOLD)
	keyspan_update_mouse (keyspan, x);
      else			/* add amount of movement (in pixels) */
	{
	  keyspan->movement += ABS (keyspan->lastxpos - event->x);
	  keyspan->lastxpos = event->x;
	}
    }
  else if (keyspan->midspecial)
    {				/* middle button special movement */
      int key;

      /* x cursor position within range? */
      key = keyspan_pixel_to_key (keyspan, x);

      if (key != keyspan->lokey || key != keyspan->hikey)
	{
	  keyspan->lokey = keyspan->hikey = key;
	  gtk_widget_draw (GTK_WIDGET (keyspan), NULL);
	  gtk_signal_emit (GTK_OBJECT (keyspan),
			   keyspan_signals[SPAN_CHANGE]);
	}
    }

  return (TRUE);
}

static void
keyspan_update_mouse (KeySpan * keyspan, gint x)
{
  guint8 keynum;

  if (keyspan->selected == 0) return;

  keynum = keyspan_pixel_to_key (keyspan, x);

  if (keyspan->selected == 1)	/* lower range handle selected? */
    {
      if (keynum == keyspan->lokey) return;
      if (keynum <= keyspan->hikey) /* Is lokey selected keynum <= hikey? */
	keyspan->lokey = keynum;
      else
	{			/* lokey > hikey so swap selection */
	  keyspan->lokey = keyspan->hikey;
	  keyspan->hikey = keynum;
	  keyspan->selected = 2;
	}
    }
  else				/* upper range handle selected */
    {
      if (keynum == keyspan->hikey) return;
      if (keynum >= keyspan->lokey) /* Is hikey selected keynum >= hikey? */
	keyspan->hikey = keynum;
      else
	{			/* hikey < lokey so swap selection */
	  keyspan->hikey = keyspan->lokey;
	  keyspan->lokey = keynum;
	  keyspan->selected = 1;
	}
    }

  gtk_signal_emit (GTK_OBJECT (keyspan), keyspan_signals[SPAN_CHANGE]);
  gtk_widget_draw (GTK_WIDGET (keyspan), NULL);
}
