#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define SHAPER_SHAPE                   0
#define SHAPER_INPUT                   1
#define SHAPER_OUTPUT                  2

LADSPA_Descriptor *shaperDescriptor = NULL;

typedef struct {
	LADSPA_Data *shape;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data run_adding_gain;
} Shaper;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return shaperDescriptor;
	default:
		return NULL;
	}
}

void cleanupShaper(LADSPA_Handle instance) {
	free(instance);
}

void connectPortShaper(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Shaper *plugin;

	plugin = (Shaper *)instance;
	switch (port) {
	case SHAPER_SHAPE:
		plugin->shape = data;
		break;
	case SHAPER_INPUT:
		plugin->input = data;
		break;
	case SHAPER_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateShaper(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Shaper *plugin_data = (Shaper *)malloc(sizeof(Shaper));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runShaper(LADSPA_Handle instance, unsigned long sample_count) {
	Shaper *plugin_data = (Shaper *)instance;

	/* Waveshape (float value) */
	LADSPA_Data shape = *(plugin_data->shape);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	int pos;
	
	if (shape < 1.0f && shape > -1.0f) {
	        shape = 1.0f;
	} else if (shape < 0) {
	        shape = -1.0f/shape;
	}
	
	for (pos = 0; pos < sample_count; pos++) {
	        if (*input < 0) {
	                buffer_write(*(output++), -pow(-(*(input++)), shape));
	        } else {
	                buffer_write(*(output++), pow(*(input++), shape));
	        }
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainShaper(LADSPA_Handle instance, LADSPA_Data gain) {
	((Shaper *)instance)->run_adding_gain = gain;
}

void runAddingShaper(LADSPA_Handle instance, unsigned long sample_count) {
	Shaper *plugin_data = (Shaper *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Waveshape (float value) */
	LADSPA_Data shape = *(plugin_data->shape);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	int pos;
	
	if (shape < 1.0f && shape > -1.0f) {
	        shape = 1.0f;
	} else if (shape < 0) {
	        shape = -1.0f/shape;
	}
	
	for (pos = 0; pos < sample_count; pos++) {
	        if (*input < 0) {
	                buffer_write(*(output++), -pow(-(*(input++)), shape));
	        } else {
	                buffer_write(*(output++), pow(*(input++), shape));
	        }
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	shaperDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (shaperDescriptor) {
		shaperDescriptor->UniqueID = 1187;
		shaperDescriptor->Label = strdup("shaper");
		shaperDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		shaperDescriptor->Name =
		 strdup("Wave shaper");
		shaperDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		shaperDescriptor->Copyright =
		 strdup("GPL");
		shaperDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		shaperDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		shaperDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		shaperDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Waveshape */
		port_descriptors[SHAPER_SHAPE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[SHAPER_SHAPE] =
		 strdup("Waveshape");
		port_range_hints[SHAPER_SHAPE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SHAPER_SHAPE].LowerBound = -10;
		port_range_hints[SHAPER_SHAPE].UpperBound = +10;

		/* Parameters for Input */
		port_descriptors[SHAPER_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SHAPER_INPUT] =
		 strdup("Input");
		port_range_hints[SHAPER_INPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SHAPER_INPUT].LowerBound = -1;
		port_range_hints[SHAPER_INPUT].UpperBound = +1;

		/* Parameters for Output */
		port_descriptors[SHAPER_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[SHAPER_OUTPUT] =
		 strdup("Output");
		port_range_hints[SHAPER_OUTPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SHAPER_OUTPUT].LowerBound = -1;
		port_range_hints[SHAPER_OUTPUT].UpperBound = +1;

		shaperDescriptor->activate = NULL;
		shaperDescriptor->cleanup = cleanupShaper;
		shaperDescriptor->connect_port = connectPortShaper;
		shaperDescriptor->deactivate = NULL;
		shaperDescriptor->instantiate = instantiateShaper;
		shaperDescriptor->run = runShaper;
		shaperDescriptor->run_adding = runAddingShaper;
		shaperDescriptor->set_run_adding_gain = setRunAddingGainShaper;
	}
}

void _fini() {
	int i;

	if (shaperDescriptor) {
		free((char *)shaperDescriptor->Label);
		free((char *)shaperDescriptor->Name);
		free((char *)shaperDescriptor->Maker);
		free((char *)shaperDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)shaperDescriptor->PortDescriptors);
		for (i = 0; i < shaperDescriptor->PortCount; i++)
			free((char *)(shaperDescriptor->PortNames[i]));
		free((char **)shaperDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)shaperDescriptor->PortRangeHints);
		free(shaperDescriptor);
	}

}
