"""Tools for the test- and benchmark functions."""

import time
import numpy as numx
from numpy.testing import assert_array_equal, assert_array_almost_equal, \
     assert_equal, assert_almost_equal

iscomplexobj = numx.iscomplexobj

#### test tools
def assert_array_almost_equal_diff(x,y,digits,err_msg=''):
    x,y = numx.asarray(x), numx.asarray(y)
    msg = '\nArrays are not almost equal'
    assert 0 in [len(numx.shape(x)),len(numx.shape(y))] \
           or (len(numx.shape(x))==len(numx.shape(y)) and \
               numx.alltrue(numx.equal(numx.shape(x),numx.shape(y)))),\
               msg + ' (shapes %s, %s mismatch):\n\t' \
               % (numx.shape(x),numx.shape(y)) + err_msg
    maxdiff = max(numx.ravel(abs(x-y)))/\
              max(max(abs(numx.ravel(x))),max(abs(numx.ravel(y)))) 
    if iscomplexobj(x) or iscomplexobj(y): maxdiff = maxdiff/2
    cond =  maxdiff< 10**(-digits)
    msg = msg+'\n\t Relative maximum difference: %e'%(maxdiff)+'\n\t'+\
          'Array1: '+str(x)+'\n\t'+\
          'Array2: '+str(y)+'\n\t'+\
          'Absolute Difference: '+str(abs(y-x))
    assert cond, msg 

def assert_type_equal(act, des):
    assert act == numx.dtype(des), \
           'dtype mismatch: "%s" (should be "%s") '%(act,des)


## random routines
def hermitian(x):
    """Compute the Hermitian, i.e. conjugate transpose, of x."""
    return x.T.conj()

def random_rot(dim, dtype='d'):
    """Return a random rotation matrix, drawn from the Haar distribution
    (the only uniform distribution on SO(n)).
    The algorithm is described in the paper
    Stewart, G.W., "The efficient generation of random orthogonal
    matrices with an application to condition estimators", SIAM Journal
    on Numerical Analysis, 17(3), pp. 403-409, 1980.
    For more information see
    http://en.wikipedia.org/wiki/Orthogonal_matrix#Randomization"""
    H = numx.eye(dim, dtype=dtype)
    D = numx.ones((dim,), dtype=dtype)
    for n in range(1, dim):
        x = numx.random.normal(size=(dim-n+1,)).astype(dtype)
        D[n-1] = numx.sign(x[0])
        x[0] -= D[n-1]*numx.sqrt((x*x).sum())
        # Householder transformation
        Hx = numx.eye(dim-n+1, dtype=dtype) \
             - 2.*numx.outer(x, x)/(x*x).sum()
        mat = numx.eye(dim, dtype=dtype)
        mat[n-1:,n-1:] = Hx
        H = numx.dot(H, mat)
    # Fix the last sign such that the determinant is 1
    D[-1] = -D.prod()
    # Equivalent to mult(numx.diag(D), H) but faster
    H = (D*H.T).T
    return H

def symrand(dim, dtype=numx.double):
    """Return a random symmetric (Hermitian) matrix.
    
    Return a NxN matrix, with eigenvalues
        uniformly distributed on (0,1].
    """
    dtype = numx.dtype(dtype)
    d = numx.random.random(dim)
    if dtype.char in ['D', 'F']:
        v = random_rot(dim, dtype=dtype) + 1j*random_rot(dim, dtype=dtype)
    else:
        v = random_rot(dim, dtype=dtype)
    h = numx.dot(numx.dot(hermitian(v), numx.diag(d)), v)
    # to avoid roundoff errors, symmetrize the matrix (again)
    return (0.5*(hermitian(h)+h)).astype(dtype)




