# $Id: omni.py,v 1.38 2003/11/18 21:31:02 stefan Exp $
#
# Copyright (C) 2000 Stefan Seefeld
# All rights reserved.
# Licensed to the public under the terms of the GNU LGPL (>= 2),
# see the file COPYING for details.
#

from Synopsis import Type, AST, Util
from omniidl import idlast, idltype, idlvisitor, idlutil
import _omniidl
import sys, getopt, os, os.path, string, types

verbose = 0
sourcefile = None

# A dummy function that doesn't modify filename. use -b to change it
def strip(filename): return filename

def strip_filename(filename):
   "This is aliased as strip if -b used and basename set"

   if len(basename) > len(filename): return filename
   if filename[:len(basename)] == basename:
      return filename[len(basename):]
   return filename

class TypeTranslator(idlvisitor.TypeVisitor):
   """maps idltype objects to Synopsis.Type objects in a Type.Dictionary"""

   def __init__(self, types):
      self.types = types
      self.__result = None
      self.__basetypes = {idltype.tk_void:       "void",
                          idltype.tk_short:      "short",
                          idltype.tk_long:       "long",
                          idltype.tk_ushort:     "unsigned short",
                          idltype.tk_ulong:      "unsigned long",
                          idltype.tk_float:      "float",
                          idltype.tk_double:     "double",
                          idltype.tk_boolean:    "boolean",
                          idltype.tk_char:       "char",
                          idltype.tk_octet:      "octet",
                          idltype.tk_any:        "any",
                          idltype.tk_TypeCode:   "CORBA::TypeCode",
                          idltype.tk_Principal:  "CORBA::Principal",
                          idltype.tk_longlong:   "long long",
                          idltype.tk_ulonglong:  "unsigned long long",
                          idltype.tk_longdouble: "long double",
                          idltype.tk_wchar:      "wchar"}

   def internalize(self, idltype):

      idltype.accept(self)
      return self.__result

   def add(self, name, type):

      self.types[name] = type

   def get(self, name):
      return self.types[name]

   def visitBaseType(self, idltype):

      type = Type.Base("IDL", (self.__basetypes[idltype.kind()],))
      self.types[type.name()] = type
      self.__result = type.name()

   def visitStringType(self, idltype):

      if not self.types.has_key(["string"]):
         self.types[["string"]] = Type.Base("IDL", ("string",))
      self.__result = ["string"]
      #if idltype.bound() == 0:
      #    self.__result_type = "string"
      #else:
      #    self.__result_type = "string<" + str(type.bound()) + ">"

   def visitWStringType(self, idltype):

      if not self.types.has_key(["wstring"]):
         self.types[["wstring"]] = Type.Base("IDL", ("wstring",))
      self.__result = ["wstring"]
      #if type.bound() == 0:
      #    self.__result_type = "wstring"
      #else:
      #    self.__result_type = "wstring<" + str(type.bound()) + ">"

   def visitSequenceType(self, idltype):

      if not self.types.has_key(["sequence"]):
         self.types[["sequence"]] = Type.Base("IDL", ("sequence",))
      idltype.seqType().accept(self)
      ptype = self.types[self.__result]
      #if type.bound() == 0:
      #    self.__result_type = "sequence<" + self.__result_type + ">"
      #else:
      #    self.__result_type = "sequence<" + self.__result_type + ", " +\
      #                         str(type.bound()) + ">"
      type = Type.Parametrized("IDL", self.types[["sequence"]], [ptype])
      name  = ["sequence<" + Util.ccolonName(ptype.name()) + ">"]
      self.types[name] = type
      self.__result = name
        
   def visitDeclaredType(self, idltype):

      self.__result = idltype.decl().scopedName()

class ASTTranslator(idlvisitor.AstVisitor):

   def __init__(self, declarations, types, mainfile_only):

      self.declarations = declarations
      self.__mainfile_only = mainfile_only
      self.__types = types
      self.__scope = []
      self.__operation = None
      self.__enum = None
        
   def scope(self): return self.__scope[-1].name()
   def addDeclaration(self, declaration): self.__scope[-1].declarations().append(declaration)
   def addType(self, name, type):

      if self.__types.types.has_key(name):
         if isinstance(self.__types.get(name), Type.Unknown):
            self.__types.add(name, type)
         else:
            pass
         return
      self.__types.add(name, type)

   def getType(self, name): return self.__types.get(name)
   def visitAST(self, node):

      self.__scope.append(AST.Scope(sourcefile, 0, "IDL", "file", []))
      # add an 'Object' Type to the Type Dictionary. Don't declare it in the AST since
      # there is no corresponding declaration
      object = AST.Class(sourcefile, 0, "IDL", "interface", ["CORBA", "Object"])
      self.addType(["CORBA", "Object"], Type.Declared("IDL", ["CORBA", "Object"], object))
      for n in node.declarations():
         n.accept(self)
      for d in self.__scope[-1].declarations():
         self.declarations.append(d)

   def visitModule(self, node):

      #if self.__mainfile_only and not node.mainFile(): return
      name = list(self.scope()) + [node.identifier()]
      module = AST.Module(sourcefile, node.line(), "IDL", "module", name)
      self.addDeclaration(module)
      self.__scope.append(module)
      self.addType(name, Type.Declared("IDL", name, module))
      if not self.__mainfile_only or node.mainFile(): 
         for c in node.comments():
            module.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for n in node.definitions():
         n.accept(self)
      self.__scope.pop()
        
   def visitInterface(self, node):

      name = list(self.scope()) + [node.identifier()]
      clas = AST.Class(sourcefile, node.line(), "IDL", "interface", name)
      self.addDeclaration(clas)
      self.__scope.append(clas)
      self.addType(name, Type.Declared("IDL", name, clas))
      if not self.__mainfile_only or node.mainFile(): 
         for c in node.comments():
            clas.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for i in node.inherits():
         parent = self.getType(i.scopedName())
         clas.parents().append(AST.Inheritance("", parent, []))
      for c in node.contents(): c.accept(self)
      self.__scope.pop()
        
   def visitForward(self, node):

      #if self.__mainfile_only and not node.mainFile(): return
      name = list(self.scope())
      name.append(node.identifier())
      forward = AST.Forward(sourcefile, node.line(), "IDL", "interface", name)
      self.addDeclaration(forward)
      self.addType(name, Type.Unknown("IDL", name))
        
   def visitConst(self, node):

      if self.__mainfile_only and not node.mainFile(): return
      name = list(self.scope())
      name.append(node.identifier())
      type = self.__types.internalize(node.constType())
      if node.constType().kind() == idltype.tk_enum:
         value = "::" + idlutil.ccolonName(node.value().scopedName())
      else:
         value = str(node.value())
      const = AST.Const(sourcefile, node.line(), "IDL", "const",
                        self.getType(type), name, value)
      self.addDeclaration(const)
      for c in node.comments():
         const.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
        
   def visitTypedef(self, node):

      #if self.__mainfile_only and not node.mainFile(): return
      # if this is an inline constructed type, it is a 'Declared' type
      # and we need to visit the declaration first
      if node.constrType():
         node.memberType().decl().accept(self)
      type = self.__types.internalize(node.aliasType())
      comments = []
      for c in node.comments():
         comments.append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for d in node.declarators():
         # reinit the type for this declarator, as each declarator of
         # a single typedef declaration can have a different type. *sigh*
         dtype = type
         if d.sizes():
            array = Type.Array("IDL", self.getType(type), map(lambda s:str(s), d.sizes()))
            dtype = map(None, type[:-1])
            dtype.append(type[-1] + string.join(map(lambda s:"["+ str(s) +"]", d.sizes()),''))
            self.addType(dtype, array)
         dname = list(self.scope())
         dname.append(d.identifier())
         typedef = AST.Typedef(sourcefile, node.line(), "IDL", "typedef", dname, self.getType(dtype), node.constrType())
         typedef.comments().extend(comments)
         for c in d.comments():
            typedef.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
         self.addType(typedef.name(), Type.Declared("IDL", typedef.name(), typedef))
         self.addDeclaration(typedef)

   def visitMember(self, node):

      if self.__mainfile_only and not node.mainFile(): return
      # if this is an inline constructed type, it is a 'Declared' type
      # and we need to visit the declaration first
      if node.constrType():
         node.memberType().decl().accept(self)
      type = self.__types.internalize(node.memberType())
      comments = []
      for c in node.comments():
         comments.append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for d in node.declarators():
         # reinit the type for this declarator, as each declarator of
         # a single typedef declaration can have a different type. *sigh*
         dtype = type
         if d.sizes():
            array = Type.Array("IDL", self.getType(type), map(lambda s:str(s), node.sizes()))
            dtype = type[:-1]
            dtype.append(type[-1] + string.join(map(lambda s:"["+s+"]", d.sizes()),''))
            self.addType(dtype, array)
         dname = list(self.scope())
         dname.append(d.identifier())
         member = AST.Variable(sourcefile, node.line(), "IDL", "variable", dname, self.getType(dtype), node.constrType())
         member.comments().extend(comments)
         for c in d.comments():
            member.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
         self.addType(member.name(), Type.Declared("IDL", member.name(), member))
         self.addDeclaration(member)

   def visitStruct(self, node):

      name = list(self.scope()) + [node.identifier()]
      if self.__mainfile_only and not node.mainFile():
         forward = AST.Forward(sourcefile, node.line(), "IDL", "struct", name)
         self.addDeclaration(forward)
         self.addType(name, Type.Declared("IDL", name, forward))
         return
      struct = AST.Class(sourcefile, node.line(), "IDL", "struct", name)
      self.addDeclaration(struct)
      self.addType(name, Type.Declared("IDL", name, struct))
      for c in node.comments():
         struct.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      self.__scope.append(struct)
      for member in node.members(): member.accept(self)
      self.__scope.pop()
        
   def visitException(self, node):

      name = list(self.scope()) + [node.identifier()]
      if self.__mainfile_only and not node.mainFile():
         forward = AST.Forward(sourcefile, node.line(), "IDL", "exception", name)
         self.addDeclaration(forward)
         self.addType(name, Type.Declared("IDL", name, forward))
         return
      exc = AST.Class(sourcefile, node.line(), "IDL", "exception", name)
      self.addDeclaration(exc)
      self.addType(name, Type.Declared("IDL", name, exc))
      self.__scope.append(exc)
      for c in node.comments():
         exc.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for member in node.members(): member.accept(self)
      self.__scope.pop()
    
    #    def visitCaseLabel(self, node):    return

   def visitUnionCase(self, node):

      if self.__mainfile_only and not node.mainFile(): return
      # if this is an inline constructed type, it is a 'Declared' type
      # and we need to visit the declaration first
      if node.constrType():
         node.caseType().decl().accept(self)
      type = self.__types.internalize(node.caseType())
      declarator = node.declarator()
      if declarator.sizes():
         array = Type.Array("IDL", self.getType(type), map(lambda s:str(s), declarator.sizes()))
         type = type[:-1]
         type.append(type[-1] + string.join(map(lambda s:"["+s+"]",node.sizes()),''))
         self.addType(type, array)
      name = list(self.scope())
      name.append(node.declarator().identifier())
      self.__scope[-1].declarations().append(
         AST.Operation(sourcefile, node.line(), "IDL", "case",
			  [], self.getType(type), name, name[-1]))

   def visitUnion(self, node):

      name = list(self.scope()) + [node.identifier()]
      if self.__mainfile_only and not node.mainFile():
         forward = AST.Forward(sourcefile, node.line(), "IDL", "union", name)
         self.addDeclaration(forward)
         self.addType(name, Type.Declared("IDL", name, forward))
         return
      clas = AST.Class(sourcefile, node.line(), "IDL", "union", name)
      self.addDeclaration(clas)
      self.__scope.append(clas)
      self.addType(name, Type.Declared("IDL", name, clas))
      for c in node.comments():
         clas.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for c in node.cases(): c.accept(self)
      self.__scope.pop()
        
   def visitEnumerator(self, node):

      if self.__mainfile_only and not node.mainFile(): return
      name = list(self.scope())
      name.append(node.identifier())
      enum = AST.Enumerator(sourcefile, node.line(), "IDL", name, "")
      self.addType(name, Type.Declared("IDL", name, enum))
      self.__enum.enumerators().append(enum)

   def visitEnum(self, node):

      name = list(self.scope()) + [node.identifier()]
      if self.__mainfile_only and not node.mainFile():
         forward = AST.Forward(sourcefile, node.line(), "IDL", "enum", name)
         self.addDeclaration(forward)
         self.addType(name, Type.Declared("IDL", name, forward))
         return
      self.__enum = AST.Enum(sourcefile, node.line(), "IDL", name, [])
      self.addDeclaration(self.__enum)
      self.addType(name, Type.Declared("IDL", name, self.__enum))
      for c in node.comments():
         self.__enum.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for enumerator in node.enumerators(): enumerator.accept(self)
      self.__enum = None
        
   def visitAttribute(self, node):

      scopename = list(self.scope())
      if self.__mainfile_only and not node.mainFile(): return
      # Add real Operation objects
      pre = []
      if node.readonly(): pre.append("readonly")
      type = self.__types.internalize(node.attrType())
      comments = []
      for c in node.comments():
         comments.append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for id in node.identifiers():
         name = scopename + [id]
         attr = AST.Operation(sourcefile, node.line(), "IDL", "attribute",
                              pre, self.getType(type), name, name[-1])
         attr.comments().extend(comments)
         self.addDeclaration(attr)

   def visitParameter(self, node):

      operation = self.__operation
      pre = []
      if node.direction() == 0: pre.append("in")
      elif node.direction() == 1: pre.append("out")
      else: pre.append("inout")
      post = []
      name = self.__types.internalize(node.paramType())
      operation.parameters().append(AST.Parameter(pre, self.getType(name), post, node.identifier()))
    
   def visitOperation(self, node):

      pre = []
      if node.oneway(): pre.append("oneway")
      returnType = self.__types.internalize(node.returnType())
      name = list(self.scope())
      name.append(node.identifier())
      self.__operation = AST.Operation(sourcefile, node.line(), "IDL", "operation", pre, self.getType(returnType), name, name[-1])
      for c in node.comments():
         self.__operation.comments().append(AST.Comment(c.text(), strip(c.file()), c.line()))
      for p in node.parameters(): p.accept(self)
      for e in node.raises():
         exception = self.getType(e.scopedName())
         self.__operation.exceptions().append(exception)
            
      self.addDeclaration(self.__operation)
      self.__operation = None
        
#    def visitNative(self, node):       return
#    def visitStateMember(self, node):  return
#    def visitFactory(self, node):      return
#    def visitValueForward(self, node): return
#    def visitValueBox(self, node):     return
#    def visitValueAbs(self, node):     return
#    def visitValue(self, node):        return

def parse(ast, file, verb, main_file_only,
          base_path, include_paths, comments_before):
   global verbose, basename, strip, sourcefile

   if verb: verbose = True
   preprocessor_args = ["-C"]
   if comments_before: _omniidl.keepComments(1)
   else: _omniidl.keepComments(0)
   if base_path:
      basename = base_path
      strip = strip_filename
   for i in include_paths:
      preprocessor_args.extend(['-I', i])

   path = string.split(os.getenv('PATH'), os.pathsep)
   # Add Synopsis' bindir
   path.insert(0, os.path.dirname(sys.argv[0]))
   if hasattr(_omniidl, "__file__"):
      # Add path to omniidl module
      dirname = os.path.dirname(_omniidl.__file__)
      path.insert(0, dirname)
      # If, eg, /usr/lib/pythonX.X/site-packages, add /usr/lib
      dirnames = string.split(dirname, os.sep)
      if len(dirnames) > 2 and dirnames[-1] == 'site-packages' \
             and dirnames[-2][:6] == 'python':
         path.insert(0, string.join(dirnames[:-2], os.sep))

   preprocessor = None
   for directory in path:
      preprocessor = os.path.join(directory, "omnicpp")
      if os.access(preprocessor, os.R_OK | os.X_OK):
         break
      preprocessor = None
   if not preprocessor:
      # Try ordinary cpp
      print "Error: unable to find omnicpp in path:"
      print string.join(path, os.pathsep)
      sys.exit(1)
   preprocessor_cmd  = preprocessor + " -lang-c++ -undef -D__OMNIIDL__=" + _omniidl.version
   preprocessor_cmd = preprocessor_cmd + " " + string.join(preprocessor_args, " ") + " " + file
   fd = os.popen(preprocessor_cmd, "r")

   _omniidl.noForwardWarning()
   tree = _omniidl.compile(fd)
   if tree == None:
      sys.stderr.write("omni: Error parsing " + file + "\n")
      sys.exit(1)

   sourcefile = AST.SourceFile(strip(file), file, "IDL")
   sourcefile.set_is_main(1)
   new_ast = AST.AST()
   new_ast.files()[sourcefile.filename()] = sourcefile
   type_trans = TypeTranslator(new_ast.types())
   ast_trans = ASTTranslator(new_ast.declarations(), type_trans, main_file_only)
   tree.accept(ast_trans)
   sourcefile.declarations()[:] = new_ast.declarations()
   ast.merge(new_ast)
   _omniidl.clear()
   return ast
