#ifndef TAGIDX_OPTIONS_H
#define TAGIDX_OPTIONS_H

/*
 * Commandline parser for tagcoll
 *
 * Copyright (C) 2003,2004,2005,2006  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Commandline.h>

namespace Tagcoll {
namespace commandline {

struct TagidxOptions : public MainParser<CommandParser>
{
	struct HelpGroup : public OptionGroup
	{
		BoolOption* help;
		BoolOption* version;

		HelpGroup()
		{
			add(help = new BoolOption("help", 'h', "help"));
			add(version = new BoolOption("version", 'V', "version"));
			help->shortNames.push_back('?');
			help->description = "print an help message and exit";
			version->description = "print the program version and exit";
			description = "Help options";
		}
		~HelpGroup()
		{
			delete help;
		}
	} helpGroup;

	struct IndexGroup : public OptionGroup
	{
		ExistingFileOption* index;

		IndexGroup()
		{
			add(index		= new ExistingFileOption("index", 'I', "index"));
			index->usage = "<dir>";
			index->description = "directory where the index files are stored";

			description = "Options used to identify the index";
		}
		~IndexGroup()
		{
			delete index;
		}
	} indexGroup;

	struct OutputGroup : public OptionGroup
	{
		BoolOption* group;
		BoolOption* quiet;
		
		OutputGroup()
		{
			add(group		= new BoolOption("group", 'g', "group"));
			add(quiet		= new BoolOption("quiet", 'q', "quiet"));
			group->description = "group items with the same tagset in the output collection";
			group->longNames.push_back("group-items");
			quiet->description = "produce no output";
			description = "Options controlling transformations of output data";
		}
		~OutputGroup()
		{
			delete group;
			delete quiet;
		}
	} outputGroup;

	struct Generic : public OptionParser
	{
		Generic(TagidxOptions* cp) : OptionParser("")
		{
			add(&cp->helpGroup);
		}
	} generic;
	struct Help : public OptionParser
	{
		Help(TagidxOptions* cp) : OptionParser("help")
		{
			usage = "[command]";
			description = "print help informations";
		}
	} help;
	struct Create : public OptionParser
	{
		Create(TagidxOptions* cp) : OptionParser("create")
		{
			usage = "[files...]";
			description = "create the index from a tagged collection";
			add(&cp->indexGroup);
			add(&cp->helpGroup);
		}
	} create;
	struct Remove : public OptionParser
	{
		Remove(TagidxOptions* cp) : OptionParser("remove")
		{
			usage = "";
			description = "deletes the disk index";
			add(&cp->indexGroup);
			add(&cp->helpGroup);
		}
	} remove;
	struct AddPatch : public OptionParser
	{
		BoolOption* force;

		AddPatch(TagidxOptions* cp) : OptionParser("addpatch")
		{
			add(force = new BoolOption("force", 'f', "force"));
			force->description = "force overwrite";

			usage = "[files...]";
			description = "installs the given files as patches to the collection";
			add(&cp->indexGroup);
			add(&cp->helpGroup);
		}
	} addpatch;
	struct Compact : public OptionParser
	{
		Compact(TagidxOptions* cp) : OptionParser("compact")
		{
			usage = "";
			description = "merge the patches into the index (can cause "
						  "race conditions if invoked concurrently with other "
						  "tagidx accessing the same index)";
			add(&cp->indexGroup);
			add(&cp->helpGroup);
		}
	} compact;
	struct Cat : public OptionParser
	{
		Cat(TagidxOptions* cp) : OptionParser("cat")
		{
			usage = "";
			description = "output the collection";
			add(&cp->indexGroup);
			add(&cp->outputGroup);
			add(&cp->helpGroup);
		}
	} cat;

	TagidxOptions() : MainParser<CommandParser>("tagidx"),
		generic(this),
		help(this), create(this), remove(this), addpatch(this), compact(this), cat(this)
	{
		add(generic);
		add(help);
		add(create);
		add(remove);
		add(addpatch);
		add(compact);
		add(cat);

		usage = "<command> [options and arguments]";
		description = "Access an indexed tagged collection";
	}
};

}
}

// vim:set ts=4 sw=4:
#endif
