/***************************************************************************
 *   Copyright (C) 2006 by Marco Martin                                    *
 *   notmart@gmail.com                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the Lesser GNU General Public                   *
 *   License as published by the Free Software Foundation;                 *
 *   either version 2 of the License, or (at your option)                  *
 *   any later version.                                                    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef TASTYLISTVIEW_H
#define TASTYLISTVIEW_H

#include <klistview.h>
#include <kiconloader.h>
#include <kservice.h>
#include <kservicegroup.h>
#include <kdesktopfile.h>
#include <qpixmap.h>
#include <qtooltip.h>
#include <qtimer.h>
#include <qheader.h>

class TastyListView;

//the space reserved for the action icon (bookmark, remove bookmark etc

/**
	@author Marco Martin <notmart@gmail.com>
 */
class TastyListViewToolTip: public QToolTip
{
  public:
    TastyListViewToolTip( QWidget *parent, TastyListView *tListView );

    void maybeTip( const QPoint &pos );

  private:
    TastyListView *listView;
};

/**
	@author Marco Martin <notmart@gmail.com>
*/
class TastyListView : public KListView
{
    Q_OBJECT
private:
    bool highLightGroups;
    QTimer *onItemTimer;
    QListViewItem *underCursorItem;
    QListViewItem *openItem;
    bool mouseDown;
    bool easyOpen;
    int actionIconSize;
    int actionIconSpace;

    TastyListViewToolTip *listItemTip;

public:
    TastyListView( QWidget * parent = 0, const char * name = 0);

    ~TastyListView();

    bool getHighLightGroups(){ return highLightGroups;}
    void setHighLightGroups(bool highLight){highLightGroups = highLight;}

    bool getEasyOpen(){ return easyOpen;}
    void setEasyOpen(bool easy){easyOpen = easy;}
    void startDrag();

    void setActionIconSize(int newSize){ actionIconSize = newSize; actionIconSpace = newSize*2; }
    int getActionIconSize(){return actionIconSize;}
    int getActionIconSpace(){return actionIconSpace;}

    QListViewItem * getOpenItem(){ return openItem;}
    void setOpenItem( QListViewItem * listItem ){openItem = listItem;}

public slots:
  virtual void clear(){openItem = underCursorItem = NULL; KListView::clear();}


protected:
    virtual void leaveEvent( QEvent * e  );
    virtual void contentsMouseMoveEvent( QMouseEvent * e  );
    virtual void contentsMouseReleaseEvent( QMouseEvent * e  );
    virtual void keyPressEvent( QKeyEvent * e );


private slots:
    void slotOnItem( QListViewItem * listItem );
    void slotTimeout();

signals:
  //Using own signal instead of clicked() in order to avoid launching two times the same app :-)
    void activated(QListViewItem *, const QPoint &, int );

};


/**
	@author Marco Martin <notmart@gmail.com>
*/
class TastyListViewItem : public KListViewItem
{
    //Q_OBJECT
friend class TastyListView;
 public:
    typedef enum
    {
     Service,
     ServiceGroup,
     DesktopFile,
     Empty
    }Type;

    typedef enum
    {
      AddBookMark,
      RemoveBookMark,
      OpenGroup,
      Expand,
      Collapse,
      NoAction
    }ActionType;

    TastyListViewItem( TastyListView * parent );
    TastyListViewItem( TastyListViewItem * parent );
    TastyListViewItem( TastyListView * parent, TastyListViewItem * after, QString label1 );
    TastyListViewItem( TastyListViewItem * parent, TastyListViewItem * after, QString label1 );
    TastyListViewItem( TastyListView * parent, TastyListViewItem * after );
    TastyListViewItem( TastyListViewItem * parent, TastyListViewItem * after );
    TastyListViewItem( TastyListView * parent, QString label1 );
    TastyListViewItem( TastyListViewItem * parent, QString label1 );

    ~TastyListViewItem();

    //QString text(int column) const {return cellText;}
    QString getCellText(int column) const {return cellText;}

    //TastyListViewItem *parent();
    void paintCell ( QPainter * p, const QColorGroup & cg, int column, int width, int align );

    Type getType(){return itemType;}
    void setType( Type newItemType ){itemType = newItemType;}

    ActionType getActionType(){return actionType;}
    void setActionType( ActionType newActionType ){ actionType = newActionType;loadPixmap();}
    void loadPixmap();

    void setPath( QString newPath){ path = newPath;}
    QString getPath(){return path;}

    void setDeskopEntryPath( QString newPath){ desktopEntryPath = newPath;}
    QString getDeskopEntryPath(){return desktopEntryPath;}

    QString getSubText(){return subText;}
    bool xOnDecoration( int x )
       { QListView *lv = listView();
         if( !lv ) return false;
         return !( x > lv->header()->sectionPos( lv->header()->mapToIndex( 0 ) ) +
                lv->treeStepSize() * ( depth() + ( lv->rootIsDecorated() ? 1 : 0) ) + lv->itemMargin() ||
                x < lv->header()->sectionPos( lv->header()->mapToIndex( 0 ) ) );}

    void setSubText(QString text) //FIXME: add the column
       {if(cellText.isEmpty())cellText=KListViewItem::text(0);
        KListViewItem::setText(0,cellText+text);subText = QString(text);}
    void setText(int column, const QString & text )
       {KListViewItem::setText(column, cellText+text); cellText = text;}
    void setDisplaySubText( bool display ){ displaySubText = display; }

    bool hasEllipsis(){return ellipsis;}
    void setHighLight( bool newHighLight ){highLight=newHighLight;}
    bool isHighLight(){return highLight;}

    void setMenuId( QString newMenuId ){ menuId = newMenuId;}
    QString getMenuId(){ return menuId; }

    QString key( int column, bool ascending ) const;

    int width( const QFontMetrics & fm, const QListView * lv, int c )
    {  TastyListView *tlv = dynamic_cast<TastyListView *>( listView() );
       if( tlv )
         return KListViewItem::width(fm, lv, c) + tlv->getActionIconSpace();
       else
         return KListViewItem::width(fm, lv, c);
    }

protected:
    virtual void setup();

private:

    Type itemType;
    ActionType actionType;
    QString path;
    QString desktopEntryPath;
    QString cellText;
    QString subText;
    QString menuId;

    bool ellipsis;
    bool highLight;
    bool displaySubText;
    QPixmap actionPix;
    KIconLoader *iconLoader;

    void commonConstructor();
    //a tiny reimplementation of max...
    int max(int a, int b){return (a>b?a:b);}
};


#endif
