/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"appUtilConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>
#   include	<time.h>

#   include	<utilPs.h>
#   include	<appGeo.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Print a bounding box.						*/
/*									*/
/*  %%BoundingBox <llx> <lly> <urx> <ury>				*/
/*									*/
/************************************************************************/

static void utilPsBoundingBoxComment(
			    const PrintingState *	ps,
			    const char *		comment,
			    const char *		orientationComment )
    {
    fprintf( ps->psFile, "%%%%%s: %d %d %d %d\n",
			    comment,
			    ps->psSheetBoundingBox.drX0,
			    ps->psSheetBoundingBox.drY1,
			    ps->psSheetBoundingBox.drX1,
			    ps->psSheetBoundingBox.drY0 );

    fprintf( ps->psFile, "%%%%%s: %s\n",
				    orientationComment, ps->psOrientation );
    }

/************************************************************************/
/*									*/
/*  Start a DSC convention document.					*/
/*									*/
/************************************************************************/

void utilPsStartDSCDocument(
			const PrintingState *		ps,
			const PostScriptFaceList *	psfl,
			const char *			title,
			const char *			creatorName,
			const char *			creatorReference )
    {
    FILE *		f= ps->psFile;

    fprintf( f, "%%!PS-Adobe-2.0\n" );
    if  ( title )
	{ utilPsFoldComment( f, "Title:", title );	}

    utilPsBoundingBoxComment( ps, "BoundingBox", "Orientation" );

    fprintf( f, "%%%%Creator: %s: %s\n",
				creatorName, creatorReference );
    fprintf( f, "%%%%Pages: (atend)\n" );
    fprintf( f, "%%%%PageOrder: Ascend\n" );

    if  ( psfl )
	{ appPsListFontNames( f, psfl );	}

    fprintf( f, "%%%%EndComments\n" );

    return;
    }

/************************************************************************/
/*									*/
/*  Initialisation for 'nup' printing.					*/
/*									*/
/*  1)  Hack: When the page is higher than the sheet, shift by the page	*/
/*	size. This gives better results in case of misconfiguration.	*/
/*									*/
/************************************************************************/

int utilPsGetNupFactor(	double *			pFac,
			int *				pRotate,
			int *				pXShift,
			int *				pYShift,
			const DocumentGeometry *	dgPage,
			const PrintGeometry *		pg )
    {
    double			xf;
    double			yf;
    int				fitsWithoutRotation;

    const DocumentGeometry *	dgSheet= &(pg->pgSheetGeometry);

    switch( pg->pgNup )
	{
	case 1:
	    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips	||
		  dgPage->dgPageWideTwips <= dgSheet->dgPageWideTwips	)
		{ fitsWithoutRotation= 1;	}
	    else{ fitsWithoutRotation= 0;	}

	    if  ( ! pg->pgRotate90 && fitsWithoutRotation )
		{
		/*  1  */
		if  ( dgPage->dgPageHighTwips > dgSheet->dgPageHighTwips )
		    { *pYShift= dgPage->dgPageHighTwips;	}
		else{ *pYShift= dgSheet->dgPageHighTwips;	}

		if  ( pg->pgCenterHorizontally )
		    {
		    *pXShift=
			( dgSheet->dgPageWideTwips- dgPage->dgPageWideTwips )/2;
		    }
		else{ *pXShift= 0;	}

		*pFac= 1.0; *pRotate= 0;
		return 0;
		}
	    else{
		if  ( dgPage->dgPageWideTwips < dgSheet->dgPageHighTwips )
		    {
		    *pYShift= dgSheet->dgPageHighTwips-
						dgPage->dgPageWideTwips;
		    }
		else{ *pYShift= 0;	}

		if  ( pg->pgCenterHorizontally )
		    {
		    *pXShift=
			( dgSheet->dgPageWideTwips- dgPage->dgPageHighTwips )/2;
		    }
		else{ *pXShift= 0;	}

		*pFac= 1.0; *pRotate= 1;
		return 0;
		}

	case 2:
	    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips )
		{
		xf= ( 1.0* dgSheet->dgPageHighTwips )/
					    ( 2.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageWideTwips )/
					    ( 1.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		*pXShift= 0; *pYShift= 0;
		*pFac= xf; *pRotate= 1; return 0;
		}
	    else{
		xf= ( 1.0* dgSheet->dgPageWideTwips )/
					    ( 1.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageHighTwips )/
					    ( 2.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		*pXShift= 0; *pYShift= 0;
		*pFac= xf; *pRotate= 0; return 0;
		}

	case 4:
	    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips )
		{
		xf= ( 1.0* dgSheet->dgPageWideTwips )/
					( 2.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageHighTwips )/
					( 2.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		*pXShift= 0; *pYShift= 0;
		*pFac= xf; *pRotate= 0;
		return 0;
		}
	    else{
		xf= ( 1.0* dgSheet->dgPageHighTwips )/
					( 2.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageWideTwips )/
					( 2.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		if  ( dgPage->dgPageWideTwips < dgSheet->dgPageHighTwips )
		    {
		    *pYShift= dgSheet->dgPageHighTwips-
						dgPage->dgPageWideTwips;
		    }
		else{ *pYShift= 0;	}

		*pXShift= 0;
		*pFac= xf; *pRotate= 1;
		return 0;
		}

	default:
	    LDEB(pg->pgNup); return -1;
	}
    }

int appPsSetNup(	PrintingState *			ps,
			const DocumentGeometry *	dgPage,
			const PrintGeometry *		pg,
			int				hasPageHeader,
			int				hasPageFooter )
    {
    int				xshift= 0;
    int				yshift= 0;
    const DocumentGeometry *	dgSheet= &(ps->psPrinterGeometry);

    NupTransform *		nt;

    double			fac;
    int				rotate;

    int				x0;
    int				y0;
    int				x1;
    int				y1;

    if  ( utilPsGetNupFactor( &fac, &rotate, &xshift, &yshift, dgPage, pg ) )
	{ LDEB(pg->pgNup); return -1;	}

    x0= dgPage->dgLeftMarginTwips;
    x1= dgPage->dgPageWideTwips- dgPage->dgRightMarginTwips;

    if  ( hasPageHeader )
	{ y0= dgPage->dgHeaderPositionTwips;	}
    else{ y0= dgPage->dgTopMarginTwips;		}

    if  ( hasPageFooter )
	{ y1= dgPage->dgPageHighTwips- dgPage->dgFooterPositionTwips;	}
    else{ y1= dgPage->dgPageHighTwips- dgPage->dgBottomMarginTwips;	}

    if  ( rotate )
	{
	ps->psOrientation= "Landscape";
	ps->psRotated= 1;
	}
    else{
	ps->psOrientation= "Portrait";
	ps->psRotated= 0;
	}

    switch( pg->pgNup )
	{
	case 1:
	    nt= &(ps->psTransform);

	    if  ( rotate )
		{
		ps->psTransform.ntAxx= 0.0;
		ps->psTransform.ntAxy= fac/ 20.0;
		ps->psTransform.ntAyx= fac/ 20.0;
		ps->psTransform.ntAyy= 0.0;
		ps->psTransform.ntTx=  0.0;
		ps->psTransform.ntTx=  xshift/20.0;
		ps->psTransform.ntTy=  yshift/20.0;

		ps->psSheetBoundingBox.drX0= NUP_X( x0, y0, nt );
		ps->psSheetBoundingBox.drY0= NUP_Y( x0, y0, nt );
		ps->psSheetBoundingBox.drX1= NUP_X( x1, y1, nt );
		ps->psSheetBoundingBox.drY1= NUP_Y( x1, y1, nt );
		}
	    else{
		ps->psTransform.ntAxx= fac/ 20.0;
		ps->psTransform.ntAxy= 0.0;
		ps->psTransform.ntAyx= 0.0;
		ps->psTransform.ntAyy= -fac/ 20.0;
		ps->psTransform.ntTx=  0.0;
		ps->psTransform.ntTx=  xshift/20.0;
		ps->psTransform.ntTy=  yshift/20.0;

		ps->psSheetBoundingBox.drX0= NUP_X( x0, y1, nt );
		ps->psSheetBoundingBox.drY0= NUP_Y( x0, y1, nt );
		ps->psSheetBoundingBox.drX1= NUP_X( x1, y0, nt );
		ps->psSheetBoundingBox.drY1= NUP_Y( x1, y0, nt );
		}


	    ps->psNup= pg->pgNup;

	    return 0;

	case 2:
	    if  ( pg->pgRotate90 || pg->pgCenterHorizontally )
		{ LLDEB(pg->pgRotate90,pg->pgCenterHorizontally);	}

	    nt= realloc( ps->psNupTransforms,
					pg->pgNup* sizeof( NupTransform ) );
	    if  ( ! nt )
		{ XDEB(nt); return -1;	}
	    ps->psNupTransforms= nt;
	    ps->psNup= pg->pgNup;

	    if  ( rotate )
		{
		nt[0].ntAxx= 0.0;
		nt[0].ntAxy= fac/ 20.0;
		nt[0].ntAyx= fac/ 20.0;
		nt[0].ntAyy= 0.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  0.0;

		nt[1]= nt[0];
		nt[1].ntTx=  0.0;
		nt[1].ntTy=  ( dgSheet->dgPageHighTwips/ 2 )/ 20.0;

		ps->psSheetBoundingBox.drX0= NUP_X( x0, y0, &(nt[0]) );
		ps->psSheetBoundingBox.drY0= NUP_Y( x0, y0, &(nt[0]) );
		ps->psSheetBoundingBox.drX1= NUP_X( x1, y1, &(nt[1]) );
		ps->psSheetBoundingBox.drY1= NUP_Y( x1, y1, &(nt[1]) );
		}
	    else{
		nt[0].ntAxx=  fac/ 20.0;
		nt[0].ntAxy=  0.0;
		nt[0].ntAyx=  0.0;
		nt[0].ntAyy= -fac/ 20.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;

		nt[1]= nt[0];
		nt[1].ntTx=  0.0;
		nt[1].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

		ps->psSheetBoundingBox.drX0= NUP_X( x0, y1, &(nt[0]) );
		ps->psSheetBoundingBox.drY0= NUP_Y( x0, y1, &(nt[1]) );
		ps->psSheetBoundingBox.drX1= NUP_X( x1, y0, &(nt[1]) );
		ps->psSheetBoundingBox.drY1= NUP_Y( x1, y0, &(nt[0]) );
		}

	    ps->psTransform= nt[0];

	    return 0;

	case 4:
	    if  ( pg->pgRotate90 || pg->pgCenterHorizontally )
		{ LLDEB(pg->pgRotate90,pg->pgCenterHorizontally);	}

	    nt= realloc( ps->psNupTransforms,
					pg->pgNup* sizeof( NupTransform ) );
	    if  ( ! nt )
		{ XDEB(nt); return -1;	}
	    ps->psNupTransforms= nt;
	    ps->psNup= pg->pgNup;

	    if  ( rotate )
		{
		nt[0].ntAxx= 0.0;
		nt[0].ntAxy= fac/ 20.0;
		nt[0].ntAyx= fac/ 20.0;
		nt[0].ntAyy= 0.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  0.0;
		}
	    else{
		nt[0].ntAxx= fac/ 20.0;
		nt[0].ntAxy= 0.0;
		nt[0].ntAyx= 0.0;
		nt[0].ntAyy= -fac/ 20.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;
		}

	    if  ( pg->pgHorizontal )
		{
		nt[1]= nt[0];
		nt[2]= nt[0];

		if  ( rotate )
		    {
		    nt[1].ntTx=  0.0;
		    nt[1].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

		    nt[2].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[2].ntTy=  0.0;
		    }
		else{
		    nt[1].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[1].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;

		    nt[2].ntTx=  0.0;
		    nt[2].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;
		    }
		}
	    else{
		nt[1]= nt[0];
		nt[2]= nt[0];

		if  ( rotate )
		    {
		    nt[1].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[1].ntTy=  0.0;

		    nt[2].ntTx=  0.0;
		    nt[2].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;
		    }
		else{
		    nt[1].ntTx=  0.0;
		    nt[1].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

		    nt[2].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[2].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;
		    }
		}

	    nt[3]= nt[0];
	    nt[3].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
	    nt[3].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

	    if  ( rotate )
		{
		ps->psSheetBoundingBox.drX0= NUP_X( x0, y0, &(nt[0]) );
		ps->psSheetBoundingBox.drY0= NUP_Y( x0, y0, &(nt[0]) );
		ps->psSheetBoundingBox.drX1= NUP_X( x1, y1, &(nt[3]) );
		ps->psSheetBoundingBox.drY1= NUP_Y( x1, y1, &(nt[3]) );
		}
	    else{
		ps->psSheetBoundingBox.drX0= NUP_X( x0, y1, &(nt[0]) );
		ps->psSheetBoundingBox.drY0= NUP_Y( x0, y1, &(nt[3]) );
		ps->psSheetBoundingBox.drX1= NUP_X( x1, y0, &(nt[3]) );
		ps->psSheetBoundingBox.drY1= NUP_Y( x1, y0, &(nt[0]) );
		}

	    ps->psTransform= nt[0];

	    return 0;

	default:
	    LDEB(pg->pgNup); return -1;
	}
    }

/************************************************************************/
/*									*/
/*  Issue a character string.						*/
/*									*/
/************************************************************************/

void appPsPrintString(	FILE *			f,
			const unsigned char *	s,
			int			len )
    {
    int		i;

    for ( i= 0; i < len; s++, i++ )
	{
	if  ( *s == '(' || *s == ')' || *s == '\\' )
	    { putc( '\\', f ); putc( *s, f ); continue;		}

	if  ( *s == '\r' )
	    { putc( '\\', f ); putc( 'r', f ); continue;	}

	if  ( *s == '\n' )
	    { putc( '\\', f ); putc( 'n', f ); continue;	}

	putc( *s, f );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Set the font, this depends on the assumption that fonts have been	*/
/*  defined for the different fonts in the document.			*/
/*									*/
/************************************************************************/

void appPsSetFont(	FILE *		f,
			const char *	prefix,
			TextAttribute	ta )
    {
    int		fontSizeTwips= 10* ta.taFontSizeHalfPoints;

    if  ( ta.taSuperSub == DOCfontSUPERSCRIPT	||
	  ta.taSuperSub == DOCfontSUBSCRIPT	)
	{ fontSizeTwips= SUPERSUB_SIZE( fontSizeTwips ); }

    if  ( fontSizeTwips == 0 )
	{ LDEB(fontSizeTwips); fontSizeTwips= 1;	}

    fprintf( f, "[ %d 0 0 %d 0 0 ] %s%d",
		fontSizeTwips, -fontSizeTwips, prefix, ta.taFontNumber );

    if  ( ta.taFontIsBold )
	{ putc( 'b', f );	}
    if  ( ta.taFontIsSlanted )
	{ putc( 'i', f );	}
    putc( '\n', f );

    return;
    }

/************************************************************************/
/*									*/
/*  Initialise a printing session.					*/
/*									*/
/************************************************************************/

void appPsInitPrintingState(	PrintingState *	ps )
    {
    int			i;

    ps->psFile= (FILE *)0;
    ps->psCurrentPhysicalFont= -1;
    ps->psPagesPrinted= 0;
    ps->psSheetsPrinted= 0;

    ps->psTransform.ntAxx= 1.0;
    ps->psTransform.ntAxy= 0.0;
    ps->psTransform.ntAyx= 0.0;
    ps->psTransform.ntAyy= 1.0;
    ps->psTransform.ntTx=  0.0;
    ps->psTransform.ntTy=  0.0;

    ps->psOrientation= (char *)0;
    ps->psRotated= 0;

    appInitDocumentGeometry( &(ps->psPrinterGeometry) );

    docInitRectangle( &(ps->psSheetBoundingBox) );

    ps->psNup= 1;
    ps->psNupTransforms= (NupTransform *)0;

    ps->psInsideLink= 0;
    ps->psUseLinkColor= 0;
    ps->psCurrentColor= PScolorNONE;

    ps->psLinkParticulesDone= 0;
    ps->psLinkRectLeft= -1;
    ps->psLinkFile= (const char *)0;
    ps->psLinkFileSize= 0;
    ps->psLinkMark= (const char *)0;
    ps->psLinkMarkSize= 0;

    for ( i= 0; i < ENCODINGps_COUNT; i++ )
	{ ps->psEncodingDefined[i]= 0;	}

    return;
    }

void appPsCleanPrintingState(	PrintingState *	ps )
    {
    if  ( ps->psNupTransforms )
	{ free( ps->psNupTransforms );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Switch to the 'Link Color'						*/
/*									*/
/************************************************************************/

void utilPsSetLinkColor(	PrintingState *	ps )
    {
    if  ( ps->psCurrentColor != PScolorLINK )
	{
	fprintf( ps->psFile, "0.0 0.0 0.8 setrgbcolor\n" );
	ps->psCurrentColor= PScolorLINK;
	}
    }

void utilPsSetTextColor(	PrintingState *	ps )
    {
    if  ( ps->psCurrentColor != PScolorTEXT )
	{
	fprintf( ps->psFile, "0 setgray\n" );
	ps->psCurrentColor= PScolorTEXT;
	}
    }

/************************************************************************/
/*									*/
/*  Go to the next page.						*/
/*									*/
/************************************************************************/

static void appPsPageOperator(		const char *		operator,
					const PrintingState *	ps,
					int			documentPage )
    {
    fprintf( ps->psFile, "%s %% Page %d # %d Sheet %d\n", operator,
				    documentPage+ 1,
				    ps->psPagesPrinted+ 1,
				    ps->psSheetsPrinted+ 1 );
    }

void appPsStartPage(	PrintingState *			ps,
			int				documentPage )
    {
    if  ( ps->psNup == 1			||
	  ps->psPagesPrinted % ps->psNup == 0	)
	{
	if  ( ps->psNup == 1 )
	    {
	    fprintf( ps->psFile, "%%%%Page: %d %d\n",
				    documentPage+ 1, ps->psSheetsPrinted+ 1 );
	    }
	else{
	    fprintf( ps->psFile, "%%%%Page: (%d ..) %d\n",
				    documentPage+ 1, ps->psSheetsPrinted+ 1 );
	    }

	utilPsBoundingBoxComment( ps, "PageBoundingBox", "PageOrientation" );

	fprintf( ps->psFile, "%%%%BeginPageSetup\n" );

	ps->psCurrentTransform= ps->psTransform;
	}
    else{
	ps->psCurrentTransform=
			ps->psNupTransforms[ps->psPagesPrinted % ps->psNup];
	}

    appPsPageOperator( "gsave", ps, documentPage );

#   if 0
    fprintf( ps->psFile, "newpath %d %d moveto\n",
		ps->psSheetBoundingBox.drX0, ps->psSheetBoundingBox.drY0 );
    fprintf( ps->psFile, "%d %d lineto\n",
		ps->psSheetBoundingBox.drX1, ps->psSheetBoundingBox.drY0 );
    fprintf( ps->psFile, "%d %d lineto\n",
		ps->psSheetBoundingBox.drX1, ps->psSheetBoundingBox.drY1 );
    fprintf( ps->psFile, "%d %d lineto\n",
		ps->psSheetBoundingBox.drX0, ps->psSheetBoundingBox.drY1 );
    fprintf( ps->psFile, "%d %d lineto\n",
		ps->psSheetBoundingBox.drX0, ps->psSheetBoundingBox.drY0 );
    fprintf( ps->psFile, "stroke\n" );
#   endif

    fprintf( ps->psFile, "[ %g %g %g %g %g %g ] concat\n",
					ps->psCurrentTransform.ntAxx,
					ps->psCurrentTransform.ntAxy,
					ps->psCurrentTransform.ntAyx,
					ps->psCurrentTransform.ntAyy,
					ps->psCurrentTransform.ntTx,
					ps->psCurrentTransform.ntTy );

    ps->psCurrentColor= PScolorNONE;
    if  ( ps->psUseLinkColor )
	{ utilPsSetLinkColor( ps );	}
    else{ utilPsSetTextColor( ps );	}

    if  ( ps->psNup == 1			||
	  ps->psPagesPrinted % ps->psNup == 0	)
	{ fprintf( ps->psFile, "%%%%EndPageSetup\n" );	}

    return;
    }

void utilPsFinishPage(	PrintingState *		ps,
			int			documentPage,
			int			asLast )
    {
    if  ( asLast					||
	  ps->psNup == 1				||
	  ( ps->psPagesPrinted+ 1 ) % ps->psNup == 0	)
	{
	appPsPageOperator( "showpage grestore", ps, documentPage );

	fprintf( ps->psFile, "%%%%PageTrailer\n" );

	ps->psPagesPrinted++;
	ps->psSheetsPrinted++;
	}
    else{
	appPsPageOperator( "grestore", ps, documentPage );

	ps->psPagesPrinted++;
	}

    if  ( asLast )
	{
	fprintf( ps->psFile, "%%%%Trailer\n" );
	fprintf( ps->psFile, "%%%%Pages: %d\n", ps->psSheetsPrinted );
	fprintf( ps->psFile, "%%%%EOF\n" );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Define a font encoding.						*/
/*									*/
/*  1)  Store an array with the desired name in the cirrent dictionary.	*/
/*  2)  Store '.notdef' in all positions of the array.			*/
/*  3)  Store glyph names in the array.					*/
/*									*/
/************************************************************************/

void appPsDefineFontEncoding(	FILE *			f,
				const char *		name,
				const char * const *	glyphNames )
    {
    int		i;

    /*  1  */
    fprintf( f, "/%s 256 array def\n", name );

    /*  2  */
    fprintf( f, "0 1 255 { %s exch /.notdef put } for\n", name );

    /*  3  */
    fprintf( f, "%s\n", name );

    for ( i= 0; i < 256; i++ )
	{
	if  ( glyphNames[i] )
	    { fprintf( f, "  dup %3d /%s put\n", i, glyphNames[i] ); }
	}

    fprintf( f, "pop\n\n" );

    return;
    }

/************************************************************************/
/*									*/
/*  Make function names for setting the fonts in a list.		*/
/*									*/
/************************************************************************/

void appPsFontNames(	FILE *				f,
			const PostScriptFaceList *	psfl,
			int *				encodingDefined,
			int				allFonts )
    {
    const PostScriptFace *	psf;

    int				i;
    int				started= 0;

    const char *		encodingSuffix= (const char *)0;
    const char *		encodingArray= (const char *)0;

    psf= psfl->psflFaces;
    for ( i= 0; i < psfl->psflFaceCount; psf++, i++ )
	{
	if  ( psf->psfEncodingUsed >= 0			&&
	      psf->psfEncodingUsed < ENCODINGps_COUNT	&&
	      ! encodingDefined[psf->psfEncodingUsed]	)
	    {
	    appPsDefineFontEncoding( f,
			PS_Encodings[psf->psfEncodingUsed].fcEncodingArray,
			PS_Encodings[psf->psfEncodingUsed].fcGlyphNames );

	    encodingDefined[psf->psfEncodingUsed]= 1;
	    }
	}

    psf= psfl->psflFaces;
    for ( i= 0; i < psfl->psflFaceCount; psf++, i++ )
	{
	const AfmFontInfo *	afi= psf->psfAfi;
	int			done;
	PostScriptFace *	psd;

	if  ( ! psf->psfAppearsInText && ! allFonts )
	    { continue;	}

	if  ( ! started )
	    { fprintf( f, "\n" ); started= 1;	}

	psd= psfl->psflFaces;
	for ( done= 0; done < i; psd++, done++ )
	    {
	    if  ( psd->psfEncodingUsed == psf->psfEncodingUsed	&&
		  ! strcmp( afi->afiFontName, psd->psfAfi->afiFontName ) )
		{ break;	}
	    }

	if  ( psf->psfEncodingUsed >= 0			&&
	      psf->psfEncodingUsed < ENCODINGps_COUNT	)
	    {
	    encodingSuffix= PS_Encodings[psf->psfEncodingUsed].fcEncodingSuffix;
	    encodingArray= PS_Encodings[psf->psfEncodingUsed].fcEncodingArray;
	    }

	if  ( encodingSuffix && done >= i )
	    {
	    fprintf( f, "/%s findfont dup length dict begin\n",
						    afi->afiFontName );
	    fprintf( f, "  {\n" );
	    fprintf( f, "    1 index /FID ne\n" );
	    fprintf( f, "      { def } { pop pop } ifelse\n" );
	    fprintf( f, "  } forall\n");
	    fprintf( f, "  /Encoding %s def currentdict\n", encodingArray );
	    fprintf( f, "end " );

	    fprintf( f, "/%s%s exch definefont pop\n\n",
					afi->afiFontName,
					encodingSuffix?encodingSuffix:"" );
	    }
	}

    psf= psfl->psflFaces;
    for ( i= 0; i < psfl->psflFaceCount; psf++, i++ )
	{
	if  ( ! psf->psfAppearsInText && ! allFonts )
	    { continue;	}

	encodingSuffix= (const char *)0;

	if  ( psf->psfEncodingUsed >= 0			&&
	      psf->psfEncodingUsed < ENCODINGps_COUNT	)
	    {
	    encodingSuffix= PS_Encodings[psf->psfEncodingUsed].fcEncodingSuffix;
	    }

	fprintf( f, "/%s\t{ /%s%s exch selectfont } def\n",
				psf->psfFontId,
				psf->psfAfi->afiFontName,
				encodingSuffix?encodingSuffix:"" );
	}

    fprintf( f, "\n" );

    return;
    }

void appPsListFontNames(	FILE *				f,
				const PostScriptFaceList *	psfl )
    {
    if  ( psfl->psflFaceCount > 0 )
	{
	int			i;
	const PostScriptFace *	psf= psfl->psflFaces;

	fprintf( f, "%%%%DocumentFonts: %s\n", psf->psfAfi->afiFontName );
	psf++;

	for ( i= 1; i < psfl->psflFaceCount; psf++, i++ )
	    {
	    int		done;

	    for ( done= 0; done < i; done++ )
		{
		if  ( ! strcmp( psfl->psflFaces[done].psfAfi->afiFontName,
						psf->psfAfi->afiFontName ) )
		    { break;	}
		}

	    if  ( done >= i )
		{
		fprintf( f, "%%%%+ %s\n", psf->psfAfi->afiFontName );
		}
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Write the header for an EPS file.					*/
/*									*/
/************************************************************************/

void utilPsFoldComment(		FILE *			f,
				const char *		name,
				const char *		value )
    {
    int         done;
    int		len= strlen( value );

    int		tl= strlen( name );

    while( len > 0 )
	{
	int     l;

	l= len;

	if  ( len+ tl > 72 )
	    {
	    int         ll;

	    l= ll= 0;
	    while( ll+ tl < 72 && ll < len )
		{
		if  ( isspace( value[ll] ) )
		    { l= ll+ 1; }

		ll++;
		}

	    if  ( l == 0 )
		{ l= ll;        }
	    else{
		while( isspace( value[l] ) && l < len )
		    { l++;      }
		}
	    }

	fprintf( f, "%%%%%s ", name );
	name= "+"; tl= 1;

	for ( done= 0; done < l; done++ )
	    {
	    putc( value[done], f );
	    }

	len -= l; value += l;
	fprintf( f, "\n" );
	}

    return;
    }

void appPsWriteEpsHeader(	FILE *			f,
				const char *		creator,
				const char *		title,
				int			pointsWide,
				int			pointsHigh )
    {
    time_t			now;

    now= time( (time_t *)0 );

    fprintf( f, "%%!PS-Adobe-3.0 EPSF-3.0\n" );

    if  ( creator )
	{ utilPsFoldComment( f, "Creator:", creator );	}

    if  ( title )
	{ utilPsFoldComment( f, "Title:", title );	}

    fprintf( f, "%%%%CreationDate: %s", ctime(&now) );
    fprintf( f, "%%%%BoundingBox: 0 0 %d %d\n", pointsWide, pointsHigh );
    fprintf( f, "%%%%EndComments\n");

    return;
    }

/************************************************************************/
/*									*/
/*  Define a procedure from an array of lines of PostScript code.	*/
/*  Empty lines and lines completely consisting of a comment are	*/
/*  skipped.								*/
/*									*/
/************************************************************************/

void utilPsDefineProcedure(	FILE *			f,
				const char **		lines,
				int			count )
    {
    int		i;

    for ( i= 0; i < count; lines++, i++ )
	{
	const char *	s= *lines;

	while( isspace( *s ) )
	    { s++;	}

	if  ( ! *s || *s == '%' )
	    { continue;	}

	fprintf( f, "%s\n", *lines );
	}

    fprintf( f, "\n" );
    }

/************************************************************************/
/*									*/
/*  Write the header for an EPS file.					*/
/*									*/
/************************************************************************/

static const char * appPsPdfmarkEmulation[]=
    {
    "%%pdfmark emulation",
    "/pdfmark where",
    "    { pop }",
    "    { userdict /pdfmark /cleartomark load put }",
    "ifelse",
    };

void appPsSetPdfmarkEmulation(	FILE *	f )
    {
    utilPsDefineProcedure( f, appPsPdfmarkEmulation,
		    sizeof(appPsPdfmarkEmulation)/sizeof(const char *) );

    return;
    }

static const char * appPsRectfillEmulation[]=
    {
    "%%rectfill emulation for one rectangle only",
    "/rectfill where",
    "    { pop }",
    "    { /rectfill",
    "        { 4 2 roll moveto 2 copy",
    "            0 exch rlineto 0 rlineto ",
    "            neg 0 exch rlineto pop closepath fill",
    "        } bind def",
    "    }",
    "ifelse",
    };

void appPsSetRectfillEmulation(	FILE *	f )
    {
    utilPsDefineProcedure( f, appPsRectfillEmulation,
		    sizeof(appPsRectfillEmulation)/sizeof(const char *) );

    return;
    }

static const char * appPsSelectfontEmulation[]=
    {
    "%%selectfont emulation",
    "/selectfont where",
    "    { pop }",
    "    { /selectfont",
    "        { exch findfont exch dup type /arraytype eq",
    "          { makefont } { scalefont } ifelse",
    "          setfont",
    "        } bind def",
    "    }",
    "ifelse",
    };

void appPsSetSelectfontEmulation(	FILE *	f )
    {
    utilPsDefineProcedure( f, appPsSelectfontEmulation,
		    sizeof(appPsSelectfontEmulation)/sizeof(const char *) );

    return;
    }

/************************************************************************/
/*									*/
/*  Define procedures to Begin/End the inclusion of an EPS graphic.	*/
/*									*/
/*  *)  Copied directly from the EPS spec.				*/
/*  1)  I spent some time wondering why the 'begin' operator was not	*/
/*	matched by an 'end' operator. The matching end is in the loop	*/
/*	as 'op_count' is remembered before the copy of 'userdict' is	*/
/*	pushed.								*/
/*									*/
/************************************************************************/

static const char * appBeginEPSF[]=
    {
    "/BeginEPSF",
    "    {",
    "    /b4_Inc_state save def          % Save state for cleanup",
    "    /dict_count countdictstack def  % Count objects on dict stack",
    "    /op_count count 1 sub def       % Count objects on operand stack",
	 /*  1  */
    "    userdict begin                  % Push userdict on dict stack",
    "    /showpage { } def               % Redefine showpage, { } = null proc",
    "    0 setgray 0 setlinecap          % Prepare graphics state",
    "    1 setlinewidth 0 setlinejoin",
    "    10 setmiterlimit [ ] 0 setdash newpath",
    "    /languagelevel where            % If level not equal to 1 then",
    "    { pop languagelevel             % set strokeadjust and",
    "        1 ne                        % overprint to their defaults.",
    "            { false setstrokeadjust false setoverprint",
    "            } if",
    "        } if",
    "    } bind def",
    };

static const char * appEndEPSF[]=
    {
    "/EndEPSF",
    "    {",
    "    count op_count sub {pop} repeat % Clean up stacks",
	 /*  1  */
    "    countdictstack dict_count sub {end} repeat",
    "    b4_Inc_state restore",
    "    } bind def",
    };

void appPsDefineEpsProcs(	FILE *		f )
    {
    utilPsDefineProcedure( f, appBeginEPSF,
				sizeof(appBeginEPSF)/sizeof(const char *) );

    utilPsDefineProcedure( f, appEndEPSF,
				sizeof(appEndEPSF)/sizeof(const char *) );

    return;
    }

void appPsBeginEpsObject(	PrintingState *		ps,
				int			x0Twips,
				int			y0Twips,
				int			llxTwips,
				int			llyTwips,
				int			urxTwips,
				int			uryTwips,
				const unsigned char *	file )
    {
    if  ( ! file )
	{ file= (const unsigned char *)"??";	}

    fprintf( ps->psFile, "BeginEPSF\n" );

    fprintf( ps->psFile, "[ %d %d %d %d %d %d ] concat\n",
				    20, 0, 0, -20, x0Twips, y0Twips );

    fprintf( ps->psFile, "newpath %d %d moveto ",
					    llxTwips/20, llyTwips/20 );
    fprintf( ps->psFile, "%d %d lineto ",
					    urxTwips/20, llyTwips/20 );
    fprintf( ps->psFile, "%d %d lineto ",
					    urxTwips/20, uryTwips/20 );
    fprintf( ps->psFile, "%d %d lineto ",
					    llxTwips/20, uryTwips/20 );
    fprintf( ps->psFile, "closepath clip\n" );

    fprintf( ps->psFile, "%%%%BeginDocument: (" );
    appPsPrintString( ps->psFile, file, strlen( (char *)file ) );
    fprintf( ps->psFile, ")\n" );

    return;
    }


void appPsEndEpsObject(	PrintingState *		ps )
    {
    fprintf( ps->psFile, "%%%%EndDocument\n" );
    fprintf( ps->psFile, "EndEPSF\n" );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert the destination of a pdfmark jump in the Printout.		*/
/*									*/
/************************************************************************/

int utilPsDestPdfmark(		PrintingState *		ps,
				int			lineTop,
				const char *		refName,
				int			refSize )
    {
    NupTransform *	nt= &(ps->psCurrentTransform);
    int			top;
    int			x= 0;
    int			y= lineTop;

    top= NUP_Y( x, y, nt );

    fprintf( ps->psFile, "[ /Dest" );
    fprintf( ps->psFile, "  /%.*s\n", refSize, refName );

    fprintf( ps->psFile, "  /View [ /XYZ null %d null ]\n", top );

    fprintf( ps->psFile, "/DEST pdfmark\n" );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initialize printing geometry.					*/
/*									*/
/************************************************************************/

void utilPsInitPrintGeometry(	PrintGeometry *	pg )
    {
    appInitDocumentGeometry( &(pg->pgSheetGeometry) );

    pg->pgRotate90= 0;
    pg->pgCenterHorizontally= 0;
    pg->pgNup= 1;
    pg->pgHorizontal= 0;

    return;
    }

