/*
 * dvi.c - a device independent DVI interpreter
 * by Hirotsugu Kakugawa
 *
 *  1 Mar 1996  Version 1.0
 * 10 Dec 1996  Version 1.1
 * 20 Oct 1996  Version 2.0    Based on VFlib 3.0
 * 19 Dev 1996  Version 2.1    Based on VFlib 3.1
 * 21 Feb 1997  Version 2.2    Added dvi property
 *  4 Mar 1997  Version 2.2.1  Added DVI_PROP_SCALE_FONT_IF_NOT_EXIST.
 *  5 Mar 1997  Version 2.2.2  Added prop_set, _add, _del args.
 *  5 Mar 1997  Version 2.3    Upgraded for VFlib 3.2
 * 22 Mar 1997  Version 2.4    
 *  2 Jun 1997  Change: put_bitmap() and put_rectangle() are not essential
 *              user functions. Added color and put_pixmap interfaces. 
 *  5 Aug 1997  Version 2.5    Based on VFlib 3.3
 * 31 Jul 1998  Version 2.5.2  VFlib 3.4 compatible. Added frame buffers.  
 * 27 Aug 1998  Version 2.5.3  Support for pTeX vertical writing mode for
 *                             Japanese text.
 * 27 Nov 1998  Version 2.5.6  Support for DVI filenames without ".dvi".
 *                Added properties DVI_PROP_SKIP_RENDERING and 
 *                DVI_PROP_SKIP_SPECIALS for fast printing of dvi specials.
 * 17 Dec 1998  Version 2.5.7  Added DVI_PROP_LIST_FONTS and 
 *                DVI_PROP_LIST_MISSING_FONTS props for fancy font list.
 *                (See also "dvifontlist" program.)
 *  3 Jun 1999  Version 2.5.8  Bug in put_rectangle() is fixed.
 *  9 Aug 1999  Version 2.5.9  VFlib 3.6 compatible.
 * 19 Sep 1999  Version 2.5.10 Added DVI_PROP_INCREMENTAL_EPS_DISPLAY prop.
 *  1 Oct 1999  Support for psfig.sty.
 *  2 Oct 1999  Added struct dvi_s_page_no in DVI obj. (page number string)
 * 13 Oct 1999  Added DVI_file_modified().
 * 28 Dec 1999  Version 2.5.11
 *  4 Jan 2000  Version 2.5.12
 * 24 May 2000  Version 2.5.14
 *  9 Oct 2001  Version 2.7.1  Enhanced DVI update check
 */
/*
 * Copyright (C) 1996-2001  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif
#ifdef HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <vararg.h>
#endif
#if defined(HAVE_STRING_H)
#  include  <string.h>
#endif
#if defined(HAVE_STRINGS_H)
#  include  <strings.h>
#endif
#include <math.h>

#include "VFlib-3_6.h"
#include "dvi-2_6.h"
#include "defs.h"
#include "cache.h"
#include "private.h"
#include "special.h"
#include "spc_ps.h"
#include "dviinst.h"


#define  HASH_TABLE_SIZE  191
#define  HASH(i)  ((i)%HASH_TABLE_SIZE)



Private int  DVI_open_font(DVI,DVI_DEVICE);
Private int  DVI_dispose(DVI,DVI_DEVICE);
Private int  DVI_draw_page(DVI,DVI_DEVICE,int,double);
Private int  DVI_file_modified(DVI,char*);
Private int  DVI_continue_to_context(DVI,DVI_DEVICE,DVI_CONTEXT,double);
Private DVI_CONTEXT DVI_get_current_context(DVI,DVI_DEVICE);
Private int         DVI_restore_context(DVI,DVI_DEVICE,DVI_CONTEXT);
Private int         DVI_free_context(DVI,DVI_DEVICE,DVI_CONTEXT);
Private DVI_CONTEXT DVI_alloc_context(DVI,DVI_DEVICE);
Private int  dvi_set_filestamp(DVI dvi, DVI_DEVICE dev);

Private int  dvi_paper_size_in_dvi(DVI dvi, DVI_DEVICE dev);
Private BOOL read_dvi_file_info(DVI,DVI_DEVICE,FILE*);
Private BOOL read_dvi_file_index(DVI,DVI_DEVICE,FILE*);
Private int  dvi_page(DVI,DVI_DEVICE,int,double);

Private int  font_define(DVI,DVI_DEVICE,UINT4,UINT4,INT4,INT4,int,int,char*);
Private int  font_do_open(DVI,DVI_DEVICE,FONT);
Private int  font_switch(DVI,DVI_DEVICE,long);
Private int  font_get_metric(DVI,DVI_DEVICE,UINT4,double*,double*,
			     double,double);

Private DVI_PROPERTY  DVI_property_dup(DVI_PROPERTY);
Private void          DVI_property_add(DVI_PROPERTY,DVI_PROPERTY);
#if 0
Private void          DVI_property_del(DVI_PROPERTY,DVI_PROPERTY);
#endif


struct s_cache_key {
  int         font_id;
  double      scale;
  double      scale_h;
  double      scale_v;
  int         rotation;
  long        code_point;
  DVI_DEVICE  dev;
};
typedef struct s_cache_key  *CACHE_KEY;
#define ENV_FONT_CACHE_SIZE  "DVILIB_FONT_CACHE_SIZE"
Private int  font_cache_size = FONT_CACHE_SIZE;
Private VF_BITMAP  font_load_bitmap(CACHE,CACHE_KEY,int);


#define UNIT2INCH(x)  \
           ((double)(x)*PD(dvi,dimen_in))
#define H_UNIT2PIXEL(x) \
           (long)(UNIT2INCH(x)*(dev->h_dpi)/(double)PD(dvi,shrink))
#define V_UNIT2PIXEL(x) \
           (long)(UNIT2INCH(x)*(dev->v_dpi)/(double)PD(dvi,shrink))
#define MAG2IMAG(m)  ((m)*1000)

Private int    altm_draw_ps_figures(DVI_DEVICE,DVI);
Private void   altm_before_ps_figure(DVI_DEVICE,DVI,char*,long,long);
Private void   altm_after_ps_figure(DVI_DEVICE,DVI);
Private int    altm_ps_figure_display_step(DVI_DEVICE,DVI,long,long);
#if 0
Private int    altm_special_command_undef(DVI_DEVICE,DVI,char*,long,long);
#endif
Private char*  altm_gs_program_path(DVI_DEVICE,DVI);
Private int    altm_gs_timeout_give_up(DVI_DEVICE,DVI);
Private int    altm_gs_timeout_value(DVI_DEVICE,DVI);
Private void   altm_message_null(DVI_DEVICE,DVI,char*,...);
Private void   altm_message_advice(DVI_DEVICE,DVI,char*,...);
Private void   altm_message_warning(DVI_DEVICE,DVI,char*,...);
Private void   altm_message_error(DVI_DEVICE,DVI,char*,...);
Private void   altm_message_fatal(DVI_DEVICE,DVI,char*,...);
Private char*  altm_temp_dir(DVI_DEVICE,DVI);

#define ENV_DEBUG    "DVILIB_DEBUG"
Glocal char  *dvi_debug_flags = NULL;
Glocal int    dvi_debug(char);

#define SCALEBOX 1



/**
 ** Initialization of DVIlib
 **/

static int  dvi_setup_flag = 0;

Public void
DVI_setup(void)
{
  if (dvi_setup_flag == 1)
    return;

  DVI_read_paper_size_db();
  DVI_read_device_mode_db();

  dvi_setup_flag = 1;
}


/**
 ** Initialization of DVIlib
 **/

static int  dvi_inited_flag = 0;

Public int 
DVI_init(char *vflibcap, char *vflib_params)
{
  int   v;
  char *s;

  if (dvi_inited_flag > 0)
    return 0;
  if (dvi_inited_flag < 0)
    return -1;

  dvi_debug_flags = getenv(ENV_DEBUG);

  font_cache_size = FONT_CACHE_SIZE;
  if ((s = getenv(ENV_FONT_CACHE_SIZE)) != NULL)
    if ((font_cache_size = atoi(s)) < 2)
      font_cache_size = FONT_CACHE_SIZE;
  if (dvi_debug('C'))
    printf("DVIlib: Cache size = %d\n", font_cache_size);

  if (dvi_setup_flag == 0){
    DVI_read_paper_size_db();
    DVI_read_device_mode_db();
    dvi_setup_flag = 1;
  }

  if (dvi_inited_flag == 0){
    v = VF_Init(vflibcap, vflib_params);
    if (dvi_debug('I')){
      printf("DVIlib: VF_Init(\"%s\", \"%s\")\n", 
	     vflibcap, vflib_params);
      printf("DVIlib:   ==> %d\n", v);
    }
    if (v < 0){
      dvi_inited_flag = -1; 
      return -1;
    }
    dvi_inited_flag = 1; 
  }

  return 0;
}


/**
 ** Create DVI Object
 **/
Public DVI 
DVI_create(DVI_DEVICE dev, char* path, DVI_PROPERTY props)
{
  DVI    dvi;
  PDATA  pd;
  char   *dvi_file_name = NULL;
  int    f_ac;
#ifdef HAVE_STAT 
  struct stat  st;
#endif

  if (dvi_inited_flag <= 0){
    if (DVI_init(NULL, NULL) < 0)
      return NULL;
  }

  if (DEV_METHOD_UNDEF(dev,draw_ps_figures))
    DEV_METHOD_SET(dev,draw_ps_figures, altm_draw_ps_figures);
  if (DEV_METHOD_UNDEF(dev,before_ps_figure))
    DEV_METHOD_SET(dev,before_ps_figure, altm_before_ps_figure);
  if (DEV_METHOD_UNDEF(dev,after_ps_figure))
    DEV_METHOD_SET(dev,after_ps_figure, altm_after_ps_figure);
  if (DEV_METHOD_UNDEF(dev,ps_figure_display_step))
    DEV_METHOD_SET(dev,ps_figure_display_step, altm_ps_figure_display_step);
  if (DEV_METHOD_UNDEF(dev,gs_program_path))
    DEV_METHOD_SET(dev,gs_program_path, altm_gs_program_path);
  if (DEV_METHOD_UNDEF(dev,gs_timeout_give_up))
    DEV_METHOD_SET(dev,gs_timeout_give_up, altm_gs_timeout_give_up);
  if (DEV_METHOD_UNDEF(dev,gs_timeout_value))
    DEV_METHOD_SET(dev,gs_timeout_value, altm_gs_timeout_value);
  if (DEV_METHOD_UNDEF(dev,message_advice))
    DEV_METHOD_SET(dev,message_advice, altm_message_null);
  if (DEV_METHOD_UNDEF(dev,message_warning))
    DEV_METHOD_SET(dev,message_warning, altm_message_null);
  if (DEV_METHOD_UNDEF(dev,message_error))
    DEV_METHOD_SET(dev,message_error, altm_message_null);
  if (DEV_METHOD_UNDEF(dev,message_fatal))
    DEV_METHOD_SET(dev,message_fatal, altm_message_null);
  if (DEV_METHOD_UNDEF(dev,temp_dir))
    DEV_METHOD_SET(dev,temp_dir, altm_temp_dir);
  
  dvi_file_name = NULL;
  ALLOC_IF_ERR(dvi, struct dvi_s_file)
    return NULL;
  ALLOC_IF_ERR(pd, struct private_data){
    free(dvi);
    return NULL;
  }


#ifdef HAVE_STAT 
# if defined(__linux__)
  f_ac = ((stat(path, &st) >= 0) && S_ISREG(st.st_mode));
# else
  f_ac = ((stat(path, &st) >= 0) && ((st.st_mode & S_IFMT) == S_IFREG));
# endif
#else
  f_ac = (access(path, R_OK) >= 0);
#endif

  if (f_ac) {
    STRDUP(dvi_file_name, path);
  } else {
    if ((dvi_file_name = (char*)malloc(strlen(path)+5)) != NULL){
      sprintf(dvi_file_name, "%s%s", path, ".dvi");
      if (access(dvi_file_name, R_OK) != 0){
	sprintf(dvi_file_name, "%s%s", path, "dvi");
	if (access(dvi_file_name, R_OK) != 0){
	  free(dvi_file_name);
	  dvi_file_name = NULL;
	}
      }
    }
  }
  if (dvi_file_name == NULL){
    free(dvi);
    free(pd);
    return NULL;
  }

  dvi->open_font             = DVI_open_font;
  dvi->dispose               = DVI_dispose;
  dvi->draw_page             = DVI_draw_page;
  dvi->file_modified         = DVI_file_modified;
  dvi->continue_to_context   = DVI_continue_to_context;
  dvi->get_current_context   = DVI_get_current_context;
  dvi->free_context          = DVI_free_context;
  dvi->pages                 = 0;
  dvi->error                 = 0;
  dvi->job_name[0]           = '\0';
  dvi->privates              = (void*)pd;

  if (props == NULL)
    PD(dvi,property) = DVI_property_alloc_default();
  else
    PD(dvi,property) = DVI_property_dup(props);
  if (PD(dvi,property) == NULL)
    goto ERROR;
  PD(dvi,font_opened)  = FALSE;
  PD(dvi,dvi_file)     = NULL;
  PD(dvi,font_cache)   = NULL;
  PD(dvi,file_name)    = dvi_file_name;
  PD(dvi,stack)        = NULL;
  PD(dvi,page_index)   = NULL;
  PD(dvi,font_table)   = NULL;
  PD(dvi,dvi_file)     = NULL;
  PD(dvi,color_stack)         = NULL;
  PD(dvi,color_background)    = NULL;
  PD(dvi,color_page_stack_fg) = NULL;
  PD(dvi,color_page_bg)       = NULL;
  PD(dvi,color_page_bg_hint)  = NULL;

  if ((PD(dvi,dvi_file) = fopen(PD(dvi,file_name), FOPEN_RD_MODE_BIN)) == NULL)
    goto ERROR;

  if (dvi_set_filestamp(dvi, dev) < 0)
    goto ERROR;

  if (read_dvi_file_info(dvi, dev, PD(dvi,dvi_file)) == FALSE)
    goto ERROR_NO_MEMORY;
  if (DVI_file_modified(dvi, PD(dvi,file_name)) == 1)
    goto ERROR;

  if (read_dvi_file_index(dvi, dev, PD(dvi,dvi_file)) == FALSE)   /*XXX*/
    goto ERROR_NO_MEMORY;
  if (DVI_file_modified(dvi, PD(dvi,file_name)) == 1)
    goto ERROR;

  ALLOCN_IF_ERR(PD(dvi,stack), struct stack_frame, PD(dvi,max_stack)+1)
    goto ERROR_NO_MEMORY;
  ALLOCN_IF_ERR(PD(dvi,font_table), FONT, HASH_TABLE_SIZE)
    goto ERROR_NO_MEMORY;
  PD(dvi,current_fnt) = 0;
  PD(dvi,current_fid) = -1;
  PD(dvi,font_cache) 
    = dvi_cache_create(font_cache_size, FONT_CACHE_HASH_SIZE, 
		       (void*(*)(CACHE,void*,int))font_load_bitmap, 
		       (void(*)(void*))VF_FreeBitmap);
  if (dvi_color_init(dvi, dev) < 0)
    goto ERROR;
  PD(dvi,gstate_stack_ptr) = 0;
  PD(dvi,gstate_transformed) = 0;
  PD(dvi,gstate_scale_h) = 1;
  PD(dvi,gstate_scale_v) = 1;
  PD(dvi,gstate_angle)   = 0;

  PD(dvi,alive) = TRUE;

  if (DVI_PROPERTY_TEST(PD(dvi,property), DVI_PROP_ASYNC_GS_INVOCATION))
    dvi_special_invoke_gs(dvi, dev);  /* invoke gs if it is not running */ 

  dvi_paper_size_in_dvi(dvi, dev);

  return dvi;


ERROR_NO_MEMORY:
ERROR:
  DVI_dispose(dvi, dev);
  return NULL;
}


Private int
dvi_set_filestamp(DVI dvi, DVI_DEVICE dev)
{
#ifdef HAVE_STAT 
  struct stat  st;
#endif

  PD(dvi,file_timestamp1) = 0;
  PD(dvi,file_timestamp2) = 0;

#ifdef HAVE_STAT 
  if (stat(PD(dvi,file_name), &st) < 0)
    return -1;

#ifndef _POSIX_SOURCE
# if defined(__svr4__)
  PD(dvi,file_timestamp1) = st.st_mtime;
  PD(dvi,file_timestamp2) = 0;
# elif defined(__linux__)
  PD(dvi,file_timestamp1) = st.st_mtime;
  PD(dvi,file_timestamp2) = 0;
# else
  PD(dvi,file_timestamp1) = st.st_mtimespec.tv_sec;
  PD(dvi,file_timestamp2) = st.st_mtimespec.tv_nsec;
# endif
#else
  PD(dvi,file_timestamp1) = st.st_mtime;
  PD(dvi,file_timestamp2) = st.st_mtimensec;
#endif

#endif /*HAVE_STAT*/

  return 0;
}



/** 
 ** Check if a DVI file is modified or not
 **/
Private int
DVI_file_modified(DVI dvi, char *f)
{
#ifndef HAVE_STAT 
  return 0;   /* not modified */
#else
  struct stat  st;

  if (stat(PD(dvi,file_name), &st) < 0)
    return 1;   /* modified (probably, removed) */
#ifndef _POSIX_SOURCE
# if defined(__svr4__)
  if (PD(dvi,file_timestamp1) != (long)st.st_mtime)
    return 1;   /* modified */
# elif defined(__linux__)
  if (PD(dvi,file_timestamp1) != (long)st.st_mtime)
    return 1;   /* modified */
# else
  if ((PD(dvi,file_timestamp1) != (long)st.st_mtimespec.tv_sec)
      || (PD(dvi,file_timestamp2) != (long)st.st_mtimespec.tv_nsec))
    return 1;   /* modified */
# endif
#else
  if ((PD(dvi,file_timestamp1) != (long)st.st_mtime)
      || (PD(dvi,file_timestamp2) != (long)st.st_mtimensec))
    return 1;   /* modified */
#endif
  return 0;  /* not modified */
#endif
}


/**
 ** Allocate DVI Device Object 
 **/
Public DVI_DEVICE
DVI_device_alloc(void)
{
  DVI_DEVICE  dev;

  if ((dev = (DVI_DEVICE)calloc(1, sizeof(struct dvi_s_device))) == NULL)
    return NULL;
  dev->h_dpi                  = DEFAULT_DPI;
  dev->v_dpi                  = DEFAULT_DPI;
  dev->put_rectangle          = NULL;
  dev->put_bitmap             = NULL;
  dev->put_pixmap_rgb         = NULL;
  dev->put_pixmap_cmyk        = NULL;
  dev->put_graymap            = NULL;
  dev->device_polling         = NULL;
  dev->special_command_hook   = NULL;
  dev->special_command_undef  = NULL;
  dev->draw_ps_figures        = altm_draw_ps_figures;
  dev->before_ps_figure       = altm_before_ps_figure;
  dev->after_ps_figure        = altm_after_ps_figure;
  dev->ps_figure_display_step = altm_ps_figure_display_step;
  dev->gs_program_path        = altm_gs_program_path;
  dev->gs_timeout_value       = altm_gs_timeout_value;
  dev->gs_timeout_give_up     = altm_gs_timeout_give_up;
  dev->color_rgb              = NULL;
  dev->color_cmyk             = NULL;
  dev->color_gray             = NULL;
  dev->color_named            = NULL;
  dev->bgcolor_rgb            = NULL;
  dev->bgcolor_cmyk           = NULL;
  dev->bgcolor_gray           = NULL;
  dev->bgcolor_named          = NULL;
  dev->message_advice         = altm_message_advice;
  dev->message_warning        = altm_message_warning;
  dev->message_error          = altm_message_error;
  dev->message_fatal          = altm_message_fatal;
  return dev;
}


/**
 ** Version of DVIlib
 **/
Public char*
DVI_version(void)
{
  return VERSION;
}




/**
 ** Reding DVI file
 **/
Private BOOL
read_dvi_file_info(DVI dvi, DVI_DEVICE dev, FILE* fp)
{
  UINT1          b1, b2, k1;
  UINT4          post_offset, p_num, p_den, p_mag;
  double         dimen;
  int            i;
  long           ofs;
  
  /** Preamble **/
  b1 = READ_UINT1(fp);
  b2 = READ_UINT1(fp);
  if ((b1 != DVIINST_PRE) || (b2 != 2)){
    dvi->error = DVI_ERR_ILL_FORMAT;
    return FALSE;
  }
  PD(dvi,num) = READ_UINT4(fp);
  PD(dvi,den) = READ_UINT4(fp);
  PD(dvi,mag) = READ_UINT4(fp);

  dimen = ( ((double)PD(dvi,num)/(double)PD(dvi,den)) / (double)10000000.0 )
          * ( (double)PD(dvi,mag)/(double)1000.0 );
  PD(dvi,dimen_m)  = dimen;
  PD(dvi,dimen_cm) = dimen * (double)100.0;
  PD(dvi,dimen_in) = dimen * ((double)100.0/(double)2.54);
  PD(dvi,dimen_pt) = dimen * ((double)100.0/(double)2.54) * (double)72.27;
  
  i = 0;
  for (k1 = READ_UINT1(fp); k1 > 0; --k1){
    if (((i%100)==0) && (DVI_file_modified(dvi, PD(dvi,file_name)) == 1))
      return FALSE;
    dvi->job_name[i] = (char)READ_UINT1(fp);
    i++;
  }
  dvi->job_name[i++] = '\0';
  PD(dvi,offset_first_page) = ftell(fp);

  /** Postamble **/
  i = 0;
  ofs = 0L;
  do {
    if (((i%100)==0) && (DVI_file_modified(dvi, PD(dvi,file_name)) == 1))
      return FALSE;
    --ofs;
    fseek(fp, ofs, SEEK_END);
    b1 = READ_UINT1(fp);
    i++;
  } while (b1 == (UINT1)DVIINST_TRAILER);
  /* check post_post arg */
  dvi->ptex_dvi = 0;
  switch ((int)b1){
  case 2:    /* normal TeX */
    break;
  case 3:    /* Japanese TeX by ASCII Coop. */
    if (DVI_PROPERTY_TEST(PD(dvi,property), DVI_PROP_ASCII_JTEX)){
      dvi->ptex_dvi = 1;
      break;
    }
    dvi->error = DVI_ERR_ILL_VERSION;
    return FALSE;
  default:
    dvi->error = DVI_ERR_ILL_VERSION;
    return FALSE;
  }

  /* check POST_POST inst */
  ofs = ofs - 5L;
  fseek(fp, ofs, SEEK_END);
  if (READ_UINT1(fp) != (UINT1)DVIINST_POSTPOST){
    dvi->error = DVI_ERR_ILL_FORMAT;
    return FALSE;
  }
  /* read pointer to POST inst */
  post_offset = (long)READ_INT4(fp);

  /** POST **/
  fseek(fp, post_offset, SEEK_SET);
  if (READ_UINT1(fp) != (UINT1)DVIINST_POST){
    dvi->error = DVI_ERR_ILL_FORMAT;
    return FALSE;
  }
  PD(dvi,offset_last_page) = READ_UINT4(fp);
  p_num = READ_UINT4(fp);
  p_den = READ_UINT4(fp);
  p_mag = READ_UINT4(fp);
  if (   (p_num != PD(dvi,num)) || (p_den != PD(dvi,den)) 
      || (p_mag != PD(dvi,mag)) ){
    dvi->error = DVI_ERR_ILL_FORMAT;
    return FALSE;
  }
  PD(dvi,max_height) = READ_INT4(fp);
  PD(dvi,max_width)  = READ_INT4(fp);
  PD(dvi,max_stack)  = READ_UINT2(fp);
  dvi->pages       = READ_UINT2(fp);
  PD(dvi,offset_fontdef) = ftell(fp);

  return TRUE;
}

Private BOOL
read_dvi_file_index(DVI dvi, DVI_DEVICE dev, FILE* fp)
{
  int  p, i;

  ALLOCN_IF_ERR(PD(dvi,page_index), long, dvi->pages){
    dvi->error = DVI_ERR_NO_MEMORY;
    return FALSE;
  }
  ALLOCN_IF_ERR(dvi->page_no, struct dvi_s_page_no, dvi->pages){
    free(PD(dvi,page_index));
    PD(dvi,page_index) = NULL;
    dvi->error = DVI_ERR_NO_MEMORY;
    return FALSE;
  }

  p = dvi->pages;
  PD(dvi,page_index)[p-1] = PD(dvi,offset_last_page);
  for ( ; p > 0; p--){
    fseek(fp, PD(dvi,page_index)[p-1]+1, SEEK_SET);
    for (i = 0; i < 10; i++)
      dvi->page_no[p-1].c[i] = READ_UINT4(fp);
    if (p >= 2)
      PD(dvi,page_index)[p-2] = READ_UINT4(fp);
    if (DEV_METHOD_DEF(dev,device_polling)){
      if (DEV_CALL(dev,device_polling)(dev, dvi, DVI_POLL_FILE) != 0)
	return FALSE;
    }
#if 0
    printf("**%d => %ld\n", p, dvi->page_no[p-1].c[0]);
    printf("INDEX: %d %ld\n", p, PD(dvi,page_index)[p-1]);
#endif
  }

  return TRUE;
}

Private int
DVI_open_font(DVI dvi, DVI_DEVICE dev)
{
  UINT1      b1;
  BOOL       val;
  int        i, k, len_dir, len_name;
  UINT4      font_id;
  UINT4      check_sum;
  INT4       design_size_f, design_size_s;
  char       *fontname = NULL;

  if (PD(dvi,font_opened) == TRUE)
    return 0;
  if (PD(dvi,alive) != TRUE){
    dvi->error = DVI_ERR_NOT_ACTIVE;
    return -1;
  }
  if (PD(dvi,dvi_file) == NULL){
    dvi->error = DVI_ERR_INTERNAL;
    return -1;
  }
  fseek(PD(dvi,dvi_file), PD(dvi,offset_fontdef), SEEK_SET);

  val = 0;
  for (;;){
    b1 = READ_UINT1(PD(dvi,dvi_file));
    switch ((int)b1){
    case DVIINST_FNTDEF1:   k = 1; break;
    case DVIINST_FNTDEF2:   k = 2; break;
    case DVIINST_FNTDEF3:   k = 3; break;
    case DVIINST_FNTDEF4:   k = 4; break;
    default:                goto END_FNTDEF;
    }
    font_id        = (UINT4)READ_UINTN(PD(dvi,dvi_file), k);
    check_sum      = READ_UINT4(PD(dvi,dvi_file));
    design_size_s  = READ_INT4(PD(dvi,dvi_file));
    design_size_f  = READ_INT4(PD(dvi,dvi_file));
    len_dir  = (int)READ_UINT1(PD(dvi,dvi_file));
    len_name = (int)READ_UINT1(PD(dvi,dvi_file));
    if (len_dir+len_name > 1024){  /* Too long. Probably DVI is broken. */
      dvi->error = DVI_ERR_NO_MEMORY;
      return -1;
    }
    MALLOC_STR_IF_ERR(fontname, len_dir+len_name+1){
      dvi->error = DVI_ERR_NO_MEMORY;
      return -1;
    }
    for (i = 0; i < len_dir+len_name; i++)
      fontname[i] = (char)READ_UINT1(PD(dvi,dvi_file));
    fontname[i] = '\0';
#if 0
    printf("%d %s %ld\n", (int)font_id, fontname, design_size_f);
#endif
    if (font_define(dvi, dev, font_id, check_sum, 
		    design_size_s, design_size_f, 
		    len_dir, len_name, fontname) < 0){
      dvi->error = DVI_ERR_FONT_NOT_FOUND;
      val = -1;
    }
    free(fontname);
#if 0 /*XXX*/
    if (DEV_METHOD_DEF(dev,device_polling)){
      if (DEV_CALL(dev,device_polling)(dev, dvi, DVI_POLL_FONT) != 0){
	dvi->error = DVI_ERR_INTERRUPTED;
	return -1;
      }
    }
#endif 
  }

END_FNTDEF:
  if (val < 0)
    return -1;

  PD(dvi,font_opened) = TRUE;

  return 0;
}


/**
 ** Dispose DVI Object
 **/
Private int 
DVI_dispose(DVI dvi, DVI_DEVICE dev)
{
  FONT  f, fn;
  int   i;

  if (PD(dvi,font_table) != NULL){
    for (i = 0; i < HASH_TABLE_SIZE; i++){
      for (f = PD(dvi,font_table)[i]; f != NULL; f = fn){
	if (f->fid >= 0){
	  VF_CloseFont(f->fid);
	  f->fid = 0;
	}
	fn = f->next;
	free(f->fname);
	free(f);
      }
    }
  }

  free_if_non_null(PD(dvi,file_name));
  DVI_property_release(PD(dvi,property));
  if (PD(dvi,dvi_file) != NULL)
    fclose(PD(dvi,dvi_file));
  free_if_non_null(PD(dvi,page_index));
  free_if_non_null(dvi->page_no);
  free_if_non_null(PD(dvi,stack));
  free_if_non_null(PD(dvi,font_table));
  dvi_cache_delete(PD(dvi,font_cache));
  dvi_color_deinit(dvi, dev);
  free_if_non_null(PD(dvi,color_background));
  free_if_non_null(dvi->privates);

  PD(dvi,alive) = FALSE;
  free(dvi);

  return 0;
}



/**
 ** Draw (a page)
 **/
Private int 
DVI_draw_page(DVI dvi, DVI_DEVICE dev, int page, double shrink)
{
  if (PD(dvi,alive) != TRUE){
    dvi->error = DVI_ERR_NOT_ACTIVE;
    return -1;
  }

  if ((page < 1) || (dvi->pages < page)){
    dvi->error = DVI_ERR_ILL_PAGE_NO;
    return -1;
  }
  fseek(PD(dvi,dvi_file), PD(dvi,page_index)[page-1], SEEK_SET);

  return dvi_page(dvi, dev, page, shrink);
}


/**
 ** Draw (continue to a context)
 **/
Private int 
DVI_continue_to_context(DVI dvi, DVI_DEVICE dev, DVI_CONTEXT context,
			double shrink)
{
  if (PD(dvi,alive) != TRUE){
    dvi->error = DVI_ERR_NOT_ACTIVE;
    return -1;
  }

  DVI_restore_context(dvi, dev, context);

  return dvi_page(dvi, dev, context->current_page, shrink);
}


/**
 ** Context
 **/
Private DVI_CONTEXT
DVI_get_current_context(DVI dvi, DVI_DEVICE dev)
{
  int          s;
  DVI_CONTEXT  context;

  if (PD(dvi,alive) != TRUE){
    dvi->error = DVI_ERR_NOT_ACTIVE;
    return NULL;
  }

  if ((context = DVI_alloc_context(dvi, dev)) == NULL){
    dvi->error = DVI_ERR_NO_MEMORY;
    return NULL;
  }
  context->current_page = dvi->current_page;
  context->offset = ftell(PD(dvi,dvi_file));
  context->depth  = PD(dvi,stackp);
  for (s = 0; s <= PD(dvi,stackp); s++)
    context->stack[s] = PD(dvi,stack)[s];
  context->fnt = PD(dvi,current_fnt);
  context->fid = PD(dvi,current_fid);
  context->color_stack = dvi_color_stack_copy(dvi, dev);
  if (PD(dvi,color_background) != NULL)
    STRDUP(context->color_bg, PD(dvi,color_background));
  else 
    context->color_bg = NULL;
  context->gstate_stack_ptr   = PD(dvi,gstate_stack_ptr);
  context->gstate_transformed = PD(dvi,gstate_transformed);
  context->gstate_scale_h     = PD(dvi,gstate_scale_h);
  context->gstate_scale_v     = PD(dvi,gstate_scale_v);
  context->gstate_angle       = PD(dvi,gstate_angle);
  for (s = 0; s < DVI_GSTATE_STACK_SIZE; s++){
    context->gstate_stack[s].type    = PD(dvi,gstate_stack[s].type);
    context->gstate_stack[s].scale_h = PD(dvi,gstate_stack[s].scale_h);
    context->gstate_stack[s].scale_v = PD(dvi,gstate_stack[s].scale_v);
    context->gstate_stack[s].angle   = PD(dvi,gstate_stack[s].angle);
  }
  return context;
}

Private int
DVI_restore_context(DVI dvi, DVI_DEVICE dev, DVI_CONTEXT context)
{
  int  s;

  if (PD(dvi,alive) != TRUE){
    dvi->error = DVI_ERR_NOT_ACTIVE;
    return -1;
  }

  dvi->current_page   = context->current_page;
  PD(dvi,current_fnt) = context->fnt;
  PD(dvi,current_fid) = context->fid;
  PD(dvi,font)        = PD(dvi,font_table)[HASH(context->fnt)];
  for (s = 0; s <= context->depth; s++)
    PD(dvi,stack)[s]  = context->stack[s];
  PD(dvi,stackp)      = context->depth;
  fseek(PD(dvi,dvi_file), context->offset, SEEK_SET);
  dvi_color_stack_resume(dvi, dev, context->color_stack, context->color_bg);
  PD(dvi,gstate_stack_ptr)   = context->gstate_stack_ptr;
  PD(dvi,gstate_transformed) = context->gstate_transformed;
  PD(dvi,gstate_scale_h)     = context->gstate_scale_h;
  PD(dvi,gstate_scale_v)     = context->gstate_scale_v;
  PD(dvi,gstate_angle)       = context->gstate_angle;
  for (s = 0; s < DVI_GSTATE_STACK_SIZE; s++){
    PD(dvi,gstate_stack[s].type)     = context->gstate_stack[s].type;
    PD(dvi,gstate_stack[s].scale_h)  = context->gstate_stack[s].scale_h;
    PD(dvi,gstate_stack[s].scale_v)  = context->gstate_stack[s].scale_v;
    PD(dvi,gstate_stack[s].angle)    = context->gstate_stack[s].angle;
  }

  return 0;
}

Private DVI_CONTEXT 
DVI_alloc_context(DVI dvi, DVI_DEVICE dev)
{
  DVI_CONTEXT  context;  

  ALLOC_IF_ERR(context, struct dvi_s_context){
    dvi->error = DVI_ERR_NO_MEMORY;
    return NULL;
  }

  context->offset        = 0;
  context->fnt           = 0;
  context->fid           = 0;
  ALLOCN_IF_ERR(context->stack, struct stack_frame, PD(dvi,max_stack)+1){ 
    free(context);
    dvi->error = DVI_ERR_NO_MEMORY;
    return NULL;
  }
  context->depth         = 0;
  context->color_stack   = NULL;
  context->color_bg      = NULL;

  return context;
}

Private int
DVI_free_context(DVI dvi, DVI_DEVICE dev, DVI_CONTEXT context)
{
  if (context != NULL){
    if (context->stack != NULL)
      free(context->stack);
    if (context->color_stack != NULL)
      dvi_color_stack_free(dvi, dev);
    free(context);
  }
  return 0;
}


/**
 ** DVI Property
 **/
Public int
DVI_property_update(DVI dvi, DVI_DEVICE dev, DVI_PROPERTY props)
{
  int  res;

  res = 0;
  if (PD(dvi,property) != NULL)
    DVI_property_release(PD(dvi,property));

  if (props == NULL){
    PD(dvi,property) = DVI_property_alloc_default();
  } else {
    PD(dvi,property) = DVI_property_dup(props);
  }
  if (PD(dvi,property) == NULL)
    return -1;

  return 0;
}

Public DVI_PROPERTY
DVI_property_alloc(void)
{
  DVI_PROPERTY  p;
  int           id;
  
  ALLOC_IF_ERR(p, struct dvi_s_property)
    return NULL;
  for (id = 0; id < sizeof(p->props); id++)
    p->props[id] = 0;
  return p;
}

Public DVI_PROPERTY
DVI_property_alloc_default(void)
{
  DVI_PROPERTY  p;
  
  if ((p = DVI_property_alloc()) == NULL)
    return NULL;
  DVI_PROPERTY_SET(p, DVI_PROP_ASCII_JTEX);
  DVI_PROPERTY_SET(p, DVI_PROP_SCALE_FONT_IF_NOT_EXIST);
  DVI_PROPERTY_SET(p, DVI_PROP_LATEX2E_GRAPHICS_STY);
  return p;
}

Public void
DVI_property_release(DVI_PROPERTY p)
{
  if (p != NULL)
    free(p);
}

Private void
DVI_property_add(DVI_PROPERTY p, DVI_PROPERTY p1)
{
  int   id;
  
  if ((p == NULL) || (p1 == NULL))
    return;
  for (id = 0; id < sizeof(p1->props); id++){
    if (p1->props[id] == 1)
      p->props[id] = 1;
  }
}

#if 0

Private void
DVI_property_del(DVI_PROPERTY p, DVI_PROPERTY p1)
{
  int   id;
  
  if ((p == NULL) || (p1 == NULL))
    return;
  for (id = 0; id < sizeof(p1->props); id++){
    if (p1->props[id] == 1)
      p->props[id] = 0;
  }
}

#endif

Private DVI_PROPERTY
DVI_property_dup(DVI_PROPERTY p0)
{
  DVI_PROPERTY  p;
  
  if ((p = DVI_property_alloc()) == NULL)
    return NULL;
  DVI_property_add(p, p0);
  return p;
}



/**
 ** The DVI Interpreter
 **/

Private void set_char(DVI,DVI_DEVICE,UINT4,int,int);
Private void set_rule(DVI,DVI_DEVICE,long,long);
Private int  special(DVI,DVI_DEVICE,long);

#define DVI_PTEX        ((dvi->ptex_dvi == 1) \
                          && PROP_TEST(dvi, DVI_PROP_ASCII_JTEX))
#define PTEX_VERT_MODE  ((DD == 1) && DVI_PTEX)


Private int
dvi_page(DVI dvi, DVI_DEVICE dev, int page, double shrink)
{
  UINT1         instr;
  UINT4         code;
  INT4          q;
  long          n, w, h, len;
  int           v;
  double        m;

  dvi->current_page = page;
  strcpy(dvi->curr_page_name, "");
  PD(dvi,shrink) = shrink;
  PD(dvi,gstate_stack_ptr) = 0;
  PD(dvi,gstate_transformed) = 0;
  PD(dvi,gstate_scale_h) = 1;
  PD(dvi,gstate_scale_v) = 1;
  PD(dvi,gstate_angle)   = 0;

  if (PD(dvi,font_opened) == FALSE){
    (dvi->open_font)(dvi, dev);
    if (PD(dvi,font_opened) == FALSE)
      return DVI_DRAW_ERROR;
  }

  dvi_color_stack_restore_page(dvi, dev, page);

  dvi->error = 0;

  v = DVI_DRAW_OK;

  for (;;){

    if (DEV_METHOD_DEF(dev,device_polling)){
      if (DEV_CALL(dev,device_polling)(dev, dvi, DVI_POLL_PAGE) != 0){
	dvi_color_reset_page(dvi, dev, NULL);
	return DVI_DRAW_INTERRUPTED;
      }
    }

    instr = READ_UINT1(PD(dvi,dvi_file));
    if (instr <= DVIINST_SET4){  /* SETCHAR0 ... SETCHAR127, SET1, ... ,SET4 */
      if ((code = instr) > (DVIINST_SETCHAR0+127)){
	n = instr - DVIINST_SET1 + 1;
	code = (UINT4)READ_UINTN(PD(dvi,dvi_file), n);
      }
      set_char(dvi, dev, code, (int)instr, 1);

    } else if ((DVIINST_FNTNUM0 <= instr) && (instr <= (DVIINST_FNTNUM0+63))){
      if (font_switch(dvi, dev, (long)instr-DVIINST_FNTNUM0) < 0) 
	return DVI_DRAW_ERROR;

    } else {

      switch (instr){

      case DVIINST_BOP:
	PD(dvi,stackp) = 0;
	HH = VV = WW = XX = YY = ZZ = 0L;
	DD = 0L;  /* for pTeX, vertical writing feature */
	sprintf(dvi->curr_page_name, "%ld", READ_INT4(PD(dvi,dvi_file)));
	SKIP_N(PD(dvi,dvi_file), 4*10);
	break;

      case DVIINST_EOP:
	if (PD(dvi,stackp) > 0){
	  dvi->error = DVI_ERR_STACK_UNDERFLOW;
	  DEV_CALL(dev,message_error)(dev, dvi, "Stack not empty at EOP");
	  return DVI_DRAW_ERROR;
	}
	v = DVI_DRAW_OK;
	if (ftell(PD(dvi,dvi_file)) > PD(dvi,offset_last_page))  
	  v = DVI_DRAW_LAST_PAGE;    /* the last page */
	goto end_page;

      case DVIINST_SETRULE:
	h = (long)READ_INT4(PD(dvi,dvi_file)); 
	w = (long)READ_INT4(PD(dvi,dvi_file)); 
#if SCALEBOX
	h = h * PD(dvi,gstate_scale_v);
	w = w * PD(dvi,gstate_scale_h);
#endif
	if (!PTEX_VERT_MODE){
	  set_rule(dvi, dev, w, h);
	  HH += w;
	} else {
	  VV += w;
	  set_rule(dvi, dev, h, w);
	}
	break;

      case DVIINST_PUTRULE:
	h = (long)READ_INT4(PD(dvi,dvi_file)); 
	w = (long)READ_INT4(PD(dvi,dvi_file)); 
#if SCALEBOX
	h = h * PD(dvi,gstate_scale_v);
	w = w * PD(dvi,gstate_scale_h);
#endif
	if (!PTEX_VERT_MODE){
	  set_rule(dvi, dev, w, h);
	} else {
	  VV = VV + w;
	  set_rule(dvi, dev, h, w);
	  VV = VV - w;
	}
	break;

      case DVIINST_PUSH:
	PD(dvi,stackp) = PD(dvi,stackp) + 1;
	if (PD(dvi,stackp) > PD(dvi,max_stack)){
	  dvi->error = DVI_ERR_STACK_OVERFLOW;
	  DEV_CALL(dev,message_fatal)(dev, dvi, "DVI stack overflow.");
	  return DVI_DRAW_ERROR;
	}
	PD(dvi,stack)[PD(dvi,stackp)] = PD(dvi,stack)[PD(dvi,stackp) - 1];
	break;

      case DVIINST_POP:
	PD(dvi,stackp) = PD(dvi,stackp) - 1;
	if (PD(dvi,stackp) < 0){
	  dvi->error = DVI_ERR_STACK_OVERFLOW;
	  DEV_CALL(dev,message_error)(dev, dvi,"DVI stack underflow.");
	  return DVI_DRAW_ERROR;
	}
	break;

      case DVIINST_PUT1: case DVIINST_PUT2: 
      case DVIINST_PUT3: case DVIINST_PUT4:
	code = (UINT4)READ_UINTN(PD(dvi,dvi_file), instr - DVIINST_SET1 + 1);
	set_char(dvi, dev, code, (int)instr, 0);
	break;	

      case DVIINST_RIGHT1: case DVIINST_RIGHT2: 
      case DVIINST_RIGHT3: case DVIINST_RIGHT4:
	q = (long)READ_INTN(PD(dvi,dvi_file), (instr-DVIINST_RIGHT1+1));
#if SCALEBOX
	q = q * PD(dvi,gstate_scale_h);
#endif
	if (!PTEX_VERT_MODE){
	  HH += q;
	} else {
	  VV += q;
	}
	break;

      case DVIINST_X0: case DVIINST_X1: 
      case DVIINST_X2: case DVIINST_X3: case DVIINST_X4:
	if (instr != DVIINST_X0){
	  XX = (long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_X0);
	}
#if 1
	m = PD(dvi,gstate_scale_h);
#else 
	m = 1.0;
#endif
	if (!PTEX_VERT_MODE){
	  HH += (XX * m);
	} else {
	  VV += (XX * m);
	}
	break;

      case DVIINST_W0: case DVIINST_W1: 
      case DVIINST_W2: case DVIINST_W3: case DVIINST_W4:
	if (instr != DVIINST_W0){
	  WW = (long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_W0);
	}
#if 1
	m = PD(dvi,gstate_scale_h);
#else 
	m = 1.0;
#endif
	if (!PTEX_VERT_MODE){
	  HH += (WW * m);
	} else {
	  VV += (WW * m);
	}
	break;

      case DVIINST_Y0: case DVIINST_Y1: 
      case DVIINST_Y2: case DVIINST_Y3: case DVIINST_Y4:
	if (instr != DVIINST_Y0){
	  YY = (long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_Y0);
	}
#if 1
	m = PD(dvi,gstate_scale_v);
#else 
	m = 1.0;
#endif
	if (!PTEX_VERT_MODE){
	  VV += (YY * m);
	} else {
	  HH -= (YY * m);
	}
	break;

      case DVIINST_Z0: case DVIINST_Z1: 
      case DVIINST_Z2: case DVIINST_Z3: case DVIINST_Z4:
	if (instr != DVIINST_Z0){
	  ZZ = (long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_Z0);
	}
#if 1
	m = PD(dvi,gstate_scale_v);
#else
	m = 1.0;
#endif
	if (!PTEX_VERT_MODE){
	  VV += (ZZ * m);
	} else {
	  HH -= (ZZ * m);
	}
	break;

      case DVIINST_DOWN1: case DVIINST_DOWN2:
      case DVIINST_DOWN3: case DVIINST_DOWN4:
	q = (long)READ_INTN(PD(dvi,dvi_file), instr-DVIINST_DOWN1+1);
#if SCALEBOX
	q = q * PD(dvi,gstate_scale_v);
#endif
	if (!PTEX_VERT_MODE){
	  VV += q;
	} else {
	  HH -= q;
	}
	break;

      case DVIINST_XXX1: case DVIINST_XXX2: 
      case DVIINST_XXX3: case DVIINST_XXX4:
	len = (long)READ_INTN(PD(dvi,dvi_file), instr-DVIINST_XXX1+1);
	if (len > 0)
	  special(dvi, dev, len);
	break;

      case DVIINST_FNT1: case DVIINST_FNT2: 
      case DVIINST_FNT3: case DVIINST_FNT4:
	if (font_switch(dvi, dev, (long)READ_UINTN(PD(dvi,dvi_file), 
						   instr-DVIINST_FNT1+1)) < 0)
	  return DVI_DRAW_ERROR;
	break;

      case DVIINST_FNTDEF1: case DVIINST_FNTDEF2: 
      case DVIINST_FNTDEF3: case DVIINST_FNTDEF4:
	SKIP_N(PD(dvi,dvi_file), instr-DVIINST_FNTDEF1+1 + 4+4+4);
	len  = READ_UINT1(PD(dvi,dvi_file)); 
	len += READ_UINT1(PD(dvi,dvi_file)); 
	SKIP_N(PD(dvi,dvi_file), len);
	break;

      case DVIINST_BEG_REFL: /* code 250 (TeX--XeT extension) */
	break;  /* not supported */

      case DVIINST_END_REFL: /* code 251 (TeX--XeT extension) */
	break;  /* not supported */

      case DVIINST_DIR: 
	/* code 255 (pTeX extension) --- change of writing direction */
	/*  DD = 0 : horizontal direction */
	/*  DD = 1 : vertical direction */
	if (DVI_PTEX){
	  DD = (long)READ_UINT1(PD(dvi,dvi_file));  
	} else {
	  DEV_CALL(dev,message_error)(dev, dvi, 
				      "Unsupported feature in DVI file.");
	  return DVI_DRAW_ERROR;
	}
	break;

      case DVIINST_NOP:
	break;

      default:
	dvi->error = DVI_ERR_ILL_FORMAT;
	if (DVI_file_modified(dvi, PD(dvi,file_name)) == 1){
	  DEV_CALL(dev,message_error)(dev, dvi, "DVI file is updated.");
	} else {
	  DEV_CALL(dev,message_error)(dev, dvi, "Illegal DVI file.");
	}
	return DVI_DRAW_ERROR;
	break;
      }
    }
  }

end_page:
  dvi_color_stack_save_page(dvi, dev, page);

  return v;
}

Private int
dvi_paper_size_in_dvi(DVI dvi, DVI_DEVICE dev)
{
  UINT1    instr, code;
  long     len;
  int      page, cont, i;
  char     *cmd, *p;
  double   w, h;
    
  page = 1;
  fseek(PD(dvi,dvi_file), PD(dvi,page_index)[page-1], SEEK_SET);

  cmd = NULL;
  for (cont = 1; cont == 1; ){

    instr = READ_UINT1(PD(dvi,dvi_file));
    switch (instr){

    case DVIINST_XXX1: case DVIINST_XXX2: 
    case DVIINST_XXX3: case DVIINST_XXX4:
      len = (long)READ_INTN(PD(dvi,dvi_file), instr-DVIINST_XXX1+1);
      if (len <= 0)
	break;
      if (cmd != NULL)
	free(cmd); 
      if ((cmd = malloc(len)) == NULL){
	fprintf(stderr, "No memory\n");
	exit(1);
      }
      for (i = 0; i < len; i++)
	cmd[i] = READ_UINT1(PD(dvi,dvi_file));
      cmd[i] = '\0';
      if (strncmp("papersize", cmd, 9) != 0)
	break;
      p = cmd + 9;
      while (strchr("= \t\0", *p) != NULL)
	p++;
      if (*p == '\0')
	break;
      if (DVI_parse_paper_size(p, &w, &h) >= 0){
	;
      }
#if 0
      printf("** Paer Size: %s ==> %.2f %.2f\n", p, w, h);
#endif
      break;

    case DVIINST_EOP:
      cont = 0;
      break;
    case DVIINST_BOP:
      SKIP_N(PD(dvi,dvi_file), 4*11);
      break;
    case DVIINST_SETRULE:
    case DVIINST_PUTRULE:
      (long)READ_INT4(PD(dvi,dvi_file)); 
      (long)READ_INT4(PD(dvi,dvi_file)); 
      break;
    case DVIINST_PUSH:
      break;
    case DVIINST_POP:
      break;
    case DVIINST_PUT1: case DVIINST_PUT2: 
    case DVIINST_PUT3: case DVIINST_PUT4:
      (UINT4)READ_UINTN(PD(dvi,dvi_file), instr - DVIINST_SET1 + 1);
      break;	
    case DVIINST_RIGHT1: case DVIINST_RIGHT2: 
    case DVIINST_RIGHT3: case DVIINST_RIGHT4:
      (long)READ_INTN(PD(dvi,dvi_file), (instr-DVIINST_RIGHT1+1));
      break;
    case DVIINST_X0: case DVIINST_X1: 
    case DVIINST_X2: case DVIINST_X3: case DVIINST_X4:
      if (instr != DVIINST_X0)
	(long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_X0);
      break;
    case DVIINST_W0: case DVIINST_W1: 
    case DVIINST_W2: case DVIINST_W3: case DVIINST_W4:
      if (instr != DVIINST_W0)
	(long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_W0);
      break;
    case DVIINST_Y0: case DVIINST_Y1: 
    case DVIINST_Y2: case DVIINST_Y3: case DVIINST_Y4:
      if (instr != DVIINST_Y0)
	(long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_Y0);
      break;
    case DVIINST_Z0: case DVIINST_Z1: 
    case DVIINST_Z2: case DVIINST_Z3: case DVIINST_Z4:
      if (instr != DVIINST_Z0)
	(long)READ_INTN(PD(dvi,dvi_file), instr - DVIINST_Z0);
      break;
    case DVIINST_DOWN1: case DVIINST_DOWN2:
    case DVIINST_DOWN3: case DVIINST_DOWN4:
      (long)READ_INTN(PD(dvi,dvi_file), instr-DVIINST_DOWN1+1);
      break;
    case DVIINST_FNT1: case DVIINST_FNT2: 
    case DVIINST_FNT3: case DVIINST_FNT4:
      break;
    case DVIINST_FNTDEF1: case DVIINST_FNTDEF2: 
    case DVIINST_FNTDEF3: case DVIINST_FNTDEF4:
      SKIP_N(PD(dvi,dvi_file), instr-DVIINST_FNTDEF1+1 + 4+4+4);
      len  = READ_UINT1(PD(dvi,dvi_file)); 
      len += READ_UINT1(PD(dvi,dvi_file)); 
      SKIP_N(PD(dvi,dvi_file), len);
      break;
    case DVIINST_BEG_REFL: /* code 250 (TeX--XeT extension) */
      break;
    case DVIINST_END_REFL: /* code 251 (TeX--XeT extension) */
      break;
    case DVIINST_DIR: 
      (long)READ_UINT1(PD(dvi,dvi_file));  
      break;
    case DVIINST_NOP:
      break;

    default:
      if (instr <= DVIINST_SET4){
	/* SETCHAR0 ... SETCHAR127, SET1, ... ,SET4 */
	if ((code = instr) > (DVIINST_SETCHAR0+127)){
	  len = instr - DVIINST_SET1 + 1;
	  code = (UINT4)READ_UINTN(PD(dvi,dvi_file), len);
	}
	;
      } else if ((DVIINST_FNTNUM0 <= instr)
		 && (instr <= (DVIINST_FNTNUM0+63))){
	;
      } else {
	cont = 0;
      }
      break;
    }
  }

  if (cmd != NULL)
    free(cmd);
  fseek(PD(dvi,dvi_file), PD(dvi,page_index)[page-1], SEEK_SET);

  return 0;
}



Private void
set_char(DVI dvi, DVI_DEVICE dev, UINT4 code_point, int instr, 
	 int need_move)
{
  struct dvi_s_bitmap  bm;
  VF_BITMAP            vfbm, vfbm0;
  int                  need_free;
  double               off_x, off_y, scale, mw, mh, mov;
  CACHE                cobj;
  struct s_cache_key   ck;

#if 0
  printf("** set_char\n");
#endif

  if (DEV_METHOD_UNDEF(dev,put_bitmap) 
      && DEV_METHOD_UNDEF(dev,put_pixmap_rgb)
      && DEV_METHOD_UNDEF(dev,put_graymap))
    return;

  if ((PD(dvi,shrink) < DVI_SHRINK_MIN)
      || (fabs((double)1.0-PD(dvi,shrink)) <= 0.02))
    scale = PD(dvi,font)->bm_mag;
  else
    scale = PD(dvi,font)->bm_mag/PD(dvi,shrink);

  cobj = PD(dvi,font_cache);
  ck.font_id    = PD(dvi,current_fid);
  ck.scale      = scale;
  ck.scale_h    = PD(dvi,gstate_scale_h);
  ck.scale_v    = PD(dvi,gstate_scale_v);
  ck.rotation   = 0;
  ck.code_point = code_point;
  ck.dev        = dev;
  if ((vfbm0 = (VF_BITMAP)(cobj->get)(cobj, &ck, sizeof(ck))) == NULL){
    return;
  }

  font_get_metric(dvi, dev, code_point, &mw, &mh, 1.0, 1.0);

#ifdef SCALEBOX
  mw = mw * PD(dvi,gstate_scale_h);
  mh = mh * PD(dvi,gstate_scale_v);
#endif

  mov = mw;

  vfbm = vfbm0;
  need_free = 0;

  if (PTEX_VERT_MODE){
    if (DD != 1){
      fprintf(stderr, "DVIlib Internal Error: %s\n", "set_char()");
      exit(0);
    }
    if (vfbm0->mv_y == 0){
      vfbm = VF_RotatedBitmap(vfbm0, VF_BM_ROTATE_270);
      need_free = 1;
      mov = mw;
    } else {
      need_free = 0;
      mov = -mh;
    }
  }
  
  if (vfbm != NULL){
    bm.width  = vfbm->bbx_width;
    bm.height = vfbm->bbx_height;
    bm.raster = vfbm->raster;
    bm.bitmap = vfbm->bitmap;
    off_x     = vfbm->off_x;
    off_y     = vfbm->off_y;
    if (PD(dvi,gstate_transformed) == 0){
      dvi_put_bitmap(dev, dvi, &bm, PD(dvi,current_fid), 
		     MAG2IMAG(PD(dvi,font)->mag), (long)code_point,
		     (long)(H_UNIT2PIXEL(HH)+off_x), 
		     (long)(V_UNIT2PIXEL(VV)-off_y));
    } else {
      dvi_put_bitmap(dev, dvi, &bm, -1, -1, -1,
		     (long)(H_UNIT2PIXEL(HH)+off_x),
		     (long)(V_UNIT2PIXEL(VV)-off_y));
    }
  }

  if (!PTEX_VERT_MODE){
    HH = HH + mov/PD(dvi,dimen_pt);
  } else {
    VV = VV + mov/PD(dvi,dimen_pt);
  }

  if ((need_free == 1) && (vfbm != NULL))
    VF_FreeBitmap(vfbm);
}

Private void
set_rule(DVI dvi, DVI_DEVICE dev, long w, long h)
{
  long   dev_x, dev_y, dev_w, dev_h;

  if (PROP_TEST(dvi, DVI_PROP_SKIP_RENDERING))
    return;

  dev_x = H_UNIT2PIXEL(HH);
  dev_y = V_UNIT2PIXEL(VV-h);
  dev_w = H_UNIT2PIXEL(w);
  dev_h = V_UNIT2PIXEL(h);
  if (dev_w == 0)
    dev_w = 1;
  if (dev_h == 0)
    dev_h = 1;
  dvi_put_rectangle(dev, dvi, dev_x, dev_y, dev_w, dev_h); 
}


Public void 
dvi_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bitmap,
	       int fontid, long code, long subcode, long x, long y)
{
  if (PROP_TEST(dvi, DVI_PROP_SKIP_RENDERING))
    return;

  if (DEV_METHOD_DEF(dev,put_bitmap)){
    DEV_CALL(dev, put_bitmap)(dev, dvi, bitmap, fontid, code, subcode, x, y);
  }
}

Public void 
dvi_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		  long dev_x, long dev_y, long dev_w, long dev_h)
{
  unsigned char *p, r;
  int           i, m;
  struct dvi_s_bitmap  bm;
  static unsigned char fil[9]
    = {0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff};
  static long cid1 = 0;
  static long cid2 = 0;

  if (PROP_TEST(dvi, DVI_PROP_SKIP_RENDERING))
    return;

  if (DEV_METHOD_DEF(dev,put_rectangle)){
    DEV_CALL(dev,put_rectangle)(dev, dvi, dev_x, dev_y, dev_w, dev_h); 
  } else {
    if (DEV_METHOD_UNDEF(dev,put_bitmap) 
	&& DEV_METHOD_UNDEF(dev,put_pixmap_rgb)
	&& DEV_METHOD_UNDEF(dev,put_graymap))
    return;
    bm.width  = dev_w;
    bm.height = dev_h;
    bm.raster = (dev_w+7)/8;
    bm.bitmap = (unsigned char*)malloc(bm.raster * dev_h);
    if (bm.bitmap == NULL)
      return;
    p = bm.bitmap;
    for (i = bm.raster * dev_h; i > 0; i--)
      *(p++) = 0xff;
    if ((m = dev_w % 8) != 0){
      p = &bm.bitmap[bm.raster-1];
      r = fil[m];
      for (i = bm.height; i > 0; i--){
	*p = m;
	p = p + bm.raster;
      }
    }
    dvi_put_bitmap(dev, dvi, &bm, -1, cid1, cid2++, dev_x, dev_y);
    free(bm.bitmap);
    if (cid2 < 0){
      cid1++;
      cid2 = 0;
    }
  }
}



Private int
special(DVI dvi, DVI_DEVICE dev, long size)
{
  char  *cmd     = NULL;
  int    i, v;

  if ((cmd = malloc(size+1)) == NULL){
    dvi->error = DVI_ERR_NO_MEMORY;
    DEV_CALL(dev,message_fatal)(dev, dvi, "No memory.");
    for (i = 0; i < size; i++)
      (void) READ_UINT1(PD(dvi,dvi_file));
    free(cmd);
    return 0;
  }
  
  for (i = 0; i < size; i++)
    cmd[i] = READ_UINT1(PD(dvi,dvi_file));
  cmd[i] = '\0';

  v = dvi_do_special(dvi, dev, cmd, H_UNIT2PIXEL(HH), V_UNIT2PIXEL(VV)); 

  free(cmd);
  return v;
}



/**
 ** Font
 **/

Private int
font_define(DVI dvi, DVI_DEVICE dev, UINT4 fnt, UINT4 check_sum,
	    INT4 design_size_s, INT4 design_size_f,
	    int len_dir, int len_name, char *fontname)
{
  FONT    f, fn;
  int     v, h;

  ALLOC_IF_ERR(f, struct s_font){
    dvi->error = DVI_ERR_NO_MEMORY;
    return -1; 
  }

  f->fname = (char*)malloc(len_name + 16);
  if (f->fname == NULL){
    dvi->error = DVI_ERR_NO_MEMORY;
    return -1; 
  }
  strncpy(f->fname, &fontname[len_dir], len_name);
  f->fname[len_name] = '\0';

  f->fid    = -1;
  f->fnt    = fnt;
  f->ds     = design_size_f;
  f->ds_s   = design_size_s;
  f->mag    = (double)design_size_s/(double)design_size_f;
  f->bm_mag = 1.0;
  f->res    = (int)(dev->h_dpi * f->mag + .5);

  if (dvi_debug('f')){
    printf("DVIlib: font definition: %s %d dpi (%.2f dpi scaled %.4f)\n", 
	   f->fname, (int)f->res, dev->h_dpi, f->mag);
  }

  if (DVI_PROPERTY_TEST(PD(dvi,property),DVI_PROP_PRINT_FONTS))
    DEV_CALL(dev,message_advice)(dev, dvi, "%s scaled %.4f", f->fname, f->mag);
  if (DVI_PROPERTY_TEST(PD(dvi,property),DVI_PROP_LIST_FONTS)
      && (DEV_METHOD_DEF(dev,font_list)))
    DEV_CALL(dev,font_list)(dev, dvi, f->fname, f->fnt, f->mag, 
			    (long)f->ds_s,(double)f->ds_s/(1<<16), 
			    (long)f->ds,(double)f->ds/(1<<16));

  if (DVI_PROPERTY_TEST(PD(dvi,property),DVI_PROP_DELAYED_FONT_OPEN)){
    v = 0;
    f->opened = FONT_OPENED_FUTURE;
  } else {
    if (dvi_debug('f')){
      printf("DVIlib: opening font: %s scaled %.4f ...",
	     f->fname, f->mag);
      fflush(stdout);
    }
    v = font_do_open(dvi, dev, f);
    if (dvi_debug('f')){
      if (v >= 0)
	printf("done\n");
      else
	printf("failed\n");
    }
  }

  if (v >= 0){
    h = HASH(f->fnt);
    fn = PD(dvi,font_table)[h];
    PD(dvi,font_table)[h] = f;
    f->next = fn;
    f->prev = NULL;
    if (fn != NULL)
      fn->prev = f;
  } else {
    if (f != NULL){
      if (f->fname != NULL)
	free(f->fname);
      free(f);
    }
  }

  return v;
}

Private int
font_do_open(DVI dvi, DVI_DEVICE dev, FONT f)
{
  char   fname[1024];

  sprintf(fname, "%s.%s", f->fname, FONT_SUFFIX);
  
  f->fid = VF_OpenFont1(fname, dev->h_dpi, dev->v_dpi, -1, f->mag, f->mag);
  if (dvi_debug('O')){
    printf("DVIlib: VF_OpenFont1(\"%s\", %.2f, %.2f, -1, %.3f, %.3f)\n",
	   fname, dev->h_dpi, dev->v_dpi, f->mag, f->mag);
    printf("DVIlib:   ==> %d\n", f->fid);
  }

  if (f->fid < 0){
    if (DVI_PROPERTY_TEST(PD(dvi,property), DVI_PROP_SCALE_FONT_IF_NOT_EXIST)){
      f->fid = VF_OpenFont1(fname, dev->h_dpi, dev->v_dpi, -1, 1, 1);
      if (dvi_debug('O')){
	printf("DVIlib: VF_OpenFont1(\"%s\", %.2f, %.2f, -1, 1, 1)\n",
	       fname, dev->h_dpi, dev->v_dpi);
	printf("DVIlib:   ==> %d\n", f->fid);
      }
      if (f->fid < 0){
	DEV_CALL(dev,message_error)
	  (dev, dvi, "Font not found: %s for %.2f dpi scaled %.3f", 
	   f->fname, dev->v_dpi, f->mag);
	dvi->error = DVI_ERR_FONT_CANNOT_OPEN;
	goto error;
      } else {
	f->bm_mag = f->mag;
	DEV_CALL(dev,message_warning)
	  (dev, dvi, "Using %s scaled by %.4f instead of %s", 
	   fname, f->bm_mag, f->fname);
      }
    } else {
      if (DVI_PROPERTY_TEST(PD(dvi,property),DVI_PROP_PRINT_MISSING_FONTS))
	DEV_CALL(dev,message_error)(dev, dvi, "%s scaled %.4f",
				    f->fname, f->mag);
      else
	DEV_CALL(dev,message_error) (dev, dvi, 
				     "Font not found: %s for %d dpi", 
				     f->fname, f->res);
      if (DVI_PROPERTY_TEST(PD(dvi,property),DVI_PROP_LIST_MISSING_FONTS)
	  && (DEV_METHOD_DEF(dev,font_list_missing)))
	DEV_CALL(dev,font_list_missing)(dev, dvi, f->fname, f->fnt, f->mag,
					(long)f->ds_s, 
					(double)f->ds_s/(1<<16), 
					(long)f->ds,
					(double)f->ds/(1<<16));
      dvi->error = DVI_ERR_FONT_CANNOT_OPEN;
      goto error;
    }
  }

  f->opened = FONT_OPENED;
  return 0;

error:
  return -1;
}

Private int
font_switch(DVI dvi, DVI_DEVICE dev, long fnt)
{
  FONT  f;
  int   h, v;

  h = HASH(fnt);
  f = PD(dvi,font_table)[h];
  while (f != NULL){
    if (f->fnt == fnt)
      break;
    f = f->next;
  }
  if (f == NULL){
    fprintf(stderr, "DVIlib Internal Error: %s %ld\n", 
	    "switch_font()", fnt);
    dvi->error = DVI_ERR_INTERNAL;
    return -1;
  }

  if (f->opened == FONT_OPENED_FUTURE){
    if (dvi_debug('f')){
      printf("DVIlib: opening font: %s scaled %.4f ...",
	     f->fname, f->mag);
      fflush(stdout);
    }
    v = font_do_open(dvi, dev, f);
    if (dvi_debug('f')){
      if (v >= 0)
	printf("done\n");
      else
	printf("failed\n");
    }
    if (v < 0){
      dvi->error = DVI_ERR_FONT_CANNOT_OPEN;
      return -1;
    }
  }

  PD(dvi,font)          = f;
  PD(dvi,current_fnt)   = fnt;
  PD(dvi,current_fid)   = f->fid;

  return 0;
} 

Private int
font_get_metric(DVI dvi, DVI_DEVICE dev, UINT4 code_point, 
		double *wdp, double *hdp, double mx, double my)
{
  double   w, h;
  struct vf_s_metric1  met;

  if (dvi_debug('M')){
    printf("DVIlib: VF_GetMetric1(%d,%ld,%p,%.3f,%.3f)\n",
	   PD(dvi,current_fid), (long)code_point, &met,mx,my);
  }

  if (VF_GetMetric1(PD(dvi,current_fid), 
		    (long)code_point, &met, mx, my) == NULL){
    if (dvi_debug('M')){
      printf("DVIlib:   ==> NULL\n");
    }
    if (wdp != NULL)
      *wdp = 0;
    if (hdp != NULL)
      *hdp = 0;
    return -1;
  }

  w = met.mv_x * PD(dvi,font)->bm_mag;
  h = met.mv_y * PD(dvi,font)->bm_mag;

  if (dvi_debug('M')){
    printf("DVIlib:   ==> w=%.3f, h=%.3f\n", w, h);
  }

  if (wdp != NULL)
    *wdp = w;
  if (hdp != NULL)
    *hdp = h;

  return 0;
}

VF_BITMAP
font_load_bitmap(CACHE cache, CACHE_KEY key, int len)
{
  VF_BITMAP  bm, bmr;

  bm = VF_GetBitmap1(key->font_id, key->code_point, 
		     key->scale * key->scale_h, key->scale * key->scale_v);

#if 0
  printf("DVIlib: VF_GetBitmap1(%d, %ld, %.3f*%.3f %.3f*%.3f);\n",
	 key->font_id, key->code_point, 
	 key->scale, key->scale_h, key->scale, key->scale_v);
  printf("DVIlib:   ==> %p\n", bm);
  VF_DumpBitmap(bm);
#endif

  if (bm == NULL)
    return NULL;

  if (key->rotation == 0)
    return bm;

  bmr = VF_RotatedBitmap(bm, VF_BM_ROTATE_270);
  if (dvi_debug('B')){
    printf("DVIlib: VF_RotatedBitmap(%p, %d);\n",
	   bm, VF_BM_ROTATE_270);
    printf("DVIlib:   ==> %p\n", bmr);
  }

  VF_FreeBitmap(bm);

  return bmr;
}



/**
 ** Alternating Methods
 **/
Private int
altm_draw_ps_figures(DVI_DEVICE dev, DVI dvi)
{
  return 1;  /* Yes, please draw PS figures */
}

Private void
altm_before_ps_figure(DVI_DEVICE dev, DVI dvi, char *spcmd, long x, long y)
{
  return; /* Do nothing before PS figure */
}

Private void 
altm_after_ps_figure(DVI_DEVICE dev, DVI dvi)
{
  return; /* Do nothing after PS figure */
}

Private int
altm_ps_figure_display_step(DVI_DEVICE dev, DVI dvi, long w, long h)
{
  return -1;
}

#if 0

Private int
altm_special_command_undef(DVI_DEVICE dev, DVI dvi, char *spcmd, 
			   long x, long y)
{
  printf("Unsupported special DVI command: ");
  if (strlen(spcmd) < 32)
    printf("%s\n", spcmd);
  else
    printf("%*s ...\n", 32, spcmd);
  return 0;
}

#endif

Private char*
altm_gs_program_path(DVI_DEVICE dev, DVI dvi)
{
  return DEFAULT_GS_PROGRAM_PATH;
}

Private int  
altm_gs_timeout_give_up(DVI_DEVICE dev, DVI dvi)
{
  return 1;  /* Yes, I give up */
}

Private int
altm_gs_timeout_value(DVI_DEVICE dev, DVI dvi)
{
  return DEFAULT_GS_TIMEOUT;        /* in sec */
}

Private void
altm_message_null(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;

  va_start(ap, fmt);
  va_end(ap);
}

Private void
altm_message_advice(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
}

Private void
altm_message_warning(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
}

Private void
altm_message_error(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
}

Private void
altm_message_fatal(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
  exit(1);
}

Private char*
altm_temp_dir(DVI_DEVICE dev, DVI dvi)
{
  char    *s;

  if ((s = getenv("TMPDIR")) != NULL)
     return  s;

  return  DEFAULT_TEMP_DIR;
}



/**
 ** Reading a Number from File
 **/
Glocal unsigned long
dvi_read_uintn(FILE* fp, int size)
{
  unsigned long  v;

  v = 0L;
  while (size >= 1){
    v = v*256L + (unsigned long)getc(fp);
    --size;
  }
  return v;
}

Glocal long
dvi_read_intn(FILE* fp, int size)
{
  long  v;

  if (size <= 0)
    return 0;

  v = (long)getc(fp) & 0xffL;
  if (v & 0x80L)
    v = v - 256L;
  --size;
  while (size >= 1){
    v = v*256L + (unsigned long)getc(fp);
    --size;
  }
  return v;
}

Glocal void
dvi_skip_n(FILE* fp, int size)
{
  while (size > 0){
    (void)getc(fp);
    --size;
  }
}


Glocal int
dvi_debug(char type)
{
  char  *p;

  if (dvi_debug_flags == NULL)
    return FALSE;

  for (p = dvi_debug_flags; *p != '\0'; p++){
    if (*p == type)
      return TRUE;
  }
  for (p = dvi_debug_flags; *p != '\0'; p++){
    if (*p == '*')
      return TRUE;
  }
  return FALSE;
}

/*EOF*/
