
/******************************************************************************
* MODULE     : edit_select.gen.cc
* DESCRIPTION: Selection handling
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <Replace/edit_select.gen.h>

#module code_edit_select
#import edit_select

/******************************************************************************
* Constructor and destructor
******************************************************************************/

edit_select_rep::edit_select_rep ():
  selecting (FALSE), mid_p (),
  selection_import ("TeXmacs"), selection_export ("TeXmacs") {}
edit_select_rep::~edit_select_rep () {}
void edit_select_rep::get_selection (path& start, path& end) {
  start= copy (start_p); end= copy (end_p); }
void edit_select_rep::set_selection (path start, path end) {
  start_p= copy (start); end_p= copy (end); }

path
common (path start, path end) {
  if (nil (start) || nil (end)) return path ();
  if (start->item != end->item) return path ();
  return path (start->item, common (start->next, end->next));
}

/******************************************************************************
* Selecting particular things
******************************************************************************/

void
edit_select_rep::select (path p) {
  select (start (et, p), end (et, p));
}

void
edit_select_rep::select (path p1, path p2) {
  start_p= p1;
  end_p  = p2;
  if (path_less_eq (end_p, start_p)) {
    start_p= p2;
    end_p  = p1;
  }
  notify_change (THE_SELECTION);
}

void
edit_select_rep::select_all () {
  select (path ());
}

void
edit_select_rep::select_line () {
  select (search_parent_upwards (DOCUMENT));
}

/******************************************************************************
* For interface with cursor movement
******************************************************************************/

void
edit_select_rep::select_from_cursor () {
  if (path_less (mid_p, tp)) {
    start_p= copy (mid_p);
    end_p  = copy (tp);
  }
  else {
    start_p= copy (tp);
    end_p  = copy (mid_p);
  }
  notify_change (THE_SELECTION);
}

void
edit_select_rep::select_from_cursor_if_active () {
  if (selecting) select_from_cursor ();
}

void
edit_select_rep::select_from_keyboard (bool flag) {
  selecting= flag;
  if (flag) mid_p= copy (tp);
}

bool
edit_select_rep::select_cursor_on_border () {
  return (tp == start_p) || (tp == end_p);
}

/******************************************************************************
* Enlarging an existing selection
******************************************************************************/

void
edit_select_rep::select_enlarge () {
  if (start_p == end_p) {
    path p = path_up (start_p);
    tree st= subtree (et, p);
    if (is_atomic (st)) {
      string s= st->label;
      if (get_env_string (MODE) == "text") {
	int i1= last_item (start_p), i2= i1;
	while (i1>0) {
	  if (s[i1-1] == ' ') break;
	  if (s[i1-1] == '>') do { i1--; } while ((i1>0) && (s[i1]!='<'));
	  i1--;
	}
	while (i2<N(s)) {
	  if (s[i2] == ' ') break;
	  if (s[i2] == '<') do { i2++; } while ((i2<N(s)) && (s[i2]!='>'));
	  i2++;
	}
	if (i1<i2) select (p * i1, p * i2);
	else {
	  if (s == "") { p= path_up (p); select (p * 0, p * 1); }
	  else select (p * 0, p * N(s));
	}
      }
      else {
	path q= path_up (p);
	if (is_concat (subtree (et, q)) || (s == ""))
	  select (q * 0, q * 1);
	else select (p * 0, p * N(s));
      }
    }
    else select (p * 0, p * 1);
  }
  else {
    path p= common (start_p, end_p);
    if (nil (p)) {
      selection_cancel ();
      set_message ("", "");
      return;
    }
    tree st= subtree (et, p);
    path q = path_up (p);
    int  i1= last_item (start_p);
    int  i2= last_item (end_p);
    if (is_atomic (st) && (!is_concat (subtree (et, q))) &&
	((i1 != 0) || (i2 != N(st->label))))
      select (p * 0, p * N(st->label));
    else select (q * 0, q * 1);
  }

  path p = common (start_p, end_p);
  tree st= subtree (et, p);
  if (is_func (st, TABLE_FORMAT) || is_func (st, DOCUMENT, 1))
    select_enlarge ();
  else {
    string s;
    if (is_atomic (st)) s= "text";
    else if (is_expand (st)) s= as_string (st[0]->label);
    else if (is_func (st, WITH)) s= "with#" * as_string (st[0]->label);
    else s= CONSTRUCTOR_NAME[L(st)];
    set_message ("selected#" * s, "enlarge selection");
  }
  selecting= FALSE;
}

static bool
stop_enlarge_environmental (tree t) {
  if (is_func (t, WITH, 3) && (t[0]==MODE) && (t[1]=="math")) return TRUE;
  if (!is_expand (t)) return FALSE;
  if (is_multi_paragraph (t)) return TRUE;
  string s= as_string (t[0]);
  return
    (s == "chapter") ||
    (s == "section") ||
    (s == "subsection") ||
    (s == "subsubsection") ||
    (s == "paragraph") ||
    (s == "subparagraph");
}

void
edit_select_rep::select_enlarge_environmental () {
  select_enlarge ();
  if (end_p == start_p) return;
  path p= common (start_p, end_p);
  tree st= subtree (et, p);
  if (stop_enlarge_environmental (st)) return;
  select_enlarge_environmental ();
}

/******************************************************************************
* Test whether selection is active
******************************************************************************/

bool
edit_select_rep::selection_active_any () {
  return end_p != start_p;
  // return made_selection;
}

bool
edit_select_rep::selection_active_normal () {
  return selection_active_any () && (!selection_active_table ());
}

bool
edit_select_rep::selection_active_table () {
  if (!selection_active_any ()) return FALSE;
  path p= common (start_p, end_p);
  if ((p == start_p) || (p == end_p)) p= path_up (p);
  tree t= subtree (et, p);
  return
    is_func (t, TABLE_FORMAT) || is_func (t, TABLE) ||
    is_func (t, ROW) || is_func (t, CELL);
}

bool
edit_select_rep::selection_active_small () {
  if (!selection_active_normal ()) return FALSE;
  path p1, p2;
  selection_get (p1, p2);
  if (p2 == p1) return FALSE;
  if (is_multi_paragraph (subtree (et, common (p1, p2)))) return FALSE;
  return TRUE;
}

bool
edit_select_rep::selection_active_enlarging () {
  return (selecting || (end_p != start_p)) && (mid_p == tp);
}

/******************************************************************************
* Subroutines for table selections
******************************************************************************/

static path
table_search_format (tree t, path p) {
  tree st= subtree (t, p);
  if (is_func (st, TABLE_FORMAT) && is_func (st[N(st)-1], TABLE)) return p;
  while ((!nil (p)) && (!is_func (subtree (t, p), TABLE))) p= path_up (p);
  if ((!nil (p)) && (is_func (subtree (t, path_up (p)), TABLE_FORMAT)))
    p= path_up (p);
  return p;
}

static void
table_search_coordinates (tree t, path p, int& row, int& col) {
  row= col= 0;
  while (TRUE) {
    if (nil (p)) p= path (1);
    if (p == path (0)) p= path (0, 0);
    if (p == path (1)) p= path (N(t)-1, 1);
    if (is_func (t, TABLE_FORMAT));
    else if (is_func (t, TABLE)) row= p->item;
    else if (is_func (t, ROW)) col= p->item;
    else return;
    t= t [p->item];
    p= p->next;
  }
}

static path
table_search_cell (tree t, int row, int col) {
  path p;
  while (is_func (t, TABLE_FORMAT)) {
    p= p * (N(t)-1);
    t= t [N(t)-1];
  }
  p= p * row;
  t= t [row];
  while (is_func (t, TABLE_FORMAT)) {
    p= p * (N(t)-1);
    t= t [N(t)-1];
  }
  p= p * col;
  t= t [col];
  while (is_func (t, TABLE_FORMAT)) {
    p= p * (N(t)-1);
    t= t [N(t)-1];
  }
  return p;
}

/******************************************************************************
* Get the selection
******************************************************************************/

static void
selection_correct (tree t, path i1, path i2, path& o1, path& o2) {
  if (i1 == i2) {
    o1= i1;
    o2= i2;
  }
  else if (atom (i1) || atom (i2)) {
    if (is_atomic (t)) {
      o1= i1;
      o2= i2;
    }
    else {
      o1= start (t);
      o2= end (t);
    }
  }
  else if (i1->item == i2->item) {
    selection_correct (t[i1->item], i1->next, i2->next, o1, o2);
    o1= path (i1->item, o1);
    o2= path (i2->item, o2);
  }
  else {
    tree_label l= L(t);
    if ((l==DOCUMENT) || (l==PARAGRAPH) || (l==CONCAT)) {
      if (is_compound (t[i1->item])) {
	path mid;
	selection_correct (t[i1->item], i1->next, end (t[i1->item]), o1, mid);
	o1= path (i1->item, o1);
      }
      else o1= i1;
      if (is_compound (t[i2->item])) {
	path mid;
	selection_correct (t[i2->item], start(t[i2->item]), i2->next, mid, o2);
	o2= path (i2->item, o2);
      }
      else o2= i2;
    }
    else {
      o1= start (t);
      o2= end (t);
    }
  }
}

tree
compute_selection (tree t, path start, path end) {
  int  i1= start->item;
  int  i2= end->item;
  path p1= start->next;
  path p2= end->next;

  if (nil (p1) || nil (p2)) {
    if (start == path (right_index (t))) return "";
    if (end == path (0)) return "";
    if (start == end) return "";
    if (nil (p1) && nil (p2)) {
      if (is_compound (t)) return copy (t);
      if (i1>=i2) return "";
      return t->label (i1, i2);
    }
    if (is_compound (t) && (!is_format (t))) return copy (t);
    if (nil (p1)) {
      i1= 0;
      p1= (start->item==0? 0: right_index (t[i1]));
    }
    if (nil (p2)) {
      i2= N(t)-1;
      p2= (end->item==0? 0: right_index (t[i2]));
    }
  }

  if (i1==i2) return compute_selection (t[i1], p1, p2);
  if (is_compound (t) && (!is_format (t))) return copy (t);

  int i;
  tree r (t, i2-i1+1);
  r[0]     = compute_selection (t[i1], p1, path (right_index (t[i1])));
  r[N(r)-1]= compute_selection (t[i2], path (0), p2);
  for (i=1; i<N(r)-1; i++) r[i]= copy (t[i+i1]);
  return r;
}

path
edit_select_rep::selection_get_subtable (
  int& row1, int& col1, int& row2, int& col2)
{
  path fp= ::table_search_format (et, common (start_p, end_p));
  tree st= subtree (et, fp);
  table_search_coordinates (st, tail (start_p, N(fp)), row1, col1);
  table_search_coordinates (st, tail (end_p, N(fp)), row2, col2);
  if (row1>row2) { int tmp= row1; row1= row2; row2= tmp; }
  if (col1>col2) { int tmp= col1; col1= col2; col2= tmp; }
  table_bound (fp, row1, col1, row2, col2);
  return fp;
}

void
edit_select_rep::selection_get (selection& sel) {
  if (selection_active_table ()) {
    int row1, col1, row2, col2;
    path fp= selection_get_subtable (row1, col1, row2, col2);
    tree st= subtree (et, fp);

    int i, j;
    rectangle r (0, 0, 0, 0);
    for (i=row1; i<=row2; i++)
      for (j=col1; j<=col2; j++) {
	path cp= fp * table_search_cell (st, i, j);
	sel= eb->find_check_selection (cp * 0, cp * 1);
	if (sel->valid) {
	  rectangles rs= sel->rs;
	  if (r != rectangle (0, 0, 0, 0)) rs= rectangles (r, rs);
	  r= least_upper_bound (rs);
	}
      }
    sel= selection (rectangles (r), fp * 0, fp * 1);
  }
  else {
    path p_start, p_end;
    selection_correct (et, start_p, end_p, p_start, p_end);
    sel= eb->find_check_selection (p_start, p_end);
  }
}

void
edit_select_rep::selection_get (path& start, path& end) {
  selection sel; selection_get (sel);
  start= sel->start;
  end  = sel->end;
}

tree
edit_select_rep::selection_get () {
  if (!selection_active_any ()) return "";
  if (selection_active_table ()) {
    int row1, col1, row2, col2;
    path fp= selection_get_subtable (row1, col1, row2, col2);
    return table_get_subtable (fp, row1, col1, row2, col2);
  }
  else {
    path start, end;
    // cout << "Selecting...\n";
    selection_get (start, end);
    // cout << "Between paths: " << start << " and " << end << "\n";
    tree t= ::compute_selection (et, start, end);
    // cout << "Selection : " << t << "\n";
    return t;
  }
}

/******************************************************************************
* Copy and paste
******************************************************************************/

void
edit_select_rep::selection_set_start () {
  bool flag= selection_active_any ();
  start_p= tp;
  if (path_less_eq (end_p, start_p) || (!flag)) end_p= start_p;
  notify_change (THE_SELECTION);
}

void
edit_select_rep::selection_set_end () {
  end_p= tp;
  if (path_less_eq (end_p, start_p)) start_p= end_p;
  notify_change (THE_SELECTION);
}

void
edit_select_rep::selection_set (string key, tree t, bool persistant) {
  selecting= FALSE;
  string mode= get_env_string (MODE);
  string lan = get_env_string (LANGUAGE (mode));
  tree sel= tuple ("edit", t, mode, lan);
  string s;
  if (key == "primary") {
    if (selection_export == "TeXmacs")
      s= tree_to_texmacs (t);
    else if (selection_export == "scheme")
      s= tree_to_scheme (t);
    else if (selection_export == "latex") {
      s = tree_to_latex (t, mode);
      if (sel[1] == "math") s= "$" * s * "$";
    }
    else s= tree_to_verbatim (t);
  }
  if (dis->set_selection (widget (this), key, sel, s) &&
      (!persistant)) selection_cancel ();
}

void
edit_select_rep::selection_set (tree t) {
  selection_set ("primary", t);
}

void
edit_select_rep::selection_copy (string key) {
  if (selection_active_any ()) {
    path old_tp= tp;
    selection sel; selection_get (sel);
    go_to (sel->end);
    selection_set (key, selection_get ());
    go_to (old_tp);
  }
}

void
edit_select_rep::selection_paste (string key) {
  tree t= copy (dis->get_selection (widget (this), key));
  // if (is_func (t, "none", 0)) return;
  if (is_tuple (t, "extern", 1)) {
    if (selection_import == "TeXmacs") {
      tree doc= texmacs_to_tree (as_string (t[1]));
      if (is_func (doc, DOCUMENT, 1)) doc= doc[0]; // temporary fix
      insert_tree (doc);
    }
    else if (selection_import == "scheme")
      insert_tree (scheme_to_tree (as_string (t[1])));
    else if (selection_import == "latex") {
      string mode= get_env_string (MODE);
      if (mode == "prog") insert_tree (verbatim_to_tree (as_string (t[1])));
      else insert_tree (latex_to_tree (as_string(t[1]), get_env_string(MODE)));
    }
    else if (selection_import == "html") {
      string mode= get_env_string (MODE);
      if (mode == "prog") insert_tree (verbatim_to_tree (as_string (t[1])));
      else insert_tree (html_to_tree (as_string(t[1]), get_env_string(MODE)));
    }
    else insert_tree (verbatim_to_tree (as_string (t[1])));
  }
  if (is_tuple (t, "edit", 3)) {
    string mode= get_env_string (MODE);
    string lan = get_env_string (LANGUAGE (mode));
    if ((mode == "prog") && (t[2] == "math")) {
      tree in= tuple (lan, t[1]), out;
      if (!eval_scheme_func_tree_tree ("cas-math-input", in, out))
	insert_tree (out);
    }
    else {
      if ((t[2] != mode) && ((t[2]=="math") || (mode=="math")))
	insert_tree (tree (WITH, copy (MODE), copy (t[2]), ""), path (2, 0));
      if (is_func (t[1], TABLE_FORMAT) || is_func (t[1], TABLE)) {
	int row, col;
	path fp= search_format (row, col);
	if (nil (fp)) insert_tree (tree (EXPAND, copy (TABULAR), t[1]));
	else table_write_subtable (fp, row, col, t[1]);
      }
      else insert_tree (t[1]);
    }
  }
}

void
edit_select_rep::selection_clear (string key) {
  dis->clear_selection (key);
}

void
edit_select_rep::selection_cancel () {
  selecting= FALSE;
  if (end_p == start_p) return;
  end_p= start_p;
  notify_change (THE_SELECTION);
}

void
edit_select_rep::selection_set_import (string fm) {
  selection_import= fm;
}

void
edit_select_rep::selection_set_export (string fm) {
  selection_export= fm;
}

/******************************************************************************
* Cutting the selection
******************************************************************************/

void
edit_select_rep::cut (path p) {
  cut (start (et, p), end (et, p));
}

void
edit_select_rep::cut (path p1, path p2) {
  if (p2 == p1) return;
  path p = common (p1, p2);
  tree t = subtree (et, p);
  int  n = N(p);
  int  i1= p1[n];
  int  i2= p2[n];

  if (is_document (t) || is_concat (t)) {
    path q1= copy (p); q1 << path (i1, end (t[i1]));
    path q2= copy (p); q2 << path (i2, start (t[i2]));
    cut (q2, p2);
    if (i2>i1+1) remove (p * (i1+1), i2-i1-1);
    cut (p1, q1);
    if (is_concat (t)) correct_concat (p);
    else remove_return (p * i1);
    return;
  }

  if (is_func (t, TABLE_FORMAT) || is_func (t, TABLE) || is_func (t, ROW)) {
    path fp= ::table_search_format (et, p);
    tree st= subtree (et, fp);
    int row1, col1, row2, col2;
    table_search_coordinates (st, tail (p1, N(fp)), row1, col1);
    table_search_coordinates (st, tail (p2, N(fp)), row2, col2);
    if (row1>row2) { int tmp= row1; row1= row2; row2= tmp; }
    if (col1>col2) { int tmp= col1; col1= col2; col2= tmp; }

    int i, j;
    for (i=row1; i<=row2; i++)
      for (j=col1; j<=col2; j++) {
	path cp= fp * table_search_cell (st, i, j);
	if (is_func (subtree (et, cp), CELL, 1)) cp= cp * 0;
	assign (cp, "");
      }
    path cp= fp * table_search_cell (st, row1, col1);
    go_to (cp * path (0, 0));

    if (is_func (st, TABLE_FORMAT))
      table_del_format (fp, row1+1, col1+1, row2+1, col2+1, "");
    return;
  }

  if (is_compound (t) && (!is_format (t))) {
    assign (p, "");
    return;
  }

  if ((N(p1) != (N(p)+1)) || (N(p2) != (N(p)+1))) {
    cerr << "t = " << t << "\n";
    cerr << "p = " << p << "\n";
    cerr << "p1= " << p1 << "\n";
    cerr << "p2= " << p2 << "\n";
    fatal_error ("invalid cut", "edit_select_rep::cut");
  }

  if (is_atomic (t)) {
    int pos= last_item (p1);
    int nr = last_item (p2)-pos;
    if (nr>0) remove (p1, nr);
  }
  else {
    if ((last_item (p1) != 0) || (last_item (p2) != 1)) {
      cerr << "t = " << t << "\n";
      cerr << "p = " << p << "\n";
      cerr << "p1= " << p1 << "\n";
      cerr << "p2= " << p2 << "\n";
      fatal_error ("invalid object cut", "edit_select_rep::cut");
    }
    assign (p, "");
  }
}

void
edit_select_rep::selection_cut (string key) {
  if (!selection_active_any ()) return;
  if (selection_active_table ()) {
    path p1= start_p, p2= end_p;
    tree sel= selection_get ();
    selection_set (key, sel);
    cut (p1, p2);
  }
  else {
    path p1, p2;
    selection_get (p1, p2);
    go_to (p2);
    if (p2 == p1) return;

    tree sel= compute_selection (et, p1, p2);
    // cout << "Selection " << sel << "\n";
    selection_set (key, sel);
    // cout << "Selected  " << well_matched (sel) << "\n";
    cut (p1, p2);
  }
}

tree
edit_select_rep::selection_get_cut () {
  tree t= selection_get ();
  selection_cut ();
  return t;
}

void
edit_select_rep::selection_move () {
  int pos= position_new ();
  tree t= selection_get_cut ();
  go_to (position_get (pos));
  insert_tree (t);
  position_delete (pos);
}

#endmodule // code_edit_select
