/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2009 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include "pdbUtils.h"
#include "pdbStats.h"
#include "CovMat.h"
#include "DLTmath.h"
#include "FragCoords.h"
#include "ProcLAPACKSVD.h"

#if defined(__APPLE__)
    #include <vecLib/clapack.h>
    #include <vecLib/cblas.h>
#endif

double
CalcE0(const Coords *coords1, const Coords *coords2,
       const double *weights, const double *axesw);

static double
CalcE0Cov(const Coords *coords1, const Coords *coords2,
          const double *axesw);

static void
CalcR(const Coords *coords1, const Coords *coords2, double **Rmat,
      const double *weights, const double *axesw);

static void
CalcRCov(const Coords *coords1, const Coords *coords2, double **Rmat,
         const double **WtMat, const double *axesw);

static int
CalcLAPACKSVD(double **Rmat, double **Umat, double *sigma, double **VTmat);

static int
CalcRotMat(double **rotmat, double **Umat, double **VTmat);


double
CalcE0(const Coords *coords1, const Coords *coords2,
       const double *weights, const double *axesw)
{   
    int             i;
    double          sum;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    double          x1i, y1i, z1i, x2i, y2i, z2i;
    const double    axeswx = axesw[0], axeswy = axesw[1], axeswz = axesw[2];

    sum = 0.0;
    i = coords1->vlen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;
        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        sum += *weights++ * (axeswx * (x1i * x1i + x2i * x2i) +
                             axeswy * (y1i * y1i + y2i * y2i) +
                             axeswz * (z1i * z1i + z2i * z2i));
    }

    return(sum);
}


static double
CalcMahFrobInnProd(const Coords *coords, const double *weights, const double *axesw)
{   
    int             i;
    double          sum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          xi, yi, zi;
    const double    axeswx = axesw[0], axeswy = axesw[1], axeswz = axesw[2];

    sum = 0.0;
    i = coords->vlen;
    while(i-- > 0)
    {
        xi = *x++;
        yi = *y++;
        zi = *z++;

        sum += *weights++ *
               (axeswx * (xi * xi) + axeswy * (yi * yi) + axeswz * (zi * zi));
    }

    return(sum);
}


double
CalcInnProdNorm(const Coords *coords)
{   
    int             i;
    double          sum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          xi, yi, zi;

    sum = 0.0;
    i = coords->vlen;
    while(i-- > 0)
    {
        xi = *x++;
        yi = *y++;
        zi = *z++;

        sum += xi * xi + yi * yi + zi * zi;
    }

    return(sum);
}


static double
CalcMahFrobInnProdRot(const Coords *coords, const double **rmat, const double *weights, const double *axesw)
{   
    int             i;
    double          sum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          xi, yi, zi, xir, yir, zir;
    const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
                    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
                    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];
    const double    axeswx = axesw[0], axeswy = axesw[1], axeswz = axesw[2];

    sum = 0.0;
    i = coords->vlen;
    while(i-- > 0)
    {
        xi = *x++;
        yi = *y++;
        zi = *z++;

        xir = xi * rmat00 + yi * rmat10 + zi * rmat20;
        yir = xi * rmat01 + yi * rmat11 + zi * rmat21;
        zir = xi * rmat02 + yi * rmat12 + zi * rmat22;

        sum += *weights++ *
               (axeswx * (xir * xir) + axeswy * (yir * yir) + axeswz * (zir * zir));
    }

    return(sum);
}


/* Should not change with rotation */
double
CalcInnProdNormRot(const Coords *coords, const double **rmat)
{   
    int             i;
    double          sum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          xi, yi, zi, xir, yir, zir;
    const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
                    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
                    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

    sum = 0.0;
    i = coords->vlen;
    while(i-- > 0)
    {
        xi = *x++;
        yi = *y++;
        zi = *z++;

        xir = xi * rmat00 + yi * rmat10 + zi * rmat20;
        yir = xi * rmat01 + yi * rmat11 + zi * rmat21;
        zir = xi * rmat02 + yi * rmat12 + zi * rmat22;

        sum += xir * xir + yir * yir + zir * zir;
    }

    return(sum);
}


static double
CalcE0Cov(const Coords *coords1, const Coords *coords2,
          const double *axesw)
{   
    int             i;
    double          sum;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *cx2 = (const double *) coords2->covx,
                   *cy2 = (const double *) coords2->covy,
                   *cz2 = (const double *) coords2->covz;
    const double   *cx1 = (const double *) coords1->covx,
                   *cy1 = (const double *) coords1->covy,
                   *cz1 = (const double *) coords1->covz;
    double          x1i, y1i, z1i, x2i, y2i, z2i,
                    cx1i, cy1i, cz1i, cx2i, cy2i, cz2i;
    const double    axeswx = axesw[0], axeswy = axesw[1], axeswz = axesw[2];

    sum = 0.0;
    i = coords1->vlen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;
        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        cx1i = *cx1++;
        cy1i = *cy1++;
        cz1i = *cz1++;
        cx2i = *cx2++;
        cy2i = *cy2++;
        cz2i = *cz2++;

        sum += (axeswx * (cx1i * x1i + cx2i * x2i) +
                axeswy * (cy1i * y1i + cy2i * y2i) +
                axeswz * (cz1i * z1i + cz2i * z2i));
    }

    return(sum);
}


/* This function assumes that the coordinates have been centered previously
   Use CenMass() and ApplyCenter() */
void
CalcRold(const Coords *coords1, const Coords *coords2, double **Rmat,
      const double *weights, const double *axesw)
{
    int             i;
    double          weight;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double    axeswx = sqrt(axesw[0]),
                    axeswy = sqrt(axesw[1]),
                    axeswz = sqrt(axesw[2]);
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    i = coords1->vlen;
    while(i-- > 0)
    {
        weight = *weights++;

        x2i = axeswx * *x2++;
        y2i = axeswy * *y2++;
        z2i = axeswz * *z2++;

        x1i = weight * axeswx * *x1++;
        y1i = weight * axeswy * *y1++;
        z1i = weight * axeswz * *z1++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcR(const Coords *coords1, const Coords *coords2, double **Rmat,
      const double *weights, const double *axesw)
{
    int             i;
    double          weight;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    const double    axeswx = axesw[0],
                    axeswy = axesw[1],
                    axeswz = axesw[2];
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

    for (i = 0; i < coords1->vlen; ++i)
    {
        weight = weights[i];

        x1i = x1[i];
        y1i = y1[i];
        z1i = z1[i];

        x2i = weight * x2[i] * axeswx;
        y2i = weight * y2[i] * axeswy;
        z2i = weight * z2[i] * axeswz;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcRvan(const Coords *coords1, const Coords *coords2, double **Rmat)
{
    int             i;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

    i = coords1->vlen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;

        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcRAx(const Coords *coords1, const Coords *coords2, double **Rmat,
        const double *weights, const double **InvAx)
{
    int             i;
    double          weight;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    double          x2i, y2i, z2i, x1i, y1i, z1i, xt, yt, zt;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

    i = coords1->vlen;
    while(i-- > 0)
    {
        weight = *weights++;

        x1i = weight * *x1++;
        y1i = weight * *y1++;
        z1i = weight * *z1++;

        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        xt = x2i * InvAx[0][0] + y2i * InvAx[1][0] + z2i * InvAx[2][0];
        yt = x2i * InvAx[0][1] + y2i * InvAx[1][1] + z2i * InvAx[2][1];
        zt = x2i * InvAx[0][2] + y2i * InvAx[1][2] + z2i * InvAx[2][2];

        Rmat00 += xt * x1i;
        Rmat01 += xt * y1i;
        Rmat02 += xt * z1i;
        
        Rmat10 += yt * x1i;
        Rmat11 += yt * y1i;
        Rmat12 += yt * z1i;
        
        Rmat20 += zt * x1i;
        Rmat21 += zt * y1i;
        Rmat22 += zt * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcRCov(const Coords *coords1, const Coords *coords2, double **Rmat,
         const double **WtMat, const double *axesw)
{
    int             i;
    const double   *x2 = (const double *) coords2->covx,
                   *y2 = (const double *) coords2->covy,
                   *z2 = (const double *) coords2->covz;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double    axeswx = sqrt(axesw[0]),
                    axeswy = sqrt(axesw[1]),
                    axeswz = sqrt(axesw[2]);
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    i = coords1->vlen;
    while(i-- > 0)
    {
        x2i = axeswx * *x2++;
        y2i = axeswy * *y2++;
        z2i = axeswz * *z2++;

        x1i = axeswx * *x1++;
        y1i = axeswy * *y1++;
        z1i = axeswz * *z1++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static int
CalcLAPACKSVD(double **Rmat, double **Umat, double *sigma, double **VTmat)
{
/*     svdcmp(Rmat, 3, 3, sigma, VTmat); */
/*     Mat3Cpy(Umat, (const double **) Rmat); */
/*     Mat3TransposeIp(VTmat); */

/*     double **tmpmat = MatInit(3, 3); */
/*     Mat3Cpy(tmpmat, (const double **) Rmat); */

/*     long int        info = 0; */
/*     char            jobu = 'A'; */
/*     char            jobvt = 'A'; */
/*     long int        m = 3, n = 3, lda = 3, ldu = 3, ldvt = 3; */
/*     long int        lwork = 100; */
/*     double         *work; */
/*  */
/*     Mat3TransposeIp(Rmat); */

/*     printf("\n\n Rmat:"); */
/*     write_C_mat((const double **)Rmat, 3, 8, 0); */

    return(dgesvd_opt_dest(Rmat, 3, 3, Umat, sigma, VTmat));

/*     work = (double *) malloc(lwork * sizeof(double)); */
/*  */
/*     DGESVD(&jobvt, &jobu, &n, &m, */
/*             &Rmat[0][0], &lda, sigma, */
/*             &VTmat[0][0], &ldvt, */
/*             &Umat[0][0], &ldu, */
/*             work, &lwork, &info); */

/*     Mat3TransposeIp(Umat); */
/*     Mat3TransposeIp(VTmat); */

/*     printf("\n\n Umat:"); */
/*     write_C_mat((const double **)Umat, 3, 8, 0); */
/*     printf("\n\n VTmat:"); */
/*     write_C_mat((const double **)VTmat, 3, 8, 0); */
/*     int i; */
/*     for (i = 0; i < 3; ++i) */
/*         printf("\n sigma[%d] = %8.2f ", i, sigma[i]); */

/*  */
/*     char            jobvl = 'V'; */
/*     char            jobvr = 'V'; */
/*     double         *wr = malloc(3 * sizeof(double)); */
/*     double         *wi = malloc(3 * sizeof(double)); */
/*     double        **vl = MatInit(3, 3); */
/*     double        **vr = MatInit(3, 3); */
/*  */
/*     DGEEV(&jobvl, &jobvr, &n, &tmpmat[0][0], &n,  */
/*            wr, wi, */
/*            &vl[0][0], &n, &vr[0][0], &n, */
/*            work, &lwork, &info); */
/*  */
/*     Mat3Print(vl); */
/*     printf("\n real %f %f %f", wr[0], wr[1], wr[2]); */
/*     printf("\n imag %f %f %f", wi[0], wi[1], wi[2]); */
/*     free(wr); */
/*     free(wi); */
/*     MatDestroy(&vl); */
/*     MatDestroy(&vr); */
/*     MatDestroy(&tmpmat); */

/*     free(work); */
}


/* Takes U and V^t on input, calculates R = VU^t */
static int
CalcRotMat(double **rotmat, double **Umat, double **Vtmat)
{   
    int         i, j, k;
    double      det;

    memset(&rotmat[0][0], 0, 9 * sizeof(double));

    det = Mat3Det((const double **)Umat) * Mat3Det((const double **)Vtmat);

    if (det > 0)
    {
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                for (k = 0; k < 3; ++k)
                    rotmat[i][j] += (Vtmat[k][i] * Umat[j][k]);

        return(1);
    }
    else
    {
        /* printf("\n * determinant of SVD U or V matrix = %f", det); */

        for (i = 0; i < 3; ++i)
        {
            for (j = 0; j < 3; ++j)
            {
                for (k = 0; k < 2; ++k)
                    rotmat[i][j] += (Vtmat[k][i] * Umat[j][k]);

                rotmat[i][j] -= (Vtmat[2][i] * Umat[j][2]);
            }
        }

        return(-1);
    }
}


/* returns sum of squared residuals, E
   rmsd = sqrt(E/atom_num)  */
double
ProcLAPACKSVDvan(const Coords *coords1, const Coords *coords2, double **rotmat,
                 double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev, term1, term2;

    term1 = CalcInnProdNorm(coords2);
    CalcRvan(coords1, coords2, Rmat);
    CalcLAPACKSVD(Rmat, Umat, sigma, VTmat);
    det = CalcRotMat(rotmat, Umat, VTmat);
    term2 = CalcInnProdNorm(coords1);
    sumdev = term1 + term2;

/*     VerifyRotMat(rotmat, 1e-5); */
/*     printf("\n*************** sumdev = %8.2f ", sumdev); */
/*     printf("\nrotmat:"); */
/*     write_C_mat((const double **)rotmat, 3, 8, 0); */

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

/*     printf("\nRmat:"); */
/*     write_C_mat((const double **)Rmat, 3, 8, 0); */
/*     printf("\nUmat:"); */
/*     write_C_mat((const double **)Umat, 3, 8, 0); */
/*     printf("\nVTmat:"); */
/*     write_C_mat((const double **)VTmat, 3, 8, 0); */
/*     int i; */
/*     for (i = 0; i < 3; ++i) */
/*         printf("\nsigma[%d] = %8.2f ", i, sigma[i]); */

    return(sumdev);
}


static void
CalcRFrag(const FragCoords *coords1, const FragCoords *coords2, double **Rmat)
{
    int             i;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

    i = coords1->fraglen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;

        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static double
CalcInnProdNormFrag(const FragCoords *coords)
{   
    int             i;
    double          sum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          xi, yi, zi;

    sum = 0.0;
    i = coords->fraglen;
    while(i-- > 0)
    {
        xi = *x++;
        yi = *y++;
        zi = *z++;

        sum += xi * xi + yi * yi + zi * zi;
    }

    return(sum);
}


/* returns sum of squared residuals, E
   rmsd = sqrt(E/atom_num)  */
double
ProcLAPACKSVDFrag(const FragCoords *frag1, const FragCoords *frag2, double **rotmat,
                  double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev, term1, term2;

    term1 = CalcInnProdNormFrag(frag2);
    CalcRFrag(frag1, frag2, Rmat);
    CalcLAPACKSVD(Rmat, Umat, sigma, VTmat);
    det = CalcRotMat(rotmat, Umat, VTmat);
    term2 = CalcInnProdNormFrag(frag1);
    sumdev = term1 + term2;

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

    return(sumdev);
}



double
ProcLAPACKSVD(const Coords *coords1, const Coords *coords2, double **rotmat,
              const double *weights, const double *axesw,
              double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev, term1, term2;

    term1 = CalcMahFrobInnProd(coords2, weights, axesw);
    CalcR(coords1, coords2, Rmat, weights, axesw);
    CalcLAPACKSVD(Rmat, Umat, sigma, VTmat);
    det = CalcRotMat(rotmat, Umat, VTmat);
    term2 = CalcMahFrobInnProdRot(coords1, (const double **) rotmat, weights, axesw);
    sumdev = term1 + term2;

/*     VerifyRotMat(rotmat, 1e-5); */
/*     printf("\n*************** sumdev = %8.2f ", sumdev); */
/*     printf("\nrotmat:"); */
/*     write_C_mat((const double **)rotmat, 3, 8, 0); */

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

/*     printf("\nRmat:"); */
/*     write_C_mat((const double **)Rmat, 3, 8, 0); */
/*     printf("\nUmat:"); */
/*     write_C_mat((const double **)Umat, 3, 8, 0); */
/*     printf("\nVTmat:"); */
/*     write_C_mat((const double **)VTmat, 3, 8, 0); */
/*     int i; */
/*     for (i = 0; i < 3; ++i) */
/*         printf("\nsigma[%d] = %8.2f ", i, sigma[i]); */

    return(sumdev);
}


double
ProcLAPACKSVDAx(const Coords *coords1, const Coords *coords2, double **rotmat,
                const double *weights, const double *axesw, const double **AxWtMat,
                double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev, term1, term2;

    term1 = CalcMahFrobInnProd(coords1, weights, axesw);
    CalcRAx(coords1, coords2, Rmat, weights, AxWtMat);
    CalcLAPACKSVD(Rmat, Umat, sigma, VTmat);
    det = CalcRotMat(rotmat, Umat, VTmat);
    term2 = CalcMahFrobInnProdRot(coords2, (const double **) rotmat, weights, axesw);
    sumdev = term1 + term2;

/*     VerifyRotMat(rotmat, 1e-5); */
/*     printf("\n*************** sumdev = %8.2f ", sumdev); */
/*     printf("\nrotmat:"); */
/*     write_C_mat((const double **)rotmat, 3, 8, 0); */

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

/*     printf("\nRmat:"); */
/*     write_C_mat((const double **)Rmat, 3, 8, 0); */
/*     printf("\nUmat:"); */
/*     write_C_mat((const double **)Umat, 3, 8, 0); */
/*     printf("\nVTmat:"); */
/*     write_C_mat((const double **)VTmat, 3, 8, 0); */
/*     int i; */
/*     for (i = 0; i < 3; ++i) */
/*         printf("\nsigma[%d] = %8.2f ", i, sigma[i]); */

    return(sumdev);
}


/* returns sum of squared residuals, E
   rmsd = sqrt(E/atom_num)  */
double
ProcLAPACKSVDCov(Coords *coords1, Coords *coords2, double **rotmat,
                 const double **WtMat, const double *axesw, double **Rmat,
                 double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev = 0.0;

    CalcCovCoords(coords1, WtMat);
    CalcCovCoords(coords2, WtMat);

    sumdev = CalcE0Cov(coords1, coords2, axesw);
    CalcRCov(coords1, coords2, Rmat, WtMat, axesw);
    CalcLAPACKSVD(Rmat, Umat, sigma, VTmat);
    det = CalcRotMat(rotmat, Umat, VTmat);

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

    return(sumdev);
}
