#include "PreviewImage.hpp"

#include "../ClusterModel.hpp"

#include "wx/dcbuffer.h"

using namespace indii::tint;
using namespace indii::tint::gui;

#define MAX_ZOOM_IN 16
#define MAX_ZOOM_OUT 16

PreviewImage::PreviewImage(wxWindow* parent, ImageResource* res,
    ClusterModel* model) : wxScrolledWindow(parent, wxID_ANY,
	  wxDefaultPosition, wxDefaultSize, wxNO_FULL_REPAINT_ON_RESIZE|
    wxCLIP_CHILDREN|wxHSCROLL|wxVSCROLL), res(res), model(model),
    zoomNumerator(1), zoomDenominator(1), clear(true) {   
  watch(model);

  SetCursor(wxCursor(wxCURSOR_HAND));
  SetBackgroundStyle(wxBG_STYLE_CUSTOM);
  updateBackground();
}

PreviewImage::~PreviewImage() {
  //
}

void PreviewImage::zoomIn() {
  if (zoomDenominator == 1) {
    if (zoomNumerator < MAX_ZOOM_IN) {
      zoomNumerator *= 2;
    }
  } else {
    zoomDenominator /= 2;
  }
  clear = true;

  int width = res->getWidth()*zoomNumerator/zoomDenominator;
  int height = res->getHeight()*zoomNumerator/zoomDenominator;
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  updateBackground();

  /* post-condition */
  assert(zoomNumerator >= 1 && zoomDenominator >= 1);
}

void PreviewImage::zoomOut() {
  if (zoomNumerator == 1) {
    if (zoomDenominator < MAX_ZOOM_OUT) {
      zoomDenominator *= 2;
    }
  } else {
    zoomNumerator /= 2;
  }
  clear = true;

  int width = res->getWidth()*zoomNumerator/zoomDenominator;
  int height = res->getHeight()*zoomNumerator/zoomDenominator;
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  updateBackground();

  /* post-condition */
  assert(zoomNumerator >= 1 && zoomDenominator >= 1);
}

void PreviewImage::zoomNormal() {
  zoomNumerator = 1;
  zoomDenominator = 1;
  clear = true;

  int width = res->getWidth();
  int height = res->getHeight();
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  updateBackground();

  /* post-condition */
  assert(zoomNumerator == 1 && zoomDenominator == 1);
}

void PreviewImage::zoomFit() {
  const int clientWidth = GetClientSize().GetWidth();
  const int clientHeight = GetClientSize().GetHeight();
  int width, height;

  res->fitInside(clientWidth, clientHeight, &width, &height);
  zoomNumerator = std::min(width / res->getWidth(), height / res->getHeight());
  zoomDenominator = std::max(res->getWidth() / width, res->getHeight() / height);
  if (zoomNumerator == 0) {
    zoomNumerator = 1;
  } else if (zoomNumerator > MAX_ZOOM_IN) {
    zoomNumerator = MAX_ZOOM_IN;
  }
  if (zoomDenominator == 0) {
    zoomDenominator = 1;
  } else if (zoomDenominator > MAX_ZOOM_OUT) {
    zoomDenominator = MAX_ZOOM_OUT;
  }
  clear = true;
  
  width = res->getWidth()*zoomNumerator/zoomDenominator;
  height = res->getHeight()*zoomNumerator/zoomDenominator;
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  updateBackground();

  /* post-condition */
  assert (zoomNumerator == 1 || zoomDenominator == 1);
}

void PreviewImage::notifyNumClustersChange() {
  Refresh();
}

void PreviewImage::notifyClusterChange(const unsigned int i) {
  Refresh();
}

void PreviewImage::notifySaturationDecayChange() {
  Refresh();
}

void PreviewImage::notifyCentroidDecayChange() {
  Refresh();
}

void PreviewImage::notifySaturationSoftnessChange() {
  Refresh();
}

void PreviewImage::notifyCentroidSoftnessChange() {
  Refresh();
}

void PreviewImage::notifyGreyscaleChange() {
  Refresh();
}

void PreviewImage::OnPaint(wxPaintEvent& evt) {
  wxPaintDC dc(this);
  //wxAutoBufferedDC dc(this); // doesn't support SetUserScale()
  DoPrepareDC(dc);

  /* scale */
  const int width = res->getWidth() / zoomDenominator;
  const int height = res->getHeight() / zoomDenominator;
  dc.SetUserScale(zoomNumerator, zoomNumerator);

  /* background */
  #ifdef __WXMSW__
  if (clear) {
    dc.SetBackground(wxBrush(wxColour(69,69,69)));
    dc.Clear();
    clear = false;
  }
  #else
  dc.SetBackground(wxBrush(wxColour(69,69,69)));
  dc.Clear();
  #endif

  /* scrolling */
  wxRect rect;
  int xUnits, yUnits, xScale, yScale, xPixels, yPixels;

  GetViewStart(&xUnits, &yUnits);
  GetScrollPixelsPerUnit(&xScale, &yScale);
  xPixels = xUnits*xScale;
  yPixels = yUnits*yScale;

  /* origin */
  //const int originX = std::max(xPixels, (GetSize().GetWidth() - width) / 2);
  //const int originY = std::max(yPixels, (GetSize().GetHeight() - height) / 2);
  //dc.SetDeviceOrigin(originX, originY);

  wxRegionIterator upd(GetUpdateRegion());
  while (upd) {
    rect.x = std::max((int)upd.GetX()+xPixels, 0) / zoomNumerator;
    rect.y = std::max((int)upd.GetY()+yPixels, 0) / zoomNumerator;
    rect.width = std::max(std::min((int)upd.GetW() / (int)zoomNumerator + 1,
        width - rect.x), 0);
    rect.height = std::max(std::min((int)upd.GetH() / (int)zoomNumerator + 1,
        height - rect.y), 0);

    if (rect.width > 0 && rect.height > 0) {
      wxImage fg(rect.width, rect.height);
      model->calcFg(rect, width, height, fg);
      dc.DrawBitmap(fg, rect.x, rect.y);
    }

    upd++;
  }
}

void PreviewImage::OnSize(wxSizeEvent& evt) {
  clear = true;
}

void PreviewImage::OnMouseWheel(wxMouseEvent& evt) {
  if (evt.GetWheelRotation() > 0) {
    zoomIn();
  } else if (evt.GetWheelRotation() < 0) {
    zoomOut();
  }
}

void PreviewImage::OnMouseDown(wxMouseEvent& evt) {
  dragging = true;
  mouseX = evt.GetX();
  mouseY = evt.GetY();
}

void PreviewImage::OnMouseUp(wxMouseEvent& evt) {
  dragging = false;
}

void PreviewImage::OnMouseMove(wxMouseEvent& evt) {
  if (evt.Dragging()) {
    int x, y, xUnit, yUnit;

    GetViewStart(&x, &y);
    GetScrollPixelsPerUnit(&xUnit, &yUnit);

    x *= xUnit;
    y *= yUnit;
    x -= evt.GetX() - mouseX;
    y -= evt.GetY() - mouseY;
    x /= xUnit;
    y /= yUnit;

    Scroll(x, y);
    mouseX = evt.GetX();
    mouseY = evt.GetY();
  }
}

void PreviewImage::OnMouseEnter(wxMouseEvent& evt) {
  //wxSetCursor(wxCursor(wxCURSOR_HAND));
}

void PreviewImage::OnMouseLeave(wxMouseEvent& evt) {
  //wxSetCursor(wxNullCursor);
  dragging = false;
}

void PreviewImage::OnEraseBackground(wxEraseEvent& evt) {
  //
}

void PreviewImage::updateBackground() {
  const int width = res->getWidth() / zoomDenominator;
  const int height = res->getHeight() / zoomDenominator;

  bg = wxBitmap(res->get(width, height)->ConvertToGreyscale());
}

BEGIN_EVENT_TABLE(PreviewImage, wxWindow)
EVT_PAINT(PreviewImage::OnPaint)
EVT_LEFT_DOWN(PreviewImage::OnMouseDown)
EVT_LEFT_UP(PreviewImage::OnMouseUp)
EVT_MOUSEWHEEL(PreviewImage::OnMouseWheel)
EVT_MOTION(PreviewImage::OnMouseMove)
EVT_ENTER_WINDOW(PreviewImage::OnMouseEnter)
EVT_LEAVE_WINDOW(PreviewImage::OnMouseLeave)
EVT_ERASE_BACKGROUND(PreviewImage::OnEraseBackground)
EVT_SIZE(PreviewImage::OnSize)
END_EVENT_TABLE()

