/***************************************************************************
 *   copyright           : (C) 2003 by Hendrik Sattler                     *
 *   mail                : pingos@hendrik-sattler.de                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ContestPropDlg.H"
#include <wx/statline.h>
#include <algorithm>

enum {
  BTN_PICT_SEL = 1,
  BTN_DESCR_SEL,
  BTN_LESSON_SEL,
  BTN_DATA_SEL,
  BTN_TEST_WIZ,
  BTN_SHOW_RES,
  BTN_SAVE
};

BEGIN_EVENT_TABLE(ContestPropDlg,wxDialog)
  EVT_BUTTON(BTN_PICT_SEL, ContestPropDlg::onUrlSelectButton)
  EVT_BUTTON(BTN_DESCR_SEL, ContestPropDlg::onUrlSelectButton)
  EVT_BUTTON(BTN_LESSON_SEL, ContestPropDlg::onUrlSelectButton)
  EVT_BUTTON(BTN_DATA_SEL, ContestPropDlg::onFileSelectButton)
  EVT_BUTTON(BTN_TEST_WIZ, ContestPropDlg::onTestWizardButton)
  EVT_BUTTON(BTN_SHOW_RES, ContestPropDlg::onShowResultsButton)
  EVT_BUTTON(BTN_SAVE, ContestPropDlg::onSaveButton)
END_EVENT_TABLE()

#ifndef __VISUALC__
  using std::max;
#endif

ContestPropDlg::ContestPropDlg (wxWindow* parent, const ContestDesc& cdata)
  : wxDialog (parent,-1,wxString(_("Contest Properties")),
	      wxDefaultPosition, wxDefaultSize,
	      wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER),
    settings(cdata),
    width(0)
{
  this->titleText = new wxStaticText(this,-1,_("Title:"));
  this->width = max(width,this->titleText->GetSize().GetWidth());
  this->pictureText = new wxStaticText(this,-1,_("Picture:"));
  this->width = max(width,this->pictureText->GetSize().GetWidth());
  this->descrText = new wxStaticText(this,-1,_("Description:"));
  this->width = max(width,this->descrText->GetSize().GetWidth());
  this->lessonText = new wxStaticText(this,-1,_("Lesson:"));
  this->width = max(width,this->lessonText->GetSize().GetWidth());
  this->dataText = new wxStaticText(this,-1,_("Data:"));
  this->width = max(width,this->dataText->GetSize().GetWidth());
  this->titleText->SetSize(this->width,-1);
  this->pictureText->SetSize(this->width,-1);
  this->descrText->SetSize(this->width,-1);
  this->lessonText->SetSize(this->width,-1);
  this->dataText->SetSize(this->width,-1);

  wxBoxSizer* dialogSizer = new wxBoxSizer(wxVERTICAL);
  dialogSizer->Add(wizardSettingsSizer(),0,
		   wxEXPAND|wxLEFT|wxTOP|wxRIGHT,10);
  dialogSizer->Add(new wxButton(this,BTN_TEST_WIZ,_("Test wizard")),
		   0,wxALIGN_RIGHT|wxRIGHT,10);
  dialogSizer->Add(0,0,1);
  dialogSizer->Add(lessonSettingsSizer(),0,
		   wxEXPAND|wxLEFT|wxTOP|wxRIGHT,10);
  dialogSizer->Add(0,0,1);
  dialogSizer->Add(resultSettingsSizer(),0,
		   wxEXPAND|wxLEFT|wxTOP|wxRIGHT,10);
  dialogSizer->Add(new wxButton(this,BTN_SHOW_RES,_("Show results")),
		   0,wxALIGN_RIGHT|wxRIGHT,10);
  dialogSizer->Add(0,0,1);
  dialogSizer->Add(new wxStaticLine(this,-1),0,
		   wxEXPAND|wxLEFT|wxTOP|wxRIGHT,10);
  dialogSizer->Add(buttonSizer(),0,wxALL|wxALIGN_CENTRE,10);

  fillFromSettings();

  SetSizer(dialogSizer);
  dialogSizer->SetSizeHints(this);
  Centre();
}

wxSizer* ContestPropDlg::buttonSizer () {
  wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
  sizer->Add(new wxButton(this,BTN_SAVE,_("Save")),
		   1,wxEXPAND|wxALIGN_CENTER|wxRIGHT,10);
  sizer->Add(new wxButton(this,wxID_CANCEL,_("Cancel"))
		   ,1,wxEXPAND|wxALIGN_CENTER);
  return sizer;
}

wxSizer* ContestPropDlg::wizardSettingsSizer () {
  wxFlexGridSizer* wSubSizer = new wxFlexGridSizer(3,5,5);

  //title line
  wSubSizer->Add(this->titleText,0,wxALIGN_CENTRE_VERTICAL);
  this->title = new wxTextCtrl(this,-1,this->settings.title);
  this->title->SetToolTip(_("This defines the window title of the contest wizard (and partially also of other contest related dialogs)."));
  wSubSizer->Add(this->title,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  wSubSizer->Add(0,0);

  //picture source selection line
  wSubSizer->Add(this->pictureText,0,wxALIGN_CENTRE_VERTICAL);
  this->picture = new wxTextCtrl(this,-1,_T(""),
				 wxDefaultPosition,wxDefaultSize,wxTE_READONLY);
  this->picture->SetToolTip(_("This is an optional parameter. It defines the source of the wizard picture. Several graphic file formats are supported."));
  wSubSizer->Add(this->picture,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  wSubSizer->Add (new wxButton(this,BTN_PICT_SEL,_("Select"),
			       wxDefaultPosition,wxDefaultSize,wxBU_EXACTFIT),
		  0,wxALIGN_CENTRE_VERTICAL);

  //description source selection line
  wSubSizer->Add(this->descrText,0,wxALIGN_CENTRE_VERTICAL);
  this->descr = new wxTextCtrl(this,-1,_T(""),
			       wxDefaultPosition,wxDefaultSize,wxTE_READONLY);
  this->descr->SetToolTip(_("This is an optional parameter. It defines the source of the introduction text that is displayed on the first page of the wizard."));
  wSubSizer->Add(this->descr,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  wSubSizer->Add (new wxButton(this,BTN_DESCR_SEL,_("Select"),
			       wxDefaultPosition,wxDefaultSize,wxBU_EXACTFIT),
		  0,wxALIGN_CENTRE_VERTICAL);
  
  //GTK1 default size is too small
//   int width = 0;
//   int height = 0;
//   this->title->GetSize(&width,&height);
//   this->title->SetSize(3*width,height);

  wSubSizer->AddGrowableCol(1);

  wxStaticBoxSizer* wSizer = new wxStaticBoxSizer(new wxStaticBox(this, -1, _("Contest wizard elements")),
						  wxVERTICAL);
  wSizer->Add(wSubSizer,1,wxEXPAND|wxALL,5);
  return wSizer;
}

wxSizer* ContestPropDlg::lessonSettingsSizer () {
  wxFlexGridSizer* lSubSizer = new wxFlexGridSizer(3,5,5);
  lSubSizer->Add(this->lessonText,0,wxALIGN_CENTRE_VERTICAL);
  this->lesson = new wxTextCtrl(this,-1,_T(""),
				wxDefaultPosition,wxDefaultSize,wxTE_READONLY);
  this->lesson->SetToolTip(_("This defines the lesson text. The content of the file is cached during a lesson."));
  lSubSizer->Add(this->lesson,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  lSubSizer->Add (new wxButton(this,BTN_LESSON_SEL,_("Select"),
			       wxDefaultPosition,wxDefaultSize,wxBU_EXACTFIT),
		  0,wxALIGN_CENTRE_VERTICAL);
  lSubSizer->AddGrowableCol(1);

  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  sizer->Add(lSubSizer,0,wxEXPAND);

  wxStaticBoxSizer* lSizer = new wxStaticBoxSizer(new wxStaticBox(this, -1, _("Lesson content")),
						  wxVERTICAL);
  lSizer->Add(sizer,1,wxEXPAND|wxALL,5);
  return lSizer;
}

wxSizer* ContestPropDlg::resultSettingsSizer () {
  wxFlexGridSizer* lSubSizer = new wxFlexGridSizer(3,5,5);
  lSubSizer->Add(this->dataText,0,wxALIGN_CENTRE_VERTICAL);
  this->data = new wxTextCtrl(this,-1,_T(""),
			      wxDefaultPosition,wxDefaultSize,wxTE_READONLY);
  this->data->SetToolTip(_("This defines the file to write the contest results to. It must be a local file."));
  lSubSizer->Add(this->data,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  lSubSizer->Add (new wxButton(this,BTN_DATA_SEL,_("Select"),
			       wxDefaultPosition,wxDefaultSize,wxBU_EXACTFIT),
		  0,wxALIGN_CENTRE_VERTICAL);

  lSubSizer->AddGrowableCol(1);

  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  sizer->Add(lSubSizer,0,wxEXPAND);
  sizer->Add(sliderSizer(),0,wxEXPAND|wxTOP,5);

  wxStaticBoxSizer* lSizer = new wxStaticBoxSizer(new wxStaticBox(this, -1, _("Result storage and calculation")),
						  wxVERTICAL);
  lSizer->Add(sizer,1,wxEXPAND|wxALL,5);
  return lSizer;
}

wxSizer* ContestPropDlg::sliderSizer () {
  this->weight = new wxSlider(this,-1,this->settings.weight-50,-50,50,wxDefaultPosition,
			      wxDefaultSize,wxSL_HORIZONTAL|wxSL_TOP|wxSL_LABELS);
  this->weight->SetToolTip(_("This slider defines the weight of speed and correctness. The result list order may be changed when this values changes."));
  wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
  sizer->Add(new wxStaticText(this,-1,_("Speed")),0,wxALIGN_CENTRE_VERTICAL);
  sizer->Add(this->weight,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL|wxLEFT|wxRIGHT,5);
  sizer->Add(new wxStaticText(this,-1,_("Correctness")),0,wxALIGN_CENTRE_VERTICAL);
  return sizer;
}

void ContestPropDlg::fillFromSettings () {
  //this->title excluded
  this->picture->SetValue(this->settings.pictureFile.getURLstr());
  this->descr->SetValue(this->settings.descrFile.getURLstr());
  this->lesson->SetValue(this->settings.lessonFile.getURLstr());
  this->data->SetValue(this->settings.dataFile.GetFullPath());
  //this->weight excluded
}

#include "dirs.H"
#include "FileUrlSelectDlg.H"

void ContestPropDlg::onUrlSelectButton (wxCommandEvent& e) {
  bool can_be_empty = true;
  bool textfile = true;
  UrlFile* old = 0;
  switch (e.GetId()) {
  case BTN_PICT_SEL:
    textfile = false;
    old = &this->settings.pictureFile;
    break;
  case BTN_DESCR_SEL:
    old = &this->settings.descrFile;
    break;
  case BTN_LESSON_SEL:
    can_be_empty = false;
    old = &this->settings.lessonFile;
    break;
  default:
    break;
  }
  if (old == 0) {
    return;
  }
  FileUrlSelectDlg* dialog;
  if (textfile) {
    dialog = new TextFileUrlSelectDlg(this,-1,*old,can_be_empty);
  } else {
    dialog = new BinaryFileUrlSelectDlg(this,-1,*old,can_be_empty);
  }
  if (dialog->ShowModal() == wxID_OK) {
    UrlFile* sel = dialog->getURL();
    *old = *sel;
    fillFromSettings();
    delete sel;
  }
  delete dialog;
}

#include <wx/file.h>

void ContestPropDlg::onFileSelectButton (wxCommandEvent& e) {
  wxFileDialog dialog(this);
  wxFileName* old = 0;
  switch (e.GetId()) {
  case BTN_DATA_SEL:
    old = &this->settings.dataFile;
    dialog.SetStyle(wxSAVE);
    break;
  default:
    return;
    break;
  }
  dialog.SetPath(old->GetFullPath());
  int r;
  do { //loop to work-around wxWindows bug
    r = dialog.ShowModal();
  } while (r == wxID_OK &&
           (dialog.GetStyle()&wxFILE_MUST_EXIST) > 0 &&
           !wxFile::Exists(dialog.GetPath()));

  if (r == wxID_OK) {
    old->Assign(dialog.GetPath());
    fillFromSettings();
  }
}

#include "ContestWizard.H"

void ContestPropDlg::onTestWizardButton (wxCommandEvent& WXUNUSED(e)) {
  wxString t = this->title->GetValue();
  ContestWizard wiz(this,t,
		    &this->settings.pictureFile,
		    &this->settings.descrFile);
  wiz.runWizard();
}

#include "ContestRankingDlg.H"

void ContestPropDlg::onShowResultsButton (wxCommandEvent& WXUNUSED(e)) {
  if (this->data->GetValue() == _T("")) {
    wxMessageBox(_("You need to specify a data storage file!"),
		 _("Important"), wxICON_WARNING | wxOK, this);    
  } else {
    ContestRankingDlg ranking(this,this->title->GetValue(),this->data->GetValue(),this->weight->GetValue()+50);
    ranking.ShowModal();
  }
}

void ContestPropDlg::onSaveButton (wxCommandEvent& WXUNUSED(e)) {
  if (this->title->GetValue() == _T("") ||
      this->lesson->GetValue() == _T("") ||
      this->data->GetValue() == _T("")) {
    wxMessageBox(_("You need to specify a title, a lesson file and a data storage file!"),
		 _("Important"), wxICON_WARNING | wxOK, this);
  } else {
    this->settings.title = this->title->GetValue();
    this->settings.weight = this->weight->GetValue()+50;
    EndModal(wxID_OK);
  }
}

ContestDesc ContestPropDlg::getContestDesc () {
  return this->settings;
}
