/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	Department of Computer Science,
 *      The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

//
// $Id: IU.h 1.1 Fri, 25 Oct 1996 18:04:04 -0600 maccabe $
//

//
// IU.h - IntegerUnit class definition
//      by Gerard Rollins
//  7/22/92

#ifndef _IU_h
#define _IU_h

class MMU;
class SystemBus;
class Instruction;
class RegisterBlock;

class IntegerUnit {
public:
    int     IU_annul;

    enum {
    	USER_TEXT	= 8,
    	SUPERVISOR_TEXT = 9,
    	USER_DATA	= 10,
    	SUPERVISOR_DATA = 11
    };

    enum State{
	RESET,
	EXECUTE,
	ERROR
    };

    enum TrapType {
	instruction_access_exception	= 0x01,		// priority 5
	illegal_instruction		= 0x02,		// priority 7
	privileged_instruction		= 0x03,		// priority 6
	fp_disabled			= 0x04,		// priority 8
	window_overflow			= 0x05,		// priority 9
	window_underflow		= 0x06,		// priority 9
	mem_address_not_aligned		= 0x07,		// priority 10
	fp_exception			= 0x08,		// priority 11
	data_access_exception		= 0x09,		// priority 13
	tag_overflow			= 0x0a,		// priority 14
	watch_point_detected		= 0x0b,		// priority 8

	interrupt_level_1		= 0x11,		// priority 31
	interrupt_level_2		= 0x12,		// priority 30
	interrupt_level_3		= 0x13,		// priority 29
	interrupt_level_4		= 0x14,		// priority 28
	interrupt_level_5		= 0x15,		// priority 27
	interrupt_level_6		= 0x16,		// priority 26
	interrupt_level_7		= 0x17,		// priority 25
	interrupt_level_8		= 0x18,		// priority 24
	interrupt_level_9		= 0x19,		// priority 23
	interrupt_level_10		= 0x1a,		// priority 22
	interrupt_level_11		= 0x1b,		// priority 21
	interrupt_level_12		= 0x1c,		// priority 20
	interrupt_level_13		= 0x1d,		// priority 19
	interrupt_level_14		= 0x1e,		// priority 18
	interrupt_level_15		= 0x1f,		// priority 17

	r_register_access_error		= 0x20,		// priority 4
	instruction_access_error	= 0x21,		// priority 3

	cp_disabled			= 0x24,		// priority 8
	unimplemented_flush		= 0x25,		// priority 8

	cp_exception			= 0x28,		// priority 11
	data_access_error		= 0x29,		// priority 12
	division_by_zero		= 0x2a,		// priority 15
	data_store_error		= 0x2b,		// priority 2
	data_access_MMU_miss		= 0x2c,		// priority 12
	instruction_access_MMU_miss	= 0x3c,		// priority 2

	impl_dependent_exception	= 0x60,		// 0x60 - 0x7f

	trap_instruction		= 0x80,		// 0x80 - 0xff

	LAST_TRAP			= 0xff
    };


    // public Constructors / Destructors
    //
    IntegerUnit(UInt32, SystemBus&, MMU&, RegisterBlock&);

    // ~IntegerUnit();

    // public Member functions
    //     
    State   IU_state();
    void    reset();
    int     execute();
    void    trapFunction(int, void (*tf)(int));


    // Register query
    //
    UInt32  PC();
    UInt32  nPC();
    UInt32  PSR();
    UInt32  WIM();
    UInt32  TBR();
    UInt32  Y();

    // Register Modify
    //
    void    PC(UInt32);
    void    nPC(UInt32);
    void    PSR(UInt32);
    void    WIM(UInt32);
    void    TBR(UInt32);
    void    Y(UInt32);

    // PSR fields query
    UInt32     N();              // PSR<23>    
    UInt32     Z();              // PSR<22>    
    UInt32     V();              // PSR<21>    
    UInt32     C();              // PSR<20>    
    UInt32     EC();             // PSR<13>    
    UInt32     EF();             // PSR<12>    
    UInt32     PIL();            // PSR<11:8>    
    UInt32     S();              // PSR<7>    
    UInt32     PS();             // PSR<6>    
    UInt32     ET();             // PSR<5>    

private:
    IntegerUnit(const IntegerUnit&);    // Declare away
    void operator = (const IntegerUnit&);    // Declare away

    enum { NOTSET = 0, SET = 1 };

    // ****** functions for instruction decode and execution ****
    //
    void    execute_trap();
    void    select_trap();
    void    initialize();
    void    do_pipelined_write_of_state_regs();
    void    do_parallel_load_of_state_regs();

    int     dispatch_instruction(const Instruction&);
    void    load(const Instruction&);
    void    store(const Instruction&);
    void    atomic_load_store(const Instruction&);
    void    swap(const Instruction&);
    void    add(const Instruction&);
    void    tagged_add(const Instruction&);
    void    subtract(const Instruction&);
    void    tagged_subtract(const Instruction&);
    void    multiply_step(const Instruction&);
    void    multiply(const Instruction&);
    void    divide(const Instruction&);
    void    logical(const Instruction&);
    void    shift(const Instruction&);
    void    sethi(const Instruction&);
    void    save_restore(const Instruction&);
    void    branch_icc(const Instruction&);
    void    call(const Instruction&);
    void    jump_link(const Instruction&);
    void    return_from_trap(const Instruction&);
    void    trap_icc(const Instruction&);
    void    read_state_reg(const Instruction&);
    void    write_state_reg(const Instruction&);
    void    unimplemented();

    // Registers
    //
    UInt32  IU_PC;
    UInt32  IU_nPC;
    UInt32  IU_WIM;    
    UInt32  IU_Y;

    // alterable fields of registers declared as UInt32 to make creating return 
    // values thru shifting and or'ing easier. (no typecasting required)
    //
    // ** alterable PSR fields **
    //
    UInt32     IU_N;              // PSR<23>    
    UInt32     IU_Z;              // PSR<22>    
    UInt32     IU_V;              // PSR<21>    
    UInt32     IU_C;              // PSR<20>    
    UInt32     IU_EC;             // PSR<13>    
    UInt32     IU_EF;             // PSR<12>    
    UInt32     IU_PIL;            // PSR<11:8>    
    UInt32     IU_S;              // PSR<7>    
    UInt32     IU_PS;             // PSR<6>    
    UInt32     IU_ET;             // PSR<5>    

    // ** alterable TBR fields **
    //
    UInt32    TBA;            // TBR<31:12>
    UInt32    tt;             // TBR<11:4>
   

    // pipelined values
    //
    UInt32 PSRp;
    UInt32 PSRpp;
    UInt32 PSRppp;
    UInt32 PSRpppp;

    UInt32 WIMp;
    UInt32 WIMpp;
    UInt32 WIMppp;
    UInt32 WIMpppp;

    UInt32 TBRp;
    UInt32 TBRpp;
    UInt32 TBRppp;
    UInt32 TBRpppp;

    UInt32 Yp;
    UInt32 Ypp;
    UInt32 Yppp;
    UInt32 Ypppp;
   
   
    // ** Internal signals used by ISP **
    //
    State   mode; 
    int     addr_space;
    int     delayed_write_count;      // not defined in ISP
    int     interrupt_level;

    int     trap;
    int     reset_trap;

    int	    trapFlag[LAST_TRAP+1];	// array of flags indexed by TrapType
    void    (*trapFunction_[LAST_TRAP+1])(int);

    const int       NWINDOWS;

    SystemBus&      sbus;
    MMU&            mmu;        // Memory management unit
    RegisterBlock&  reg;

};

#endif /* _IU_h */
