/* Tower Toppler - Nebulus
 * Copyright (C) 2000-2006  Andreas Rver
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "menusys.h"

#include "string.h"
#include "screen.h"
#include "sprites.h"
#include "configuration.h"
#include "event.h"

#include <vector>
#include <string>

#include <ctype.h>

void draw_input_box(int x, int y, int len, int cursor, char *txt);
static int input_box_cursor_state = 0;


menubg_callback_proc Menusystem::bgproc=0;


void Menusystem::set_bgproc(menubg_callback_proc proc) {
    bgproc=proc;
}

void Menusystem::set_timeproc(long t, menuopt_callback_proc proc) {
    mtime=t;
    timeproc=proc;
    curr_mtime=0;
}

void Menusystem::add_option(const char *name,menuopt_callback_proc pr,
		 SDLKey quickkey, menuoptflags flags, int state) {
  Menuoption *opt=new Menuoption(name,pr,quickkey,flags,state);
  if(quickkey!=SDLK_UNKNOWN) {
    quickkeys[quickkey]=options.size();
  }
  options.push_back(opt);
  int olen=0;
  if(name) {
    olen=scr_textlength(name);
  } else if(pr) {
    const char *nm=(*pr)(NULL);
    if(nm) {
      olen=scr_textlength(nm);
    }
  } 
  if(maxoptlen<olen) {
    maxoptlen=olen;
  }
}


void Menusystem::draw(int dx,int dy) {
  static int color_r = 0, color_g = 20, color_b = 70;
  unsigned int y=dy;
  int offs = 0, realy, minx, miny, maxx, maxy, scrlen,
    newhilite = -1, yz, titlehei=0;


  if(hilited<0) {
    move_hilite_up();
  }
  if (hilited >= (int) options.size()) {
    move_hilite_down();
  }
  
  draw_background();

  titlehei=0;

  if(!title.empty()) {
    char ntitle[255];
    strncpy(ntitle,title.c_str(),254);
    int pos = 0;
    int start = 0;
    int len = strlen(ntitle);

    while (pos <= len) {

      if ((ntitle[pos] == '\n') || (ntitle[pos] == 0)) {

        bool end = ntitle[pos] == 0;

        ntitle[pos] = 0;
        scr_writetext_center(ystart + titlehei * FONTHEI, ntitle + start);
        titlehei ++;

        if (!end)
          ntitle[pos] = '\n';

        start = pos + 1;
      }
      pos++;
    }

    titlehei++;
  }

  yz = ystart + (titlehei) * FONTHEI;
  for (y = 0; (yz+y+1 < SCREENHEI) && (y+offs < options.size()); y++) {
    realy = yz + y * FONTHEI;
    scrlen = scr_textlength(options[y]->get_name().c_str(), options[y]->get_name().length());
    minx = (SCREENWID - scrlen) / 2;
    miny = realy;
    maxx = (SCREENWID + scrlen) / 2;
    maxy = realy + FONTHEI;
    if (!options[y]->get_name().empty()) {
      if (dx >= minx && dx <= maxx && dy >= miny && dy <= maxy) {
        newhilite = y + offs;
        curr_mtime = 0;
      }
      if ((int)y + offs == hilited) {
        if (yhilitpos == -1) {
          yhilitpos = miny;
        } else {
          if (yhilitpos < miny) {
            yhilitpos += ((miny - yhilitpos + 3) / 4)+1;
            if (yhilitpos > miny) yhilitpos = miny;
          } else if (yhilitpos > miny) {
            yhilitpos -= ((yhilitpos - miny + 3) / 4)+1;
            if (yhilitpos < miny) yhilitpos = miny;
          }
        }
        scr_putbar((SCREENWID - maxoptlen - 8) / 2, yhilitpos - 3,
                   maxoptlen + 8, FONTHEI + 3,
                   color_r, color_g, color_b, (config.use_alpha_darkening())?128:255);
      }
    }
  }

  for (y = 0; y < options.size(); y++) {
    if (!options[y+offs]->is_empty()) {
      miny = ystart + (y + titlehei)*FONTHEI;
      if ((options[y+offs]->get_flags() & MOF_LEFT))
        scr_writetext((SCREENWID - maxoptlen) / 2 + 4, miny,
                      options[y+offs]->get_name().c_str());
      else
        if ((options[y+offs]->get_flags() & MOF_RIGHT))
          scr_writetext((SCREENWID + maxoptlen) / 2 - 4
                        - scr_textlength(options[y+offs]->get_name().c_str()), miny,
                        options[y+offs]->get_name().c_str());
        else
          scr_writetext_center(miny, options[y+offs]->get_name().c_str());
    }
  }

  if (newhilite >= 0) hilited = newhilite;

  scr_color_ramp(&color_r, &color_g, &color_b);

  scr_swap();
  dcl_wait();

}

// draw_background.  This will draw the background proc.  If there isn't one,
// it will walk up the parents to see if one of the parents have one.
void Menusystem::draw_background() {
 
  Menusystem *m=this;
  while(m && (!m->callback) && (m->parent)) {
      m=m->parent;
  } 

  if(m->callback)
    (*m->callback)(this);
  else if (bgproc)
    (*bgproc) ();
}


void Menusystem::run(Menusystem *mparent) {
  int x,y;

  parent=mparent;

  reset_hilited();

  do {
    bool do_menu=false;
    x = y = 0;

    if ((curr_mtime++ >= mtime) && timeproc) {
      (void) (*timeproc) (this);
      curr_mtime = 0;
    }

    draw(x,y);
    Event *e=events.get_input();
    if(e) {
	set_last_event(e); // set_last_event delets old one to avoid mem leak
	curr_mtime=0; // reset timer 
    }
	
    if(captured>-1) {
      hilited=captured;
      do_menu=true;
    } else if(e) {
      switch(e->get_type()) {
	case down_ev:
	  move_hilite_down();
	  break;
	case up_ev:
	  move_hilite_up();
	  break;
	case fire_ev:
	case left_ev:
	case right_ev:
	  do_menu=true;
	  break;
	case break_ev: 
	  exitmenu = true; 
	  break;
	case key_ev:
	  if(quickkeys[e->get_keysym()]) { 
	    hilited=quickkeys[e->get_keysym()];
	  } 
	  break;
	default:
	  break;
      }
    }
    if ((do_menu)&&(hilited >= 0) && (hilited < (int)options.size()) &&
	options[hilited]->has_proc()) {
      const char *tmpbuf = options[hilited]->run_proc(this);
      if (tmpbuf) {
	options[hilited]->set_name(tmpbuf);
	if (maxoptlen < (int)options[hilited]->get_name().length()) 
	  maxoptlen = options[hilited]->get_name().length();
      }
    }
  } while(!exitmenu);
  exitmenu=false;
}

void Menusystem::reset_hilited() {
  hilited=0;
  while((options[hilited]->is_empty())&&(hilited<(int)options.size())) {
    hilited++;
  }
}

// move_hilite_down and move_hilite_up:  These abstract moving the hilited
// around
void Menusystem::move_hilite_down() {
  int n=hilited+1;
  while((n<(int)options.size()) && (options[n]->is_empty())) {
    n++;
    if((n==(int)options.size()&&(wraparound))) {
	n=0;
    } 
  }
  // if we have no wraparound and we didn't find any more options
  if((n>=(int)options.size())||(options[n]->is_empty())) {
    n=hilited;
  }
  hilited=n;
}

void Menusystem::move_hilite_up() {
  int n=hilited-1;
  while((n>=0) && (options[n]->is_empty())) {
    n--;
    if((n<0)&&(wraparound)) {
	n=options.size()-1;
    } 
  }
  // if we have no wraparound and we didn't find any more options
  if((n<0)||(options[n]->is_empty())) {
    n=hilited;
  }
  hilited=n;
}

void Menusystem::add_spacer() {
  add_option(NULL,NULL);
}


Menusystem::Menusystem(char const* ntitle, menuopt_callback_proc ncallback, int nlen, int nystart) : hilited(0), captured(-1), yhilitpos(-1), maxoptlen(nlen), curr_mtime(0), exitmenu(false), wraparound(false), mtime(-1), opt_steal_control(-1), ystart(nystart), callback(ncallback), timeproc(NULL), last_ev(0), state(0) {
  if(ntitle) {
    title=ntitle;   
  } else {
    title="";
  } 
}

// men_input
// This function pops up an input box
bool men_input(char* origs, int max_len, int xpos, int ypos, 
                           const char *allowed) {
 
  char inpc;
  static int pos = strlen(origs);
  int ztmp;
  static char s[256];
  static bool copy_origs = true;
  bool restore_origs = false;
  bool ende = false;

  if ((strlen(origs) >= 256)) return true;

  if (copy_origs) {
    strcpy(s, origs);
    copy_origs = false;
    pos = strlen(origs);
  }

 
  Menusystem::call_bgproc();

  draw_input_box(xpos,ypos,max_len,pos,s);
  scr_swap();
  dcl_wait();
  
  SDLKey sdlinp=events.get_nextkey(&inpc);

  switch (sdlinp) {
  case SDLK_RIGHT: if ((unsigned)pos < strlen(s)) pos++; break;
  case SDLK_LEFT: if (pos > 0) pos--; break;
  case SDLK_ESCAPE:if (strlen(s)) {
    s[0] = '\0';
    pos = 0;
    restore_origs = false;
  } else {
    restore_origs = true;
    ende = true;
  }
  break;
  case SDLK_RETURN: restore_origs = false; copy_origs = true; ende = true;
  break;
  case SDLK_DELETE:
    if (strlen(s) >= (unsigned)pos) {
      for (ztmp = pos; ztmp < max_len-1; ztmp++) s[ztmp] = s[ztmp+1];
      s[ztmp] = '\0';
    }
    break;
  case SDLK_BACKSPACE:
    if (pos > 0) {
      if (pos <= max_len) {
        for (ztmp = pos-1; ztmp < max_len-1; ztmp++) s[ztmp] = s[ztmp+1];
        s[ztmp] = '\0';
      }
      pos--;
    }
    break;
  default:
    if (pos >= max_len || (inpc < ' ')) break;
    if (allowed) {
      if (!strchr(allowed, inpc)) {
        if (strchr(allowed, toupper(inpc))) inpc = toupper(inpc);
        else
          if (strchr(allowed, tolower(inpc))) inpc = tolower(inpc);
          else break;
      }
    } else {
      if (inpc < ' ' || inpc > 'z') break;
    }
    if ((strlen(s) >= (unsigned)pos) &&
        (strlen(s) < (unsigned)max_len)) {
      for (ztmp = max_len-1; ztmp >= pos; ztmp--) s[ztmp+1] = s[ztmp];
      s[pos] = inpc;
      s[max_len] = '\0';
      pos++;
    }
    break;
  }
  if (ende) {
    if (!restore_origs) strcpy(origs, s);
    s[0] = 0;
    copy_origs = true;
  } else {
    copy_origs = false;
  }
  return ende;
}


Menuoption::Menuoption(const char *nname,menuopt_callback_proc pr,
                       SDLKey nquickkey, menuoptflags nflags, int nstate) :
                       callback(pr), quickkey(nquickkey), flags(nflags), state(nstate) {
  if(nname) {
    name=nname; 
  } else if(!nname && pr) {
    const char *nm=(*pr)(NULL); // get name from callback
    if(nm) { 
      name=nm;
    } else {
      name="";
    }
  } else {
    name="";
  }
}

void draw_input_box(int x, int y, int len, int cursor, char *txt)
{
  static int col_r = 0, col_g = 200, col_b = 120;
  int nlen = len, slen = len;
  int arrows = 0;

  if ((len+3)*FONTMAXWID > SCREENWID)
    nlen = (SCREENWID / FONTMAXWID) - 3;

  if (x < 0) x = (SCREENWID / 2) - nlen * (FONTMAXWID / 2);
  if (x < 0) x = 0;
  if (y < 0) y = (SCREENHEI / 2) - (FONTHEI / 2);

  scr_putbar(x, y, nlen * FONTMAXWID, FONTHEI, 0, 0, 0, (config.use_alpha_darkening())?128:255);

  if (scr_textlength(txt) >= nlen*FONTMAXWID) {
    while ((cursor >= 0) &&
           (scr_textlength(txt, cursor+(nlen/2)) >= (nlen)*FONTMAXWID)) {
      cursor--;
      txt++;
      arrows = 1;
    }
  }
  if (scr_textlength(txt) >= nlen*FONTMAXWID) {
    arrows |= 2;
    while ((slen > 0) && (scr_textlength(txt, slen) >= nlen*FONTMAXWID)) slen--;
  }

  scr_writetext(x+1,y, txt, slen);

  if ((input_box_cursor_state & 4) && (cursor >= 0))
    scr_putbar(x + scr_textlength(txt, cursor) + 1, y, FONTMINWID, FONTHEI,
               col_r, col_g, col_b, (config.use_alpha_darkening())?128:255);
  scr_putrect(x,y, nlen * FONTMAXWID, FONTHEI, col_r, col_g, col_b, 255);

  if ((arrows & 1)) scr_writetext(x-FONTMAXWID,y, "\x08"); //fontptrright
  if ((arrows & 2)) scr_writetext(x+(nlen*FONTMAXWID),y, "\x06"); //fontptrleft

  input_box_cursor_state++;

  scr_color_ramp(&col_r, &col_g, &col_b);
}

