// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef HMM_INC
#define HMM_INC

#include "Distribution.h"

namespace Torch {

/** This class implements a Hidden Markov Model distribution. In fact,
    it also implements the Input Output version (IOHMM). It can be trained
    either by EM, Viterbi, or Gradient Descent.

    Note that this kind of HMM always contain one initial state and
    one final state. Both are non-emitting.

    @author Samy Bengio (bengio@idiap.ch)
*/
class HMM : public Distribution
{
  public:
    /// keep the training data to compute the size of the longest sequence
    SeqDataSet* data;

    /** The number of states of the HMM.
        the first model is the initial state,
        the last model is the final (absorbing) state,
        (neither of them are emitting).
        hence, n_states > 2
    */
    int n_states;

    /// a prior on the transition probabilities
    real prior_transitions;

    /// keep the emission distributions
    Distribution** states;

    /** in the case where the emission distribution share their
        parameters, #unique_states# contains the prototypes which
        will be used for updates (otherwise, it is a pointer to
        #states#). Note that if given, and as for #states#,
        #unique_states# contains an empty state for positions
        0 (initial state) and #n_states# - 1 (final state)
    */
    Distribution** unique_states;

    /// if #unique_states# is given, n_unique_states is its size
    int n_unique_states;

    /// the initial transitions between states are kept as a matrix
    real** transitions;
    /// in fact, we keep the transitions in log
    real** log_transitions;
    /// the derivative of the log transitions for gradient descent
    real** dlog_transitions;
    /// the accumulators of the transitions for EM
    real** transitions_acc;
    
    /// accumulator used in the forward phase to compute log likelihood 
    real** log_alpha;
    /// accumulator used in the backward phase to compute log likelihood 
    real** log_beta;
    /// for each state, for each time step, keep the best predecessor 
    int** arg_viterbi;
    /// for each time step, keep the best state 
    int* viterbi_sequence;

    /// keep for each time step and each model its emission log probability
    real** log_probabilities_s;

    HMM(int n_states_, Distribution **states_, real prior_transitions_,SeqDataSet* data_,real** transitions, Distribution** unique_states_ = NULL, int n_unique_states_ = 0);

    virtual void reset();
    virtual int numberOfParams();
    virtual void allocateMemory();
    virtual void freeMemory();
    virtual void loadFILE(FILE *file);
    virtual void saveFILE(FILE *file);

    /// this method can be used for debugging purpose to see the transitions
    virtual void printTransitions(bool real_values=false,bool transitions_only=false);

    /// computes the log_alpha during forward phase of EM
    virtual void logAlpha(SeqExample* ex);
    /// computes the log_beta during backward phase of EM
    virtual void logBeta(SeqExample* ex);
    /// computes the log_viterbi during forward phase of Viterbi
    virtual void logViterbi(SeqExample* ex);

    /// this method returns the state sequence associated to the input
    virtual void decode(List* input);

    /** computes for each state and each time step of the sequence #inputs#
        its associated emission probability.
    */
    virtual void logProbabilities(List *inputs);
   
    virtual real logProbability(List *inputs);
    virtual real viterbiLogProbability(List *inputs);

    virtual void  iterInitialize();
    virtual void  eMIterInitialize();
    virtual void  eMSequenceInitialize(List* inputs);
    virtual void  sequenceInitialize(List* inputs);
    virtual void  eMAccPosteriors(List *inputs, real log_posterior);
    virtual void  viterbiAccPosteriors(List *inputs, real log_posterior);
    virtual void  eMUpdate();

    virtual void backward(List *inputs, real *alpha);
    virtual void viterbiBackward(List *inputs, real *alpha);

    virtual ~HMM();
};


}

#endif
