// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef TRAINER_INC
#define TRAINER_INC

#include "Object.h"
#include "Machine.h"
#include "DataSet.h"
#include "List.h"

namespace Torch {

// Methodes utiles

/** Make a table of measurers from a #List#.
    
    Given a #List# of #measurers#,
    and, if you want, a #train# #DataSet# (else NULL)
    \begin{itemize}
      \item Returns all datasets associated to the measurers in #datas#.
        For i != j, (*datas)[i] != (*datas)[j].
        Moreover, if #train# != NULL, (*datas)[0] = #train#.
      \item Returns the list of measurers associated to (*datas)[i] in (*mes)[i].
      \item Returns the number of measureurs associated to (*datas)[i] in (*n_mes)[i].
      \item Returns in *n_datas the number of datasets in *datas.
    \end{itemize}
    
    (Memory allocations are made by the fonction: use
    #deleteExtractedMeasurers()# to free memory)

    @see deleteExtractedMeasurers
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void extractMeasurers(List *measurers, DataSet *train, DataSet ***datas, Measurer ****mes, int **n_mes, int *n_datas);

/** Free memory allocations did by #extractMeasurers()#.
  
    @see extractMeasurers
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void deleteExtractedMeasurers(DataSet **datas, Measurer ***mes, int *n_mes, int n_datas);

/** Trainer.
  
    A trainer takes a #Machine# and a #DataSet#,
    and is able to train this machine on this dataset.
*/
class Trainer : public Object
{
  public:
    DataSet *data;
    Machine *machine;

    //-----

    ///
    Trainer(Machine *machine_, DataSet *data_);

    //-----

    /** Train the machine.
        The Trainer has to call the measurers
        when it want.
    */
    virtual void train(List *measurers) = 0;

    /** Test the machine.
        This method call all the measurers,
        for all the examples of their associated
        dataset.
        It's already written...
    */
    virtual void test(List *measurers);

    /** Test on one example.
        It supposes that all the measurers
        have the same dataset, and call the
        measurers for the example #t# of this
        dataset.
    */
    virtual void testExample(List *measurers, int t);

    /** K-fold cross-validation.
        Do a K-fold cross-validation on #data#.
        \begin{itemize}
          \item #k_fold# is the number of subsets that
          it will make on #data#.
          \item #train_measurers# are the measurers called
          during the train phase. (NULL if nothing).
          \item #test_measurers# are the measurers called
          during the test phase.
          \item #cross_valid_measurers# are the measurers
          called after each iteration of the cross-validation.
          (these last measurers should'nt use #measureEx()#) 
        \end{itemize}
    */
    virtual void crossValidate(int k_fold, List *train_measurers, List *test_measurers, List *cross_valid_measurers=NULL);

    /// Load the parameters of the machine and the dataset
    virtual void loadFILE(FILE *file);

    /// Save the parameters of the machine and the dataset
    virtual void saveFILE(FILE *file);

    //-----

    virtual ~Trainer();
};


}

#endif
