// cpu_inst.c - TMS9900 Series Instruction Functions
//
// Copyright (c) 2001, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "ti99/defs.h"

// ******************** Opernad Decoder *******************

inline uint16 ti99_GetOperand(register TI99_CPU *ti99cpu, 
	uint16 op, uint16 len)
{
	register uint16 reg  = (op & 0xF) << 1;
	register uint16 type = (op >> 4) & 3;
	register uint16 ea;

	switch (type) {
		case 0: // Rn   - Register
			return WP + reg;

		case 1: // *Rn  - Indirect Register
			return ReadW(WP + reg);

		case 2: // @nn     - Symbolic or Indexed Register
			     // @nn(Rn) - Indexed Register (Rn > 0)
			ea = ReadW(PC) + ((reg > 0) ? ReadW(WP + reg) : 0);
			INC_PC;
			return ea;

		case 3: // *Rn+ - Indirect Auto Increment
			ea = ReadW(WP + reg);
			WriteW(WP + reg, ea + len);
			return ea;

#ifdef DEBUG
		default:
			dbg_Printf("%s: Unknown register type: %d\n",
				ti99cpu->Unit.keyName, (reg >> 4));
#endif /* DEBUG */
	}
	return 0;
}

// Format 1 (Arthimetic Instructions)
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// | Opcode |B |  Td |   Dest    |  Ts |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT1(len) \
	src = ti99_GetOperand(TI99, IR, len); \
	dst = ti99_GetOperand(TI99, IR >> 6, len);

// Format 2 (Jump/CRU Instructions)
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |       Opcode          |     Displacement      |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15


// Format 3
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |      Opcode     |   Dest    |  Ts |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT3 \
	src = ti99_GetOperand(TI99, IR, WSIZE); \
	dst = WP + (((IR >> 6) & 0xF) << 1);

// Format 4 - CRU instructions
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |      Opcode     |   Count   |  Ts |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT4 \
	src = ti99_GetOperand(TI99, IR, WSIZE); \
	cnt = (IR >> 6) & 0xF;

// Format 5 - Shift instructions
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |         Opcode        |   Count   |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT5 \
	src = WP + ((IR & 0xF) << 1);        \
	if ((cnt = (IR >> 4) & 0xF) == 0) {  \
		if ((cnt = ReadW(R0)) == 0)       \
			cnt = 16;                      \
	}

// Format 6 (One-register Instructions)
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |             Opcode          |  Ts |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT6 \
	src = ti99_GetOperand(TI99, IR, WSIZE);

// Format 8 (One-register Instructions)
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |             Opcode                |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
// |                  Immediate                    |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT8 FORMAT8_REG FORMAT8_IMM

#define FORMAT8_REG \
	src = WP + ((IR & 0xF) << 1);

#define FORMAT8_IMM \
	imm = ReadW(PC); INC_PC;

// Format 9 (XOP/Multiply/Divide Instructions)
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |      Opcode     |   Dest    |  Ts |   Src     |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define FORMAT9 \
	src = ti99_GetOperand(TI99, IR, WSIZE); \
	dst = WP + (((IR >> 6) & 0xF) << 1);

#define FORMAT9_XOP \
	ea  = ti99_GetOperand(TI99, IR, WSIZE); \
	xop = ((IR >> 6) & 0xF) << 1;

// ********************* Instructions *********************

// A    Add
INSFUNC(ti99, A)
{
	register uint16 src, dst;
	register int16  dsrc, ddst, res;

	// Decode operands.
	FORMAT1(WSIZE);

	// Do a math for add word.
	dsrc = ReadW(src);
	ddst = ReadW(dst);
	WriteW(dst, res = dsrc + ddst);

	// Update status register.
	CC_ADD_W(res, dsrc, ddst);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("A: (>%04X) %04X + (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// AB   Add Byte
INSFUNC(ti99, AB)
{
	register uint16 src, dst;
	register int8   dsrc, ddst, res;

	// Decode operands.
	FORMAT1(BSIZE);

	// Do a math for add byte.
	dsrc = ReadB(src);
	ddst = ReadB(dst);
	WriteB(dst, res = dsrc + ddst);

	// Update status register.
	CC_ADD_B(res, dsrc, ddst);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("AB: (>%04X) %02X + (>%04X) %02X => %02X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// ABS  Aboslute Value
INSFUNC(ti99, ABS)
{
	register uint16 src;
	register int16  dsrc, res;

	// Decode an operand.
	FORMAT6;

	// Do an absolute value of number.
	dsrc = ReadW(src);
	res = (dsrc < 0) ? (~dsrc + 1) : dsrc;
	WriteW(src, res);

	// Update status register
//	CC_IIIIPP(res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("ABS: (>%04X) %04X => %04X\n", src, dsrc, res);
#endif /* DEBUG */
}

// AI   Add Immediate
INSFUNC(ti99, AI)
{
	register uint16 src;
	register int16  imm, dsrc, res;

	// Decode operands.
	FORMAT8;

	// Do a math for add.
	dsrc = ReadW(src);
	WriteW(src, res = dsrc + imm);

	// Update status register.
	CC_ADD_B(res, dsrc, imm);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("AI: (>%04X) %04X + %04X => %04X\n",
			src, dsrc, imm, res);
#endif /* DEBUG */
}

// ANDI AND Immediate
INSFUNC(ti99, ANDI)
{
	register uint16 src,  imm;
	register uint16 dsrc, res;

	// Decode operands.
	FORMAT8;

	// Do an AND gate.
	dsrc = ReadW(src);
	res  = dsrc & imm;
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("ANDI: (>%04X) %04X & %04X => %04X\n",
			src, dsrc, imm, res);
#endif /* DEBUG */
}

// B    Branch
INSFUNC(ti99, B)
{
	register uint16 src;

	FORMAT6;
	PC = src;
}

// BL   Branch and Link
INSFUNC(ti99, BL)
{
	register uint16 src;

	FORMAT6;
	WriteW(R11, PC);
	PC = src;
}

// BLWP Branch and Load Workplace Pointer
INSFUNC(ti99, BLWP)
{
	register uint16 src;
	register uint16 oldWP;

	FORMAT6;
	oldWP = WP;
	WP = ReadW(src);
	WriteW(R13, oldWP);
	WriteW(R14, PC);
	WriteW(R15, ST);
	PC = ReadW(src+2);
}

// C    Compare
INSFUNC(ti99, C)
{
	register uint16 src, dst;
	register int16  dsrc, ddst;

	// Decode operands
	FORMAT1(WSIZE);

	// Do a comparsion between two values.
	dsrc = ReadW(src);
	ddst = ReadW(dst);
	CC_CMP_W(dsrc, ddst);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("C: Compare (>%04X) %04X with (>%04X) %04X\n",
			src, dsrc, dst, ddst);
#endif /* DEBUG */
}

// CB   Compare Byte
INSFUNC(ti99, CB)
{
	register uint16 src, dst;
	register int8   dsrc, ddst;

	// Decode operands.
	FORMAT1(BSIZE);

	// Do a comparsion between two values.
	dsrc = ReadB(src);
	ddst = ReadB(dst);
	CC_CMP_B(dsrc, ddst);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("CB: Compare (>%04X) %02X with (>%04X) %02X\n",
			src, dsrc, dst, ddst);
#endif /* DEBUG */
}

// CI   Compare Immediate
INSFUNC(ti99, CI)
{
	register uint16 src, dsrc, imm;

	// Decode operands.
	FORMAT8;

	// Do a comparsions between two values.
	dsrc = ReadW(src);
	CC_CMP_W(dsrc, imm);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("CI: Compare (>%04X) %04X with %04X\n",
			src, dsrc, imm);
#endif /* DEBUG */
}

// CKOF Clock Off
INSFUNC(ti99, CKOF)
{
}

// CKON Clock On
INSFUNC(ti99, CKON)
{
}

// CLR  Clear
INSFUNC(ti99, CLR)
{
	register uint16 src;

	// Decode operands and clear content
	// at its source address.
	FORMAT6; // Decode Operands.
	WriteW(src, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("CLR: Clear >%04X\n", src);
#endif /* DEBUG */
}

// COC  Compare Ones Corresponding
INSFUNC(ti99, COC)
{
	register uint16 src, dst;
	register uint16 dsrc, ddst, res;

	// Decode operands.
	FORMAT3;

	dsrc = ReadW(src);
	ddst = ReadW(dst);
	res = dsrc & ddst;

	// Update status register.
	if (res == dsrc)  CC |= CC_EQ;
	else              CC &= ~CC_EQ;

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("COC: (>%04X) %04X & (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// CZC  Compare Ones Corresponding
INSFUNC(ti99, CZC)
{
	register uint16 src, dst;
	register uint16 dsrc, ddst, res;

	// Decode operands.
	FORMAT3;

	dsrc = ReadW(src);
	ddst = ReadW(dst);
	res = dsrc & ddst;

	// Update status register.
	if (res == 0)  CC |= CC_EQ;
	else           CC &= ~CC_EQ;

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("CZC: (>%04X) %04X & (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// DEC  Decrement
INSFUNC(ti99, DEC)
{
	register uint16 src;
	register int16  dsrc, res;

	FORMAT6;
	dsrc = ReadW(src);
	res  = dsrc - 1;
	WriteW(src, res);

	// Update status register.
	CC_SUB_W(res, dsrc, 1);
}

// DECT Decrement by Two
INSFUNC(ti99, DECT)
{
	register uint16 src;
	register int16  dsrc, res;

	FORMAT6;
	dsrc = ReadW(src);
	res  = dsrc - 2;
	WriteW(src, res);

	// Update status register.
	CC_SUB_W(res, dsrc, 2);
}

// DIV  Divide
INSFUNC(ti99, DIV)
{
	register uint16 src, dst;
	register uint32 dvd, dvr;
	register uint16 quo, rem;

	// Decode operands
	FORMAT9;

	// Do a math for 32-bit divide.
	dvr = ReadW(src);
	dvd = ReadW(dst);
	if (dvr > dvd) {
		dvd = (dvd << 16) | ReadW(dst+2);
		quo = dvd / dvr;
		rem = dvd % dvr;
		WriteW(dst,   quo);
		WriteW(dst+2, rem);

		// Clear overflow flag
		CC &= ~CC_OV;

#ifdef DEBUG
		if (dbg_Check(DBG_DATA))
			dbg_Printf("DIV: %08X / %04X => %04X R %04X\n",
				dvd, dvr, quo, rem);
#endif /* DEBUG */
	} else {
		// Set overflow flag
		CC |= CC_OV;
	}
}

// IDLE Computer Idle
INSFUNC(ti99, IDLE)
{
}

// INC  Increment
INSFUNC(ti99, INC)
{
	register uint16 src;
	register int16  dsrc, res;

	FORMAT6;
	dsrc = ReadW(src);
	res  = dsrc + 1;
	WriteW(src, res);

	// Update status register.
	CC_ADD_W(res, dsrc, 1);
}

// INCT Increment by Two
INSFUNC(ti99, INCT)
{
	register uint16 src;
	register int16  dsrc, res;

	FORMAT6;
	dsrc = ReadW(src);
	res  = dsrc + 2;
	WriteW(src, res);

	// Update status register.
	CC_ADD_W(res, dsrc, 2);
}

// INV  Invert
INSFUNC(ti99, INV)
{
	register uint16 src, dsrc, res;

	// Decode an operand.
	FORMAT6;

	// Invert data.
	dsrc = ReadW(src);
	res  = ~dsrc;
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("INV: (>%04X) %04X => %04X\n", src, dsrc, res);
#endif /* DEBUG */
}

// JEQ  Jump if Equal
INSFUNC(ti99, JEQ)
{
	if (CC & CC_EQ)
		JUMP;
}

// JGT  Jump if Greater Than
INSFUNC(ti99, JGT)
{
	if ((CC & (CC_AGT|CC_EQ)) == CC_AGT)
		JUMP;
}

// JH   Jump if High
INSFUNC(ti99, JH)
{
	if ((CC & (CC_LGT|CC_EQ)) == CC_LGT)
		JUMP;
}

// JHE  Jump if High or Equal
INSFUNC(ti99, JHE)
{
	if (CC & (CC_LGT|CC_EQ))
		JUMP;
}

// JL   Jump if Low
INSFUNC(ti99, JL)
{
	if ((CC & (CC_LGT|CC_EQ)) == 0)
		JUMP;
}

// JLE  Jump if Low or Equal
INSFUNC(ti99, JLE)
{
	if ((CC & (CC_LGT|CC_EQ)) == CC_EQ)
		JUMP;
}

// JLT  Jump if Less Than
INSFUNC(ti99, JLT)
{
	if ((CC & (CC_AGT|CC_EQ)) == 0)
		JUMP;
}

// JMP  Jump unconditionally
INSFUNC(ti99, JMP)
{
	JUMP;
}

// JNC  Jump if No Carry
INSFUNC(ti99, JNC)
{
	if ((CC & CC_C) == 0)
		JUMP;
}

// JNE  Jump if Not Equal
INSFUNC(ti99, JNE)
{
	if ((CC & CC_EQ) == 0)
		JUMP;
}

// JNO  Jump if No Overflow
INSFUNC(ti99, JNO)
{
	if ((CC & CC_OV) == 0)
		JUMP;
}

// JOC  Jump On Carry
INSFUNC(ti99, JOC)
{
	if (CC & CC_C)
		JUMP;
}

// JOP  Jump if Odd Parity
INSFUNC(ti99, JOP)
{
	if (CC & CC_OP)
		JUMP;
}

// LDCR Load Communication Register
INSFUNC(ti99, LDCR)
{
}

// LI   Load Immediate
INSFUNC(ti99, LI)
{
	register uint16 src, imm;

	FORMAT8;
	WriteW(src, imm);

	// Update status register
	CC_IIIPPP_W((int16)imm, 0);
}

// LIMI Load Interrupt Mask Immediate
INSFUNC(ti99, LIMI)
{
	// Format 8 - Decode Operand
	// Get an immediate.
	IPL = (1u << (ReadW(PC) & 0xF)) - 1;
	INC_PC;

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("LIMI: Set IPL to %d\n", IPL);
#endif /* DEBUG */
}

// LREX Load or Restart Execution
INSFUNC(ti99, LREX)
{
}

// LWPI Load Workplace Pointer Immediate
INSFUNC(ti99, LWPI)
{
	// Get an immediate and load
	// a new workplace register.
	WP = ReadW(PC); INC_PC;
}

// MOV  Move
INSFUNC(ti99, MOV)
{
	register int16 src, dst, data;

	// Decode operands.
	FORMAT1(WSIZE);

	// Move a value.
	WriteW(dst, data = ReadW(src));

	// Update status register.
	CC_IIIPPP_W(data, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("MOV: Move >%04X from >%04X to >%04X\n",
			(uint16)data, (uint16)src, (uint16)dst);
#endif /* DEBUG */
}

// MOVB Move Byte
INSFUNC(ti99, MOVB)
{
	register int16 src, dst;
	register int8  data;

	// Decode operands.
	FORMAT1(BSIZE);

	// Move a value.
	WriteB(dst, data = ReadB(src));

	// Update status register.
	CC_IIIPPP_B(data, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("MOVB: Move >%02X from >%04X to >%04X\n",
			(uint8)data, (uint16)src, (uint16)dst);
#endif /* DEBUG */
}

// MPY  Multiply
INSFUNC(ti99, MPY)
{
	register uint16 src, dst;
	register uint32 mpc, mpr, prod;

	// Decode operands
	FORMAT9;

	// Do a math for 32-bit mutliply.
	mpr = ReadW(src);
	mpc = ReadW(dst);
	prod = mpc * mpr;
	WriteW(dst,   (uint16)(prod >> 16));
	WriteW(dst+2, (uint16)prod);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("MPY: %04X * %04X => %08X\n", mpc, mpr, prod);
#endif /* DEBUG */
}

// NEG  Negate
INSFUNC(ti99, NEG)
{
	register uint16 src;
	register int16  dsrc, res;

	FORMAT6;

	dsrc = ReadW(src);
	res = -dsrc;
	WriteW(src, res);

	// Update status register
}

// ORI  OR Immediate
INSFUNC(ti99, ORI)
{
	register uint16 src,  imm;
	register uint16 dsrc, res;

	// Decode operands.
	FORMAT8;

	// Do an OR gate.
	dsrc = ReadW(src);
	res = dsrc | imm;
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("ORI: (%>04X) %04X | %04X => %04X\n",
			src, dsrc, imm, res);
#endif /* DEBUG */
}

// RSET Reset
INSFUNC(ti99, RSET)
{
}

// RTWP Return Workplace Pointer
INSFUNC(ti99, RTWP)
{
	// Return back to caller.
	ST = ReadW(R15);
	PC = ReadW(R14);
	WP = ReadW(R13);
}

// S    Subtract
INSFUNC(ti99, S)
{
	register uint16 src, dst;
	register int16  dsrc, ddst, res;

	// Decode operands.
	FORMAT1(WSIZE);

	// Do a math for subtract word.
	dsrc = ReadW(src);
	ddst = ReadW(dst);
	WriteW(dst, res = dsrc - ddst);

	// Update status register.
	CC_SUB_W(res, dsrc, ddst);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("S: (>%04X) %04X - (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// SB   Subtract Byte
INSFUNC(ti99, SB)
{
	register uint16 src, dst;
	register int8   dsrc, ddst, res;

	// Decode operands.
	FORMAT1(BSIZE);

	// Do a math for subtract byte.
	dsrc = ReadB(src);
	ddst = ReadB(dst);
	WriteB(dst, res = dsrc - ddst);

	// Update status register.
	CC_SUB_B(res, dsrc, ddst);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SB: (>%04X) %02X - (>%04X) %02X => %02X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// SBO  Set Bit to One
INSFUNC(ti99, SBO)
{
}

// SBZ  Set Bit to Zero
INSFUNC(ti99, SBZ)
{
}

// SETO Set to Ones
INSFUNC(ti99, SETO)
{
	register uint16 src;

	// Decode operands and set ones to
	// content at its source address.
	FORMAT6;
	WriteW(src, WMASK);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SETO: Set Ones to >%04X\n", src);
#endif /* DEBUG */
}

// SLA  Shift Left Arithmetic
INSFUNC(ti99, SLA)
{
	register uint16 src,  cnt;
	register int16  dsrc, res;

	// Decode operands
	FORMAT5;

	// Shift Left
	dsrc = ReadW(src);
	res = dsrc << cnt;
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W(res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SLA: (>%04X) %04X << %d => %04X\n",
			src, dsrc, cnt, res);
#endif /* DEBUG */
}

// SOC  Set Ones Corresponding
INSFUNC(ti99, SOC)
{
	register uint16 src,  dst;
	register uint16 dsrc, ddst, res;

	// Decode operands.
	FORMAT1(WSIZE);

	// Set ones corresponding.
	dsrc = ReadW(src);
	ddst = ReadW(dst);
	res  = ddst | dsrc;
	WriteW(dst, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SOC: (>%04X) %04X | (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// SOCB Set Ones Corresponding Byte
INSFUNC(ti99, SOCB) 
{
	register uint16 src,  dst;
	register uint8  dsrc, ddst, res;

	// Decode operands.
	FORMAT1(BSIZE);

	// Set ones corresponding.
	dsrc = ReadB(src);
	ddst = ReadB(dst);
	res  = ddst | dsrc;
	WriteB(dst, res);

	// Update status register.
	CC_IIIPPP_B((int8)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SOCB: (>%04X) %02X | (>%04X) %02X => %02X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// SRA  Shift Right Arithmetic
INSFUNC(ti99, SRA)
{
	register uint16 src,  cnt;
	register int16  dsrc, res;

	// Decode operands
	FORMAT5;

	// Shift Right Arithmetic
	dsrc = ReadW(src);
	res = dsrc >> cnt;
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W(res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SRA: (>%04X) %04X >> %d => %04X\n",
			src, dsrc, cnt, res);
#endif /* DEBUG */
}

// SRC  Shift Right Circular
INSFUNC(ti99, SRC)
{
	register uint16 src,  cnt;
	register uint16 dsrc, res;

	// Decode operands
	FORMAT5;

	// Rotate Right
	dsrc = ReadW(src);
	res = (dsrc >> cnt) | (dsrc << cnt);
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SRC: (>%04X) %04X >> %d => %04X\n",
			src, dsrc, cnt, res);
#endif /* DEBUG */
}

// SRL  Shift Right Logical
INSFUNC(ti99, SRL)
{
	register uint16 src,  cnt;
	register uint16 dsrc, res;

	// Decode operands
	FORMAT5;

	// Shift Right Logical
	dsrc = ReadW(src);
	res = dsrc >> cnt;
	WriteW(src, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SRA: (>%04X) %04X >> %d => %04X\n",
			src, dsrc, cnt, res);
#endif /* DEBUG */
}

// STCR Store Communication Register
INSFUNC(ti99, STCR)
{
}

// STST Store Status Register
INSFUNC(ti99, STST)
{
	register uint16 src;

	// Decode an operand and store ST register.
	FORMAT8_REG;
	WriteW(src, ST | CC);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("STST: Store SR %04X to >%04X\n", ST|CC, src);
#endif /* DEBUG */
}

// STWP Store Workplace Pointer
INSFUNC(ti99, STWP)
{
	register uint16 src;

	// Decode an operand and store WP register.
	FORMAT8_REG;
	WriteW(src, WP);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("STWP: Store WP %04X to >%04X\n", WP, src);
#endif /* DEBUG */
}

// SWPB Swap Bytes
INSFUNC(ti99, SWPB)
{
	register uint16 src,  dst;
	register uint16 dsrc, res;

	// Decode operands.
	FORMAT6;

	// Swap two bytes
	dsrc = ReadW(src);
	res  = (dsrc >> 8) | (dsrc << 8);
	WriteW(src, res);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SWPB: (>%04X) %04X => %04X\n", src, dsrc, res);
#endif /* DEBUG */
}

// SZC  Set Zeros Corresponding
INSFUNC(ti99, SZC)
{
	register uint16 src,  dst;
	register uint16 dsrc, ddst, res;

	// Decode operands.
	FORMAT1(WSIZE);

	// Set Zeros Corresponding.
	dsrc = ReadW(src);
	ddst = ReadW(dst);
	res  = ddst & ~dsrc;
	WriteW(dst, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SZC: (>%04X) ~%04X & (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// SZCB Set Zeros Corresponding Byte
INSFUNC(ti99, SZCB)
{
	register uint16 src,  dst;
	register uint8  dsrc, ddst, res;

	// Decode operands.
	FORMAT1(BSIZE);

	// Set Zeros Corresponding.
	dsrc = ReadB(src);
	ddst = ReadB(dst);
	res  = ddst & ~dsrc;
	WriteB(dst, res);

	// Update status register.
	CC_IIIPPP_B((int8)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("SZCB: (>%04X) ~%02X & (>%04X) %02X => %02X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// TB   Test Bit
INSFUNC(ti99, TB)
{
}

// X    Execute
INSFUNC(ti99, X)
{
}

// XOP  Extended Operation
INSFUNC(ti99, XOP)
{
	register uint16 ea, xop;
	register uint16 oldWP;

	// Decode operands
	FORMAT9_XOP;

	oldWP = WP;
	WP = ReadW(0x0040 + xop);
	WriteW(R11, ea);
	WriteW(R13, oldWP);
	WriteW(R14, PC);
	WriteW(R15, ST);
	PC = ReadW(0x0042 + xop);

	// Enable XOP flag in status register.
	ST |= SR_X;
}

// XOR  Exclusive OR
INSFUNC(ti99, XOR)
{
	register uint16 src,  dst;
	register uint16 dsrc, ddst, res;

	// Decode operands.
	FORMAT3;

	// Exclusive OR (XOR)
	dsrc = ReadW(src);
	ddst = ReadW(dst);
	res = dsrc ^ ddst;
	WriteW(dst, res);

	// Update status register.
	CC_IIIPPP_W((int16)res, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("XOR: (>%04X) %04X ^ (>%04X) %04X => %04X\n",
			src, dsrc, dst, ddst, res);
#endif /* DEBUG */
}

// Illegal Instruction
INSFUNC(ti99, Illegal)
{
#ifdef DEBUG
//	dbg_Printf("%s: Illegal Instruction (%04X) at PC %04X\n",
//		ti99cpu->cpu.keyName, IR, faultPC);
#endif /* DEBUG */
}
