// defs.h - TI990 Emulator Definitions
//
// Copyright (c) 2001, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "emu/defs.h"

#define TI99_OK      EMU_OK
#define TI99_ARG     EMU_ARG
#define TI99_OPENERR EMU_OPENERR

#define TI99_RUN     EMU_RUN
#define TI99_HALT    EMU_HALT

#define TI99_TICK    16666
#define TI99_SECOND  60

#define BSIGN  0x80
#define BMASK  0xFF
#define BSIZE  1

#define WSIGN  0x8000
#define WMASK  0xFFFF
#define WSIZE  2

typedef struct ti99_InstTable TI99_INST;
typedef struct ti99_Processor TI99_CPU;
typedef struct ti99_System    TI99_SYSTEM;

struct ti99_InstTable {
	char   *Name;      // Mnemonic Name
	char   *Descrip;   // Description
	uint16 opCode;     // Opcode
	uint16 opMask;     // Opcode Mask
	uint8  opFormat;   // Opcode Format
	uint32 Flags;      // Flags

	void   (*Execute)(register TI99_CPU *);
};

struct ti99_Processor {
	// Description Information
	char   *devName;    // Device Name
	char   *keyName;    // Type (Key) Name
	char   *emuName;    // Emulator Name
	char   *emuVersion; // Emulator Version

	uint16 wp;      // Workplace Pointer
	uint16 pc;      // Program Counter
	uint16 ir;      // Instruction Register
	uint16 st;      // Status Register
	uint16 cc;      // Condition Code (Part of Status Register)

	uint16 irqMask;    // Interrupt Priority Level
	uint16 irqRequest; // Interrupt Requests

	uint16 faultPC; // faulting PC pointer
	uint32 ips;     // Instructions Per Second
	uint32 State;   // Processor State

	TI99_SYSTEM *System;

	// Opcode Table for execution
//	void   (*Opcodes[65536])(register TI99_CPU *);
	void   (**Opcodes)(register TI99_CPU *);

	// Function Calls
	uint16 (*ReadW)(register TI99_CPU *, uint16);
	void   (*WriteW)(register TI99_CPU *, uint16, uint16);
	uint8  (*ReadB)(register TI99_CPU *, uint16);
	void   (*WriteB)(register TI99_CPU *, uint16, uint8);
	uint16 (*ReadCRU)(register TI99_CPU *, uint16);
	void   (*WriteCRU)(register TI99_CPU *, uint16, uint16);

	void   (*EnableTimer)(register TI99_CPU *);
	void   (*DisableTimer)(register TI99_CPU *);
};

struct ti99_System {
	// Description Information
	char   *devName;    // Device Name
	char   *keyName;    // Type (Key) Name
	char   *emuName;    // Emulator Name
	char   *emuVersion; // Emulator Version

	TI99_CPU *Processor;
};

#define TI99    ti99cpu

#define WP      TI99->wp
#define PC      TI99->pc
#define IR      TI99->ir
#define ST      TI99->st
#define CC      TI99->cc

#define R0      (WP)
#define R1      (WP + 2)
#define R2      (WP + 4)
#define R3      (WP + 6)
#define R4      (WP + 8)
#define R5      (WP + 10)
#define R6      (WP + 12)
#define R7      (WP + 14)
#define R8      (WP + 16)
#define R9      (WP + 18)
#define R10     (WP + 20)
#define R11     (WP + 22)
#define R12     (WP + 24)
#define R13     (WP + 26)
#define R14     (WP + 28)
#define R15     (WP + 30)

#define IPL     TI99->irqMask
#define IRQ     TI99->irqRequest

#define faultPC TI99->faultPC
#define IPS     TI99->ips

// Function Calls
#define ReadW(addr)          TI99->ReadW(TI99, addr)
#define WriteW(addr, data)   TI99->WriteW(TI99, addr, data)
#define ReadB(addr)          TI99->ReadB(TI99, addr)
#define WriteB(addr, data)   TI99->WriteB(TI99, addr, data)
#define ReadCRU(addr)        TI99->ReadCRU(TI99, addr)
#define WriteCRU(addr, data) TI99->WriteCRU(TI99, addr, data)

#define EnableTimer          TI99->EnableTimer(TI99)
#define DisableTimer         TI99->DisableTimer(TI99)

#define INC_PC PC += 2
#define JUMP   PC += ((int8)IR << 1) // Implies Format II field

// Status Register
//
// +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
// |L>|A>|EQ|C |OV|OP|X |              | Interrupt |
// +--+--+--+--^--+--+--+--^--+--+--+--^--+--+--+--+
//  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15

#define SR_L    0x8000    // Logical Greater Than
#define SR_A    0x4000    // Arithmetic Greater Than
#define SR_E    0x2000    // Equal Bit
#define SR_C    0x1000    // Carry Bit
#define SR_V    0x0800    // Overflow Bit
#define SR_P    0x0400    // Odd Parity Bit
#define SR_X    0x0200    // Extended Operation Bit
#define SR_I    0x000F    // Interrupt Mask

#define SR_CC   (SR_L|SR_A|SR_E|SR_C|SR_V|SR_P)

#define CC_LGT SR_L // Logical Greater Than
#define CC_AGT SR_A // Arithmetic Greater Than
#define CC_EQ  SR_E // Equal
#define CC_C   SR_C // Carry
#define CC_OV  SR_V // Overflow
#define CC_OP  SR_P // Odd Parity

// Instruction Function Call Definition
#define INSNAME(cpu, rtn) cpu##_op##rtn
#define INSFUNC(cpu, rtn) \
	void INSNAME(cpu, rtn)(register TI99_CPU *TI99)

// Update Condition Codes

#define CC_IIIZZZ_B(r, c) \
	if (r < c)      CC = CC_LGT;        \
	else if (r > c) CC = CC_LGT|CC_AGT; \
	else            CC = CC_EQ;

#define CC_IIIZZP_W(r, c) \
	CC &= ~(CC_LGT|CC_AGT|CC_EQ|CC_C|CC_OV); \
	if (r < c)      CC |= CC_LGT;            \
	else if (r > c) CC |= CC_LGT|CC_AGT;     \
	else            CC |= CC_EQ;


#define CC_IIIPPP_B(r, c) \
	CC &= ~(CC_LGT|CC_AGT|CC_EQ);        \
	if (r < c)      CC |= CC_LGT;        \
	else if (r > c) CC |= CC_LGT|CC_AGT; \
	else            CC |= CC_EQ;

#define CC_IIIPPP_W(r, c) \
	CC &= ~(CC_LGT|CC_AGT|CC_EQ);        \
	if (r < c)      CC |= CC_LGT;        \
	else if (r > c) CC |= CC_LGT|CC_AGT; \
	else            CC |= CC_EQ;

// Condition codes for compare instructions (C, CB, CI)

#define CC_CMP_B(r, c) \
	CC &= ~(CC_LGT|CC_AGT|CC_EQ);               \
	if (r == c)                 CC |= CC_EQ;    \
	else {                                      \
		if ((int8)r > (int8)c)   CC |= CC_AGT;   \
		if ((uint8)r > (uint8)c) CC |= CC_LGT;   \
	}

#define CC_CMP_W(r, c) \
	CC &= ~(CC_LGT|CC_AGT|CC_EQ);                 \
	if (r == c)                   CC |= CC_EQ;    \
	else {                                        \
		if ((int16)r > (int16)c)   CC |= CC_AGT;   \
		if ((uint16)r > (uint16)c) CC |= CC_LGT;   \
	}

// Condition codes for add instructions

#define V_ADD_B(r, s1, s2) \
	if (((~(s1) ^ (s2)) & ((s1) ^ (r))) & BSIGN) CC |= CC_OV;
#define V_ADD_W(r, s1, s2) \
	if (((~(s1) ^ (s2)) & ((s1) ^ (r))) & WSIGN) CC |= CC_OV;

#define C_ADD_B(r, s1, s2) \
	if ((uint8)(r)  < (uint8)(s2))  CC |= CC_C;
#define C_ADD_W(r, s1, s2) \
	if ((uint16)(r) < (uint16)(s2)) CC |= CC_C;

#define CC_ADD_B(r, s1, s2) \
	CC_IIIZZZ_B(r, 0);       \
	V_ADD_B(r, s1, s2);      \
	C_ADD_B(r, s1, s2);

#define CC_ADD_W(r, s1, s2) \
	CC_IIIZZP_W(r, 0);       \
	V_ADD_W(r, s1, s2);      \
	C_ADD_W(r, s1, s2);

// Condition codes for subtract instructions

#define V_SUB_B(r, s1, s2) \
	if ((((s1) ^ (s2)) & (~(s1) ^ (r))) & BSIGN) CC |= CC_OV;
#define V_SUB_W(r, s1, s2) \
	if ((((s1) ^ (s2)) & (~(s1) ^ (r))) & WSIGN) CC |= CC_OV;

#define C_SUB_B(r, s1, s2) \
	if ((uint8)(s1)  < (uint8)(s2))  CC |= CC_C;
#define C_SUB_W(r, s1, s2) \
	if ((uint16)(s1) < (uint16)(s2)) CC |= CC_C;

#define CC_SUB_B(r, s1, s2) \
	CC_IIIZZZ_B(r, 0);       \
	V_SUB_B(r, s1, s2);      \
	C_SUB_B(r, s1, s2);

#define CC_SUB_W(r, s1, s2) \
	CC_IIIZZP_W(r, 0);       \
	V_SUB_W(r, s1, s2);      \
	C_SUB_W(r, s1, s2);


// External Defintions
#include "ti99/proto.h"
