/*
 * @(#)Destination.h 3.00 21 August 2000
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tse3/ins/Destination.h"

#include "tse3/ins/Instrument.h"

#include <vector>
#include <algorithm>

using namespace TSE3::Ins;

/******************************************************************************
 * DestinationImpl class
 *****************************************************************************/

class TSE3::Ins::DestinationImpl
{
    public:

        DestinationImpl(size_t np) : noPorts(np) {}

        size_t                      noPorts;
        bool                       *_allChannels;
        Instrument                **instruments;
        std::vector<Instrument *>   ilist;

};


/******************************************************************************
 * Destination class
 *****************************************************************************/

Destination::Destination(size_t noPorts)
: pimpl(new DestinationImpl(noPorts))
{
    pimpl->_allChannels = new bool[noPorts];
    pimpl->instruments  = new Instrument*[(noPorts+1)*16];
    for (size_t n = 0; n < noPorts; n++)
    {
        setPort(n, 0);
    }
}


Destination::~Destination()
{
    delete pimpl->_allChannels;
    delete pimpl->instruments;
    delete pimpl;
}


size_t Destination::noPorts() const
{
    return pimpl->noPorts;
}


bool Destination::allChannels(size_t port)
{
    return pimpl->_allChannels[port];
}


Instrument *Destination::port(size_t port)
{
    if (port >= pimpl->noPorts || !pimpl->_allChannels[port])
    {
        return 0;
    }

    return pimpl->instruments[index(port, 0)];
}


void Destination::setPort(size_t port, Instrument *instrument)
{
    if (port < pimpl->noPorts)
    {
        pimpl->_allChannels[port]          = true;
        pimpl->instruments[index(port, 0)] = instrument;
        notify(&DestinationListener::Destination_Altered, 0, port, instrument);
    }
}


Instrument *Destination::channel(size_t channel, size_t port)
{
    if (port >= pimpl->noPorts || channel > 15)
    {
        return 0;
    }

    if (pimpl->_allChannels[port])
    {
        channel = 0;
    }

    return pimpl->instruments[index(port, channel)];
}


void Destination::setChannel(size_t channel, size_t port,
                             Instrument *instrument)
{
    if (port < pimpl->noPorts && channel < 16)
    {
        if (pimpl->_allChannels[port])
        {
            for (size_t n = 0; n < 16; n++)
            {
                pimpl->instruments[index(port, n)]
                    = pimpl->instruments[index(port, 0)];
                if (n != channel)
                {
                    notify(&DestinationListener::Destination_Altered, n, port,
                           pimpl->instruments[index(port, 0)]);
                }
            }
            pimpl->_allChannels[port] = false;
        }
        pimpl->instruments[index(port, channel)] = instrument;
        notify(&DestinationListener::Destination_Altered, channel, port,
               instrument);
    }
}


/******************************************************************************
 * Destination class: managing the ilist
 *****************************************************************************/

size_t Destination::noInstruments()
{
    return pimpl->ilist.size();
}


Instrument *Destination::instrument(size_t index)
{
    if (index < pimpl->ilist.size())
    {
        return pimpl->ilist[index];
    }
    else
    {
        return 0;
    }
}


Instrument *Destination::instrument(const std::string &title)
{
    std::vector<Instrument*>::iterator i = pimpl->ilist.begin();
    while (i != pimpl->ilist.end() && (*i)->title() != title)
    {
        ++i;
    }
    return (i == pimpl->ilist.end()) ? 0 : *i;
}


void Destination::addInstrument(Instrument *instrument)
{
    std::vector<Instrument*>::iterator i = pimpl->ilist.begin();
    while (i != pimpl->ilist.end() && (*i)->title() < instrument->title())
    {
        ++i;
    }

    if (i == pimpl->ilist.end() || instrument->title() != (*i)->title())
    {
        pimpl->ilist.insert(i, instrument);
        notify(&DestinationListener::Destination_InstrumentAdded, instrument);
    };
}


void Destination::removeInstrument(Instrument *instrument)
{
    std::vector<Instrument*>::iterator i
        = find(pimpl->ilist.begin(), pimpl->ilist.end(), instrument);
    if (i == pimpl->ilist.end()) return;

    // Check every destination
    for (size_t port = 0; port < pimpl->noPorts; port++)
    {
        size_t maxc = (pimpl->_allChannels[port]) ? 1 : 16;
        for (size_t channel = 0; channel < maxc; channel++)
        {
            if (pimpl->instruments[index(port, channel)] == instrument)
            {
                pimpl->instruments[index(port, channel)] = 0;
                notify(&DestinationListener::Destination_Altered,
                       channel, port, (Instrument*)0);
            }
        }
    }

    pimpl->ilist.erase(i);
    notify(&DestinationListener::Destination_InstrumentRemoved, instrument);
}

