/*
 *     TreeParser for TTCN-3
 *
 * Copyright (C) 2001 Institute for Telematics
 *
 *    Michael Schmitt <schmitt@itm.mu-luebeck.de>
 *    Roman Koch <rkoch@itm.mu-luebeck.de>
 *
 *    Medical University of Luebeck,
 *    Ratzeburger Allee 160,
 *    23538 Luebeck,
 *    Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: TTCNTreeParser.g,v 1.29 2001/08/07 18:46:19 schmitt Exp $
 *
 * This file is intended to be used with ANTLR 2.7.1 or higher. ANTLR is a parser generator that
 * is based on 'linear approximate' lookahead analysis (a kind of weak LL(k) analysis).
 * ANTLR is free software provided by jGuru.com (MageLang Institute). For further information
 * please refer to http://www.antlr.org. My special thanks go to Terence Parr from jGuru.com.
 *
 * If you make use of this TTCN-3 grammar specification, please send a little email in which
 * you tell us what you are using it for and whether you consider it useful. Any grammar
 * corrections and extensions (e.g. for semantics checks and abstract syntax tree transformation)
 * are welcome. For the latest release of all TTCN-3 grammar files please have a look at
 * http://www.itm.mu-luebeck.de/english/research/specification/ttcn3parser. We also invite you to
 * visit the WWW pages of the Institute for Telematics at http://www.itm.mu-luebeck.de.
 *
 *
 * This grammar is based on the final draft of the TTCN-3 standard (version 1.0.11 of May 2001)
 *
 */

header "pre_include_hpp"
{
  #include "EnvAST.hpp"

  // $Id: TTCNTreeParser.g,v 1.29 2001/08/07 18:46:19 schmitt Exp $
}


header "post_include_cpp"
{
  #include <string>
  #include "InfoAST.hpp"
  #include "ScopeAST.hpp"
  #include "TTCNSymbolParser.hpp"

  using std::string;

  static string   ttcnTreeParserVersion = "$Id: TTCNTreeParser.g,v 1.29 2001/08/07 18:46:19 schmitt Exp $";

  #define ENTER_ENV(ast)   { currentEnv = static_cast<antlr::RefScopeAST>( ast ) -> getEnv(); }
  #define LEAVE_ENV        { currentEnv = currentEnv -> getParentEnv(); }
}


options
{
  language  = "Cpp";
  namespace = "TTCNGrammar";
  namespaceStd = "std";
  namespaceAntlr = "antlr";
  genHashLines = true;
}


class TTCNTreeParser extends TreeParser;

//
// Node types not introduced in the parser because of nondeterminisms:
//
// ExtendedFieldReference
// FormalPortPar  (-> SomeFormalPar)
// FormalValuePar  (-> SomeFormalPar)
// FunctionActualParList  (-> SomeActualParList)
// FunctionInstance   (-> SomeInstance)
// FunctionRef
// GlobalModuleId
// InLineTemplate
// TeststepInstance   (-> SomeInstance)
// TeststepRef
// Port
// ReferencedType
// ReferencedValue
// RunningOp  (-> SomeRunningOp)
// RunningTimerOp  (-> SomeRunningOp)
// Signature
// SingleValueOrAttrib
// StartStatement  (-> SomeStartStatement)
// StartTimerStatement  (-> SomeStartStatement)
// StopStatement  (-> SomeStopStatement)
// StopTimerStatement  (-> SomeStopStatement)
// TemplateActualParList  (-> SomeActualParList)
// TemplateRefWithParList   (-> SomeInstance (partially))
// TimerRef
// TypeReference
// ValueReference
// VariableRef

options
{
  k = 1;
  importVocab = TTCNParser;
  defaultErrorHandler = true;
  buildAST = false;
}


{
 public:

   antlr::RefEnvAST getEnvAST() { return currentEnv; }

 private:

   antlr::RefEnvAST currentEnv;
}


ttcn3Document :

  #( root:TTCN3Document
     { ENTER_ENV( #root ) }
     ( ttcn3Module )+
     { LEAVE_ENV } )

;


/*****  SECTION A.1.6.1 - TTCN Module *****/

ttcn3Module :

  #( root:TTCN3Module
     { ENTER_ENV( #root ) }
     ttcn3ModuleId ( moduleParList )? ( moduleDefinitionsPart )? ( moduleControlPart )?
     ( withStatement )?
     { LEAVE_ENV } )

;


ttcn3ModuleId :

  #( TTCN3ModuleId
     /*module*/ Identifier ( definitiveIdentifier )? )

;


definitiveIdentifier :

  #( DefinitiveIdentifier
     ( definitiveObjIdComponent )+ )

;


definitiveObjIdComponent :

    nameForm
  | definitiveNumberForm
  | definitiveNameAndNumberForm

;


definitiveNumberForm :

  Number

;


definitiveNameAndNumberForm :

  #( DefinitiveNameAndNumberForm
     Identifier definitiveNumberForm )

;


moduleParList :

  #( ModuleParList
     ( modulePar )+ )

;


modulePar :

  #( ModulePar
     /*modulePar*/ type /*modulePar*/ Identifier ( constantExpression )? )

;


/*****  SECTION A.1.6.2 - Module Definitions Part *****/

moduleDefinitionsPart :

  #( ModuleDefinitionsPart
     ( moduleDefinition )+ )

;


moduleDefinition :

  #( ModuleDefinition
     (   typeDef
       | constDef
       | templateDef
       | functionDef
       | signatureDef
       | testcaseDef
       | teststepDef
       | importDef
       | groupDef
       | extFunctionDef
       | extConstDef )
     ( withStatement )? )

;


/*****  SECTION A.1.6.2.1 - Typedef Definitions *****/

typeDef :

    structuredTypeDef
  | subTypeDef

;


structuredTypeDef :

    recordDef
  | unionDef
  | setDef
  | recordOfDef
  | setOfDef
  | enumDef
  | portDef
  | componentDef

;


recordDef :

  #( RecordDef
     structDefBody )

;


structDefBody :   // no root node needed

  (   /*structType*/ Identifier ( structDefFormalParList )?
    | Address )
  ( structFieldDef )*

;


structDefFormalParList :

  #( StructDefFormalParList
     ( structDefFormalPar )+ )

;


structDefFormalPar :

    formalValuePar
  | formalTypePar

;


structFieldDef :

  #( StructFieldDef
     type /*structField*/ Identifier ( arrayDef )? ( subTypeSpec )? ( Optional )? )

;


unionDef :

  #( UnionDef
     (   /*structType*/ Identifier ( structDefFormalParList )?
       | Address )
     ( unionFieldDef )+ )

;


unionFieldDef :

  #( UnionFieldDef
     type /*structField*/ Identifier ( arrayDef )? ( subTypeSpec )? )

;


setDef :

  #( SetDef
     structDefBody )

;


recordOfDef :

  #( RecordOfDef
     ( stringLength )? structOfDefBody )

;


structOfDefBody :   // no root node needed

  type
  (   /*structType*/ Identifier
    | Address )
  ( subTypeSpec )?

;


setOfDef :

  #( SetOfDef
     ( stringLength )? structOfDefBody )

;


enumDef :

  #( EnumDef
     (   /*enumType*/ Identifier
       | Address )
     ( namedValue )+ )

;


namedValue :

  #( NamedValue
     /*namedValue*/ Identifier ( Number )? )

;


subTypeDef :

  #( SubTypeDef
     type
     (   /*subType*/ Identifier
       | Address )
     ( arrayDef )? ( subTypeSpec )? )

;


subTypeSpec :

    allowedValues
  | stringLength

;


allowedValues :

  #( AllowedValues
     ( valueOrRange )+ )

;


valueOrRange :

    integerRangeDef
  | singleConstExpression

;


integerRangeDef :

  #( IntegerRangeDef
     lowerBound upperBound )

;


stringLength :

  #( StringLength
     singleConstExpression ( upperBound )? )

;


portType :

  #( PortType
     ( ( globalModuleId Identifier ) => globalModuleId )? /*portType*/ Identifier )

;


portDef :

  #( PortDef
     /*portType*/ Identifier portDefAttribs )

;


portDefAttribs :

    messageAttribs
  | procedureAttribs
  | mixedAttribs

;


messageAttribs :

  #( MessageAttribs
     ( messageList )+ )

;


messageList :

  #( MessageList
     direction ( All | ( type )+ ) )

;


direction :

    In
  | Out
  | InOut

;


procedureAttribs :

  #( ProcedureAttribs
     ( procedureList )+ )

;


procedureList :

  #( ProcedureList
     direction ( All | ( signature )+ ) )

;


mixedAttribs :

  #( MixedAttribs
     ( mixedList )+ )

;


mixedList :

  #( MixedList
     direction ( All | ( procOrType )+ ) )

;


procOrType :

    ( type ) =>
    type
  | signature

;


componentDef :

  #( ComponentDef
     /*componentType*/ Identifier ( componentElementDef )* )

;


componentType :

  #( ComponentType
     ( ( globalModuleId Identifier ) => globalModuleId )? /*componentType*/ Identifier )

;


componentElementDef :

    portInstance
  | varInstance
  | timerInstance
  | constDef

;


portInstance :

  #( PortInstance
     portType ( portElement )+ )

;


portElement :

  #( PortElement
     /*port*/ Identifier ( arrayDef )? )

;


/*****  SECTION A.1.6.2.2 - Constant Definitions  *****/

constDef :

  #( ConstDef
     type ( singleConstDef )+ )

;


singleConstDef :

  #( SingleConstDef
     /*const*/ Identifier ( arrayDef )? constantExpression )

;


/*****  SECTION A.1.6.2.3 - Template Definitions  *****/

templateDef :

  #( TemplateDef
     baseTemplate ( derivedDef )? templateBody )

;


baseTemplate :

  #( BaseTemplate
     ( ( type ) => type | signature ) /*template*/ Identifier ( templateFormalParList )? )

;


derivedDef :

  #( DerivedDef
     (   ( ( globalModuleId Identifier ) => globalModuleId )? /*template*/ Identifier
       | /*templatePar*/ Identifier ) )

;


templateFormalParList :

  #( TemplateFormalParList
     ( templateFormalPar )+ )

;


templateFormalPar :

    formalValuePar
  | formalTemplatePar

;


templateBody :

    simpleSpec
  | fieldSpecList
  | arrayValueOrAttrib

;


simpleSpec :

  singleValueOrAttrib

;


fieldSpecList :

  #( FieldSpecList
     ( fieldSpec )* )

;


fieldSpec :

  #( FieldSpec
     fieldReference templateBody )

;


fieldReference :

    recordRef
  | arrayOrBitRef
  | parRef

;


recordRef :

  /*structField*/ Identifier

;


parRef :

  signatureParIdentifier

;


signatureParIdentifier :

  /*valuePar*/ Identifier

;


arrayOrBitRef :

  #( ArrayOrBitRef
     singleExpression )

;


singleValueOrAttrib :

    ( templateRefWithParList ) =>
    templateRefWithParList
  | matchingSymbol ( extraMatchingAttributes )?
  | singleExpression ( extraMatchingAttributes )?

;


arrayValueOrAttrib :

  #( ArrayValueOrAttrib
     ( arrayElementSpec )+ )

;


arrayElementSpec :

    NotUsed
  | templateBody

;


matchingSymbol :

    complement
  | Omit
  | AnyValue
  | AnyOrOmit
  | valueList
  | integerRange
  | bitStringMatch
  | hexStringMatch
  | octetStringMatch
  | charStringMatch

;


extraMatchingAttributes :

    lengthMatch
  | IfPresentMatch

;


bitStringMatch :

  BStringMatch

;


hexStringMatch :

  HStringMatch

;


octetStringMatch :

  OStringMatch

;


charStringMatch :

  #( CharStringMatch
     charStringValue )

;


complement :

  #( Complement
     ( singleConstExpression | valueList ) )

;


valueList :

  #( ValueList
     ( singleConstExpression )+ )

;


lengthMatch :

  stringLength

;


integerRange :

  #( IntegerRange
     lowerBound upperBound )

;


lowerBound :

    singleConstExpression
  | MinusInfinity

;


upperBound :

    singleConstExpression
  | Infinity

;


templateInstance :

  inLineTemplate

;


templateRefWithParList :

    #( SomeInstance   // -> TemplateRefWithParList
       ( ( globalModuleId Identifier ) => globalModuleId )? /*template*/ Identifier
       templateActualParList )
  | ( ( globalModuleId Identifier ) => globalModuleId )? /*template*/ Identifier
  | /*templatePar*/ Identifier

;


inLineTemplate :

  ( ( type Identifier ) => type | ( signature Identifier ) => signature )? ( derivedDef )? templateBody

;


templateActualParList :

  #( SomeActualParList   // -> TemplateActualParList
     ( templateActualPar )+ )

;


templateActualPar :

  templateInstance

;


templateOps :

    matchOp
  | valueofOp

;


matchOp :

  #( MatchOp
     expression templateInstance )

;


valueofOp :

  #( ValueofOp
     templateInstance )

;


/*****  SECTION A.1.6.2.4 - Function Definitions  *****/

functionDef :

  #( root:FunctionDef
     { ENTER_ENV( #root ) }
     /*function*/ Identifier ( functionFormalParList )? ( runsOnSpec )? ( returnType )?
     functionBody
     { LEAVE_ENV } )

;


functionFormalParList :

  #( FunctionFormalParList
     ( functionFormalPar )+ )

;


functionFormalPar :

    formalValuePar
  | formalTimerPar
  | formalTemplatePar
  | formalPortPar

;


returnType :

  #( ReturnType
     type )

;


runsOnSpec :

  #( RunsOnSpec
     ( componentType | MTCOp ) )

;


functionBody :

  #( FunctionBody
     ( functionStatementOrDef )* )

;


functionStatementOrDef :

    functionLocalDef
  | functionLocalInst
  | functionStatement

;


functionLocalInst :

    varInstance
  | timerInstance

;


functionLocalDef :

  constDef

;


functionStatement :

    configurationStatements
  | timerStatements
  | communicationStatements
  | basicStatements
  | behaviourStatements
  | verdictStatements
  | sutStatements

;


functionInstance :

  #( SomeInstance   // -> FunctionInstance
     functionRef ( functionActualParList )? )

;


functionRef :

  ( ( globalModuleId Identifier ) => globalModuleId )? /*function*/ Identifier

;


functionActualParList :

  #( SomeActualParList   // -> FunctionActualParList
     ( functionActualPar )+ )

;


functionActualPar :

    ( templateInstance ) =>
    templateInstance
  | timerRef
  | port
  | componentRef

;


/*****  SECTION A.1.6.2.5 - Signature Definitions  *****/

signatureDef :

  #( SignatureDef
     /*signature*/ Identifier ( signatureFormalParList )? ( returnType | Noblock )? ( exceptionSpec )? )

;


signatureFormalParList :

  #( SignatureFormalParList
     ( signatureFormalPar )+ )

;


signatureFormalPar :

  formalValuePar

;


exceptionSpec :

  #( ExceptionSpec
     ( type )+ )

;


signature :

  ( ( globalModuleId Identifier ) => globalModuleId )? /*signature*/ Identifier

;


/*****  SECTION A.1.6.2.6 - Testcase Definitions  *****/

testcaseDef :

  #( root:TestcaseDef
     { ENTER_ENV( #root ) }
     /*testcase*/ Identifier ( testcaseFormalParList )? configSpec functionBody
     { LEAVE_ENV } )

;


testcaseFormalParList :

  #( TestcaseFormalParList
     ( testcaseFormalPar )+ )

;


testcaseFormalPar :

    formalValuePar
  | formalTemplatePar

;


configSpec :

  #( ConfigSpec
     runsOnSpec ( systemSpec )? )

;


systemSpec :

  #( SystemSpec
     componentType )

;


testcaseInstance :

  #( TestcaseInstance
     testcaseRef ( testcaseActualParList )? ( timerValue )? )

;


testcaseRef :

  #( TestcaseRef
     ( ( globalModuleId Identifier ) => globalModuleId )? /*testcase*/ Identifier )

;


testcaseActualParList :

  #( TestcaseActualParList
     ( testcaseActualPar )+ )

;


testcaseActualPar :

  templateInstance

;


/*****  SECTION A.1.6.2.7 - Teststep Definitions  *****/

teststepDef :

  #( TeststepDef
     /*teststep*/ Identifier ( teststepFormalParList )? ( runsOnSpec )? altGuardList )

;


teststepFormalParList :

  functionFormalParList

;


teststepInstance :

  #( SomeInstance   // -> TeststepInstance
     teststepRef ( functionActualParList )? )

;


teststepRef :

  ( ( globalModuleId Identifier ) => globalModuleId )? /*teststep*/ Identifier

;


/*****  SECTION A.1.6.2.8 - Import Definitions  *****/

importDef :

    importAllSpec
  | importGroupSpec
  | importTypeDefSpec
  | importTemplateSpec
  | importConstSpec
  | importTestcaseSpec
  | importTeststepSpec
  | importFunctionSpec
  | importSignatureSpec

;


importAllSpec :

  #( ImportAllSpec
     ( defKeyword )? importFromSpec )

;


importFromSpec :

  #( ImportFromSpec
     moduleId ( Nonrecursive )? )

;


moduleId :

  #( ModuleId
     globalModuleId ( extModuleActualParList )? ( languageSpec )? )

;


languageSpec :

  #( LanguageSpec
     freeText )

;


globalModuleId :

  /*module*/ Identifier ( objectIdentifierValue )?

;


extModuleActualParList :

  #( ExtModuleActualParList
     (   ( parExpressionSpec )+
       | ( notUsedOrModuleParExpression )+ ) )

;


parExpressionSpec :

  #( ParExpressionSpec
     /*extModulePar*/ Identifier /*modulePar*/ expression )

;


notUsedOrModuleParExpression :

    /*modulePar*/ expression
  | NotUsed

;


defKeyword :

    DefType
  | DefConst
  | DefTemplate
  | DefTestcase
  | DefFunction
  | DefSignature
  | DefTeststep

;


importGroupSpec :

  #( ImportGroupSpec
     ( /*group*/ Identifier )+ importFromSpec )

;


importTypeDefSpec :

  #( ImportTypeDefSpec
     ( typeDefIdentifier )+ importFromSpec )

;


typeDefIdentifier :

  /*structType,enumType,portType,componentType,subType*/ Identifier

;


importTemplateSpec :

  #( ImportTemplateSpec
     ( /*template*/ Identifier )+ importFromSpec )

;


importConstSpec :

  #( ImportConstSpec
     ( /*const*/ Identifier )+ importFromSpec )

;


importTestcaseSpec :

  #( ImportTestcaseSpec
     ( /*testcase*/ Identifier )+ importFromSpec )

;


importFunctionSpec :

  #( ImportFunctionSpec
     ( /*function*/ Identifier )+ importFromSpec )

;


importSignatureSpec :

  #( ImportSignatureSpec
     ( /*signature*/ Identifier )+ importFromSpec )

;


importTeststepSpec :

  #( ImportTeststepSpec
     ( /*teststep*/ Identifier )+ importFromSpec )

;


/*****  SECTION A.1.6.2.9 - Group Definitions  *****/

groupDef :

  #( GroupDef
     /*group*/ Identifier ( moduleDefinitionsPart )? )

;


/*****  SECTION A.1.6.2.10 - External Function Definitions  *****/

extFunctionDef :

  #( ExtFunctionDef
     /*extFunction*/ Identifier ( functionFormalParList )? ( returnType )? )

;

/*****  SECTION A.1.6.2.11 - External Constant Definitions  *****/

extConstDef :

  #( ExtConstDef
     type /*extConst*/ Identifier )

;


/*****  SECTION A.1.6.3 - Control Part  *****/

moduleControlPart :

  #( root:ModuleControlPart
     { ENTER_ENV( #root ) }
     moduleControlBody ( withStatement )?
     { LEAVE_ENV } )

;


moduleControlBody :

  #( ModuleControlBody
     ( controlStatementOrDef )* )

;


controlStatementOrDef :

    functionLocalInst
  | controlStatement
  | functionLocalDef

;


controlStatement :

    timerStatements
  | basicStatements
  | behaviourStatements
  | sutStatements

;


/*****  SECTION A.1.6.3.1 - Variable Instantiation  *****/

varInstance :

  #( VarInstance
     type ( singleVarInstance )+ )

;


singleVarInstance :

  #( SingleVarInstance
     /*var*/ Identifier ( arrayDef )? ( expression )? )

;


variableRef :

  /*var,valuePar*/ id:Identifier ( extendedFieldReference )?

  {
    TTCNGrammar::TTCNSymbolParser sp;
    sp.startStatement( static_cast<antlr::RefAST>( currentEnv ), id -> getText() );
  }

;


/*****  SECTION A.1.6.3.2 - Timer Instantiation  *****/

timerInstance :

  #( TimerInstance
     /*timer*/ Identifier ( arrayDef )? ( timerValue )? )

;


timerValue :

  expression

;


timerRef :

  /*timer,timerPar*/ Identifier ( arrayOrBitRef )?

;


/*****  SECTION A.1.6.3.3 - Component Operations  *****/

configurationStatements :

    connectStatement
  | mapStatement
  | disconnectStatement
  | unmapStatement
  | doneStatement
  | startTCStatement
  | stopTCStatement

;


configurationOps :

    createOp
  | selfOp
  | systemOp
  | mtcOp
  | runningOp

;


createOp :

  #( CreateOp
     componentType )

;


systemOp :

  SystemOp

;


selfOp :

  SelfOp

;


mtcOp :

  MTCOp

;


doneStatement :

  #( DoneStatement
     componentId )

;


componentId :

    componentIdentifier
  | AnyComponent
  | AllComponent

;


runningOp :

  #( SomeRunningOp   // -> RunningOp
     componentId )

;


connectStatement :

  #( ConnectStatement
     portSpec )

;


portSpec :   // no root node needed

  portRef portRef

;


portRef :

  #( PortRef
     componentRef port )

;


componentRef :

    componentIdentifier
  | systemOp
  | selfOp
  | mtcOp

;


disconnectStatement :

  #( DisconnectStatement
     portSpec )

;


mapStatement :

  #( MapStatement
     portSpec )

;


unmapStatement :

  #( UnmapStatement
     portSpec )

;


startTCStatement :

  #( StartTCStatement
     componentIdentifier functionInstance )

;


stopTCStatement :

  StopTCStatement

;


componentIdentifier :

    functionInstance
  | variableRef

;


/*****  SECTION A.1.6.3.4 - Port Operations  *****/

port :

  /*port,portPar*/ Identifier ( arrayOrBitRef )?

;


communicationStatements :

    sendStatement
  | callStatement
  | replyStatement
  | raiseStatement
  | receiveStatement
  | triggerStatement
  | getCallStatement
  | getReplyStatement
  | catchStatement
  | checkStatement
  | clearStatement
  | startStatement
  | stopStatement

;


sendStatement :

  #( SendStatement
     port sendParameter ( toClause )? )

;


sendParameter :

  templateInstance

;


toClause :

  #( ToClause
     addressRef )

;


addressRef :

    functionInstance
  | variableRef

;


callStatement :

  #( CallStatement
     port callParameters ( toClause )? ( portCallBody )? )

;


callParameters :

  #( CallParameters
     templateInstance ( callTimerValue )? )

;


callTimerValue :

    timerValue
  | Nowait

;


portCallBody :

  #( PortCallBody
     ( callBodyStatement )+ )

;


callBodyStatement :

  #( CallBodyStatement
     callBodyGuard statementBlock )

;


callBodyGuard :

  #( CallBodyGuard
     altGuardChar callBodyOps )

;


callBodyOps :

    getReplyStatement
  | catchStatement

;


replyStatement :

  #( ReplyStatement
     port templateInstance ( replyValue )? ( toClause )? )

;


replyValue :

  #( ReplyValue
     expression )

;


raiseStatement :

  #( RaiseStatement
     port signature templateInstance ( toClause )? )

;


receiveStatement :

  #( ReceiveStatement
     portOrAny portReceiveOp )

;


portOrAny :

    port
  | AnyPort

;


portReceiveOp :

  #( PortReceiveOp
     ( receiveParameter )? ( fromClause )? ( portRedirect )? )

;


receiveParameter :

  templateInstance

;


fromClause :

  #( FromClause
     addressRef )

;


portRedirect :

  #( PortRedirect
     (   valueSpec ( senderSpec )?
       | senderSpec ) )

;


valueSpec :

  #( ValueSpec
     variableRef )

;


senderSpec :

  #( SenderSpec
     variableRef )

;


triggerStatement :

  #( TriggerStatement
     portOrAny ( receiveParameter )? ( fromClause )? ( portRedirect )? )

;


getCallStatement :

  #( GetCallStatement
     portOrAny portGetCallOp )

;


portGetCallOp :

  #( PortGetCallOp
     ( receiveParameter )? ( fromClause )? ( portRedirectWithParam )? )

;


portRedirectWithParam :

  #( PortRedirectWithParam
     redirectSpec )

;


redirectSpec :

  #( RedirectSpec
     (   valueSpec ( paraSpec )? ( senderSpec )?
       | paraSpec ( senderSpec )?
       | senderSpec ) )

;


paraSpec :

  #( ParaSpec
     paraAssignmentList )

;


paraAssignmentList :

  #( ParaAssignmentList
     ( assignmentList | variableList ) )

;


assignmentList :

  #( AssignmentList
     ( variableAssignment )+ )

;


variableAssignment :

  #( VariableAssignment
     variableRef parameterIdentifier )

;


parameterIdentifier :

  /*valuePar,timerPar,templatePar,portPar*/ Identifier

;


variableList :

  #( VariableList
     ( variableEntry )+ )

;


variableEntry :

    variableRef
  | NotUsed

;


getReplyStatement :

  #( GetReplyStatement
     portOrAny portGetReplyOp )

;


portGetReplyOp :

  #( PortGetReplyOp
     ( receiveParameter ( valueMatchSpec )? )? ( fromClause )? ( portRedirectWithParam )? )

;


valueMatchSpec :

  #( ValueMatchSpec
     templateInstance )

;


checkStatement :

  #( CheckStatement
     portOrAny ( checkParameter )? )

;


checkParameter :

    portReceiveOp
  | portGetCallOp
  | portGetReplyOp
  | portCatchOp
  | #( CheckParameter
       ( fromClause )? ( senderSpec )? )

;


catchStatement :

  #( CatchStatement
     portOrAny portCatchOp )

;


portCatchOp :

  #( PortCatchOp
     ( catchOpParameter )? ( fromClause )? ( portRedirect )? )

;


catchOpParameter :

  #( CatchOpParameter
     ( signature templateInstance | Timeout ) )

;


clearStatement :

  #( ClearStatement
     portOrAll )

;


portOrAll :

    port
  | AllPort

;


startStatement :

  #( SomeStartStatement   // -> StartStatement
     portOrAll )

;


stopStatement :

  #( SomeStopStatement   // -> StopStatement
     portOrAll )

;


/*****  SECTION A.1.6.3.5 - Timer Operations  *****/

timerStatements :

    startTimerStatement
  | stopTimerStatement
  | timeoutStatement

;


timerOps :

    readTimerOp
  | runningTimerOp

;


startTimerStatement :

  #( SomeStartStatement
     timerRef ( timerValue )? )

;


stopTimerStatement :

  #( SomeStopStatement   // -> StopTimerStatement
     timerRefOrAll )

;


timerRefOrAll :

    timerRef
  | AllTimer

;


readTimerOp :

  #( ReadTimerOp
     timerRef )

;


runningTimerOp :

  #( SomeRunningOp   // -> RunningTimerOp
     timerRefOrAny )

;


timeoutStatement :

  #( TimeoutStatement
     timerRefOrAny )

;


timerRefOrAny :

    timerRef
  | AnyTimer

;


/*****  SECTION A.1.6.4 - Type  *****/

type :

    predefinedType
  | referencedType

;


predefinedType :

    TypeBitstring
  | TypeBoolean
  | TypeCharstring
  | TypeUniversalCharstring
  | TypeChar
  | TypeUniversalChar
  | TypeInteger
  | TypeOctetstring
  | TypeObjId
  | TypeHexstring
  | TypeVerdictType
  | TypeFloat
  | TypeAddress
  | TypeDefault

;


referencedType :

  ( ( globalModuleId typeReference ) =>
    globalModuleId )?
  typeReference ( extendedFieldReference )?

;


typeReference :

    /*structType*/ Identifier ( typeActualParList )?
  | /*enumType,subType,typePar,componentType*/ Identifier

;


typeActualParList :

  #( TypeActualParList
     ( typeActualPar )+ )

;


typeActualPar :

    singleConstExpression
  | type

;


/*****  SECTION A.1.6.4.1 - Array Types *****/

arrayDef :

  #( ArrayDefList
     ( arrayDef2 )+ )

;


arrayDef2 :

  #( ArrayDef
     arrayBounds ( arrayBounds )? )

;


arrayBounds :

  singleConstExpression

;


/*****  SECTION A.1.6.5 - Value  *****/

value :

    referencedValue   // temporarily reversed order
  | predefinedValue

;


predefinedValue :

    bitStringValue
  | booleanValue
  | charStringValue
  | integerValue
  | octetStringValue
  | objectIdentifierValue
  | hexStringValue
  | verdictTypeValue
  | enumeratedValue
  | FloatValue
  | addressValue

;


bitStringValue :

  BString

;


booleanValue :

    True
  | False

;


integerValue :

  Number

;


octetStringValue :

  OString

;


objectIdentifierValue :

  #( ObjectIdentifierValue
     ( objIdComponent )+ )

;


objIdComponent :

    nameForm
  | numberForm
  | nameAndNumberForm

;


numberForm :

    Number
  | referencedValue

;


nameAndNumberForm :

  #( NameAndNumberForm
     Identifier numberForm )

;


nameForm :

  Identifier

;


hexStringValue :

  HString

;


verdictTypeValue :

    Pass
  | Fail
  | Inconc
  | None
  | Error

;


enumeratedValue :

  /*namedValue*/ Identifier

;


charStringValue :

    CString
  | quadruple

;


quadruple :

  #( Quadruple
     /*group*/ Number /*plane*/ Number /*row*/ Number /*cell*/ Number )

;


referencedValue :

  valueReference ( extendedFieldReference )?

;


valueReference :

    ( ( globalModuleId Identifier ) => globalModuleId )?
    /*const*/ Identifier
  | /*extConst,valuePar,modulePar,var*/ Identifier

;


freeText :

  CString  // temporary workaround - see TTCNParser.g

;


addressValue :

  Null

;


/*****  SECTION A.1.6.6 - Parameterisation  *****/

formalValuePar :

  #( SomeFormalPar   // -> FormalValuePar
     ( InOut | Out )? type /*valuePar*/ Identifier )

;


formalTypePar :

  #( FormalTypePar
     /*typePar*/ Identifier )

;


formalPortPar :

  #( SomeFormalPar   // -> FormalPortPar
     /*portType*/ Identifier /*portPar*/ Identifier )

;


formalTimerPar :

  #( FormalTimerPar
     /*timerPar*/ Identifier )

;


formalTemplatePar :

  #( FormalTemplatePar
     type /*templatePar*/ Identifier )

;


/*****  SECTION A.1.6.7 - The With Statement  *****/

withStatement :

  #( WithStatement
     ( singleWithAttrib )+ )

;


singleWithAttrib :

  #( SingleWithAttrib
     attribKeyword ( Override )? ( attribQualifier )? attribSpec )

;


attribKeyword :

    Encode
  | Display
  | Extension

;


attribQualifier :

  #( AttribQualifier
     ( defOrFieldRef )+ )

;


defOrFieldRef :

    fieldReference
  | definitionRef   // temporarily reversed order

;


definitionRef :

    /*structType,enumType,portType,componentType,subType,
      const,template,teststep,testcase,function,signature*/ Identifier

;


attribSpec :

  freeText

;


/*****  SECTION A.1.6.8 - Behaviour Statements  *****/

behaviourStatements :

    testcaseInstance
  | functionInstance
  | returnStatement
  | altConstruct
  | interleavedConstruct
  | labelStatement
  | gotoStatement
  | repeatStatement
  | deactivateStatement
  | teststepInstance

;


verdictStatements :

  setLocalVerdict

;


verdictOps :

  getLocalVerdict

;


setLocalVerdict :

  #( SetLocalVerdict
     singleExpression )

;


getLocalVerdict :

  GetLocalVerdict

;


sutStatements :

  #( SUTStatements
     ( freeText | templateRefWithParList ) )

;


returnStatement :

  #( ReturnStatement
     ( expression )? )

;


altConstruct :

  #( AltConstruct
     altGuardList )

;


altGuardList :   // no root node needed

  ( guardStatement )+ ( elseStatement )?

;


guardStatement :

  #( GuardStatement
     altGuardChar
     (   teststepInstance
       | guardOp statementBlock ) )

;


elseStatement :

  #( ElseStatement
     statementBlock )

;


altGuardChar :

  #( AltGuardChar
     ( booleanExpression )? )

;


guardOp :

    timeoutStatement
  | receiveStatement
  | triggerStatement
  | getCallStatement
  | catchStatement
  | checkStatement
  | getReplyStatement
  | doneStatement

;


statementBlock :

  #( StatementBlock
     ( functionStatementOrDef )* )

;


interleavedConstruct :

  #( InterleavedConstruct
     ( interleavedGuardElement )+ )

;


interleavedGuardElement :

  #( InterleavedGuardElement
     interleavedGuard interleavedAction )

;


interleavedGuard :

  #( InterleavedGuard
     guardOp )

;


interleavedAction :

  statementBlock

;


labelStatement :

  #( Label
     /*label*/ Identifier )

;


gotoStatement :

  #( GotoStatement
     /*label*/ Identifier )

;


repeatStatement :

  RepeatStatement

;


activateOp :

  #( ActivateOp
     teststepInstance )

;


deactivateStatement :

  #( DeactivateStatement
     ( expression )? )

;


/*****  SECTION A.1.6.9 - Basic Statements  *****/

basicStatements :

    assignment
  | logStatement
  | loopConstruct
  | conditionalConstruct

;


expression :

    singleExpression
  | compoundExpression

;


compoundExpression :

    fieldExpressionList
  | arrayExpression

;


fieldExpressionList :

  #( FieldExpressionList
     ( fieldExpressionSpec )+ )

;


fieldExpressionSpec :

  #( FieldExpressionSpec
     fieldReference expression )

;


arrayExpression :

  #( ArrayExpression
     (   expression
       | NotUsed )* )

;


constantExpression :

    singleConstExpression
  | compoundConstExpression

;


singleConstExpression :

  singleExpression

;


booleanExpression :

  singleExpression

;


compoundConstExpression :

    fieldConstExpressionList
  | arrayConstExpression

;


fieldConstExpressionList :

  #( FieldConstExpressionList
     ( fieldConstExpressionSpec )+ )

;


fieldConstExpressionSpec :

  #( FieldConstExpressionSpec
     fieldReference constantExpression )

;


arrayConstExpression :

  #( ArrayConstExpression
     ( constantExpression )* )

;


assignment :

  #( Assignment
     variableRef expression )

;


singleExpression :

    #( OpBitwiseAnd singleExpression singleExpression )
  | #( OpBitwiseXor singleExpression singleExpression )
  | #( OpBitwiseOr singleExpression singleExpression )
  | #( OpLogicalAnd singleExpression singleExpression )
  | #( OpLogicalXor singleExpression singleExpression )
  | #( OpLogicalOr singleExpression singleExpression )
  | #( OpConcatenate singleExpression singleExpression )
  | #( OpEqual singleExpression singleExpression )
  | #( OpLess singleExpression singleExpression )
  | #( OpGreater singleExpression singleExpression )
  | #( OpNotEqual singleExpression singleExpression )
  | #( OpGreaterOrEqual singleExpression singleExpression )
  | #( OpLessOrEqual singleExpression singleExpression )
  | #( OpShiftLeft singleExpression singleExpression )
  | #( OpShiftRight singleExpression singleExpression )
  | #( OpRotateLeft singleExpression singleExpression )
  | #( OpRotateRight singleExpression singleExpression )
  | #( OpBinaryPlus singleExpression singleExpression )
  | #( OpBinaryMinus singleExpression singleExpression )
  | #( OpTimes singleExpression singleExpression )
  | #( OpDivide singleExpression singleExpression )
  | #( OpMod singleExpression singleExpression )
  | #( OpRem singleExpression singleExpression )
  | #( OpUnaryPlus singleExpression )
  | #( OpUnaryMinus singleExpression )
  | #( OpLogicalNot singleExpression )
  | #( OpBitwiseNot singleExpression )
  | primary

;


primary :

    ( opCall ) =>
    opCall
  | value

;


extendedFieldReference :

  (   /*structField*/ Identifier
    | arrayOrBitRef )+

;


opCall :

    configurationOps
  | verdictOps
  | timerOps
  | testcaseInstance
  | functionInstance
  | templateOps
  | activateOp

;


logStatement :

  #( LogStatement
     ( freeText )? )

;


loopConstruct :

    forStatement
  | whileStatement
  | doWhileStatement

;


forStatement :

  #( ForStatement
     ( varInstance | assignment ) booleanExpression assignment
     statementBlock )

;


whileStatement :

  #( WhileStatement
     booleanExpression statementBlock )

;


doWhileStatement :

  #( DoWhileStatement
     statementBlock booleanExpression )

;


conditionalConstruct :

  #( ConditionalConstruct
     booleanExpression statementBlock ( elseIfClause )* ( elseClause )? )

;


elseIfClause :

  #( ElseIfClause
     booleanExpression statementBlock )

;


elseClause :

  #( ElseClause
     statementBlock )

;
