/* vim:ts=4:sw=4:noet
 * (tabspace=4)
 * 
 * Copyright (C) 2004, 2005 Walter Doekes, <walter@djcvt.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "getopt.h"
#include "tthsum.h"
#include <locale.h>
#include <stdio.h>
#include <string.h>

void help(int error);
void version();

int main(int argc, char* const argv[]) {
	const char* stdin_file[] = {NULL, NULL};
	struct tthsum_options opt;
	int ret;
	int digest = 0;
	int quit = 0;
	char* cp;
	memset(&opt, 0, sizeof(struct tthsum_options));

	/* set up proper locale */
	if((cp = setlocale(LC_CTYPE, "")) == NULL &&
			(cp = setlocale(LC_CTYPE, "C")) == NULL)
		fprintf(stderr, "tthsum: warning: Check your locale settings "
				"(the LANG and/or LC_CTYPE environment variables).\n");
	if(cp && strcmp(cp, "C") != 0 && strcmp(cp, "POSIX") != 0)
		opt.has_locale = 1;
	
	/* read options */
	while((ret = getopt(argc, argv, "bchmpvVw")) >= 0) {
		switch(ret) {
		case 'b':
			/* md5sum compatibility, do nothing */
			break;
		case 'c':
			digest = 1;
			break;
		case 'm':
			opt.use_mmap = 1;
			break;
		case 'p':
			opt.progress_every_mib = 10;
			break;
		case 'v':
			opt.verbose = 1;
			break;
		case 'w':
			opt.warn = 1;
			break;
		case 'V':
			version();
			quit = 1;
			break;
		case 'h':
			help(0);
			quit = 1;
			break;
		case '?':
			help(1);
			return 2;
		default:
			/* shouldn't happen */
			return -1;
		}
	}

	if(quit)
		return 0;

	/* check digest? */
	if(digest) {
		/* only accept one digest file */
		if(optind > argc + 1) {
			help(1);
			return 2;
		}
		if(optind == argc)
			ret = tthsum_check_digest(NULL, &opt);
		else
			ret = tthsum_check_digest(argv[optind], &opt);
	/* generate digest? */
	} else {
		if(optind == argc) {
			ret = tthsum_generate_roots(stdin_file, 1, &opt);
		} else {
			ret = tthsum_generate_roots((const char**)argv + optind,
					argc - optind, &opt);
		}
	}

	/* done */
	return ret == 0 ? 0 : 1;
}

void help(int error) {
	fprintf(error ? stderr : stdout,
"usage: tthsum [-bhmpvVw] [-c [file]] | [file...]\n"
"Generates or checks TTH Message Digests (root of the Tiger/THEX hash tree)\n"
"    -b  md5sum compatibility option (does absolutely nothing)\n"
"    -c  check message digests (default is generate)\n"
"    -h  show the small help and exit\n"
"    -m  use mmap for reading files\n"
"    -p  show the hashing progress in MiB\n"
"    -v  verbose, print file names when checking a digest\n"
"    -V  print the version and exit\n"
"    -w  warn on improperly formatted lines checking a digest\n"
"The input for -c should be the list of message digests and file names\n"
"that is printed on stdout by this program when it generates digests.\n"
"If you don't specify a file, input will be read from stdin.\n"
	);
}

void version() {
	printf(
"tthsum 1.1.0\n"
"\n"
"Copyright 2004, 2005 Walter Doekes.\n"
"This is free software; see the source for copying conditions. There is NO\n"
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
	);
}
