/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
void plot_takeoff(void);

#include "header.h"

#define T_H 20
#define I_W 180

struct gfx *gfx;

#ifdef HAVE_SDL

#include <SDL.h>

#ifndef HAVE_SDL_CLIP_RECT
SDL_Rect dummy_cr;
#endif

#undef USE_RECT

/* !!!!!! */
#define CLIPPED(clip, _x, _y) ((_x)<(clip)->x || (_y)<(clip)->y || (_x)>=(clip)->x+(clip)->w || (_y)>=(clip)->y+(clip)->h) 


struct zoom zooms[]={
    {0, 1,  NULL,    0, 0},
    
    {1, 50, NULL, 2000, 0},
    {2, 40, NULL, 2000, 0},
    {2, 30, NULL, 1000, 0},
    {3, 15, NULL, 1000, 0},
    {4, 10, NULL,  500, 0},
    {5, 7,  NULL,  500, 1},
    {6, 5,  NULL,  500, 1},
    {7, 3,  NULL,  200, 1},
    {8, 2,  NULL,  100, 1},
    {9, 1,  NULL,  100, 1},
    
    {0, 1,  NULL,    0, 0},
};


SDL_Color pal[256];
int mouse_x, mouse_y;
static gchar *title_str=NULL;

int makecol(int r, int g, int b){
    SDL_PixelFormat *format;

    format=gfx->screen->format;
    return (((r>>format->Rloss)<<format->Rshift)&format->Rmask) |
           (((g>>format->Gloss)<<format->Gshift)&format->Gmask) |
           (((b>>format->Bloss)<<format->Bshift)&format->Bmask);
}

SDL_Rect *new_rect(int x, int y, int w, int h){
    SDL_Rect *rect;

    rect = g_new0(SDL_Rect, 1);
    rect->x = x;
    rect->y = y;
	/* TODO */
    rect->w = w;
    rect->h = h;
    return rect;
}

void free_rect(SDL_Rect *rect){
    g_free(rect);
}

void fast_putpixel8(SDL_Rect *clip, int x, int y, int color){
    Uint8 *p;

    if CLIPPED(clip, x, y) return;
    p = (Uint8 *) gfx->screen->pixels + y*gfx->screen->pitch + x;
    *p = color;
}

void fast_putpixel16(SDL_Rect *clip, int x, int y, int color){
    Uint16 *p;

    if CLIPPED(clip, x, y) return;
    p = (Uint16 *) gfx->screen->pixels + y*gfx->screen->pitch/2 + x;
    *p = color;
}

void fast_putpixel24(SDL_Rect *clip, int x, int y, int color){
    Uint16 *p16;
    Uint8  *p8;

    if CLIPPED(clip, x, y) return;
    p16 = (Uint16 *) gfx->screen->pixels + y*gfx->screen->pitch/3 + x;
    p8  = (Uint8 *)  (p16+1); 
    *p16 = color;     /* TODO Big endian? */
    *p8  = color>>16; 
}

void fast_putpixel32(SDL_Rect *clip, int x, int y, int color){
    Uint32 *p;

    if CLIPPED(clip, x, y) return;
    p = (Uint32 *) gfx->screen->pixels + y*gfx->screen->pitch/4 + x;
    *p = color;
}

void (*fast_putpixel)(SDL_Rect *clip, int x, int y, int color);


int fast_getpixel8(int x, int y){
    Uint8 *p;

    p = (Uint8 *) gfx->screen->pixels + y*gfx->screen->pitch + x;
    return *p;
}

int fast_getpixel16(int x, int y){
    Uint16 *p;

    p = (Uint16 *) gfx->screen->pixels + y*gfx->screen->pitch/2 + x;
    return *p;
}

int fast_getpixel24(int x, int y){
    Uint16 *p16;
    Uint8  *p8;

    p16 = (Uint16 *) gfx->screen->pixels + y*gfx->screen->pitch/3 + x;
    p8  = (Uint8 *)  (p16+1); 
	return (*p16) + ((*p8)<<16);
/*    *p16 = color; */    /* TODO Big endian? */
/*    *p8  = color>>16; */
}

int fast_getpixel32(int x, int y){
    Uint32 *p;

    p = (Uint32 *) gfx->screen->pixels + y*gfx->screen->pitch/4 + x;
    return *p;
}

int (*fast_getpixel)(int x, int y);


void line(SDL_Rect *clip, int x1, int y1, int x2, int y2, int color){
    int dx, dy, p;	
    int inc, tmp;
    
    dx=abs(x1-x2);
    dy=abs(y1-y2);
    if (dx>=dy){
    	p = 2*dy-dx;
    	if (x1 >= x2){   
            tmp=x1;x1=x2;x2=tmp;
	    	tmp=y1;y1=y2;y2=tmp;
		}
        fast_putpixel(clip, x1, y1, color);
    	if (y2>=y1) inc=1;
        else inc=-1;
    	
        while(x1 < x2){	
    	    x1++;   	
            if (p<0) 
                p += 2*dy;
            else{
                y1+=inc;
                p += 2*(dy-dx);
            }
            fast_putpixel(clip, x1, y1, color);
        }
    }else{
    	p = 2*dx-dy;
        if (y1 >= y2){
            tmp=x1;x1=x2;x2=tmp;
	    	tmp=y1;y1=y2;y2=tmp;
        }
        fast_putpixel(clip, x1, y1, color);
    	
    	if (x2>=x1) inc=1;
        else inc=-1;
    	
        while(y1 < y2){	
    	    y1++;   	
            if (p<0) 
                p += 2*dx;
            else{
                x1 += inc;
                p += 2*(dx-dy);
            }
            fast_putpixel(clip, x1,y1, color);
        }
    }
} 

void do_line(SDL_Rect *clip, int x1, int y1, int x2, int y2, int color, 
        void (*func)(SDL_Rect *clip, int x, int y, int d) ){

    int dx, dy, p;	
    int inc, tmp;
    
    dx=abs(x1-x2);
    dy=abs(y1-y2);
    if (dx>=dy){
    	p = 2*dy-dx;
    	if (x1 >= x2){   
            tmp=x1;x1=x2;x2=tmp;
	    	tmp=y1;y1=y2;y2=tmp;
		}
        func(clip, x1, y1, color);
    	if (y2>=y1) inc=1;
        else inc=-1;
    	
        while(x1 < x2){	
    	    x1++;   	
            if (p<0) 
                p += 2*dy;
            else{
                y1+=inc;
                p += 2*(dy-dx);
            }
            func(clip, x1, y1, color);
        }
    }else{
    	p = 2*dx-dy;
        if (y1 >= y2){
            tmp=x1;x1=x2;x2=tmp;
	    	tmp=y1;y1=y2;y2=tmp;
        }
        func(clip, x1, y1, color);
    	
    	if (x2>=x1) inc=1;
        else inc=-1;
    	
        while(y1 < y2){	
    	    y1++;   	
            if (p<0) 
                p += 2*dx;
            else{
                x1 += inc;
                p += 2*(dx-dy);
            }
            func(clip, x1,y1, color);
        }
    }
} 

#define SWAP(a,b){ int tmp; \
    tmp=x##a; x##a=x##b; x##b=tmp;\
    tmp=y##a; y##a=y##b; y##b=tmp;}
    
void triangle(SDL_Rect *rect, int x1, int y1, int x2, int y2, int x3, int y3, int c){
    int a1,b1,c1, a2,b2,c2, a3,b3,c3, xx1, xx2,xx3,y;
    
    
    if (y2<y1) SWAP(1,2);
    if (y3<y1) { SWAP(1,3); SWAP(2,3); }
    if (y3<y2) SWAP(2,3);

    if (y1==y2 && y2==y3){ /* singularity, horizontal line */
#ifdef USE_RECT 
        SDL_Rect r;
        r.x=x1<x2?x1:x2;
        if (x3<r.x) r.x=x3;
        r.y=y1;
        r.w=x1>x2?x1:x2;
        if (x3>r.x) r.w=x3;
        r.w-=r.x;
        r.h=1;
        SDL_SetClipRect(gfx->screen, rect);   
        SDL_FillRect(gfx->screen, &r, c);
        SDL_SetClipRect(gfx->screen, NULL);   
#else        
        /* i don't want to sort X's :-) */
        line(rect, x1, y1, x2, y2, c);
        line(rect, x1, y1, x3, y3, c);
        line(rect, x3, y3, x2, y2, c);
#endif        
        return;
    }
    
    a1=y2-y3;
    b1=x3-x2;
    c1=y3*x2-x3*y2;
    
    a2=y1-y2;
    b2=x2-x1;
    c2=y2*x1-x2*y1;
    
    a3=y1-y3;
    b3=x3-x1;
    c3=y3*x1-x3*y1;

#ifdef USE_RECT 
    SDL_SetClipRect(gfx->screen, rect);   
#endif    
    for (y=y1;y<y2;y++){
#ifdef USE_RECT        
        SDL_Rect r;
#endif        
        xx2=-(b2*y+c2)/a2; /* a2=0 not reached because y1=y2 -> no iteration */
        xx3=-(b3*y+c3)/a3; /* a3=0 not reached because y1=y2 -> y1=y2=y3 */
#ifdef USE_RECT
        r.x=xx2<xx3?xx2:xx3;
        r.w=xx2<xx3?xx3-xx2:xx2-xx3;
        r.y=y;
        r.h=1;
        SDL_FillRect(gfx->screen, &r, c);
#else        
        line(rect, xx2, y, xx3, y, c);
#endif        
    }
    
    for (y=y2;y<y3;y++){
#ifdef USE_RECT        
        SDL_Rect r;
#endif        
        xx1=-(b1*y+c1)/a1; /* a1=0 not reached because y2=y3 -> no iteration */
        xx3=-(b3*y+c3)/a3;
#ifdef USE_RECT        
        r.x=xx1<xx3?xx1:xx3;
        r.w=xx1<xx3?xx3-xx1:xx1-xx3;
        r.y=y;
        r.h=1;
        SDL_FillRect(gfx->screen, &r, c);
#else        
        line(rect, xx1, y, xx3, y, c);
#endif        
    }
#ifdef USE_RECT    
    SDL_SetClipRect(gfx->screen, NULL);   
#endif    
    fast_putpixel(rect, x3, y3, c);
    
}

void circle(SDL_Rect *rect, int x, int y, int r, int c){
    int xx, yy, p;

/*    dbg("circle([%d,%d,%d,%d], %d, %d, %d, %x\n", rect->x, rect->y, rect->w, rect->h, x, y, r, c);*/
    p=1-r;
    yy=r;
    
    for (xx=0; xx<=yy; xx++){
        fast_putpixel(rect, x+yy, y+xx, c);
        fast_putpixel(rect, x+yy, y-xx, c);
        fast_putpixel(rect, x-yy, y+xx, c);
        fast_putpixel(rect, x-yy, y-xx, c);
                              
        fast_putpixel(rect, x+xx, y+yy, c);
        fast_putpixel(rect, x+xx, y-yy, c);
        fast_putpixel(rect, x-xx, y+yy, c);
        fast_putpixel(rect, x-xx, y-yy, c);

        if (p<0){
            p+=2*xx + 3;
        }else{
            p+=2*xx - 2*yy + 1;
            yy--;
        }
    }
}

void rect(SDL_Rect *rect, int x1, int y1, int x2, int y2, int c){
    line(rect, x1, y1, x2, y1, c);
    line(rect, x2, y1, x2, y2, c);
    line(rect, x1, y2, x2, y2, c);
    line(rect, x1, y1, x1, y2, c);
}

static int dashcnt=0;

void dashfce(SDL_Rect *clip, int x, int y, int d) {
    if (dashcnt++%6<3) return;
    fast_putpixel(clip, x, y, d);
}
    

/********************************************/

int SetVideoMode(int w, int h){
    if ((gfx->screen = SDL_SetVideoMode(w, h, 0, SDL_SWSURFACE|SDL_RESIZABLE|SDL_HWPALETTE))==NULL){
        return 1;    
    }

    if (gfx->title) free_rect(gfx->title);
    if (gfx->map) free_rect(gfx->map);
    if (gfx->info) free_rect(gfx->info);
    gfx->title = new_rect(4,4, w-8, T_H);
    gfx->map   = new_rect(4, T_H+8,w-I_W-12, h-12-T_H);
    gfx->info  = new_rect(w-I_W-4, T_H+8, I_W, h-12-T_H);
    
    gfx->m_x=gfx->map->x + gfx->map->w/2;
    gfx->m_y=gfx->map->y + gfx->map->h/2;
    
    if (gfx->screen->format->palette){
        int i;
        SDL_PixelFormat *fmt;
        
        for (i=0; i<256; i++){
            gfx->colors[i].r=(i&0xe0)+31; 
            gfx->colors[i].g=((i<<3)&0xe0)+31;
            gfx->colors[i].b=((i<<6)&0xc0)+63;
        }
        SDL_SetColors(gfx->screen, gfx->colors, 0, 256);

        fmt=gfx->screen->format;
        fmt->Rmask=0xe0;
        fmt->Gmask=0x1c;
        fmt->Bmask=0x03;
        fmt->Rshift=5;
        fmt->Gshift=2;
        fmt->Bshift=0;
        fmt->Rloss=5;
        fmt->Gloss=5;
        fmt->Bloss=6;
        
        
    }
    
    invalidate_bkg(NULL);
    invalidate_cache(aband);
    return 0;
}

struct gfx *init_gfx(){
    gchar *c;
    int i;
   /* PALETTE pal;*/
    int w,h;

    gfx = g_new0(struct gfx, 1);
    gfx->zoomint = 8; /*1 */
        
    c = g_strconcat(getenv("HOME"),"/tucnak/tucnakcor", NULL);
    gfx->cor = load_cor(c);
    g_free(c);
    if (!gfx->cor){
        gfx->cor = load_cor(PREFIX"/share/"PACKAGE"/tucnakcor");
    }
    
    if (!gfx->cor){
        log_addf(TEXT(T_CANT_READ_S), "tucnakcor");
        free_gfx(); 
        gfx=NULL;
        return NULL;       
    }
    
    setenv("SDL_VIDEO_X11_WMCLASS", "Tucnak", 1);
    if (SDL_Init(SDL_INIT_VIDEO|SDL_INIT_NOPARACHUTE)){
        log_addf(TEXT(T_CANT_SET_GR_S), "SDL_Init");
        free_gfx(); 
        gfx=NULL;
        return NULL;
    }

    gfx->icon = SDL_LoadBMP(PREFIX"/share/"PACKAGE"/tucnak.bmp");
    if (gfx->icon) SDL_WM_SetIcon(gfx->icon, NULL);
    

    w=800;
    h=600;
    
    if (SetVideoMode(w, h)){
        log_addf(TEXT(T_CANT_SET_GR_S), "SetVideoMode");
        free_gfx(); 
        gfx=NULL;
        return NULL;
    }
    
    
    gfx->is_graphics = 1;
    gfx->bpp = gfx->screen->format->BitsPerPixel;
#ifdef HAVE_SDL_CLIP_RECT    
    gfx->scr = &gfx->screen->clip_rect;
#else 
    dummy_cr.x = gfx->screen->clip_minx; 
    dummy_cr.y = gfx->screen->clip_miny; 
    dummy_cr.w = gfx->screen->clip_maxx-gfx->screen->clip_minx; 
    dummy_cr.h = gfx->screen->clip_maxy-gfx->screen->clip_miny; 
    gfx->scr = &dummy_cr;   
#endif
    
    gfx->zoom = zooms+gfx->zoomint;

    pal[0].r = pal[0].g = pal[0].b = 0;
    gfx->gr[0] = makecol(0, 0, 0);
    for (i=1; i<16; i++){
        pal[i].r = pal[i].g = pal[i].b = i*4+3;
        gfx->gr[i] = makecol(i*16+15, i*16+15, i*16+15);

    }
    
    gfx->mapcache = SDL_CreateRGBSurface(SDL_SWSURFACE, gfx->map->w, gfx->map->h, gfx->bpp, 
                gfx->screen->format->Rmask, gfx->screen->format->Gmask, gfx->screen->format->Gmask, 0);
    SDL_UpdateRect(gfx->screen, 0, 0, 0, 0);
    gfx->yellow  = makecol(255, 255, 0);
    gfx->green   = makecol(0, 255, 0);
    gfx->red     = makecol(255,0,0);
    gfx->magenta = makecol(254,0,254);
    gfx->mapbg   = makecol(0,35,35); 
    gfx->wkmap   = makecol(255,70,0);

    switch(gfx->screen->format->BytesPerPixel){
        case 1:
            fast_putpixel = fast_putpixel8;
			fast_getpixel = fast_getpixel8;
            break;
        case 2:
            fast_putpixel = fast_putpixel16;
			fast_getpixel = fast_getpixel16;
            break;
        case 3:
            fast_putpixel = fast_putpixel24;
			fast_getpixel = fast_getpixel24;
            break;
        case 4:
            fast_putpixel = fast_putpixel32;
			fast_getpixel = fast_getpixel32;
            break;
    }

    for_printing(gfx, 0);
    gfx->draw=1;
    gfx->func_timer_id = install_timer_(100, (void (*)(void *)) gfx_func, (void *)gfx);
    
    gfx_update_qth();
    SDL_UpdateRect(gfx->screen, 0, 0, 0, 0);
    
    SDL_SetEventFilter(FilterEvents);
    SDL_EventState(SDL_KEYUP, SDL_IGNORE);
    SDL_WM_SetCaption(title_str, NULL);
    rot_update_colors(gfx);
    return gfx;
}
                                                                             



void free_gfx(void){
    int i;
    /*dbg("free_gfx  gfx=%p\n", gfx);*/
    
    if (!gfx) return;

    if (gfx->func_timer_id){
        kill_timer(gfx->func_timer_id);
    }
            
    if (gfx->cor) free_cor(gfx->cor);
    if (gfx->icon) SDL_FreeSurface(gfx->icon);
    
    if (gfx->title) free_rect(gfx->title);
    if (gfx->map) free_rect(gfx->map);
    if (gfx->info) free_rect(gfx->info);
    
    if (gfx->screen) SDL_FreeSurface(gfx->screen);

    for (i=1; ;i++){
        if (zooms[i].id==0) break;
        if (!zooms[i].bkg) continue;
        SDL_FreeSurface(zooms[i].bkg);
        zooms[i].bkg=NULL;
    }
    SDL_FreeSurface(gfx->mapcache);
    
    CONDGFREE(gfx->pwwlo);
/*    CONDGFREE(title_str); yes, small memory leak, it's freeed by kernel*/
    
    SDL_Quit();

    g_free(gfx);
}


void compute_cache(struct qso *qso){
    double a;
    
    if (qso->gfx_x) return;
    a = qso->qtf*MY_PI/180.0;
    qso->gfx_x = gfx->m_x + qso->qrb/gfx->zoom->value*sin(a);
    qso->gfx_y = gfx->m_y - qso->qrb/gfx->zoom->value*cos(a);
}

struct qso *find_nearest(struct band *b, int mouse_x, int mouse_y){
    int i;
    struct qso *minq, *q;
    gdouble d, min;

    if (!b) return NULL;
    min=G_MAXDOUBLE;
    minq=NULL;
    for (i=0; i<b->qsos->len; i++){
        q = (struct qso *)g_ptr_array_index(b->qsos, i);
        if (q->error) continue;
        
        compute_cache(q);
        d = sqrt((double)(sqr(q->gfx_x-mouse_x)+sqr(q->gfx_y-mouse_y)));
/*        dbg("  %s %d %d %6.3f\n", q->callsign, q->gfx_x-mouse_x,  q->gfx_y-mouse_y, d);*/
        if (d<min){
            min=d;
            minq=q;
        }
    }
/*    if (minq) dbg("nearest: %s\n", minq->callsign);
    else dbg("nearest: NULL\n");*/
    return minq;
}

void invalidate_cache(struct band *b){
    int i;
    struct qso *q;
    
    if (!b) return;
    for (i=0; i<b->qsos->len; i++){
        q = (struct qso *)g_ptr_array_index(b->qsos, i);
        q->gfx_x=0;
    }
    if (gfx) gfx->minq=NULL;
}

void invalidate_bkg(struct zoom *zoom){
    int i;
    
    if (zoom && zoom->bkg){
        SDL_FreeSurface(zoom->bkg);
        zoom->bkg=NULL;
        gfx->draw=1; 
        return;
    }

    for (i=1; ;i++){
        if (zooms[i].id==0) break;
        if (!zooms[i].bkg) continue;
        SDL_FreeSurface(zooms[i].bkg);
        zooms[i].bkg=NULL;
    }
    gfx->draw=1; 
}

int not_on_screen(struct gfx *gfx, double h, double w){
    double w2, h2, qrb, qtf;
    int x,y;
    SDL_Rect rect;
    
    /*dbg("nos[%4d,%4d] ", h, w);*/
    
    w2=(MY_PI*(double)w)/180;
    h2=(MY_PI*(double)h)/180;
    
    hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
    x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
    y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);

    rect.x=x; rect.y=y; rect.w=3; rect.h=3;

        
    if (x<gfx->map->x) {
        return 1;
    }
    if (x>gfx->map->x+gfx->map->w) {
        return 1;
    }
    
    if (y<gfx->map->y) {
        return 1;
    }
    if (y>gfx->map->y+gfx->map->h) {
        return 1;
    }
#if 0    
    dbg("os[%4d,%4d] x=%d y=%d\n", h, w, x, y);
    line(gfx->map, x, y-4,x,y+4, makecol(255,0,0));
    line(gfx->map, x-4, y,x+4,y, makecol(255,0,0));
#endif    
    return 0;
}

int plot_cor(){
    int i, w, h;
    double  h2, w2, qrb, qtf;
    double oldx, oldy, x, y;
    struct corarray *ca;
    struct cpoint *cp;
    int color, c;
    double h1, w1;

    oldx=oldy=color=-1;
    
    h1 = gfx->myh;
    w1 = gfx->myw;
    
    for (h=0; h<COR_H; h++)
    for (w=0; w<COR_W; w++){
        if (
             not_on_screen(gfx, (h+1-COR_H/2)*10,      (w+1-COR_W/2)*10      ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10+9.99, (w+1-COR_W/2)*10      ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10,      (w+1-COR_W/2)*10+5    ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10+9.99, (w+1-COR_W/2)*10+5    ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10,      (w+1-COR_W/2)*10+9.99 ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10+9.99, (w+1-COR_W/2)*10+9.99 ) && 
             not_on_screen(gfx, (h+1-COR_H/2)*10+5,    (w+1-COR_W/2)*10+5    ) 
           ) continue;
        
        ca = gfx->cor->hash[h][w];
        
        for (i=0;i<ca->len;i++){
            cp = ca->data+i;
            
            w2=(MY_PI*(double)cp->w)/18000;
            h2=(MY_PI*(double)cp->h)/18000;

            hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
            x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
            y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
            
            if (cp->c<0) {
                oldx = x;
                oldy = y;
                switch((int)cp->c){
                    case -128: /* coast */
                        color=makecol(0, 255, 255);
                        break;
                    case -127: /* isle */
                        color=makecol(0, 255, 255);
                        break;
                    case -126: /* lake */
                        color=makecol(0, 195, 195);
                        break;
                    case -125: /* border */
                        color=makecol(0, 255, 255);
                        break;
                    default:
                        dbg("unknown char %d\n", (int)cp->c);    
                        return 0;
                        break;
                        
                }
                continue;
            }

            c = color;
            if (abs(oldx-x) + abs(oldy-y) < 100) 
                line(gfx->map, oldx, oldy, x, y, c);

            oldx = x;
            oldy = y;
            
            /*fast_putpixel(gfx->map, x+1, y+1, makecol(255,0,0));*/
        
        }
/*        SDL_UpdateRect(gfx->screen, 0, 0, 0, 0);*/
        
    }


#if 0    
    oldx=oldy=color=-1;
    
    h1 = gfx->myh;
    w1 = gfx->myw;
    
    for (h=0; h<COR_H; h++)
    for (w=0; w<COR_W; w++){
/*        if ( not_on_screen(gfx, (h+1-COR_H/2)*10,   (w+1-COR_W/2)*10   ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10+9, (w+1-COR_W/2)*10   ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10,   (w+1-COR_W/2)*10+9 ) &&
             not_on_screen(gfx, (h+1-COR_H/2)*10+9, (w+1-COR_W/2)*10+9 )
           ) continue;*/
        
        ca = gfx->cor->hash[h][w];
        
        for (i=0;i<ca->len;i++){
            cp = ca->data+i;
            
            w2=-(MY_PI*(double)cp->w)/18000;
            h2=MY_PI+(MY_PI*(double)cp->h)/18000;

            hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
            x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
            y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
            
            if (cp->c<0) {
                oldx = x;
                oldy = y;
                switch((int)cp->c){
                    case -128: /* coast */
                        color=makecol(255, 0, 0);
                        break;
                    case -127: /* isle */
                        color=makecol(255, 0, 0);
                        break;
                    case -126: /* lake */
                        color=makecol(255, 0, 0);
                        break;
                    case -125: /* border */
                        color=makecol(255, 0, 0);
                        break;
                    default:
                        dbg("unknown char %d\n", (int)cp->c);    
                        return 0;
                        break;
                        
                }
                continue;
            }

            c = color;
            if (abs(oldx-x) + abs(oldy-y) < 100) 
                line(gfx->map, oldx, oldy, x, y, c);

            oldx = x;
            oldy = y;
        }
/*        SDL_UpdateRect(gfx->screen, 0, 0, 0, 0);*/
        
    }
#endif    
    return 0;
}

int plot_gridsq(int h, int w){ /* -89..90, -180..180 left down corner then clockwise */
    double ww0, ww1, hh0, hh1, ww5, hh5, qrb, qtf;
    int x[4], y[4], tx[4], ty[4], x5, y5;
    double a,b,c, v;
    int i,j, in;
    unsigned int col;
    char s[10];
    
    hh0 = h*MY_PI/180;
    ww0 = w*MY_PI/180;
    hh1 = (h+2)*MY_PI/180;
    ww1 = (w+1)*MY_PI/180;
    hh5 = (h+1)*MY_PI/180;
    ww5 = (w+0.5)*MY_PI/180;
    
    hw2qrbqtf(gfx->myh, gfx->myw, hh0, ww0, &qrb, &qtf);
    x[0] = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
    y[0] = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
    
    hw2qrbqtf(gfx->myh, gfx->myw, hh0, ww1, &qrb, &qtf);
    x[1] = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
    y[1] = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
    
    hw2qrbqtf(gfx->myh, gfx->myw, hh1, ww1, &qrb, &qtf);
    x[2] = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
    y[2] = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
    
    hw2qrbqtf(gfx->myh, gfx->myw, hh1, ww0, &qrb, &qtf);
    x[3] = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
    y[3] = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
    
    mkwwl4(s, w, h/2);

    /* FIXME buggy */
    if (CLIPPED(gfx->map, x[1], y[1]) && 
        CLIPPED(gfx->map, x[2], y[2]) && 
        CLIPPED(gfx->map, x[3], y[3]) &&
        CLIPPED(gfx->map, x[4], y[4])) return 1;
    
    
    col=gfx->mapbg;
    if (gfx->workedwwl &&
        aband && aband->stats && aband->stats->wwls &&
        g_hash_table_lookup(aband->stats->wwls, s)){
            col=gfx->wkmap;
    }
    /*triangle(gfx->map, x[0],y[0], x[1],y[1], x[2],y[2], col);
    triangle(gfx->map, x[0],y[0], x[3],y[3], x[2],y[2], col);*/
    triangle(gfx->map, x[0],y[0], x[2],y[2], x[1],y[1], col);
    triangle(gfx->map, x[0],y[0], x[2],y[2], x[3],y[3], col);

    if (!gfx->wwlnames) return 1;
    
    hw2qrbqtf(gfx->myh, gfx->myw, hh5, ww5, &qrb, &qtf);
    x5 = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
    y5 = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
    
    
    in=1;
    tx[0]=x5-fontwidth(s)/2;
    ty[0]=y5-fontheight(s)/2;
    tx[2]=x5+fontwidth(s)/2;
    ty[2]=y5+fontheight(s)/2;
    tx[1]=tx[0];
    ty[1]=ty[2];
    tx[3]=tx[2];
    ty[3]=ty[0];
    
    
    for (j=0; j<4; j++){
        for (i=0; i<4; i++){
            a=y[i] - y[(i+1)%4];
            b=x[(i+1)%4] - x[i];
            c=y[(i+1)%4] * x[i] - x[(i+1)%4] * y[i];
            
            v=a*tx[j]+b*ty[j]+c;
            if (v<0) {
                in=0;
                goto govno;
            }
        } 
    }
    
    fontout(gfx->map, x5, y5, gfx->gr[12], FONT_TRANSP|FONT_CENTER, s);
    
    
govno:;    
    return 0;
}

int qrv_int,qrv_now;

void plot_one_qrv(gpointer key, gpointer value, gpointer user_data){
    struct cw_item *cwi;
    double a,qrb,qtf;
    int x,y;
    
    cwi  = (struct cw_item *) value;
    if ((cwi->qrv&qrv_int)==0) return;
    if (!cwi->wwl0) return;
    qrbqtf(ctest->pwwlo, cwi->wwl0, &qrb, &qtf, NULL);
    a = qtf*MY_PI/180.0;
    x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(a);
    y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(a);
    fast_putpixel(gfx->map, x, y, gfx->yellow); 
    if (cwi->stamp0+10010 < qrv_now) return;
    fast_putpixel(gfx->map, x+1, y, gfx->yellow); 
    fast_putpixel(gfx->map, x-1, y, gfx->yellow); 
    fast_putpixel(gfx->map, x, y+1, gfx->yellow); 
    fast_putpixel(gfx->map, x, y-1, gfx->yellow); 
    

}
int plot_qrv(void){
    time_t now;
    struct tm *tm;

    if (!ctest || !aband) return 1;
    qrv_int=1<<(aband->bandchar-'A');
    
    now = time(NULL);
    tm = gmtime(&now);
    qrv_now =  (tm->tm_year+1900)*10000 + (tm->tm_mon+1)*100 + tm->tm_mday;
    t_hash_table_foreach(cw->cw, plot_one_qrv, NULL);
    return 0;
}

int plot_gridsqs(void){
    int w, h;
    if (!gfx->zoom->showbigwwls) return 0;

    if (!gfx->wwls){

    }
    
    for (h=-180;h<180;h+=2){
        for (w=-90; w<90; w++){
            plot_gridsq(h, w);
        }
    }
    return 0;
}


int plot_grid(){
    double  h2, w2, qrb, qtf;
    double oldx=-1, oldy=-1, x, y;
    int init;
    int w, h;
    
    
    if (gfx->zoom->showbigwwls){
        for (w=-89; w<90; w+=1){
            init=0;
            for (h=-180; h<=180; h++){

                w2 = w*MY_PI/180;
                h2 = h*MY_PI/180;
                
                hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
                x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
                y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
                
                if (init){                       
                    if (w%10!=0 && abs(oldx-x)+abs(oldy-y) < 150) 
                        line(gfx->map, oldx, oldy, x, y, gfx->gr[9]);
/*                    if (h%2==0) plot_cross(x,y,makecol(255,255,255));*/
                }else{
                    init=1;
                }
                oldx = x;
                oldy = y;
            }
        }           
        
        for (h=-180; h<=180; h+=2){
            init=0;
            for (w=-90; w<=90; w++){
                w2 = ((double)w)*MY_PI/180.0;
                h2 = ((double)h)*MY_PI/180.0;

                hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
                x = gfx->o_x + gfx->m_x + (qrb/gfx->zoom->value)*sin(qtf);
                y = gfx->o_y + gfx->m_y - (qrb/gfx->zoom->value)*cos(qtf);
                
                if (init){                       
                    if (h%20!=0 && abs(oldx-x)+abs(oldy-y) <150) {
                        line(gfx->map, oldx, oldy, x, y, gfx->gr[9]);
                    }
                }else{
                    init=1;
                }
                oldx = x;
                oldy = y;
            }
        }       
    }

    for (w=-80; w<90; w+=10){
        init=0;
        for (h=-180; h<=180; h+=1){

            w2 = w*MY_PI/180;
            h2 = h*MY_PI/180;
            
            hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
            x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
            y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
                 
            if (init){                       
                if (abs(oldx-x)+abs(oldy-y) < 150) 
                    line(gfx->map, oldx, oldy, x, y, gfx->gr[12]);
            }else{
                init=1;
            }
            oldx = x;
            oldy = y;
        }
    }           
    
    for (h=-180; h<=180; h+=20){
        init=0;
        for (w=-90; w<=90; w++){

            w2 = ((double)w)*MY_PI/180.0;
            h2 = ((double)h)*MY_PI/180.0;

            hw2qrbqtf(gfx->myh, gfx->myw, h2, w2, &qrb, &qtf);
            x = gfx->o_x + gfx->m_x + qrb/gfx->zoom->value*sin(qtf);
            y = gfx->o_y + gfx->m_y - qrb/gfx->zoom->value*cos(qtf);
               
            if (init){                       
                if (abs(oldx-x)+abs(oldy-y) <150) 
                    line(gfx->map, oldx, oldy, x, y, gfx->gr[12]);
            }else{
                init=1;
            }
            oldx = x;
            oldy = y;
        }
    }       
    return 0;
}

int plot_star(){
    int i,col; 
    double qrb,qtf,r;
    double minqrb, maxqrb, minqtf, maxqtf;

    minqrb=20000.0;
    maxqrb=-1;
    minqtf=360;
    maxqtf=0;

    if (xy2qrbqtf(gfx, gfx->map->x, gfx->map->y, &qrb, &qtf)==0){
        if (qrb<minqrb) minqrb=qrb;
        if (qrb>maxqrb) maxqrb=qrb;
        if (qtf<minqtf) minqtf=qtf;
        if (qtf>maxqtf) maxqtf=qtf;
    }
    if (xy2qrbqtf(gfx, gfx->map->x+gfx->map->w, gfx->map->y, &qrb, &qtf)==0){
        if (qrb<minqrb) minqrb=qrb;
        if (qrb>maxqrb) maxqrb=qrb;
        if (qtf<minqtf) minqtf=qtf;
        if (qtf>maxqtf) maxqtf=qtf;
    }
    if (xy2qrbqtf(gfx, gfx->map->x, gfx->map->y+gfx->map->h, &qrb, &qtf)==0){
        if (qrb<minqrb) minqrb=qrb;
        if (qrb>maxqrb) maxqrb=qrb;
        if (qtf<minqtf) minqtf=qtf;
        if (qtf>maxqtf) maxqtf=qtf;
    }
    if (xy2qrbqtf(gfx, gfx->map->x+gfx->map->w, gfx->map->y+gfx->map->h, &qrb, &qtf)==0){
        if (qrb<minqrb) minqrb=qrb;
        if (qrb>maxqrb) maxqrb=qrb;
        if (qtf<minqtf) minqtf=qtf;
        if (qtf>maxqtf) maxqtf=qtf;
    }
    /*dbg("qrb=%5.3f..%5.3f  qtf=%5.3f..%5.3f\n", minqrb, maxqrb, minqtf*180/MY_PI, maxqtf*180/MY_PI);*/

    /*r=sqrt(sqr(gfx->map->w)+sqr(gfx->map->h))/2;*/
    r=maxqrb;
    if (r>MY_PI*R_EARTH/gfx->zoom->value)
        r=MY_PI*R_EARTH/gfx->zoom->value;
    
    for (i=0; i<18; i++){
        line(gfx->map, 
                gfx->o_x+gfx->m_x-r*cos((MY_PI*i)/18), 
                gfx->o_y+gfx->m_y-r*sin((MY_PI*i)/18), 
                gfx->o_x+gfx->m_x+r*cos((MY_PI*i)/18), 
                gfx->o_y+gfx->m_y+r*sin((MY_PI*i)/18), 
                i%3==0?gfx->gr[13]:gfx->gr[6]);
    }

    i=1;
    for (qrb=gfx->zoom->qrbring; 
         qrb/gfx->zoom->value<=maxqrb; 
         qrb+=gfx->zoom->qrbring, i++){

/*        dbg("qrb=%f RE=%f\n", qrb, MY_PI*R_EARTH);*/
        
        if (i%5==0) col=gfx->gr[13];
        else col=gfx->gr[8];
        
        if (qrb>MY_PI*R_EARTH) {
            qrb=MY_PI*R_EARTH;
            col=gfx->gr[13];
            break;
        }
        circle(gfx->map, 
               gfx->o_x + gfx->m_x, 
               gfx->o_y + gfx->m_y, 
               qrb/gfx->zoom->value, col);
        if (qrb==MY_PI*R_EARTH) break;
    }
    return 0;
            
}

void plot_cross(int x, int y, int color){
    
    line(gfx->map, x-4, y-4, x+4, y+4, color);
    line(gfx->map, x-3, y-4, x+4, y+3, color);
    line(gfx->map, x-4, y-3, x+3, y+4, color);
    line(gfx->map, x-4, y+4, x+4, y-4, color);
    line(gfx->map, x-3, y+4, x+4, y-3, color);
    line(gfx->map, x-4, y+3, x+3, y-4, color);
}

int plot_qso(struct qso *qso){
    int x, y;

    if (!qso) {
        plot_info(NULL);
        plot_qrb_qth();
        return 0;
    }
    
    compute_cache(qso);
    x = gfx->o_x + qso->gfx_x;  
    y = gfx->o_y + qso->gfx_y;   

    /*dbg("plot_qso:   %s=%d %d  %d\n", qso->callsign, x, y, qso==gfx->minq);*/
    

    if (qso==gfx->minq){
        plot_cross(x, y, gfx->red);
        plot_info(qso);
        if (gfx->lines) line(gfx->map, gfx->m_x+gfx->o_x, gfx->m_y+gfx->o_y, x, y, gfx->red);
    }else{
        if (gfx->lines) line(gfx->map, gfx->m_x+gfx->o_x, gfx->m_y+gfx->o_y, x, y, gfx->yellow);
        plot_cross(x, y, gfx->yellow);
    }
    gfx->dirty=1;
    
    return 0;
}

int plot_qsos(void){
    gfx->minq=NULL;
    if (aband){
        int i;
        for (i=0; i<aband->qsos->len; i++){
            struct qso *qso;
            qso = (struct qso *)g_ptr_array_index(aband->qsos, i);
            if (qso->error) continue;

            plot_qso(qso);
        }
        gfx->minq=find_nearest(aband, 
                               mouse_x - gfx->o_x, 
                               mouse_y - gfx->o_y);
        if (gfx->minq) plot_qso(gfx->minq);
    }
    return 0;
}

int plot_info(struct qso *qso){
    SDL_Rect rect;
    char s[10];
 
    rect.x=gfx->info->x;
    rect.y=gfx->info->y + 4;
    rect.w=gfx->info->w;
    rect.h=9*18;
    SDL_FillRect(gfx->screen, &rect, makecol(0,0,35)) ;
    if (!qso) return 0;

    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 4,      gfx->green, FONT_TRANSP, TEXT(T_GCALL), q0(qso->callsign));
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 1*18+4, gfx->green, FONT_TRANSP, TEXT(T_GWWL),  q0(qso->locator));
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 2*18+4, gfx->green, FONT_TRANSP, TEXT(T_GQRB),  (int)qso->qrb);
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 3*18+4, gfx->green, FONT_TRANSP, TEXT(T_GQTF),  qso->qtf);
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 4*18+4, gfx->green, FONT_TRANSP, TEXT(T_GSEN),  q0(qso->rsts), q0(qso->qsonrs));
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 5*18+4, gfx->green, FONT_TRANSP, TEXT(T_GRCV),  q0(qso->rstr), q0(qso->qsonrr));
    /*fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 6*18+4, gfx->green, FONT_TRANSP, TEXT(T_GTIM),  q0(qso->time_str));*/
    strcpy(s,qso->time_str);
    s[5]=s[4];
    s[4]=s[3];
    s[3]=s[2];
    s[2]='.';
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 6*18+4, gfx->green, FONT_TRANSP, TEXT(T_GTIM),  s);
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 7*18+4, gfx->green, FONT_TRANSP, TEXT(T_GOPE),  q0(qso->operator));
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 8*18+4, gfx->green, FONT_TRANSP, TEXT(T_GREM),  q0(qso->remark));
    return 0;                           
}

int plot_qrb_qth(){
    double qrb,qtf, h2, w2; 
    SDL_Rect rect;
    char buf[100];
 
/*    dbg("plot_qtb_qtf\n");*/
    rect.x=gfx->info->x;
    rect.y=gfx->info->y + 10*18+4;
    rect.w=gfx->info->w;
    rect.h=6*18;
    
    xy2qrbqtf(gfx, mouse_x, mouse_y, &qrb, &qtf);
    
    SDL_FillRect(gfx->screen, &rect, makecol(0,0,35)) ;
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 10*18+4, gfx->green, FONT_TRANSP, TEXT(T_GQRB), (int)(qrb));
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 11*18+4, gfx->green, FONT_TRANSP, TEXT(T_GQTF), (int)(qtf*180/MY_PI));
    /*fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 12*18+4, gfx->green, FONT_TRANSP, TEXT(T_GINC), (int) gst[(int)(qtf*180/MY_PI)%360]*maxcnt/400);*/
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 12*18+4, gfx->green, FONT_TRANSP, TEXT(T_GINC), (int) (gst[(int)(qtf*180/MY_PI)%360]/400*100));

    if (qrbqtf2hw(gfx->myh, gfx->myw, qrb, qtf, &h2, &w2)==0){
        fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 13*18+4, gfx->green, FONT_TRANSP, TEXT(T_GLON), h2*180.0/MY_PI, x2gramin(buf, h2*180/MY_PI, "EW"));
        fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 14*18+4, gfx->green, FONT_TRANSP, TEXT(T_GLAT), w2*180.0/MY_PI, x2gramin(buf, w2*180/MY_PI, "NS"));
        fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 15*18+4, gfx->green, FONT_TRANSP, TEXT(T_GLOC), hw2loc(buf, h2*180/MY_PI, w2*180/MY_PI));
    }
    
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 17*18+4, gfx->green, FONT_TRANSP, "Borders, Gain, ");
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 18*18+4, gfx->green, FONT_TRANSP, "worKedwwl, Lines, ");
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 19*18+4, gfx->green, FONT_TRANSP, "wwlNames, qsOs, ");
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 20*18+4, gfx->green, FONT_TRANSP, "Printing, Rotars,");
    fontoutf(gfx->info, gfx->info->x+4, gfx->info->y+ 21*18+4, gfx->green, FONT_TRANSP, "Star, qrV, Wwls ");
    
    
    gfx->dirty=1;

    return 0;
}


void plot_rot(struct rotar *rot){
    double rad,r;
    int vx,vy,vx1,vy1,vx2,vy2;

    if (!rot) return;

    if (gfx->map->w>gfx->map->h)
        r=gfx->map->h/2.3;
    else
        r=gfx->map->w/2.3;

    /*r=sqrt(sqr(gfx->map->w)+sqr(gfx->map->h))/2;*/
    /*r=200;*/
    rad=MY_PI*(double)(rot->qtf-90)/180.0;
    vx=gfx->o_x+gfx->m_x+r*cos(rad); 
    vy=gfx->o_y+gfx->m_y+r*sin(rad);
    vx1=gfx->o_x+gfx->m_x+0.95*r*cos(rad+0.02),
    vy1=gfx->o_y+gfx->m_y+0.95*r*sin(rad+0.02),
    vx2=gfx->o_x+gfx->m_x+0.95*r*cos(rad-0.02),
    vy2=gfx->o_y+gfx->m_y+0.95*r*sin(rad-0.02),
    
    line(gfx->map, 
            gfx->o_x+gfx->m_x, 
            gfx->o_y+gfx->m_y, 
            vx,vy,rot->color);
    triangle(gfx->map,vx,vy,vx1,vy1,vx2,vy2,rot->color);
    
    if (rot->beamwidth<=0) return;
    
    rad=MY_PI*(double)(rot->qtf-90+rot->beamwidth/2)/180.0;
    vx1=gfx->o_x+gfx->m_x+0.9*r*cos(rad); 
    vy1=gfx->o_y+gfx->m_y+0.9*r*sin(rad);
    dashcnt=0;
    do_line(gfx->map, 
            gfx->o_x+gfx->m_x, 
            gfx->o_y+gfx->m_y, 
            vx1,vy1,rot->color,dashfce);

    rad=MY_PI*(double)(rot->qtf-90-rot->beamwidth/2)/180.0;
    vx2=gfx->o_x+gfx->m_x+0.9*r*cos(rad); 
    vy2=gfx->o_y+gfx->m_y+0.9*r*sin(rad);
    dashcnt=0;
    do_line(gfx->map, 
            gfx->o_x+gfx->m_x, 
            gfx->o_y+gfx->m_y, 
            vx2,vy2,rot->color,dashfce);
    
    do_line(gfx->map, vx1, vy1, vx2, vy2, rot->color, dashfce);

    
    
}

void test_font(void){
    int i,j;
    char s[2];

    s[1]='\0';
    for (i=0;i<16;i++)
        for (j=0;j<16;j++){
            s[0]=i*16+j;
            fontout(gfx->scr, 16+j*10, 16+i*20, gfx->gr[15], 0, s);
        }
            
}

#define RECT(bmp) rect(gfx->scr, gfx->bmp->x-1, gfx->bmp->y-1, gfx->bmp->x + gfx->bmp->w, gfx->bmp->y + gfx->bmp->h, gfx->gr[15])

int gfx_redraw(){
    gchar *c;
    int i;
    struct rotar *rot;

    fontout(gfx->scr, 
        gfx->map->x +  (gfx->map->w + fontwidth(TEXT(T__PLESE_WAIT_)))/2, 
            gfx->map->y +  (gfx->map->h + fontheight(TEXT(T__PLESE_WAIT_)))/2,  
            gfx->gr[15], FONT_CENTER|FONT_SYNC, TEXT(T__PLESE_WAIT_));
    
    if (!gfx->zoom->bkg){
    
        SDL_FillRect(gfx->screen, NULL, gfx->gr[8]); 
        SDL_FillRect(gfx->screen, gfx->info, makecol(0,0,35));
        plot_info(NULL);
        plot_qrb_qth();

        SDL_FillRect(gfx->screen, gfx->map, gfx->mapbg);
        SDL_FillRect(gfx->screen, gfx->title, makecol(35,0,0));
/*      rect(gfx->scr, gfx->title->x-1, gfx->title->y-1, gfx->title->x + gfx->title->w, gfx->title->y + gfx->title->h, gfx->gr[15]);*/
        RECT(title);
        RECT(map); 
        RECT(info);
        rect(gfx->scr, 0, 0, gfx->screen->w-1, gfx->screen->h-1, gfx->gr[15]);
    
        if (gfx->star) plot_star();
        if (gfx->cors) plot_cor();     
        
        gfx->zoom->bkg = SDL_CreateRGBSurface(SDL_SWSURFACE, gfx->map->w, gfx->map->h, gfx->bpp, 
                gfx->screen->format->Rmask, gfx->screen->format->Gmask, gfx->screen->format->Gmask, 0);
/*        gfx->zoom->bkg = SDL_CreateRGBSurface(0, gfx->map->w, gfx->map->h, gfx->bpp, 0, 0, 0, 0);*/
        if (gfx->screen->format->palette){
            SDL_SetColors(gfx->zoom->bkg, gfx->colors, 0, 256);
        }
        SDL_BlitSurface(gfx->screen, gfx->map, gfx->zoom->bkg, NULL);
        
    }else{
        SDL_FillRect(gfx->screen, gfx->map, gfx->mapbg);
    }
/*    SDL_BlitSurface(gfx->zoom->bkg, NULL, gfx->screen, gfx->map);*/
    
    SDL_FillRect(gfx->screen, gfx->title, makecol(35,0,0));
    plot_gridsqs();
    plot_takeoff();
    if (gfx->wwls) plot_grid();
    if (gfx->qrv) plot_qrv();
    
    c = g_strdup_printf("%dx%d-%d zoom=%d (%3.1fx)", gfx->screen->w, gfx->screen->h, gfx->bpp, gfx->zoom->id, gfx->zoom->value);
    fontout(gfx->title, gfx->title->w-8, 5, gfx->gr[15], FONT_RIGHT|FONT_TRANSP, c);
    g_free(c);
    
    if (title_str)
        fontout(gfx->title, 8, 5, gfx->gr[15], FONT_TRANSP, title_str);
    else
        fontout(gfx->title, 8, 5, gfx->gr[15], FONT_TRANSP, "Tucnak");

    compute_gfxstats(aband);
    if (gfx->gain) plot_gfxstats();
   
    /* blit with alpha-blending */
    SDL_SetAlpha(gfx->zoom->bkg, SDL_SRCALPHA, 128);
    SDL_BlitSurface(gfx->zoom->bkg, NULL, gfx->screen, gfx->map);
    if (gfx->qsos) plot_qsos();
    
    SDL_BlitSurface(gfx->screen, gfx->map, gfx->mapcache, NULL);
    for (i=0;i<rotars->len;i++){
        rot=(struct rotar *)g_ptr_array_index(rotars, i);
        if (gfx->shrot) plot_rot(rot);
    }
    gfx->dirty=1;
    /*test_font();*/
    return 0;
}

void redraw_rot(void){
    int i;
    struct rotar *rot;
    
    SDL_BlitSurface(gfx->mapcache, NULL, gfx->screen, gfx->map);
    for (i=0;i<rotars->len;i++){
        rot=(struct rotar *)g_ptr_array_index(rotars, i);
        if (gfx->shrot) plot_rot(rot);
    }
    gfx->dirty=1;
}

void gfx_update_qth(){
    int recalc=0;

    if (!gfx) return;

    CONDGFREE(gfx->pwwlo);
    
    if (!gfx->pwwlo && ctest && ctest->pwwlo) {
        gfx->pwwlo = g_strdup(ctest->pwwlo);
        recalc=1;
    }
       
    if (!gfx->pwwlo && cfg && cfg->pwwlo) {
        gfx->pwwlo = g_strdup(cfg->pwwlo);
        recalc=1;
    }
    
    
    if (gfx->pwwlo && ctest && ctest->pwwlo && strcmp(gfx->pwwlo, ctest->pwwlo)!=0){
        g_free(gfx->pwwlo);
        gfx->pwwlo = g_strdup(ctest->pwwlo);
        recalc=1;
    }

    if (recalc){
        gfx->o_x=0; 
        gfx->o_y=0; 
        gfx->myh = qth(gfx->pwwlo, 0);
        gfx->myw = qth(gfx->pwwlo, 1);
        invalidate_bkg(NULL);
    }
}

void mouse_moved(){
    struct qso *tmpq;
    
/*    dbg("mouse_moved\n");*/
    
    tmpq=gfx->minq;
    gfx->minq=find_nearest(aband, 
                           mouse_x - gfx->o_x, 
                           mouse_y - gfx->o_y);
    if (tmpq!=gfx->minq){
        if (gfx->qsos){
            plot_qso(tmpq);
            plot_qso(gfx->minq);
            gfx->dirty=1;
        }else{
            plot_info(gfx->minq);
        }
    }
    plot_qrb_qth();
    
}
void gfx_screenshot(void){
    gchar *c; 
    int i;

    for (i=0;1;i++){  
        struct stat st;

        if (ctest)
            c=g_strdup_printf("%s/snap%c%d.%s",ctest->directory,aband->bandchar,i, SHOTEXT);
        else
            c=g_strdup_printf("%s/tucnak/snap%d.%s",getenv("HOME"),i,SHOTEXT);
        if (stat(c, &st)!=0) break;
        g_free(c);
    }
    if (SHOTSAVE(gfx->screen, c)){
        log_addf(TEXT(T_CANT_WRITE_S), c);
        errbox(TEXT(T_CANT_WRITE), errno);
    }
    g_free(c);
}


int update_mouse=0;

int FilterEvents(const SDL_Event *ev) {
    switch (ev->type){
        case SDL_MOUSEMOTION:
            update_mouse=1;
            /*dbg("update_mouse=1\n");*/
            return 0;
            break;
    }

            
    return 1;
}
int gfx_func(void *xxx){
    static int old_mx, old_my;
    double oldzoom;
    SDL_Event event, *ev;
    int move_pixels;
    
    
    
    if (!gfx) return 0;
    ev=&event;
    
    gfx->func_timer_id = 0;
    
    if (SDL_PollEvent(ev)) {
    
/*      dbg("event: %d\n", ev->type);*/
        switch (ev->type){
            case SDL_KEYDOWN:
                move_pixels=100;
                if (ev->key.keysym.mod & (KMOD_LSHIFT|KMOD_RSHIFT)) move_pixels=20;
/*                dbg("Pressed %d 0x%x\n", ev->key.keysym.sym, ev->key.keysym.sym);*/
                switch(ev->key.keysym.sym){
                    case SDLK_F1:
                        gfx_screenshot();
                        break;
                    case SDLK_UP:
                        gfx->o_y+=move_pixels; 
                        invalidate_bkg(NULL);
                        update_mouse=1;
                        break;  
                    case SDLK_DOWN:
                        gfx->o_y-=move_pixels; 
                        invalidate_bkg(NULL);
                        update_mouse=1;
                        break;  
                    case SDLK_LEFT:
                        gfx->o_x+=move_pixels; 
                        invalidate_bkg(NULL);
                        update_mouse=1;
                        break;  
                    case SDLK_RIGHT:
                        gfx->o_x-=move_pixels; 
                        invalidate_bkg(NULL);
                        update_mouse=1;
                        break;  
                    case SDLK_EQUALS:
                    case SDLK_KP_PLUS:
                        if (zooms[gfx->zoomint+1].id==0) break;
                        oldzoom=gfx->zoom->value;
                        gfx->zoomint++;
                        gfx->zoom = zooms+gfx->zoomint;
                        gfx->draw = 1;
                        gfx->o_x*=oldzoom/gfx->zoom->value;
                        gfx->o_y*=oldzoom/gfx->zoom->value;
                        invalidate_cache(aband);
                        update_mouse=1;
                        break;
                    case SDLK_MINUS:
                    case SDLK_KP_MINUS:
                        if (zooms[gfx->zoomint-1].id==0) break;
                        oldzoom=gfx->zoom->value;
                        gfx->zoomint--;
                        gfx->zoom = zooms+gfx->zoomint;
                        gfx->draw = 1;
                        gfx->o_x*=oldzoom/gfx->zoom->value;
                        gfx->o_y*=oldzoom/gfx->zoom->value;
                        invalidate_cache(aband);
                        update_mouse=1;
                        break;
                    case SDLK_b:
                        gfx->cors=!gfx->cors;    
                        invalidate_bkg(gfx->zoom);
                        break;
                    case SDLK_c:
                        gfx->o_x=0; 
                        gfx->o_y=0; 
                        invalidate_cache(aband);
                        invalidate_bkg(NULL);
                        update_mouse=1;
                        gfx->draw = 1;
                        break;
                    case SDLK_f:
                        SDL_WM_ToggleFullScreen(gfx->screen);
                        break;
                    case SDLK_g:
                        gfx->gain=!gfx->gain;    
                        gfx->draw=1;    
                        break;
                    case SDLK_k:
                        gfx->workedwwl=!gfx->workedwwl;
                        gfx->draw=1;    
                        break;
                    case SDLK_l:
                        gfx->lines=!gfx->lines;    
                        gfx->draw=1;    
                        break;
                    case SDLK_n:
                        gfx->wwlnames=!gfx->wwlnames;    
                        gfx->draw=1;    
                        break;
                    case SDLK_o:
                        gfx->qsos=!gfx->qsos;    
                        gfx->draw=1;    
                        break;
                    case SDLK_p:
                        for_printing(gfx, 1);
                        gfx->draw = 1;
                        break;
                    case SDLK_q:
                        gfx->konec=1;
                        break;
                    case SDLK_r:
                        gfx->draw=1;
                        break;    
                    case SDLK_s:
                        gfx->star=!gfx->star;    
                        invalidate_bkg(gfx->zoom);
                        break;
                    case SDLK_t:
                        gfx->shrot=!gfx->shrot;;
                        gfx->draw=1;
                        break;    
                    case SDLK_v:
                        gfx->qrv=!gfx->qrv;
                        gfx->draw=1;    
                        break;
                    case SDLK_w:
                        gfx->wwls=!gfx->wwls;    
                        gfx->draw=1;    
                        break;
                    case SDLK_TAB:
                        rxtx();
                        break;
                    case SDLK_ESCAPE:
                        rx();
                        break;
                    case SDLK_F5:
                        cq_run_by_number(0);
                        break;
                    case SDLK_F6:
                        cq_run_by_number(1);
                        break;
                    case SDLK_F7:
                        cq_run_by_number(2);
                        break;
                    case SDLK_F8:
                        cq_run_by_number(3);
                        break;
                    case SDLK_F11:
                        cq_run_by_number(4);
                        break;
                    case SDLK_F12:
                        cq_run_by_number(5);
                        break;
                    default: /* for warning */
                        break;
                }
                break /* SDL_KEYDOWN */;
            case SDL_VIDEORESIZE:
                SetVideoMode(ev->resize.w, ev->resize.h);
                break;
			case SDL_QUIT:
                gfx->konec=1;
				break;
				
                
        }
    }

    if (gfx->draw) 
        gfx_redraw();
    else{
        if (gfx->dirtyrot) redraw_rot();
    }
    gfx->dirtyrot=0;

    if (update_mouse){
        update_mouse=0;
        SDL_GetMouseState(&mouse_x, &mouse_y);
        if (mouse_x!=old_mx || mouse_y!=old_my){
            mouse_moved();
            old_mx=mouse_x;
            old_my=mouse_y;
        }
    }
            
    
    if (gfx->dirty){
        SDL_UpdateRect(gfx->screen, 0, 0, 0, 0);
        gfx->draw=0;
        gfx->dirty=0;
    }

    if (gfx->konec) {
        free_gfx();
        gfx=NULL;
        return -1;
    }
    gfx->func_timer_id = install_timer_(100, (void (*)(void *)) gfx_func, (void *)gfx);
    return 0;
}

int gfx_reload(struct gfx *gfx){
    if (!gfx) return 0;
    /*dbg("gfx_reload\n");*/
    gfx_update_qth();
    invalidate_bkg(NULL);
    invalidate_cache(aband);
    clear_gfxstats();
    gfx->dirty=1;
    gfx->draw=1;
    return 0;
}

int gfx_add_qso(struct qso *qso){
    if (!gfx || !gfx->qsos) return 0;
    plot_qso(qso);
    gfx->dirty=1;
    return 0;
}

int gfx_set_title(gchar *title){
    
    CONDGFREE(title_str);
    title_str=g_strdup_printf("Map: %s",title);

    if (!gfx) return 0;
    SDL_WM_SetCaption(title, NULL);
    gfx->draw=1;
    return 0;
}



int gstcnt1[360];
double gst[360];
double maxcnt;
/*double gsttmp[360];*/
double antchar[181];
double beamwidth;

void compute_gfxstats(struct band *b){
    int i,j;
    struct qso *q;
    

    beamwidth=15;
    for (i=0;i<=180;i++){
        antchar[i]=cos(MY_PI/180.0*(double)i*120.0/beamwidth);
       /* dbg("i=%3d %5.3f\n", i, antchar[i]);*/
        if (antchar[i]<=0) break;
    }
    for (;i<=180;i++) antchar[i]=0;

   
    memset(gstcnt1,0,sizeof(gstcnt1));
    
    if (!b) return;

    for (i=0; i<b->qsos->len; i++){
        q = (struct qso *)g_ptr_array_index(b->qsos, i);
        if (q->error) continue;
        if (q->qtf<0 || q->qtf>=360) continue;
        /*gstcnt1[q->qtf/10]+=q->qrb;*/
        for (j=0;j<=180;j++){
            int qtf;
            qtf=q->qtf+j;
            if (qtf<0) qtf+=360;
            gst[qtf%360]+=q->qrb*antchar[j];
        }
        for (j=1;j<=180;j++){
            int qtf;
            qtf=q->qtf-j;
            if (qtf<0) qtf+=360;
            gst[qtf%360]+=q->qrb*antchar[j];
        }
    }
    maxcnt=0;
    for (i=0; i<360;i++) {
        if (gst[i]>maxcnt) maxcnt=gst[i];
    }
    if (maxcnt){
        for (i=0; i<360;i++) {
            gst[i]=400*gst[i]/maxcnt;;
        }
    }
}

void clear_gfxstats(void){
    int i;
    for (i=0; i<360;i++){ gst[i]=0; }
}

void putsqr(SDL_Rect *clip, int x, int y, int d){
    fast_putpixel(clip, x-1, y-1, d);
    fast_putpixel(clip, x-1, y  , d);
    fast_putpixel(clip, x-1, y+1, d);
    fast_putpixel(clip, x  , y-1, d);
    fast_putpixel(clip, x  , y  , d);
    fast_putpixel(clip, x  , y+1, d);
    fast_putpixel(clip, x+1, y-1, d);
    fast_putpixel(clip, x+1, y  , d);
    fast_putpixel(clip, x+1, y+1, d);
}

void plot_gfxstats(void){
    int i, oldx,oldy, x,y;
    
    if (!maxcnt) return;
    oldx = gfx->o_x + gfx->m_x + (gst[359]/gfx->zoom->value)*sin(359*MY_PI/180);
    oldy = gfx->o_y + gfx->m_y - (gst[359]/gfx->zoom->value)*cos(359*MY_PI/180);
    for (i=0; i<360;i++) {
        x = gfx->o_x + gfx->m_x + gst[i]/gfx->zoom->value*sin(i*MY_PI/180);
        y = gfx->o_y + gfx->m_y - gst[i]/gfx->zoom->value*cos(i*MY_PI/180);
        do_line(gfx->map, oldx,oldy, x,y, gfx->green, putsqr);
        oldx=x;
        oldy=y;
    }
}

void plot_takeoff(void){
    int i,j,col;
    
    col=makecol(255,0,0);
    for (i=0;i<cfg->takeoff->len;i++){
        struct takeoff *toff;
        double from,to,r1,r2;
        int x0,y0,x1,y1,x2,y2,x3,y3;
        
        if (gfx->map->w>gfx->map->h)
            r1=gfx->map->h/2.3+5;
        else
            r1=gfx->map->w/2.3+5;
        
        toff=(struct takeoff *)g_ptr_array_index(cfg->takeoff, i);
        for (j=toff->from;j<=toff->to;j++){

            r2 = r1 + toff->value;
            from = (j-0.5)*MY_PI/180;
            to   = (j+0.5)*MY_PI/180;
            
            x0 = gfx->o_x + gfx->m_x + r1 * sin(from);
            y0 = gfx->o_y + gfx->m_y - r1 * cos(from);
                
            x1 = gfx->o_x + gfx->m_x + r2 * sin(from);
            y1 = gfx->o_y + gfx->m_y - r2 * cos(from);
                
            x2 = gfx->o_x + gfx->m_x + r2 * sin(to);
            y2 = gfx->o_y + gfx->m_y - r2 * cos(to);
                
            x3 = gfx->o_x + gfx->m_x + r1 * sin(to);
            y3 = gfx->o_y + gfx->m_y - r1 * cos(to);
                
            
            triangle(gfx->map, x0,y0, x2,y2, x1,y1, col);
            triangle(gfx->map, x0,y0, x2,y2, x3,y3, col);
        }
    }
}

int for_printing(struct gfx *gfx, int on){
/*    int r;*/

    if (!on){
        gfx->cors = 1;
        gfx->gain = 1;
        gfx->lines = 0;
        gfx->qrv = 1;
        gfx->qsos = 1;
        gfx->shrot = 1;
        gfx->star = 1;
        gfx->workedwwl = 1;
        gfx->wwlnames = 1;
        gfx->wwls = 1;
        return 0;
    }
    gfx->o_x=0; 
    gfx->o_y=0; 
    invalidate_cache(aband);
    invalidate_bkg(NULL);
    update_mouse=1;
    
/*    gfx->zoomint=1;
    gfx->zoom = zooms+gfx->zoomint;
    if (gfx->map->w>gfx->map->h) r=gfx->map->h/2;
    else r=gfx->map->w/2;
    gfx->zoom->value=MY_PI*R_EARTH/(r-5);*/
    
    gfx->cors = 1;
    gfx->gain = 0;
    gfx->lines = 0;
    gfx->qrv = 0;
    gfx->qsos = 0;
    gfx->shrot = 0;
    gfx->workedwwl = 0;
    gfx->wwlnames = 0;
    gfx->wwls = 0;
    return 0;
}
                        


#else
struct gfx *init_gfx(void){
    return NULL;
}

void free_gfx(void){
    return;
}

int gfx_func(void *xxx){
    return 0;
}

int gfx_reload(void){
    return 0;
}

int gfx_set_title(gchar *title){
    return 0;
}

void gfx_update_qth(){
    return;
}

int gfx_add_qso(struct qso *qso){
    return 0;
}

#endif /* HAVE_SDL */
