#include "TypeInfo.hh"
#include "std_standardPkg.hh"
#include "UniversalReal.hh"

TypeInfo &
TypeInfo::getNullTypeInfo(){
  static TypeInfo NULL_TYPE_INFO;
  return NULL_TYPE_INFO;
}

VHDLType*
TypeInfo::createObject(ObjectBase::ObjectType) const {
  cerr << "Error - TypeInfo::createObject(ObjectBase::ObjectType) called.\n";
  return NULL;
}

int
TypeInfo::getCharacterOffset() const {
  cerr << "Error - TypeInfo::getPosition() const called.\n";
  return -1;
}

char**
TypeInfo::createImageMap(int count, ...) {
  va_list ap;
  int counter;
  char **retValue = new char*[count];
  
  va_start(ap, count);
  
  for(counter = 0; (counter < count); counter++) {
    char *temp = va_arg(ap, char*);
    retValue[counter] = new char[strlen(temp) + 1];
    strcpy(retValue[counter], temp);
  }
  
  va_end(ap);
  
  return retValue;
}

bool 
TypeInfo::is_character_type() const {
  // Overidden in enumeration type
  return false;
}

//The constructor for IntegerTypeInfo
IntegerTypeInfo::IntegerTypeInfo() {
  left  = 0;
  right = -1;
  dirn  = ArrayInfo::to;
}

IntegerTypeInfo::IntegerTypeInfo(const IntegerTypeInfo &ri) :
  TypeInfo() {
  left  = ri.left;
  dirn  = ri.dirn;
  right = ri.right;
}

IntegerTypeInfo::IntegerTypeInfo(int l, ArrayInfo::ArrayDirn_t d, int r){
  left  = l;
  dirn  = d;
  right = r;
}

IntegerTypeInfo::IntegerTypeInfo( IntegerType &lbound,
				  ArrayInfo::ArrayDirn_t d,
				  IntegerType &rbound ){
  left = lbound.getIntValue();
  dirn = d;
  right = rbound.getIntValue();
}

//Member functions of IntegerTypeInfo
int
IntegerTypeInfo::get_left() const{
  return left;
}

ArrayInfo::ArrayDirn_t 
IntegerTypeInfo::get_direction() const{
  return dirn;
}

int
IntegerTypeInfo::get_right() const{
  return right;
}

IntegerTypeInfo&
IntegerTypeInfo::operator=(const IntegerTypeInfo &ri) {
  left  = ri.left;
  dirn  = ri.dirn;
  right = ri.right;
  
  return *this;
}

int
IntegerTypeInfo::get_range() const {
  if (dirn == ArrayInfo::to) {
    return ((right - left) + 1);
  }

  return ((left - right) + 1);
}

VHDLType*
IntegerTypeInfo::createObject(ObjectBase::ObjectType objType) const {
  return new IntegerType(objType, *this);
}


//The constructor for RealTypeInfo
RealTypeInfo::RealTypeInfo() {
  left  = 0;
  right = -1;
  dirn  = ArrayInfo::to;
}

RealTypeInfo::RealTypeInfo(const RealTypeInfo &ri) :
  TypeInfo(){
  left  = ri.left;
  right = ri.right;
  dirn  = ri.dirn;
}

RealTypeInfo&
RealTypeInfo::operator=(const RealTypeInfo &ri) {
  left  = ri.left;
  right = ri.right;
  dirn  = ri.dirn;

  return *this;
}

RealTypeInfo::RealTypeInfo(double l, ArrayInfo::ArrayDirn_t d, double r){
  left  = l;
  dirn  = d;
  right = r;
}

RealTypeInfo::RealTypeInfo(RealType& lbound, ArrayInfo::ArrayDirn_t d, RealType& rbound){
  left = (UniversalReal&) lbound.readVal();
  dirn = d;
  right = (UniversalReal&)rbound.readVal();
}

//Member functions of RealTypeInfo
double
RealTypeInfo::get_left() const{
  return left;
}

ArrayInfo::ArrayDirn_t 
RealTypeInfo::get_direction() const{
  return dirn;
}

double
RealTypeInfo::get_right() const{
  return right;
}

VHDLType*
RealTypeInfo::createObject(ObjectBase::ObjectType objType) const {
  return new RealType(objType, *this);
}

//The constructors for PhysicalTypeInfo
PhysicalTypeInfo::PhysicalTypeInfo() {
  dirn  = ArrayInfo::to;
  left  = 0;
  right = -1;
  no_of_elements = 0;

  unitInfo  = NULL;
  scaleInfo = NULL;
}

PhysicalTypeInfo::PhysicalTypeInfo(const PhysicalTypeInfo &pi) :
  TypeInfo(){
  no_of_elements = pi.no_of_elements;
  unitInfo       = pi.unitInfo;
  scaleInfo      = pi.scaleInfo;
  dirn           = pi.dirn;
  left           = pi.left;
  right          = pi.right;
}

PhysicalTypeInfo&
PhysicalTypeInfo::operator=(const PhysicalTypeInfo &pi) {
  no_of_elements = pi.no_of_elements;
  unitInfo       = pi.unitInfo;
  scaleInfo      = pi.scaleInfo;
  dirn           = pi.dirn;
  left           = pi.left;
  right          = pi.right;

  return *this;
}

PhysicalTypeInfo::PhysicalTypeInfo(int n, LONG l, ArrayInfo::ArrayDirn_t d, LONG r,
		 LONG* sInfo, char** uInfo){
  dirn           = d;
  left           = l;
  right          = r;
  no_of_elements = n;
  unitInfo       = uInfo;
  scaleInfo      = sInfo;
}

int
PhysicalTypeInfo::get_no_of_elements() const {
  return no_of_elements;
}

LONG
PhysicalTypeInfo::get_left() const {
  return left;
}

LONG
PhysicalTypeInfo::get_right() const {
  return right;
}

ArrayInfo::ArrayDirn_t
PhysicalTypeInfo::get_direction() const {
  return dirn;
}

LONG*
PhysicalTypeInfo::get_scale_info() const {
  return scaleInfo;
}

char **
PhysicalTypeInfo::get_unit_info() const {
  return unitInfo;
}

char *
PhysicalTypeInfo::get_unit(int i) const{
  return unitInfo[i];
}

LONG
PhysicalTypeInfo::get_scale(int i) const {
  return scaleInfo[i];
}

LONG*
PhysicalTypeInfo::createScaleMap(int count, ...) {
  va_list ap;
  int counter;
  LONG *retValue = new LONG[count];
  
  va_start(ap, count);
  
  for(counter = 0; (counter < count); counter++) {
    LONG temp = va_arg(ap, LONG);
    retValue[counter] = temp;
  }
  
  va_end(ap);
  
  return retValue;
}


VHDLType*
PhysicalTypeInfo::createObject(ObjectBase::ObjectType objType) const {
  return new PhysicalType(objType, *this);
}

RecordTypeInfo::RecordTypeInfo() {
  fields     = NULL;
  noOfFields = 0;
}
  
RecordTypeInfo::RecordTypeInfo(int fieldCount, bool compositeFlag, ResolutionFnId_t resFn, ...) {
  noOfFields            = fieldCount;
  fields                = new TypeInfo*[noOfFields];
  compositeResolvedType = compositeFlag;
  resolutionFunctionId  = resFn;

  va_list ap;
  va_start(ap, resFn);
  int counter;

  for(counter = 0; (counter < fieldCount); counter++) {
    TypeInfo *temp  = va_arg(ap, TypeInfo *);
    fields[counter] = temp;
  }

  va_end(ap);
}

RecordTypeInfo::RecordTypeInfo(const RecordTypeInfo& rhs) :
  TypeInfo(){
  noOfFields            = rhs.get_number_of_fields();
  fields                = new TypeInfo*[noOfFields];
  compositeResolvedType = rhs.isCompositeResolvedType();
  resolutionFunctionId  = rhs.getResolutionFunctionId();

  int counter;

  for(counter = 0; (counter < noOfFields); counter++) {
    fields[counter] = rhs.get_field_info(counter);
  }
}

TypeInfo*
RecordTypeInfo::get_field_info(const int fieldId) const {
  if ((fieldId < 0) || (fieldId >= noOfFields)) {
    cerr << "Error: RecordTypeInfo::get_field_info(const int) const called with"
	 << " an field number that is not within range.\n";
    return NULL;
  }
  
  return fields[fieldId];
}

int
RecordTypeInfo::get_number_of_fields() const {
  return noOfFields;
}

RecordTypeInfo&
RecordTypeInfo::operator=(const RecordTypeInfo &rhs) {
  noOfFields            = rhs.get_number_of_fields();
  fields                = new TypeInfo*[noOfFields];
  compositeResolvedType = rhs.isCompositeResolvedType();
  resolutionFunctionId  = rhs.getResolutionFunctionId();
  
  int counter;
  
  for(counter = 0; (counter < noOfFields); counter++) {
    fields[counter] = rhs.get_field_info(counter);
  }

  return *this;
}

VHDLType*
RecordTypeInfo::createObject(ObjectBase::ObjectType objType) const {
  return new RecordType(objType, *this, -1);
}

VHDLType*
AccessTypeInfo::createObject(ObjectBase::ObjectType) const {
  return new AccessType();
}



/**
  This is here solely for detection of the library when searching via
  "configure".
*/
extern "C" {
  int isTyvis(){ return 0; }
}

