//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.0, Copyright (C) Peter A. Buhr 2003
// 
// mcpcom.cc -- 
// 
// Author           : Richard C. Bilson
// Created On       : Thu Feb 26 11:31:54 2004
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Jul 25 14:53:21 2005
// Update Count     : 6
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#include <stdio.h>					// tempnam, freopen, perror
#include <unistd.h>					// execvp, fork, unlink
#include <sys/wait.h>					// wait
#include <string>
using std::string;
#include <vector>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <iterator>

//#define __U_DEBUG_H__

#ifdef __U_DEBUG_H__
#include <iostream>
using std::cerr;
using std::endl;
#endif


string compiler_name( CCAPP );				// path/name of C compiler


void checkEnv( const char *args[], int &nargs ) {
    char *value;

    value = getenv( "__U_COMPILER__" );
    if ( value != NULL ) {
	compiler_name = value;
#ifdef __U_DEBUG_H__
	cerr << "env arg:\"" << compiler_name << "\"" << endl;
#endif // __U_DEBUG_H__
    } // if

    value = getenv( "__U_GCC_MACHINE__" );
    if ( value != NULL ) {
	args[nargs] = ( *new string( value ) ).c_str(); // pass the argument along
#ifdef __U_DEBUG_H__
	cerr << "env arg:\"" << args[nargs] << "\"" << endl;
#endif // __U_DEBUG_H__
	nargs += 1;
    } // if

    value = getenv( "__U_GCC_VERSION__" );
    if ( value != NULL ) {
	args[nargs] = ( *new string( value ) ).c_str(); // pass the argument along
#ifdef __U_DEBUG_H__
	cerr << "env arg:\"" << args[nargs] << "\"" << endl;
#endif // __U_DEBUG_H__
	nargs += 1;
    } // if
} // checkEnv


bool isprefix( string arg, string prefix ) {
    return arg.substr( 0, prefix.size() ) == prefix;
} // isprefix


template< class ArrayType >
void Stage1( const int argc, ArrayType &argv ) {
    int code;
    int i;

    string arg;
    string bprefix;

    const char *cpp_in = NULL;
    const char *cpp_out = NULL;
    const char *depend_out = NULL;

    bool upp_flag = false;
    bool cpp_flag = false;
    bool start = false;
    bool do_depends = false;

    const char *args[argc + 100];			// leave space for 100 additional cpp command line values
    int nargs = 1;					// number of arguments in args list; 0 => command name
    const char *uargs[20];				// leave space for 20 additional u++-cpp command line values
    int nuargs = 1;					// 0 => command name

    // process all the arguments

    checkEnv( args, nargs );				// arguments passed via environment variables

    for ( i = 1; i < argc; i += 1 ) {
#ifdef __U_DEBUG_H__
	cerr << "argv[" << i << "]:\"" << argv[i] << "\"" << endl;
#endif // __U_DEBUG_H__
	arg = argv[i];
#ifdef __U_DEBUG_H__
	cerr << "arg:\"" << arg << "\"" << endl;
#endif // __U_DEBUG_H__
	if ( arg.substr(0,1) == "-" ) {
	    if ( isprefix( arg, "-mGLOB_obj_output_file=" ) && cpp_out == NULL ) {
		cpp_out = &argv[ i ][ 23 ];
#ifdef __U_DEBUG_H__
		cerr << "cpp_out:\"" << cpp_out << "\""<< endl;
#endif // __U_DEBUG_H__

	    // strip u++ flags controlling cpp step

	    } else if ( arg == "-D__U_CPP__" ) {
		cpp_flag = true;
	    } else if ( arg == "-D__U_UPP__" ) {
		upp_flag = true;
	    } else if ( isprefix( arg, "-D__U_GCC_BPREFIX__" ) ) {
		bprefix = arg.substr( 19+1 );

	    // u++ flags controlling the u++-cpp step

	    } else if ( arg == "-D__U_YIELD__" || arg == "-D__U_VERIFY__" || arg == "-D__U_PROFILE__" ) {
		args[nargs] = argv[i];			// pass the flag along to cpp
		nargs += 1;
		uargs[nuargs] = argv[i];		// pass the flag along to upp
		nuargs += 1;

	    // all other flags

	    } else if ( arg == "-MD" ) {
		do_depends = true;
		i += 1;
#ifdef __U_DEBUG_H__
		cerr << "argv[" << i << "]:\"" << argv[i] << "\"" << endl;
#endif // __U_DEBUG_H__
	    } else {
		args[nargs] = argv[i];			// pass the flag along
		nargs += 1;
		// CPP flags with an argument
		if ( arg == "-I" || arg == "-MF" || arg == "-MT" || arg == "-MQ" ||
		     arg == "-include" || arg == "-imacros" || arg == "-idirafter" || arg == "-iprefix" ||
		     arg == "-iwithprefix" || arg == "-iwithprefixbefore" || arg == "-isystem" ) {
		    i += 1;
		    args[nargs] = argv[i];		// pass the argument along
		    nargs += 1;
#ifdef __U_DEBUG_H__
		    cerr << "argv[" << i << "]:\"" << argv[i] << "\"" << endl;
#endif // __U_DEBUG_H__
		} // if
	    } // if
	} else {					// obtain input and possibly output files
	    if ( cpp_in == NULL ) {
		cpp_in = argv[i];
#ifdef __U_DEBUG_H__
		cerr << "cpp_in:\"" << cpp_in << "\"" << endl;
#endif // __U_DEBUG_H__
	    } else if ( cpp_out == NULL ) {
		cpp_out = argv[i];
#ifdef __U_DEBUG_H__
		cerr << "cpp_out:\"" << cpp_out << "\""<< endl;
#endif // __U_DEBUG_H__
	    } else {
		fprintf( stderr, "Usage: %s input-file [output-file] [options]\n", argv[0] );
		exit( 1 );
	    } // if
	} // if
    } // for

#ifdef __U_DEBUG_H__
    cerr << "args:";
    for ( i = 1; i < nargs; i += 1 ) {
	cerr << " " << args[i];
    } // for
    if ( cpp_in != NULL ) cerr << " " << cpp_in;
    if ( cpp_out != NULL ) cerr << " " << cpp_out;
    cerr << endl;
#endif // __U_DEBUG_H__

    if ( cpp_in == NULL ) {
	fprintf( stderr, "Usage: %s input-file [output-file] [options]\n", argv[0] );
	exit( 1 );
    } // if

    if ( do_depends ) {
	string depend_file;
	if ( cpp_out != NULL ) {
	    depend_file = string( cpp_out );
	} else {
	    depend_file = string( cpp_in );
	} // if
	int lastdot = depend_file.rfind( '.' );
	int lastslash = depend_file.rfind( '/' );
	if ( lastdot != string::npos && ( lastslash == string::npos || lastdot > lastslash ) ) {
	    depend_file.replace( lastdot + 1, string::npos, "d" );
	} else {
	    depend_file += ".d";
	}
	depend_out = (new string( depend_file ))->c_str();
    } // if

    if ( cpp_flag ) {
	// The -E flag is specified on the u++ command so only run the
	// preprocessor and output is written to standard output. The call to
	// u++ has a -E so it does not have to be added to the argument
	// list.

	args[0] = compiler_name.c_str();
	args[nargs] = cpp_in;
	nargs += 1;
	if ( do_depends ) {
	    args[nargs] = "-MD";
	    nargs += 1;
	    args[nargs] = "-MF";
	    nargs += 1;
	    args[nargs] = depend_out;
	    nargs += 1;
	    args[nargs] = "-MT";
	    nargs += 1;
	    args[nargs] = cpp_out;
	    nargs += 1;
	} // if
	args[nargs] = NULL;				// terminate argument list

#ifdef __U_DEBUG_H__
	cerr << "nargs: " << nargs << endl;
	for ( i = 0; args[i] != NULL; i += 1 ) {
	    cerr << args[i] << " ";
	} // for
	cerr << endl;
#endif // __U_DEBUG_H__

	execvp( args[0], (char *const *)args );		// should not return
	perror( "uC++ Translator error: cpp level, execvp" );
	exit( -1 );
    } // if

    // Create a temporary file to store output of the C preprocessor.

    char tmpname[] = P_tmpdir "/uC++XXXXXX";
    int tmpfile = mkstemp( tmpname );
    if ( tmpfile == -1 ) {
	perror( "uC++ Translator error: cpp level, mkstemp" );
	exit( -1 );
    } // if

#ifdef __U_DEBUG_H__
    cerr << "tmpname:" << tmpname << " tmpfile:" << tmpfile << endl;
#endif // __U_DEBUG_H__

    // Run the C preprocessor and save the output in tmpfile.

    if ( fork() == 0 ) {				// child process ?
	// -o xxx.ii cannot be used to write the output file from cpp because
	// no output file is created if cpp detects an error (e.g., cannot find
	// include file). Whereas, output is always generated, even when there
	// is an error, when cpp writes to stdout. Hence, stdout is redirected
	// into the temporary file.
	if ( freopen( tmpname, "w", stdout ) == NULL ) { // redirect stdout to tmpname
	    perror( "uC++ Translator error: cpp level, freopen" );
	    exit( -1 );
	} // if
	args[0] = compiler_name.c_str();
	if ( do_depends ) {
	    args[nargs] = "-MD";
	    nargs += 1;
	    args[nargs] = "-MF";
	    nargs += 1;
	    args[nargs] = depend_out;
	    nargs += 1;
	    args[nargs] = "-MT";
	    nargs += 1;
	    args[nargs] = cpp_out;
	    nargs += 1;
	} // if
	args[nargs] = "-E";
	nargs += 1;
	args[nargs] = cpp_in;				// input to cpp
	nargs += 1;
	args[nargs] = NULL;				// terminate argument list

#ifdef __U_DEBUG_H__
	cerr << "cpp nargs: " << nargs << endl;
	for ( i = 0; args[i] != NULL; i += 1 ) {
	    cerr << args[i] << " ";
	} // for
	cerr << endl;
#endif // __U_DEBUG_H__

	execvp( args[0], (char *const *)args );		// should not return
	perror( "uC++ Translator error: cpp level, execvp" );
	exit( -1 );
    } // if

    wait( &code );					// wait for child to finish

#ifdef __U_DEBUG_H__
    cerr << "return code from cpp:" << WEXITSTATUS(code) << endl;
#endif // __U_DEBUG_H__

    if ( WIFSIGNALED(code) ) {				// child failed ?
	unlink( tmpname );				// remove tmpname
	fprintf( stderr, "uC++ Translator error: cpp failed with signal %d\n", WTERMSIG(code) );
	exit( -1 );
    } // if

    if ( WEXITSTATUS(code) != 0 ) {			// child error ?
	unlink( tmpname );				// remove tmpname
	exit( WEXITSTATUS(code) );			// do not continue
    } // if

    // If -U++ flag specified, run the u++-cpp preprocessor on the temporary
    // file, and output is written to standard output.  Otherwise, run the
    // u++-cpp preprocessor on the temporary file and save the result into the
    // output file.

    // Create a temporary file to store output of the u++ preprocessor.
    
    char tmpname_u[] = P_tmpdir "/uC++XXXXXX";
    if ( !upp_flag ) {
        int tmpfile_u = mkstemp( tmpname_u );
        if ( tmpfile_u == -1 ) {
            perror( "uC++ Translator error: cpp level, mkstemp" );
            exit( -1 );
        } // if
    } // if
    
    if ( upp_flag || fork() == 0 ) {			// conditional fork ?
	uargs[0] = ( *new string( bprefix + "/u++-cpp" ) ).c_str();

	uargs[nuargs] = tmpname;
	nuargs += 1;
	if ( ! upp_flag ) {				// run u++-cpp ?
	    uargs[nuargs] = tmpname_u;
	    nuargs += 1;
	} // if
	uargs[nuargs] = NULL;				// terminate argument list

#ifdef __U_DEBUG_H__
	cerr << "u++-cpp nuargs: " << nuargs << endl;
	for ( i = 0; uargs[i] != NULL; i += 1 ) {
	    cerr << uargs[i] << " ";
	} // for
	cerr << endl;
#endif // __U_DEBUG_H__

	execvp( uargs[0], (char * const *)uargs );	// should not return
	perror( "uC++ Translator error: cpp level, execvp" );
	exit( -1 );
    } // if

    wait( &code );					// wait for child to finish

#ifdef __U_DEBUG_H__
    cerr << "return code from u++-cpp:" << WEXITSTATUS(code) << endl;
#endif // __U_DEBUG_H__

    // Must unlink here because file must exist across execvp.
    if ( unlink( tmpname ) == -1 ) {
	perror( "uC++ Translator error: cpp level, unlink" );
	exit( -1 );
    } // if

    if ( WIFSIGNALED(code) ) {				// child failed ?
	fprintf( stderr, "uC++ Translator error: u++-cpp failed with signal %d\n", WTERMSIG(code) );
	exit( -1 );
    } // if

    if ( WEXITSTATUS(code) != 0 ) {			// child error ?
	unlink( tmpname );				// remove tmpname
	unlink( tmpname_u );				// remove tmpname_u
	exit( WEXITSTATUS(code) );			// do not continue
    } // if

    // Run the C compiler

    if ( fork() == 0 ) {				// child process ?
	args[0] = compiler_name.c_str();
        args[nargs] = "-c";
        nargs += 1;
        args[nargs] = "-o";
        nargs += 1;
        args[nargs] = cpp_out;
        nargs += 1;
        args[nargs] = "-x";
        nargs += 1;
        args[nargs] = "c++-cpp-output";
        nargs += 1;
	args[nargs] = tmpname_u;			// input to cpp
	nargs += 1;
	args[nargs] = NULL;				// terminate argument list

#ifdef __U_DEBUG_H__
	cerr << "icc nargs: " << nargs << endl;
	for ( i = 0; args[i] != NULL; i += 1 ) {
	    cerr << args[i] << " ";
	} // for
	cerr << endl;
#endif // __U_DEBUG_H__

	execvp( args[0], (char *const *)args );		// should not return
	perror( "uC++ Translator error: cpp level, execvp" );
	exit( -1 );
    } // if

    wait( &code );					// wait for child to finish

#ifdef __U_DEBUG_H__
    cerr << "return code from icc:" << WEXITSTATUS(code) << endl;
#endif // __U_DEBUG_H__

    // Must unlink here because file must exist across execvp.
    if ( unlink( tmpname_u ) == -1 ) {
	perror( "uC++ Translator error: cpp level, unlink" );
	exit( -1 );
    } // if

    exit( WEXITSTATUS(code) );
} // Stage1


template< class Istream >
class line_iterator {
    Istream *in;
    string next;
    void do_next() { getline( *in, next ); }
public:
    line_iterator( Istream *is = NULL ) : in( is ) { if( in ) do_next(); }
    const char *operator*() { string *tmp = new string( next ); return tmp->c_str(); }
    line_iterator operator++() { do_next(); return *this; }
    line_iterator operator++(int) { line_iterator< Istream > tmp( *this ); do_next(); return tmp; }
    bool operator==( const line_iterator &other ) { return ( in == other.in ) || ( other.in == NULL && !in->good() ); }
    bool operator!=( const line_iterator &other ) { return ( in == other.in ) || ( other.in == NULL && in->good() ); }
};

template< class Istream >
line_iterator< Istream > line_iterator_begin( Istream &is ) { return line_iterator< Istream >( &is ); }

template< class Istream >
line_iterator< Istream > line_iterator_end( Istream &is ) { return line_iterator< Istream >( NULL ); }

template< class InputIterator, class OutputIterator >
void gatherArgs( InputIterator begin, InputIterator end, OutputIterator out, bool &ipo_flag, bool preprocess = false ) {
    for ( InputIterator in = begin; in != end; in++ ) {
	string arg = *in;
#ifdef __U_DEBUG_H__
	cerr << "gatherArgs: " << arg << endl;
#endif // __U_DEBUG_H__
	if ( arg[ 0 ] == '@' ) {
#ifdef __U_DEBUG_H__
	    cerr << "mcpcom reading argument file " << &(*in)[1] << endl;
#endif // __U_DEBUG_H__
	    std::ifstream argfile( &(*in)[1] );
	    gatherArgs( line_iterator_begin( argfile ), line_iterator_end( argfile ), out, ipo_flag, preprocess );
	} else if ( arg[ 0 ] != '-' ||
		    isprefix( arg, "-mGLOB_obj_output_file=" ) ) {
	    *out++ = *in;
	} else if ( isprefix( arg, "-mGLOB_options_string=" ) ) {
	    std::istringstream iss( arg.substr( 22 ) );
	    string s;
	    while( iss >> s ) {
		if ( s == "-o" ) {
		    // Ignore the output name specified, because it might be for the
		    // link step
		    iss >> s;
		} else if ( ! isprefix( s, "-Qlocation" ) ) {
		    string *tmp = new string( s );
		    *out++ = tmp->c_str();
		} // if
	    } // while
	} else if ( arg == "--sys_include" ) {
	    // skip filename argument
	    in++;
	} else if ( arg == "-mIPOPT_link" ) {
	    ipo_flag = true;
	} else if ( arg == "-E" ) {
	    preprocess = true;
	    *out++ = *in;
	} else if ( preprocess && ( isprefix( arg, "-D" ) || isprefix( arg, "-I" ) ) ) {
	    // icpc doesn't provide "-mGLOB_options_string" for preprocessing-only calls
	    // This logic assumes that the important -D and -I options come after -E on the
	    // command line, which seems to be true.
	    *out++ = *in;
	} // if
    } // for
} // gatherArgs


int main( const int argc, const char *argv[], const char * const env[] ) {
/// #ifdef __U_DEBUG_H__
///     for ( int i = 0; env[i] != NULL; i += 1 ) {
/// 	cerr << env[i] << endl;
///     } // for
/// #endif // __U_DEBUG_H__

    std::vector< const char* > args;
    bool ipo_flag = false;

    gatherArgs( &argv[0], &argv[argc], std::back_inserter( args ), ipo_flag );
    
    if ( ipo_flag ) {
	// For the ipo link step we need to pass args unmolested to the real mcpcom.
	// It's a basic precondition of icpc that icpc and mcpcom be in the same directory.
	string mcpcom_name = compiler_name;
	int lastslash = mcpcom_name.rfind( '/' );
	if ( lastslash != string::npos ) {
	    mcpcom_name.replace( lastslash + 1, string::npos, "mcpcom" );
	} else {
	    mcpcom_name = "mcpcom";
	} // if
	argv[0] = mcpcom_name.c_str();
	execvp( argv[0], (char *const *)&argv[0] );	// should not return
	perror( "uC++ Translator error: cpp level, execvp" );
	exit( -1 );
    } // if

    Stage1( args.size(), args );
} // main


// Local Variables: //
// compile-command: "gmake install" //
// End: //
