/* ==================================================== ======== ======= *
 *
 *  uunatima.cc : native images and pixmaps
 *  Ubit Project [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:00] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uunatima.cc ubit:b1.6.1"
#include <ubrick.hh>
#include <ucolor.hh>
#include <ufont.hh>
#include <unat.hh>
#include <upix.hh>
#include <X11/Xmu/Drawing.h>
#ifndef XPM_NOT_AVAILABLE
#include <X11/xpm.h>
#endif
#ifndef UNGIF_NOT_AVAILABLE
#include <ungif.hh>
#endif

#define GIF_COLORTABLE_SIZE 256

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
// ximashape == null ==> opaque image
// ximashape != null ==> transparent background

UNatIma::UNatIma(UNatDisp *nd, UX_Image ima, UX_Image imashape) {
  natdisp   = nd;
  xima      = ima;
  ximashape = imashape;
  // extensions used by class UIma
  lscale    = 0;     //LOGICAL scale
  next      = null;
}

UNatIma::~UNatIma() {
  if (xima) XDestroyImage(xima);
  if (ximashape) XDestroyImage(ximashape);
  xima = null;
  ximashape = null;
}

u_dim UNatIma::getWidth() const {
  if (xima) return xima->width;
  else {
    uerror("UNatIma::getWidth", "Unrealized image");
    return -1;
  }
}

u_dim UNatIma::getHeight() const {
  if (xima) return xima->height;
  else {
    uerror("UNatIma::getHeight", "Unrealized image");
    return -1;
  }
}

u_bool UNatIma::isRealized() const {
  return xima != null;
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

UNatPix::UNatPix(UNatDisp *nd, UX_Pixmap p, UX_Pixmap m,
		 u_dim ww, u_dim hh, int dd) {
  natdisp = nd;
  height    = hh; 
  width     = ww;
  depth     = dd;
  xpix      = p;
  xpixshape = m;
  lscale    = 0;     //LOGICAL scale
}

// creates the pixmaps from the images
// ximashape == null ==> opaque image
// ximashape != null ==> transparent background

void UNatPix::set(UX_Image xima, UX_Image ximashape) {
  lscale    = 0;     //LOGICAL scale
  if (!xima) {
    height    = 0; 
    width     = 0;
    depth     = 0;
    xpix      = None;
    xpixshape = None;
  }

  else if (xima->width == 0 || xima->height == 0) {
    uwarning("UNatPix::set", 
	     "image with null width or null height --> ignored");
    // test important car X plante sur XCreatePixmap si une dimension
    // est nulle
    return;
  }
  else {
    width  = xima->width;
    height = xima->height;
    depth  = xima->depth;
    xpix = XCreatePixmap(natdisp->getXDisplay(), natdisp->getXWindow(),
			 width, height, xima->depth);
    XGCValues values;
    values.foreground = BlackPixelOfScreen(natdisp->getXScreen());
    values.background = WhitePixelOfScreen(natdisp->getXScreen());
    
    GC gc = XCreateGC(natdisp->getXDisplay(), xpix,
		      GCForeground | GCBackground, &values);

    XPutImage(natdisp->getXDisplay(), xpix, gc,
	      xima, 0, 0, 0, 0, width, height);
    XFreeGC(natdisp->getXDisplay(), gc);

    if (!ximashape) xpixshape = None;
    else {
      xpixshape = XCreatePixmap(natdisp->getXDisplay(), 
				natdisp->getXWindow(),
				width, height, ximashape->depth);
      values.foreground = 1;
      values.background = 0;
      //!ATT: le GC est de depth 1 dans ce cas !
      GC gc = XCreateGC(natdisp->getXDisplay(), xpixshape,
		     GCForeground | GCBackground, &values);

      XPutImage(natdisp->getXDisplay(), xpixshape, gc, ximashape, 
		0, 0, 0, 0, width, height);
      XFreeGC(natdisp->getXDisplay(), gc);
    }
  }
}

UNatPix::UNatPix(UNatDisp *nd, UX_Image xima, UX_Image ximashape) {
  natdisp = nd;
  xpix = xpixshape = null;
  set(xima, ximashape);
}

UNatPix::UNatPix(UNatDisp *nd, UNatIma *ima) {
  natdisp = nd;
  xpix = xpixshape = null;
  set(ima->xima, ima->ximashape);
}

UNatPix::~UNatPix() {
  if (natdisp
      && natdisp->getXDisplay() 
      //obs: && this != natdisp->getUnknownPixmap()
      ) {
    if (xpix != None) XFreePixmap(natdisp->getXDisplay(), xpix);
    if (xpixshape != None) XFreePixmap(natdisp->getXDisplay(), xpixshape);
  }
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

#ifndef XPM_NOT_AVAILABLE

static void initXPMattributes(UNatDisp *nd, 
			      Pixel fg, Pixel bg, u_bool transp_bg,
			      XpmAttributes  &attributes, 
			      XpmColorSymbol &symbol) {

  // cherche couleurs approchees avec une large tolerance
  attributes.exactColors = false;
  attributes.closeness   = 100000;

  // specifier la bonne combinaison depth / visual / colormap c'est-a-dire
  // celle qui est definie par de UNatDisp

  attributes.depth     = nd->getDepth();
  attributes.colormap  = nd->getXColormap();
  attributes.valuemask = XpmExactColors | XpmCloseness | XpmDepth | XpmColormap;

  // !att: visual can be null in which case it should not be given 
  // as an attribute to Xpm functions
  // (note that the visual also specifies the screen number)

  if ((attributes.visual = nd->getXVisual()) != null)
    attributes.valuemask |= XpmVisual;

  // no transparent -> take specified color as bg
  if (!transp_bg) {
    symbol.name  = NULL;
    symbol.value = (char*)"none";
    symbol.pixel = bg;
    attributes.colorsymbols = &symbol;
    attributes.numsymbols   = 1;
    attributes.valuemask   |= XpmColorSymbols;
  }
}

static int decodeXPMstatus(int stat) {
  if (stat >= 0) return UIma::Loaded;
  else switch(stat) {
  case XpmFileInvalid:
    return UIma::InvalidFile;
  case XpmOpenFailed:
    return UIma::CantOpenFile;
  case XpmNoMemory:
    return UIma::NoMoreMemory;
  default:
    return UIma::CantReadFile;
  }
}

/* ==================================================== ======== ======= */

int UXpmFileReader(UNatDisp *nd, const char *fpath,
		   Pixel fg, Pixel bg, u_bool transp_bg,
		   UX_Image &ima, UX_Image &shapeima,
		   u_dim &width, u_dim &height) {
  width  = 0;
  height = 0;
  ima    = null;
  shapeima = null; 

  if (!fpath || !fpath[0]) return UIma::InvalidFile;

  XpmAttributes attr;
  XpmColorSymbol symbol;  
  initXPMattributes(nd, fg, bg, transp_bg, attr, symbol);

  int stat = XpmReadFileToImage(nd->getXDisplay(),
				(char*)fpath,
				&ima, 
				(transp_bg ? &shapeima : NULL),
				&attr);
  stat = decodeXPMstatus(stat);
  if (stat >= 0) {
    width  = attr.width;
    height = attr.height;
  }
  return stat;
}

/* ==================================================== ======== ======= */

int UXpmDataReader(UNatDisp *nd, const char *xpmdata,
		   Pixel fg, Pixel bg, u_bool transp_bg,
		   UX_Image &ima, UX_Image &shapeima,
		   u_dim &width, u_dim &height) {
  width  = 0;
  height = 0;
  ima    = null;
  shapeima = null; 

  char **data = (char**)xpmdata;
  if (!data || !data[0]) return UIma::InvalidFile;

  XpmAttributes attr;
  XpmColorSymbol symbol;  
  initXPMattributes(nd, fg, bg, transp_bg, attr, symbol);

  int stat = XpmCreateImageFromData(nd->getXDisplay(),
				    data,
				    &ima, 
				    (transp_bg ? &shapeima : NULL),
				    &attr);
  stat = decodeXPMstatus(stat);
  if (stat >= 0) {
    width  = attr.width;
    height = attr.height;
  }
  return stat;
}

#endif
/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
// creates a native pixmap from a X11-XBM file

#if KKKK

int XbmFileReader(UNatDisp *nd, const char *fpath,
		  Pixel fg, Pixel bg, u_bool transp_bg,
		  UX_Image &ima, UX_Image &shapeima
		  u_dim &width, u_dim &height) {

  //NB: XReadBitmapFileData PAS DANS X11r5 !!!
  unsigned char *read_data = NULL;
  Pixmap pix = None;

  if (fpath ) {
    unsigned int ww, hh;
    int x_hot, y_hot;
    int result = XmuReadBitmapDataFromFile((char*)fpath,
					   &ww, &hh, &read_data,
					   &x_hot, &y_hot);
    if (result != BitmapSuccess || !read_data)
      uwarning("UNatPix::createPixmap", "Could not read XBitmap file '%s'", fpath);
    else {
      xpm_data = (char*)read_data;
      width = ww, height = hh;
    }
  }

  /*
    The XCreatePixmapFromBitmapData function creates a pixmap of
    the given depth and then does a bitmap-format XPutImage of
    the data into it.  The depth must be supported by the screen
    of the specified drawable, or a BadMatch error results.
    */
  if (xpm_data) {
    pix = XCreatePixmapFromBitmapData(nd->getXDisplay(), nd->getXWindow(),
				      xpm_data, width, height,
				      nd->getXPixel(fg), nd->getXPixel(bg),
				      nd->getDepth());
  }
  if (read_data) XFree(read_data);

  if (pix == None) return null;
  else return new UNatPix(nd, pix, None, width, height);
}


static void JPGwrapper() {
  // JPEG image assumed
 
  char *comm =
    (char*)malloc((80 + strlen(pixmap_path)) * sizeof(char));
  sprintf(comm, "convert %s "TMP_PIX_FILE,
	  pixmap_path);
  system(comm);
  natpix = g.createPixmap(TMP_PIX_FILE, null, props->bgcolor, bgc);
  system("/bin/rm -f "TMP_PIX_FILE);
  free(comm);
}

#endif

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
//forward defs.
static void AllocColors(UX_Display display, UX_Colormap cmap,
			unsigned long *pixelTable,
			XColor *colorTable, int colorCount);

#ifndef UNGIF_NOT_AVAILABLE

int UGifFileReader(UNatDisp *nd, const char *fpath,
		   Pixel fg, Pixel bg, u_bool transp_bg,
		   UX_Image &ima, UX_Image &shapeima,
		   u_dim &width, u_dim &height) {
  width  = 0;
  height = 0;
  ima    = null;
  shapeima = null; 

  UngifAttributes attr;
  attr.from_stdin = 0;

  // !att: visual can be null in which case it should not be given 
  // as an attribute to Xpm functions
  // NB: the visual also specifies the screen number

  if (!fpath || !fpath[0]) return UIma::InvalidFile;
  if (!readGifFile(fpath, &attr)) 
    return UIma::CantReadFile;
 
  //getDepth returns value in bits and they are 8 bits in a byte
  //ca marche pas car X fait ce qu'il veut => acll XCreateImge puis voir taille
  //size_t datasize = (int)
  //  ((float)attr.width * attr.height * nd->getDepth() / 8. + 1);

  int depth = nd->getDepth();
  int bitmap_pad;
  if (depth > 16) bitmap_pad = 32;
  else if (depth > 8) bitmap_pad = 16;
  else bitmap_pad = 8;

  ima = XCreateImage(nd->getXDisplay(), nd->getXVisual(), 
		     depth, ZPixmap, 
		     0,   //offset (# of pixels to ignore at the
		          // beginning of the scanline)
		     NULL,	// data will be created later
		     attr.width, attr.height,  // requested size
		     bitmap_pad,   // bitmap_pad (quantum of a scanlin)
		     0);  //bytes_per_line (0 : continuous scanlines 
                          //=> calculated auto.)
  if (!ima) {
    // faudrait detruire attr.buffer[] !!!
    if (attr.buffer) free(attr.buffer);
    return UIma::NoMoreMemory;
  }
  
  // once ima is created, we can get the actual line width via field 
  // 'bytes_per_line' (note: it is just impossible to guess this value
  // as X may attribute more bit per pixel than what we requested
  // (for instance a 32bit-per-pix graphics card will probably attribute
  // 32 bits even if we also ask for 24 !)

  ima->data = (char*)malloc(ima->bytes_per_line * attr.height);
  if (!ima->data) {
    XDestroyImage(ima);  
    // faudrait detruire attr.buffer[] !!!
    if (attr.buffer) free(attr.buffer);
    return UIma::NoMoreMemory;
  }
  
  // Note the table has 256 entry which is the maximum allowed in GIF format
  unsigned long pixelTable[GIF_COLORTABLE_SIZE];
  AllocColors(nd->getXDisplay(), nd->getXColormap(), 
	      pixelTable, attr.colorTable, attr.colorCount);
  
  for (int i = 0; i < attr.height; i++) {
    for (int j = 0; j < attr.width; j++)
      XPutPixel(ima, j, i, pixelTable[attr.buffer[i][j]]);
    free(attr.buffer[i]);
  }
  free(attr.buffer);

  width  = attr.width;
  height = attr.height;
  return UIma::Loaded;
}

#endif
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// NOTE: UNatIma::createScaledImage was adapted from gifrsize.c from 
// the Gif-Lib Written by Gershon Elber (Ver 0.1, Jul. 1989)

//forward def.
static void resizeLine(XImage *in_ima, XImage *out_ima, 
		       int in_y, int out_y, float xscale);

UNatIma* UNatIma::clone(UNatDisp *nd, float xscale, float yscale) {
  UX_Image xima_clone = null, ximashape_clone = null;
  if (xima) 
    xima_clone = createScaled_XImage(nd, xima, xscale, yscale);
  if (ximashape) 
    ximashape_clone = createScaled_XImage(nd, ximashape, xscale, yscale);

  return new UNatIma(nd, xima_clone, ximashape_clone);
}

UNatIma* UNatIma::createScaledIma(UNatDisp *nd, 
				  UX_Image xima, UX_Image ximashape, 
				  float xscale, float yscale) {
  UX_Image xima_clone = null, ximashape_clone = null;
  if (xima) 
    xima_clone = createScaled_XImage(nd, xima, xscale, yscale);
  if (ximashape) 
    ximashape_clone = createScaled_XImage(nd, ximashape, xscale, yscale);

  return new UNatIma(nd, xima_clone, ximashape_clone);
}

/* ==================================================== ======== ======= */

UX_Image UNatIma::createScaled_XImage(UNatDisp *nd, UX_Image xima, 
				     float xscale, float yscale) {
  if (xima->depth > nd->getDepth()) {
    uerror("UNatIma::createScaledImage", 
	   "Can't create image:\nthe source image and the destination drawable have incompatible depths");
    return null;
  }

  int in_h  = xima->height;
  int out_w = (int) (xima->width  * xscale + 0.5);
  int out_h = (int) (xima->height * yscale + 0.5);

  //getDepth returns value in bits and they are 8 bits in a byte
  //size_t datasize = (int)((float)out_w * out_h * nd->getDepth() / 8. + 1);
  //char *data = (char*)malloc(datasize);

  int depth = xima->depth;
  int bitmap_pad;
  if (depth > 16) bitmap_pad = 32;
  else if (depth > 8) bitmap_pad = 16;
  else bitmap_pad = 8;

  UX_Image out_ima = XCreateImage(nd->getXDisplay(), nd->getXVisual(), 
				//nd->getDepth(), ZPixmap, 
				  depth, ZPixmap,
				  0, //offset (# of pixels to ignore at the 
				  //beginning of the scanline)
				  NULL,	// data will be created later
				  out_w, out_h,
				  bitmap_pad,  //bitmap_pad (quantum of a scanline)
				  0); //bytes_per_line (0 : continuous scanlines 
                                    // => calculated auto.)
  if (!out_ima) {
    uwarning("UNatIma::createScaledImage", 
	     "Could not create image: not enough memory");
    return null;
  }

  // see note above in UGifFileReader
  out_ima->data = (char*)malloc(out_ima->bytes_per_line * out_h);
  if (!out_ima->data) {
    XDestroyImage(out_ima);  
    uwarning("UNatIma::createScaledImage", 
	     "Could not create image: not enough memory");
    return null;
  }
  
  int in_y, out_y, out_x;
  float y = 0.0;
  int last_out_y = 0;

  for (in_y = 0; in_y < in_h; y += yscale, in_y++) {
    out_y = (int)y;

    //NB: don't call resizeLine(...,out_y..) if out_y >= out_h !
    if (out_y >= out_h) break;

    if (last_out_y < out_y) {
      resizeLine(xima, out_ima, in_y, out_y, xscale);
      
      // duplicate additionnal lines 
      for ( ; last_out_y < out_y; last_out_y++) {
	for (out_x = 0; out_x < out_w; out_x++) 
	  XPutPixel(out_ima, out_x, last_out_y, 
		    XGetPixel(out_ima, out_x, out_y));
      }
    }
  }
  
  // add additionnal lines at the end if scale is not dividable
  ++last_out_y; //skip out_y
  if (last_out_y < out_h) {
    out_y = last_out_y;
    resizeLine(xima, out_ima, in_y, out_y, xscale);

    while (++last_out_y < out_h) {
      for (out_x = 0; out_x < out_w; out_x++)
	XPutPixel(out_ima, out_x, last_out_y, 
		  XGetPixel(out_ima, out_x, out_y));
    }
  }

  return out_ima;
}

/* ==================================================== ======== ======= */

static void resizeLine(XImage *in_ima, XImage *out_ima, 
		       int in_y, int out_y, float xscale) {
  int in_w  = in_ima->width;
  int out_w = out_ima->width;

  int in_x, out_x;
  float out_flx = 0.0;
  int last_out_x = 0;

  for (in_x = 0; in_x < in_w; out_flx += xscale, in_x++) {

    out_x = (int)out_flx;
    if (out_x > out_w) out_x = out_w;

    for (; last_out_x < out_x; last_out_x++) {
      XPutPixel(out_ima, last_out_x, out_y, 
		XGetPixel(in_ima, in_x, in_y));
    }
  }

  // fill the line if needed
  for (in_x = in_w - 1; last_out_x < out_w; last_out_x++) {
    XPutPixel(out_ima, last_out_x, out_y, 
	      XGetPixel(in_ima, in_x, in_y));
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UX_Image UNatIma::blend_XImages(UNatDisp *nd, 
			       UX_Image xima1, UX_Image xima2,
			       float alpha) {
  if (alpha == 0.0)
    return xima1;
  else if (alpha == 1.0)
    return xima2;

  if (xima1->depth != xima2->depth) {
    uerror("UNatIma::createScaledImage", 
	   "Images have different depths");
    return null;
  }
  if (xima1->depth > nd->getDepth()) {
    uerror("UNatIma::createScaledImage", 
	   "the source image and the destination drawable have incompatible depths");
    return null;
  }  
  if (xima1->depth < 16) {  //TESTER TRueColor!!
     uerror("UNatIma::createScaledImage", 
	   "Image is not in TrueColors");
    return null;
  }

  float alpha_compl = 1.0 - alpha;
  //test: float alpha_compl = 0.0;
  int ww = U_MIN(xima1->width, xima2->width);
  int hh = U_MIN(xima1->height, xima2->height);

  //-- !!TRANSP: rajouter VisualInfo pour eviter d'avoir a recalculer
  // a chaque fois   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  XVisualInfo vi;
  int dd = nd->matchVisualProps(&vi, TrueColor, nd->getDepth(), 0);
  if ((dd = 0)) {
    uerror("UNatIma::blend", "TRueColor Visual not found: can't blend images");
    return xima2;
  }
  //printf ("visual found %d -- ww=%d hh=%d \n", kk, ww, hh);
  //printf("xv->bpgb %ld %d\n\n", vi.bits_per_rgb);
  //printf("xv->red_mask %ld\n", vi.red_mask);
  //printf("xv->green_mask %ld\n", vi.green_mask);
  //printf("xv->blue_mask %ld\n\n", vi.blue_mask);

  //-- !!TRANSP: rajouter VisualInfo pour eviter d'avoir a recalculer
  // a chaue fois   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  unsigned long blend, red, blue, green;
  register unsigned long 
    pix1, pix2,
    red_mask = vi.red_mask, 
    green_mask = vi.green_mask,
    blue_mask = vi.blue_mask;
 
  for (int y = 0; y < hh; y++)
    for (int x = 0; x < ww; x++) {
      pix1 = XGetPixel(xima1, x, y); 
      pix2 = XGetPixel(xima2, x, y);

      red   = long((pix1 & red_mask) * alpha_compl + (pix2 & red_mask) * alpha);
      green = long((pix1 & green_mask) * alpha_compl + (pix2 & green_mask) * alpha);
      blue  = long((pix1 & blue_mask) * alpha_compl  + (pix2 & blue_mask) * alpha);

      blend = (red & red_mask) + (green & green_mask) + (blue & blue_mask);
      XPutPixel(xima1, x, y, blend);
    }

  return xima1;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// AllocColors was adapted from the XXL GUI Builder
// [copyright E. Lecolinet - ENST]

// essaye de tout caser dans la Colormap donnee en arg.

static void AllocColors(UX_Display display, UX_Colormap cmap,
			unsigned long *pixelTable,
			XColor *colorTable, int colorCount) {
  register int k;
  // on essaye d'allouer les couleurs dans la cmap et on voit ce qui reste
  int todoCount = 0;
  int todo[GIF_COLORTABLE_SIZE];
  XColor col;
  col.flags = DoRed | DoGreen | DoBlue;

  for (k = 0; k < colorCount; k++) {
    col.red   = colorTable[k].red;
    col.green = colorTable[k].green;
    col.blue  = colorTable[k].blue;

    if (XAllocColor(display, cmap, &col)) 
      pixelTable[k] = col.pixel;
    else {
      pixelTable[k] = 0;
      todo[todoCount] = k;
      todoCount++;
    }
  }

  //printf("colorCount = %d, reste = %d\n", colorCount, todoCount);

  // -- si on a reussi a tout caser : c'est fini!
  if (todoCount == 0) return;

  // -- sinon: on associe aux couleurs non-allouees la couleur la plus proche 
  // dans la colormap

  // pas toujours vrai si Visuel specifique ... !!
  // mais comment retrouver la vraie taille de cmap ???
  int num_xcolors = DisplayCells(display, DefaultScreen(display));
  //printf("num_xcolors = %d, \n", num_xcolors);

  XColor *xcolors = (XColor*)malloc(num_xcolors * sizeof(XColor));
  for (k = 0; k < num_xcolors; k++) xcolors[k].pixel = k;
  XQueryColors(display, cmap, xcolors, num_xcolors);

  unsigned long green, red, blue;

  for (k = 0; k < todoCount; k++) {
    red   = colorTable[todo[k]].red;
    green = colorTable[todo[k]].green;
    blue  = colorTable[todo[k]].blue;

    unsigned long dist, mindist = 0xffffffff;
    int jmin = 0;

    for (int j = 0; j < num_xcolors; j++) {
      long r = red   - xcolors[j].red;
      long g = green - xcolors[j].green;
      long b = blue  - xcolors[j].blue;
      if (r <0) r = -r;
      if (g <0) g = -g;
      if (b <0) b = -b;

      //NB: on pourrait faire nettement mieux: voir par exemple create.c
      //dans la distrib XPM

      dist = r + g + b;
      if (dist < mindist) {mindist = dist; jmin = j;}
      //printf("[r%d g%d b%d -> %d] ", r, g, b, dist);
    }

    pixelTable[todo[k]] = jmin;
  }

  free(xcolors);
}

/* ==================================================== ======== ======= */
#if TEST
//version plus sophistiquee qui essaie d'abord d'alleour les couleurs
//les plus frequentes dans l'image
//en pratique: resultat similaire (= aussi mediocre)

static void AllocColors2(Display *display, Colormap cmap,
			int width, int height,
			unsigned char **buffer,
			unsigned long *pixelTable,
			XColor *colorTable, int colorCount) {
  register int j, k;
  long compteur[256];

  for (k = 0; k < 256; k++) {
    compteur[k] = 0;
    pixelTable[k] = 0;
  }

  /* On cherche les couleurs les plus utilisees */
  for (int i = 0; i < height; i++)
    for (j = 0; j < width; j++)
      // valeurs entre 0 et 255
      compteur[buffer[i][j]]++;

  // On reorganise les couleurs par ordre decroissant d'occurence
  // et on essaye de les allouer dans la cmap
  int nb_reste = 0;
  int reste[256];
  XColor col;
  col.flags = DoRed | DoGreen | DoBlue;

  for (k = 0; k < 256; k++) {
    long max_count = 0;
    int jmax;
    for (j = 0; j < 256; j++)
      if (compteur[j] > max_count) { 
	max_count = compteur[j];
	jmax = j;
      }
    compteur[jmax] = 0;	 //cas traite
    col.red   = colorTable[jmax].red;
    col.green = colorTable[jmax].green;
    col.blue  = colorTable[jmax].blue;

    if (XAllocColor(display, cmap, &col)) pixelTable[jmax] = col.pixel;
    else reste[nb_reste++] = jmax;
  }

  printf("colorCount = %d, reste = %d\n", colorCount, nb_reste);

  // On associe aux couleurs non-allouees la couleur la plus proche 
  // dans la colormap
  XColor colors[256];
  unsigned long green, red, blue;

  // 256 --> mumColors = DisplayCells(xdisplay, DefaultScreen(xdisplay));
  for (k = 0; k < 256; k++) colors[k].pixel = k;
  XQueryColors(display, cmap, colors, 256);

  for (k = 0; k < nb_reste; k++) {
    red   = colorTable[reste[k]].red;
    green = colorTable[reste[k]].green;
    blue  = colorTable[reste[k]].blue;

    unsigned long dist, mindist = 0xffffffff;
    int jmin = 0;

    for (j = 0; j < 256; j++) {
      //!att:: le cas (signed long) est necessaire!
      dist = U_ABS((signed long)red - colors[j].red)
	+ U_ABS((signed long)green - colors[j].green)
	+ U_ABS((signed long)blue - colors[j].blue);
      if (dist < mindist) {mindist = dist; jmin = j;}
    }
    pixelTable[reste[k]] = jmin;
  }
}
#endif

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:00] ======= */
