/* ==================================================== ======== ======= *
 *
 *  uustr.cc
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uustr.cc	ubit:b1.11.1"
#include <udefs.hh>
#include <ubrick.hh>
#include <uconfig.hh>
#include <uprop.hh>
#include <ucontext.hh>
#include <ugraph.hh>
#include <uctrl.hh>
#include <uview.hh>
#include <ustr.hh>
#include <uflow.hh>
#include <uappli.hh>
#include <ucolor.hh>

#include <uint.hh>


const UClass UStr::uclass("UStr");

UStr UStr::newline("\n", UMode::UCONST);
UStr UStr::goBOL("\15", UMode::UCONST); // retour a la ligne si necessaire

UStr& ustr(const char *s)        {return *(new UStr(s));}
UTitle& utitle(const char *s)    {return *(new UTitle(s));}

/* ==================================================== [Elc:01] ======= */
/* === useful functions =============================== ======== ======= */

// creates a duplicate of s1 (similar to strdup when this function exists)
// - note: argument can be null

char *u_strdup(const char *s1) {
  if (!s1) return null;
  else {
    char *p = (char*)malloc((strlen(s1) + 1) * sizeof(char));
    if (p) strcpy(p, s1);
    return(p);
  }
}

// creates a duplicate of s1 + s2
// - note: arguments can be null

char *u_strdupcat(const char *s1, const char *s2) {
  if (!s1 || !*s1) return u_strdup(s2);
  else if (!s2 || !*s2) return u_strdup(s1);
  else {
    int l1 = strlen(s1);
    char *p = (char*)malloc((l1 + strlen(s2) + 1) * sizeof(char));
    if (p) {
      strcpy(p, s1);
      strcpy(p+l1, s2);
    }
    return(p);
  }
}

char* u_strdupcat(const char *s1, char sep, const char *s2) {
  if (!s1 || !*s1) return u_strdup(s2);
  else if (!s2 || !*s2) return u_strdup(s1);
  else {
    int l1 = strlen(s1);
    char *p = (char*)malloc((l1 + strlen(s2) + 2) * sizeof(char));
    if (p) {
      strcpy(p, s1);
      p[l1] = sep;
      strcpy(p + l1 + 1, s2);
    }
    return(p);
  }
}

const char *u_strext(const char *path) {
  const char *p = path + strlen(path);
  //probleme: etre sur que le . est apres le dernier /
  while (p >= path) {
    if (*p == '.' || *p == '/') break;
    p--;
  }
  // bes sure its a . and eliminate . and .xxxrc and ..
  if (p <= path || *p != '.') return null;
  else if (p == path+1 && *path == '.') return null;
  else return p+1;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UStr::UStr(const char *chs, u_modes m) : UItem(m) {
  s = null; //!dont forget: checked by _set() for freeing memory!
  model = null;
  setImpl(chs, (chs ? strlen(chs) : 0), false, false);
}

UStr::UStr(const UStr &str) : UItem(0) {
  s = null; //!!
  model = null;
  setImpl(str.s, str.len, false, false);
}

UStr::~UStr() {
  if (s) free(s);
  model = null;
  s = null; len = 0;
}

/* ==================================================== ======== ======= */
// NB: duplicates the char string (!ATT: length MUST be the EXACT length)

u_bool UStr::setImpl(const char *_s, int _len, u_bool chgd, u_bool upd) {
  if (s) free(s);
  if (!_s) syncVals(null, 0);
  else {
    s = (char*)malloc((_len + 1) * sizeof(char));
    if (s) {strcpy(s, _s); syncVals(s, _len);}
    else {
      syncVals(null, 0);
      error("setImpl", UError::NO_MEMORY);
      return false;
    }
  }

  if (chgd) changed(upd);
  return true;
}

/* ==================================================== ======== ======= */

void UStr::set(const UStr &str, u_bool upd) {
  //voir ci-dessus
  if (equals(str)) return;
  setImpl(str.s, str.len, true, upd);
  //return *this;
}

//!att: ne doit pas planter si str est null!
void UStr::set(const UStr *str, u_bool upd) {
  //if (!str) /*return*/ set((char*)null);
  //else /*return*/ set(*str);

  if (equals(str)) return;
  else if (!str) setImpl(null, 0, true, upd);
  else setImpl(str->s, str->len, true, upd);
  //return *this;
}

void UStr::set(const char *str, u_bool upd) {
  //tester valeur inchangee sinon risque de boucles infinies
  //ou alloc memoire incorrecte dans le cas ou c'est la meme UStr 
  //(or or commence justement par faire un free, aie, ouille..)
  if (equals(str)) return;
  setImpl(str, (str ? strlen(str) : 0), true, upd);
  //return *this;
}

void UStr::reset(u_bool upd) {
  set((char*)0, upd);
}

//NB: noms modifies pour eviter ambiguites
void UStr::setInt(long ii, u_bool upd) {
  char str[50] = {0}; 
  sprintf(str, "%ld", ii);
  set(str, upd);
}

void UStr::setFloat(double xx, u_bool upd) {
  char str[50] = {0}; 
  sprintf(str, "%f", xx);
  set(str, upd);
}

/*
void UStr::setInt(int i, u_bool upd) {
  if (s) {
    if (*s && i == atoi(s)) return;
    free(s);
  }
  char tmp[50]; sprintf(tmp, "%d", i);
  s = u_strdup(tmp);
  len = strlen(s);
  changed();
  //return *this;
}
void UStr::setFloat(double x, u_bool upd) {
  if (s) {
    if (*s && x == atof(s)) return;
    free(s);
  }
  char tmp[100]; sprintf(tmp, "%f", x);
  s = u_strdup(tmp);
  len = strlen(s);
  changed();
  //return *this;
}
*/
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

u_bool UStr::equals(const UStr&str) const {
  return (str.s == s || (str.s && s && strcmp(str.s, s)==0));
}

u_bool UStr::equals(const UStr*str) const {
  if (!str) return !s ? true : false; 
  else return (str->s == s || (str->s && s && strcmp(str->s, s)==0));
}

u_bool UStr::equals(const char*chs) const  {
  return (chs == s || (chs && s && strcmp(chs, s)==0));
}

/* ==================================================== ======== ======= */

int UStr::compareTo(const char *str) const {
  if (!s) {
    if (!str) return 0;
    else return -2;
  }
  else if (!str) return 2;
  return strcmp(s, str);
}

int UStr::compareTo(const UStr *str) const {
  if (!str) {
    if (!s) return 0;
    else return 2;
  }
  return compareTo(str->s);
}

int UStr::compareTo(const UStr &str) const {
  return compareTo(str.s);
}

int compare(const UStr &us1, const UStr &us2) {
  return us1.compareTo(us2.s);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/*
u_bool UStr::checkChar(int pos, char _c) {
  return (_c != '\0');      //don't insert the NUL char!
}
u_bool UStr::checkStr(int pos, const char *_s) {
  return (_s != null);
}

void UStr::syncVals(char *news, int newlen) {
  s = news;
  len = newlen;
}
*/
u_bool UStr::checkFormat(int pos, char _c) {
  if (_c == '\0') return false;      //don't insert the NUL char!
  else if (model) return model->checkFormat(pos, _c);
  else return true;
}

u_bool UStr::checkFormat(int pos, const char *_s) {
  if (_s == null) return false;      //don't insert the NULL string!
  else if (model) return model->checkFormat(pos, _s);
  else return true;
}

void UStr::syncVals(char *news, int newlen) {
  s = news;
  len = newlen;
  /**** a completer
  if (model) model->set(this);
  ****/
}

  /**** a completer
	void UStr::setStrModel(class UEditable *m) {
  model = m;
  chainage et cas null et enlever existant;
  }
  ****/

void UStr::update() {
  // size changed horizontally ??          // !!!CFCFCF
  parents.updateParents(UUpdate::layout);
}

// calls UOn::change callbacks of this object when its value is changed
// then UOn::changeItem AND UOn::changeStr callbacks of its parents

void UStr::changed(u_bool update_now) {
  if (update_now) update();

  if (cache) {
    UEvent e(UEvent::change, null, NULL);
    e.setItemSource(this);

    fire(e, UOn::change);
    // ensuite on lance les callbacks des parents
    parents.fireParents(e, UOn::itemChange);
    // NE PAS OUBLIER changeStr !!!
    parents.fireParents(e, UOn::strChange);
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

// returns characters at offset 'pos'
char UStr::charAt(int pos) const {
  if (!s || pos >= len) return 0;
  if (pos < 0) pos = len-1; // last char
  return s[pos];
}
 
// changes character at offset 'pos'
u_bool UStr::setCharAt(int pos, char newchar, u_bool upd) {

  if (!s || pos >= len) return false;
  if (!checkFormat(pos, newchar)) return false; //!!

  if (pos < 0) pos = len-1; // last char
  s[pos] = newchar;

  syncVals(s, len);
  changed(upd);
  return true; 
}

/* ==================================================== ======== ======= */

void UStr::append(const UStr &str, u_bool upd) {
  insertImpl(-1, str.s, upd);
}
void UStr::append(const UStr *str, u_bool upd) {
  if (str) insertImpl(-1, str->s, upd);
}
void UStr::append(const char *chs, u_bool upd) {
  insertImpl(-1, chs, upd);
}
void UStr::insert(int pos, const UStr &str,  u_bool upd) {
  insertImpl(pos, str.s, upd);
}
void UStr::insert(int pos, const UStr *str,  u_bool upd) {
  if (str) insertImpl(pos, str->s, upd);
}
void UStr::insert(int pos, const char *str, u_bool upd) {
  insertImpl(pos, str, upd);
}
void UStr::insert(int pos, char newchar, u_bool upd) {
  insertImpl(pos, newchar, upd);
}

/* ==================================================== ======== ======= */

u_bool UStr::insertImpl(int pos, const char *str, u_bool upd) {

  if (!checkFormat(pos, str)) return false; //!!

  if (pos < 0 || pos > len) pos = len; // append
  int nbc = strlen(str);

  char *news = (char*) malloc(len + nbc + 1); // 0 final
  if (!news) {
    error("insertImpl", UError::NO_MEMORY);
    return false;		// str et strLen inchanges !
  }

  if (!s) strcpy(news, str);
  else {
    if (pos>0) strncpy(news, s, pos);
    strncpy(news+pos, str, nbc);
    strcpy(news+pos+nbc, s+pos);
    free(s);
  }

  syncVals(news, len+nbc);
  changed(upd);
  return true;
}

/* ==================================================== ======== ======= */

u_bool UStr::insertImpl(int pos, char c, u_bool upd) {

  if (!checkFormat(pos, c)) return false; //!!

  if (pos < 0 || pos > len) pos = len; // append

  char *news = (char*) malloc(len + 1 + 1); // 0 final
  if (!news) {
    error("insertImpl", UError::NO_MEMORY);
    return false;		// str et strLen inchanges !
  }

  if (!s) {
    news[0] = c;
    news[1] = '\0';
  }
  else {
    if (pos>0) strncpy(news, s, pos);
    news[pos] = c;
    strcpy(news+pos+1, s+pos);
    free(s);
  }

  syncVals(news, len+1);
  changed(upd);
  return true;
}

/* ==================================================== ======== ======= */

UStr* UStr::split(int pos, u_bool upd) {
  if (!s || pos<0 || pos>=len) 
    return null;

  UStr *s2 = new UStr(s+pos);

  if (pos == 0) {
    free(s); 
    syncVals(null, 0);
  }
  else {
    len = pos;
    s = (char*)realloc(s, sizeof(char)*(pos+1));
    s[pos] = '\0';
    syncVals(s, len);
  }

  changed(upd);
  return s2;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UStr::removeN(int pos, int nrem, u_bool upd) {
  replaceImpl(pos, nrem, (char*)null, upd);
}
void UStr::remove(int pos, int nrem, u_bool upd) {
  replaceImpl(pos, nrem, (char*)null, upd);
}

void UStr::replace(int pos, int nrem, const UStr& s2, u_bool upd) {
  replaceImpl(pos, nrem, s2.s, upd);
}
void UStr::replace(int pos, int nrem, const UStr* s2, u_bool upd) {
  replaceImpl(pos, nrem, s2->s, upd);
}
void UStr::replace(int pos, int nrem, const char *str, u_bool upd) {
  replaceImpl(pos, nrem, str, upd);
}

/* ==================================================== ======== ======= */

u_bool UStr::replaceImpl(int pos, int nrem, const char *str, u_bool upd) {

  // revient a un insert() ou un append() dans ce cas
  // (rien a detruire dans la string d'origine)
  if (nrem <= 0 || pos < 0 || pos > len) {
    return insertImpl(pos, str, upd);
  }

  // ne pas depasser taille actuelle de str
  if (pos + nrem > len) nrem = len - pos;

  int nadd = str ? strlen(str) : 0;
  int newlen = len - nrem + nadd;

  if (newlen <= 0) {	  // theoriquement jamais < 0 mais == 0
    syncVals(null, 0);
  }
  
  else {
    char *news = (char*)malloc(newlen + 1); // 0 final
    if (!news) {
      error("replaceImpl", UError::NO_MEMORY);
      return false;		  // str et strLen inchanges !
    }

    // cas (!s ou !*s)  ==>  equivalent a remove()
    if (nadd==0) {	
      if (!s) return false;	  // securite: deja teste par: newlen <= 0
      if (pos>0) strncpy(news, s, pos);
      strcpy(news+pos, s+pos+nrem);
      free(s); s = null;
    }

    // il y a qq chose a ajouter
    else {
      if (!checkFormat(pos, str)) return false; //!!

      if (!s) strcpy(news, str);
      else {
	if (pos>0) strncpy(news, s, pos);
	strncpy(news+pos, str, nadd);
	strcpy(news+pos+nadd, s+pos+nrem);
	free(s); s = null;
      }
    }

    syncVals(news, newlen);
  }

  changed(upd);
  return true;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

char *UStr::get() const {
  if (!s) return null;
  else return u_strdup(s);  // strdup !!
}

char *UStr::getN(int pos, int nbc) const {
  if (!s || nbc <= 0) return null;

  // hors bornes
  if (pos < 0 || pos >= len) return null;

  // ne pas depasser taille actuelle de str
  if (pos + nbc > len) nbc = len - pos;

  char *news = (char*) malloc(nbc + 1); // 0 final
  if (!news) {
    error("getN", UError::NO_MEMORY);
    return null;
  }

  strncpy(news, s+pos, nbc);
  news[nbc] = '\0';		// 0 final
  return news;
}

/* ==================================================== ======== ======= */
#if TEST
// API a definir

  // copies the [pos1,pos2[ subpart of this UStr to 'dest'
  // -- the character located at 'pos2' is not copied
  // -- pos2 == -1 means "end of string"
  // -- 'dest' is automatically augmented
  int copyTo(UStr& dest, int pos1 = 0, int pos2 = -1) const;


  // -- the copy is truncated if 'busize' if too small
  // -- the final '\0' character is always copied
  //    (ie. 'bufsize' must be >= length()+1 to copy the whole UStr)
  // -- returns the number of chars. that were actually copied
  //    (except the final '\0' that is not counted)


  // -- the char. located at pos2 is excluded
  int copyTo(char* buffer, int bufsize, int pos1, int pos2) const;

int UStr::copyTo(char* buffer, int bufsize, int pos1, int pos2) const {
  if (pos2 < 0) pos2 = len; 
  int nbc = pos2 - pos1;
  if (!s || nbc <= 0) return null;

  // hors bornes
  if (pos1 < 0 || pos1 >= len) return null;

  // ne pas depasser taille actuelle de str
  if (pos1 + nbc > len) nbc = len - pos1;

  // ne pas depasser taille du buffer
  if (nbc >= bufsize) nbc = bufsize-1;

  strncpy(buffer, s+pos1, nbc);
  buffer[nbc] = '\0';		// 0 final
  return nbc;
}

int UStr::copyTo(char* buffer, int bufsize) const {
  return copyTo(buffer, bufsize, 0, len);
}
#endif
/* ==================================================== ======== ======= */
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UStr::getSize(UContext *props, u_dim *w, u_dim *h) const {
  if (s && *s) {
    props->winview->wg().getTextSize(&props->fontdesc, s, len, w, h);
    // sinon il n'y aura pas de place pour afficher le caret en fin de ligne
    *w += 1;
  }
  else {
    //NOTES: 
    // -1- conserver une hauteur stable
    // -2- faire en sorte que la surface ne soit jamais de taille nulle 
    //     (sinon on ne pourra plus la selectionner en cliquant)
    // 
    *h = props->winview->wg().getTextHeight(&props->fontdesc);
    *w = 2;
  }
}


void UStr::getSize(UContext*props, u_dim maxwidth, u_dim chw, 
		   class UFlowCell*cell) const {
  if (s && *s) {
    cell->len = 
      props->winview->wg().getSubTextSize(&props->fontdesc, 
					  s + cell->offset,
					  len - cell->offset,
					  maxwidth, chw, 
					  &(cell->width), &(cell->height));
    // sinon il n'y aura pas de place pour afficher le caret en fin de ligne
    cell->width += 1;
    cell->truncated = (cell->offset + cell->len < len);
  }
  else {    // voir NOTES ci-dessus
    cell->len = 0;
    cell->height = props->winview->wg().getTextHeight(&props->fontdesc);
    cell->truncated = false;
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UStr::paint2(UWinGraph &g, UContext *props, const URegion &r,
		  int offset, int cellen) const {
  // !! don't confuse cellen and len!
  UTextsel &ts = g.getAppli()->getTextsel();

  //ce qui serait cool c'est d'economiser les setColor et setFont!!!

  g.setFont(&props->fontdesc);
  g.setColor(props->color);

  // cas ou il n'y a pas de TextSel, ou l'objet pointe est null
  // ou ce n'est pas le parent de str 
  // -> affichage de base sans selection
  
  if ((bmodes & UMode::IN_TEXTSEL) == 0
      || !ts.inObj
      || ts.inObj != props->obj
      || !ts.fromLink
      || !ts.toLink
      ) {
    g.drawString(s + offset, cellen, r.x, r.y);
  }

  // -> sinon affichage avec selection
  else {
    int deb = 0, fin = cellen;
    int ww = 0;

    // IL FAUDRAIT tester LEs LIENs (pas les UStr)
    // sinon les strings partagees vont s'allumer a plusieurs endroits!!

    if (this == dynamic_cast<const UStr*>(ts.fromLink->brick())) {
      deb = ts.fromPos - offset;
      if (deb < 0) deb = 0;  //inclus dans selection
      else if (deb >= cellen) deb = cellen;   //exclus (avant) selection
    }

    // pas de else: this peut etre a la fois fromLink et toLink
    // (c'est meme le cas le plus frequent!)

    if (this == dynamic_cast<const UStr*>(ts.toLink->brick())) {
      fin = ts.toPos - offset;
      if (fin < 0) fin = 0;   //exclus (apres) selection
      else if (fin >= cellen) fin = cellen; //inclus dans selection
    }

    if (deb > 0) {
      g.drawString(s+offset, deb, r.x, r.y);
      ww = props->winview->wg().getTextWidth(&props->fontdesc, s+offset, deb);
    }
    
    if (fin > deb) {
      if (ts.color)   g.setColor(ts.color);
      if (ts.bgcolor) g.setBgcolor(ts.bgcolor);
      UFontDesc *fd = &props->fontdesc;
      if (ts.fontdesc) {fd = ts.fontdesc; g.setFont(fd);}
      
      g.drawString(s+deb+offset, fin-deb, r.x + ww, r.y);
      ww += props->winview->wg().getTextWidth(fd, s+deb+offset, fin-deb);
    }

    if (fin < cellen) {
      g.setFont(&props->fontdesc);
      g.setColor(props->color);
      g.drawString(s+fin+offset, cellen-fin, r.x + ww, r.y);
    }
  }
} 


void UStr::paint(UWinGraph &g, UContext *props, const URegion &r) const {
  //!ATT: meme si !s penser afficher le Caret
  if (s && *s) paint2(g, props, r, 0, len);

  //!att: meme si !this->s penser afficher le Caret
  //nb: caret affiche "par dessus" le texte
  
  if (props->local.edit && props->local.edit->getCaretStr() == this) {
    props->local.edit->drawCaret(this, props, r, null, g);
  }
}


void UStr::paint(UWinGraph &g, UContext *props, const URegion &r,
		 UFlowCell *cell) const {
  if (s && *s) {
    paint2(g, props, r, cell->offset, cell->len);
    //g.drawString(s + cell->offset, cell->len, r.x, r.y);
  }

  //ATT: meme si !s penser afficher le Caret
  //NB: curseur affiche "par dessus" le texte
  int pos_in_string;
  if (props->local.edit
      && props->local.edit->getCaretStr() == this
      && (pos_in_string = props->local.edit->getCaretPosInCS()) >= cell->offset

      //pbm subtil: 2 cas differents
      //-1- newline (explicite par \n ou implicite par taille)
      //    au milieu d'une UStr --> test pos_in_string <strict
      //    sinon le caret s'affiche 2 fois (debut+fin de ligne)
      //-2- newline implicite juste a la fin physique de la UStr
      //    --> test >= sinon le caret va disparaitre

      && (pos_in_string < cell->offset + cell->len
	  || (pos_in_string == len && pos_in_string <= cell->offset+cell->len)
	  )
      ) {
    //printf("pos_in_string %d / cell->offset %d / cell->len %d / total %d\n",
    //     pos_in_string, cell->offset, cell->len, cell->offset+ cell->len);
    props->local.edit->drawCaret(this, props, r, cell, g);
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UTitle::uclass("UTitle");

UTitle::UTitle(const char *str) : UStr(str) {}

void UTitle::getSize(UContext *props, u_dim *w, u_dim *h) const {
  *h = *w = 0;
  // exbug: rien a faire!  update();
}

void UTitle::getSize(UContext*, u_dim maxwidth, u_dim chw, 
		     class UFlowCell*cell) const {
  // voir NOTES ci-dessus
  cell->len = 0;
  cell->height = 0;
  cell->truncated = false;
  // exbug: rien a faire: update();
}

void UTitle::paint(UWinGraph &g, UContext *props, const URegion &r) const {}

void UTitle::paint(UWinGraph &g, UContext *props, const URegion &r,
		   UFlowCell *cell) const {}

void UTitle::update() {
  UUpdate upd(UUpdate::TITLE);
  upd.paintTitle(this);
  parents.updateParents(upd);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
