/* ==================================================== ======== ======= *
 *
 *  uwin.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _uwin_hh
#define	_uwin_hh
//pragma ident	"@(#)uwin.hh	ubit:b1.11.3"
#include <ubox.hh>

//shortcuts (see notes below):

class UDialog& udialog(UArgs = UArgs::none);
class UFrame& uframe(UArgs = UArgs::none);
//Note: see umenu.hh for making pulldown and popup menus

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UWin : public UBox {
  friend class UAppli;
  friend class UBoxLink;
  friend class UHardwinImpl;
  friend class USoftwinImpl;
  friend class UNatWin;
  friend class UNatDisp;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}
  virtual class UWin* winCast() {return this;}

  UWin(UArgs = UArgs::none);
  virtual ~UWin();

  //NOTE: the following functions return false if the requested action 
  //      couldn't be performed

  // changes window location relatively to the specified object
  virtual u_bool move(class UWin*,  u_pos x_in_win,  u_pos y_in_win);
  virtual u_bool move(class UView*, u_pos x_in_view, u_pos y_in_view);
  // the 2nd arg. specify placement constraints (see UPlacement) 
  virtual u_bool move(class UView*, class UPlacement&);

  // changes window location relatively to an Event position
  virtual u_bool move(class UEvent*, u_pos x_offset = 0, u_pos y_offset = 0);

  // changes window location on screen
  virtual u_bool moveOnScreen(u_pos x_in_screen, u_pos y_in_screen);
  // centers the window on the screen
  virtual u_bool moveAndCenter();


  // gets window location relatively to the specified object
  virtual u_bool where(class UWin*, u_pos &x, u_pos &y) const;
  virtual u_bool where(class UView*, u_pos &x, u_pos &y) const;

  // gets window location on screen
  virtual u_bool whereOnScreen(u_pos &x, u_pos &y) const;

  // changes current window size
  virtual u_bool resize(u_dim width, u_dim height);

  // gets current window size
  virtual u_bool getSize(u_dim &width, u_dim &height) const;
  virtual u_dim  getWidth() const;
  virtual u_dim  getHeight() const;


  // forces the window to appear if arg. is true and disappear if false
  virtual void show(u_bool state) {UBox::show(state);}

  // is the window currently shown ?
  virtual u_bool isShown() const;

  //ATT: cette fct doit etre appelee a la creation de la UWin
  virtual UWin&  setSoftwinMode(u_bool state = true);
  virtual u_bool isSoftwinMode() const;

  // Updates object and chidren layout then repaints
  // !NOTE: must be called after add() or remove()
  virtual void update();

  // updates object according to upmode (see UUpdate class)
  virtual void update(class UUpdate upmode); 

  // closes the window (actual action depends on subclass)
  //NOTE: this function is called when clicking on the "Close" button
  //      of the Window Manager decoration
  virtual void close() = 0;

  // UWin(s) have only one (shared) view
  virtual UView*  getWinView() const {return winview;}
  virtual u_bool  isHardwin()  const {return is_hardwin;}
  virtual UAppli* getAppli()   const;
  virtual UDisp*  getDisp()    const;

  //package_private: ====[Ubit Intrinsics]==============================

  // returns a pointer to the Window Graphics (see uugraph.hh)
  class UWinGraph* getWinGraph() const;
  // returns a pointer to the Native Window Class (implementation dependant)
  class UNatWin* getNatWin() const;

  UGroup* getSubSoftwins();
  ULink* getSubSoftwinLinks();

  // creates the native X Window and initializes the children of this UWin 
  // (by creating their corresponding UViews)
  // NOTE: it is NOT necessary to call this function explicitely as it is 
  // automatically called the first time the UWin is shown on the screen 
  // (so that unused windows won't consume unnecessary resources)
  virtual u_bool realize() = 0;  //abstract

protected:
  u_bool is_hardwin;
  class UView *winview;
  union {
    void* initialized;   // null if not initialized
    class USoftwinImpl* soft;
    class UHardwinImpl* hard;
  } impl;

  // internal realization function
  virtual u_bool realizeSoftwin();
  virtual u_bool realizeHardwin(u_bool (*xrealize)(class UNatDisp*, 
						   class UNatWin*, UWin*));
  // called by realize() in order to create the children views of the UWin
  virtual void realizeChildren();

  // link and view management
  virtual void init(ULink *selflink, ULink *parlink, UView*, UDisp*);
  virtual void init(ULink *selflink, ULink *parlink, UView* parview);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UDialog: dialog box
// - UDialog objects are AUTOMATICALLY OPENED by their parent(s) when they
//   are clicked
// - the show() method can be used for displaying or hiding dialogs
//   while the move() methods make it possible to change their location

class UDialog : public UWin {
  class UCall *open_call;
public:
  static const UClass uclass;
  static UStyle *style;

  virtual const UClass* getClass() const {return &uclass;}
  virtual const UStyle* getStyle(const UBox *parent);

  UDialog(UArgs = UArgs::none);
  virtual ~UDialog() {clean();}

  // closes the dialog
  virtual void close() {show(false);}

  //package_private: ====[Ubit Intrinsics]==============================

  virtual u_bool realize();
  virtual void addingTo(ULink *selflink, UGroup *parent);
  //NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UFrame : main frame of the application (or secondary windows)
// NOTE:
// - the "Main Frame" is the first UFrame that is added to the UAppli object

class UFrame : public UDialog {
friend class UAppli;
protected:
  u_bool is_main_frame;
public:
  static const UClass   uclass;
  virtual const UClass* getClass()  const {return &uclass;}

  UFrame(UArgs = UArgs::none);

  // kills the appli if this frame is the main frame (= the first frame
  // that was realized) and just hides other frames
  // **NOTE**
  // this function is called when clicking on the "Close" button
  // of the Window Manager decoration. It can be redefined if you want
  // a specific behavior (such as opening a confirmation dialog)

  virtual void close() {isMainFrame() ? exit(0) : show(false);}

  u_bool isMainFrame() const {return is_main_frame;}
  virtual u_bool realize();
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// Constrainst for automatic Window Placement

class UPlacement {
public:
  // Horizontal placement. 
  // -- Explicit if &uleft() or &uright()
  // -- Default placement if null or other UHalign values.
  class UHalign *halign;

  // vertical placement. Explicit if &utop() or &ubottom()
  class UValign *valign;

  // Relative location from reference object:
  //
  // if oppositeBorder is true:
  //    the specified border (eg. uleft()) of the reference object is 
  //    aligned with the OPPOSITE border of 'this' object (eg. right)
  //    --> this implies that 'this' object will be located OUTSIDE
  //    the ref. object (exemple: a cascaded submenu located on the
  //    right side of its parent menu)
  //
  // if oppositeBorder is false:
  //    same borders are used for both objects (eg. both left
  //    sides are aligned together)
  //    --> this implies that 'this' object will be located INSIDE
  //    the ref. object (in the same way as for usual UH/Valign rules)

  u_bool hoppositeBorder, voppositeBorder;

  // relative 'distance' between reference and 'this' object borders.
  // Usually 1 or 0. Can be < 0 as it is a relative 'distance'.
  u_dim hdist, vdist;

  UPlacement();
};

// Examples (R = reference object, T = this object)
// --  RR TT   hplace = uright(),  hoppositeBorder = true,  hdist = 9
//     RR      vplace = utop(),    hoppositeBorder = false, vdist = 0
//
// --  RR      hplace = uleft(),   hoppositeBorder  = false, hdist = 0
//     RR      vplace = ubottom(), hoppositeBorder  = true,  vdist = 1
//     TT
//
// --  TT      hplace = uleft(),   hoppositeBorder = true,  hdist = 0
//     TTRR    vplace = ubottom(), hoppositeBorder = false, vdist = 0


/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// X Window incrustation in an Ubit UWin, UFrame, UDialog, etc. :
// -- an external program or any specific routine can perform customized
//    object rendering of this object (through the X API or whatever).

// -- Note:
//    the X Window is created by Ubit in the first case, it is created
//    by an external application and provided "as it" in the second case

class UIncrust& uincrust(UArgs = UArgs::none);

class UIncrust : public UWin {
  friend class UNatWin;
protected:
  u_bool is_external_win;
  unsigned long xwin;
public:
  static const UClass uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // incrusts an X Window in the containing UWin
  // NOTE: for this reason UIncrust can only control ONE window 
  //       and can not have several perents
  UIncrust(UArgs = UArgs::none);
  virtual ~UIncrust();

  // Specifies the external X Window to be incrusted in the Ubit GUI
  // If this fct. is not called, the X Window will be implicitely created 
  // by Ubit when shown the fist time
  virtual void setExternalXWin(unsigned long external_xwin);

  // true if this is an exteranl X Window
  // false if the window was created by Ubit
  virtual u_bool isExternalXWin() const {return is_external_win;}

  // returns the associated (internal or external) X Window
  virtual unsigned long getXWin() const {return xwin;}

  virtual void close() {show(false);}
  virtual u_bool realize();
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */


