/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <string.h>
#include <memory.h>
#include <ug_path.h>

UgPathPart* ug_path_part_new_full (const char* path, int path_len, const char folder_separator)
{
	UgPathPart* spart;

	spart = g_malloc (sizeof (UgPathPart));
	ug_path_part_full (spart, path, path_len, folder_separator);

	return spart;
}

void  ug_path_part_free (UgPathPart* spart)
{
	g_free (spart);
}

/* Return TRUE if path had drive or folder, otherwise return FALSE */
gboolean ug_path_part_full (UgPathPart* spart, const char* path, int path_len, const char folder_separator)
{
	int  index, index_file;
	int  dot_len;
	char chr;

	/* initialize */
	memset (spart, 0, sizeof (UgPathPart));
	if (path_len == -1)
		path_len = (int) strlen (path);
	spart->path     = path;
	spart->path_len = path_len;
	index_file = 0;	// begin index of file

	/* drive */
	for (index=0; index < path_len; index++) {
		chr = path[index];
		if (chr == folder_separator || chr == '.')
			break;
		if (chr == ':') {
			spart->path_drive_len = index;
			index_file = index + 1;
			break;
		}
	}

	/* folder */
	for (index = path_len -1;  index >= 0;  index--) {
		if (path[index] == folder_separator) {
			spart->folder     = path;
			spart->folder_len = index + 1;
			if (spart->path_drive_len) {
				spart->folder     += spart->path_drive_len + 1;
				spart->folder_len -= spart->path_drive_len + 1;
			}
			break;
		}
	}
	index_file += spart->folder_len;

	/* check special folder "." and ".." */
	for (index = index_file, dot_len = 0;  index < path_len;  index++) {
		chr = path[index];
		if (chr == '.')
			dot_len++;
		if (chr != '.' || dot_len > 2) {
			dot_len = 0;
			break;
		}
	}
	if (dot_len) {
		if (spart->folder == NULL)
			spart->folder = path + index_file;
		spart->folder_len += dot_len;
		index_file += dot_len;
	}

	/* primary filename and filename extension */
	if (index_file < path_len) {
		spart->file = path + index_file;
		spart->file_len = path_len - index_file;
		spart->file_pri_len = spart->file_len;

		/* find filename extension */
		for (index = path_len -1;  index >= index_file;  index--) {
			if (path[index] == '.') {
				spart->file_ext = path + index + 1;
				spart->file_ext_len = path_len - index - 1;
				spart->file_pri_len = index - index_file;
				break;
			}
		}
	}

	if (spart->path_drive_len || spart->folder)
		return TRUE;
	return FALSE;
}

// ---------------------------------------------------------------------------
/* other utilities for path */

unsigned int ug_path_element_full (const char* path, int path_len, unsigned int offset, unsigned int* len, const char folder_separator)
{
	const gchar*	path_end;
	const gchar*	element_beg;	// path element
	const gchar*	element_end;

	if (path_len == -1)
		path_len = strlen (path);
	path_end = path + path_len;

	for (element_beg = path+offset;  element_beg < path_end;  element_beg++) {
		if (*element_beg != folder_separator)
			break;
	}
	for (element_end = element_beg;  element_end < path_end;  element_end++) {
		if (*element_end == folder_separator)
			break;
	}

	if (len)
		*len = (unsigned int) (element_end - element_beg);
	return element_beg - path;
}

