/*
 *  update-cluster-removeline / Remove a host information based on 
 * a specified key.
 *  Copyright 2001-2003 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <dancer-xml.h>
#include "config.h"
#include "update-cluster-i18n.h"


typedef struct assoclist
{
  char * topic;
  char * content;  
  struct assoclist * next;
}assoclist;  

static assoclist * remove_parameter_list = NULL;
static int switch_notfound = 0 ;/* optional NOT option*/
static int switch_failfound = 0 ;/* optional failfound option*/

static void 
outofmemory(void)
{
  fprintf(stderr, _("Out of memory\n"));
  exit (EXIT_FAILURE);
}

static assoclist*
add_to_list(assoclist * a, char * topic, char * content)
{
  assoclist * tmp = malloc (sizeof (assoclist));
  if (!tmp)
    outofmemory () ;
  tmp->topic = strdup(topic);
  tmp->content = strdup(content);  
  tmp->next = a;
  return tmp;  
}


/**
 * Option parsing.
 */
void
parse_options ( int ac, char ** av)
{
  int index_point;  
  int c;			/* option */
  
  static struct option long_options[]=
  {
    {"ip", required_argument, 0, 'i'},
    {"master", no_argument, 0, 'm'},
    {"mac", required_argument, 0, 'c'},
    {"not", no_argument, 0, 'n'},
    {"failfound", no_argument, 0, 'f'},
    {"bootmaster", required_argument, 0, 'b'},
    {"help", no_argument, 0, 'h'},
    {0,0,0,0}
  };
  
  while((c = getopt_long (ac, av, "c:i:mnhfb:", long_options, &index_point)) != -1)
    {
      switch (c)
	{
	case 'i':
	  remove_parameter_list = add_to_list (remove_parameter_list, "ip", optarg);
	  break;	  
	case 'm':
	  remove_parameter_list = add_to_list (remove_parameter_list, "notes", "master");
	  break;
	case 'n':
	  switch_notfound=1;
	  break;
	case 'f':
	  switch_failfound=1;
	  break;
	case 'c':
	  remove_parameter_list = add_to_list (remove_parameter_list, "mac", optarg);
	  break;
	case 'b':
	  remove_parameter_list = add_to_list (remove_parameter_list, "bootmaster", optarg);
	  break;
	case 'h':
	  fprintf(stderr, _("update-cluster-remove by Junichi Uekawa 2001-2003\n\n"
		  "update-cluster-remove [--ip ipaddr] [--not] [--master] [--bootmaster xx]...\n"
		  "  parses an XML content, and outputs well-formatted XML output\n"
		  
		  "\n"
		  "please read the manpage for details.\n"));
	  exit (1);
	  break;	  
	default:
	  fprintf (stderr, _("Unhandled option specified\n"));
	  break;	  
	}
    }
}

/**
 * Print host info
 */
static 
void print_hostinfo_recurse(dxml_element * e)
{
  for (; e; e=e->next)
    {
      printf ("    <%s>%s</%s>\n", 
	      e->element_name,
	      e->child->element_name,
	      e->element_name);
    }
}


/**
   Match item
 */
int
match_item(const dxml_element*e, assoclist * param)
{
  int match_found = 0;

  if (!e)
    fprintf (stderr, _("Warning: Data extected to be non-NULL is NULL\n"));
  
  for (param = remove_parameter_list; param; param=param->next)
    for ( ; e; e = e->next)
      {
	if (e->element_type != element_type_element)
	  {
	    fprintf (stderr, _("Data extected to be a ELEMENT not ELEMENT\n"));
	    exit (EXIT_FAILURE);
	  }
	if (e->child->element_type != element_type_pcdata)
	  {
	    fprintf (stderr, _("Data expected to be a PCDATA not PCDATA\n"));
	    exit (EXIT_FAILURE);
	  }

	if (!strcmp(param->topic, e->element_name))
	  if (!strcmp(param->content, e->child->element_name))
	    match_found = 1;
      }
  return match_found;
}
    
    

/**
 * Process each element.
 *
 * @return 1 on fail
 */
int 
process_element(dxml_element*e_root)
{
  dxml_element*e;
  if (e_root->child == NULL) 
    {
      fprintf (stderr, _("No XML element available\n"));
      return 1;
    }
  if (strcmp (e_root->element_name, "cluster"))
    {
      fprintf (stderr, _("Expected element <cluster> at root\n"));
      return 1;
    }
  
  for (e=e_root->child; e; e=e->next)
    {
      int match_found = 0;
      
      /* this is <host> element, I think */
      if (strcmp (e->element_name, "host"))
	{
	  fprintf (stderr, _("Expected element <host> at each host, but got <%s> skipping\n"), e->element_name);
	  continue;
	}
      match_found=match_item (e->child, remove_parameter_list);
      if (((!match_found) && (!switch_notfound)) ||
	  (match_found && switch_notfound))
	{
	  printf("  <host>\n");
	  print_hostinfo_recurse(e->child);
	  printf ("  </host>\n");
	  if (switch_failfound)
	    exit (1);
	}

    }
  return 0;
}


int
main(int argc, char **argv) 
{
  dxml_element * e;
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);
  textdomain(PACKAGE_NAME);

  e = dxml_read_xml (stdin);
  if (!e) 
    {
      fprintf(stderr, _("Failed initialization of the parser\n"));
      exit(EXIT_FAILURE);
    }

  parse_options(argc, argv);
  
  printf ("<?xml version =\"1.0\"?>\n"
	  "<!DOCTYPE cluster SYSTEM \"%s/cluster.dtd\">\n"
	  "<cluster>\n",
	  PKGDATADIR
	  );

  if (process_element(e))
    exit (EXIT_FAILURE);
  
  printf ("</cluster>\n");  

  exit (EXIT_SUCCESS);
}  

