/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef TOOLSHADE_H
#define TOOLSHADE_H

#include <glib.h>
#include <coreTools/toolMatrix.h>

/**
 * Shade_struct
 *
 * Opaque structure to store linear shade informations.
 */
struct Shade_struct;
/**
 * Shade:
 *
 * Short name to address Shade_struct objects.
 */
typedef struct Shade_struct Shade;

/**
 * ShadeColorMode:
 * @shade_colorModeRGB: variation described in the shade are applied to RGB coding colors ;
 * @shade_colorModeHSV: variation described in the shade are applied to HSV coding colors ;
 * @shade_nb_colorMode: number of modes available.
 *
 * Defines color mode : Red-Green-Blue or Hue-Saturation-Value.
 */
typedef enum
  {
    shade_colorModeRGB,
    shade_colorModeHSV,
    shade_nb_colorMode
  } ShadeColorMode;

/**
 * ShadeMode:
 * @shade_modeLinear: all channels are defined by a linear variation
 * Ax+B ;
 * @shade_modeArray: all channels are defined by a given array of
 * values ;
 * @shade_nb_mode: the number of different shade mode.
 *
 * Defines the storage of the shade mode.
 */
typedef enum
  {
    shade_modeLinear,
    shade_modeArray,
    shade_nb_mode
  } ShadeMode;

/**
 * shadeNew:
 * @labelUTF8: a UTF8 string that shortly named this new shade ;
 * @vectA: an array of three floating point values ;
 * @vectB: an array of three floating point values ;
 * @colorMode: an integer that describes the color code (see #ShadeColorMode enumeration).
 *
 * Create a linear shade. Its profile is given by an AX+B formula,
 * dealing on three channels. These channels are defined by the
 * @colorMode parameter. All given values are copied when the new
 * shade is created.
 *
 * Returns: the newly created Shade.
 */
Shade* shadeNew(const gchar* labelUTF8, float vectA[3], float vectB[3],
		ShadeColorMode colorMode);
/**
 * shadeNew_fromData:
 * @labelUTF8: a UTF8 string that shortly named this new shade ;
 * @len: the size of arguments @vectCh1, @vectCh2 and @vectCh3 ;
 * @vectCh1: an array of floating point values for the first channel ;
 * @vectCh2: an array of floating point values for the first channel ;
 * @vectCh3: an array of floating point values for the first channel ;
 * @colorMode: an integer that describes the color code (see
 * #ShadeColorMode enumeration).
 *
 * Create a #Shade from direct data for three channels. These channels
 * are defined by the @colorMode parameter. All given values are
 * copied when the new shade is created.
 *
 * Returns: the newly created Shade.
 */ 
Shade* shadeNew_fromData(const gchar* labelUTF8, guint len, float *vectCh1,
			 float *vectCh2, float *vectCh3, ShadeColorMode colorMode);
/**
 * shadeFree:
 * @shade: a #Shade.
 *
 * Free all dynamic memory from @shade and free @shade itself.
 */
void shadeFree(Shade *shade);
/**
 * shadeCopy:
 * @shade: a #Shade.
 *
 * Create a new shade deep copy of the first.
 *
 * Returns: a newly created shade.
 */
Shade* shadeCopy(Shade *shade);
/**
 * shadeCompare:
 * @sh1: a #Shade ;
 * @sh2: a #Shade.
 *
 * Compare if the two shade are identical (first, smae mode, then same
 * values).
 *
 * Returns: TRUE if @shade1 is equivalent to @shade2.
 */
gboolean shadeCompare(Shade* sh1, Shade *sh2);
/**
 * shadeGet_label:
 * @shade: a valid #Shade object.
 *
 * Get the name (in UTF8) of the shade.
 *
 * Returns: a string naming the shade.
 */
gchar* shadeGet_label(Shade *shade);
/**
 * shadeGet_colorMode:
 * @shade: a valid #Shade object.
 *
 * Get the color mode of the shade (RGB or HSV).
 *
 * Returns: the color mode.
 */
ShadeColorMode shadeGet_colorMode(Shade *shade);
/**
 * shadeSet_colorMode:
 * @shade: a #Shade ;
 * @mode: a new mode for the shade.
 * 
 * Change the mode of the shade, see #ShadeColorMode.
 *
 * Returns: TRUE if @mode is different from previous @shade mode.
 */
gboolean shadeSet_colorMode(Shade *shade, ShadeColorMode mode);
/**
 * shadeGet_mode:
 * @shade: a valid #Shade object.
 *
 * Get the mode of the shade (linear, array...).
 *
 * Returns: the mode.
 */
ShadeMode shadeGet_mode(Shade *shade);
/**
 * shadeGet_linearCoeff:
 * @shade: a valid #Shade object ;
 * @vectA: a pointer to a floating point values array to store vect in AX+B ;
 * @vectB: a pointer to a floating point values array to store vect in AX+B.
 *
 * This methods can get the linear color transformation. The given
 * arrays (@vectA, @vectB) are read-only. This method return
 * FALSE if the @shade is not in a #shade_modeLinear state.
 *
 * Returns: TRUE if @vectA, @vectB and @vectX have been set correctly.
 */
gboolean shadeGet_linearCoeff(Shade *shade, float **vectA, float **vectB);
/**
 * shadeSet_linearCoeff:
 * @shade: a #Shade ;
 * @coeff: a new value ;
 * @channel: either RGBA (from 0 to 3) ;
 * @order: the order in the linear approx (0 means constant and 1 is
 * the linear coeeficient).
 *
 * Change one value @coeff of the linear mode for the given @shade.
 *
 * Returns: TRUE if the new value changes anything.
 */
gboolean shadeSet_linearCoeff(Shade *shade, float coeff, int channel, int order);
/**
 * shadeGet_valueTransformedInRGB:
 * @shade: a valid #Shade object ;
 * @rgba: an array of size [4] ;
 * @value: the value ranged in [0;1].
 *
 * Give a RGBA vector for the given value.
 */
void shadeGet_valueTransformedInRGB(const Shade *shade, float rgba[4], float value);
/**
 * shadeGet_RGBFromValues:
 * @shade: a #Shade ;
 * @rgba: a location to store the result of the colour transformation ;
 * @values: inout values.
 *
 * Like shadeGet_valueTransformedInRGB() but here, the three values
 * are applied respectivly for the Red, the Green and the Blue
 * channel.
 */
void shadeGet_RGBFromValues(const Shade *shade, float rgba[4], float values[3]);
/**
 * shadeDraw_legend:
 * @shade: a valid #Shade object ;
 * @widthVal: a floating point value ;
 * @heightVal: a floating point value ;
 * @minMax: the min & max value of the represented shaded color ;
 * @marks: an array of values bounded by @minMax ;
 * @nbMarks: the number of elements in the @marks array ;
 * @scale: the kind of transformation.
 *
 * This method calls OpenGL primitive to draw a coloured legend on the rendering
 * surface. If @widthVal is negative then the default value in pixel is used. If @widthVal
 * is smaller than 1., the width is taken as a percentage of the width of the rendering area.
 * Finaly, if @widthVal is larger than 1., the width of the legend in pixel is the given value.
 * If @marks is not NULL, some marks are set on the legend for the
 * given values.
 *
 */
void shadeDraw_legend(Shade *shade, float widthVal, float heightVal,
		      float minMax[2], float *marks, int nbMarks,
		      ToolMatrixScalingFlag scale);



/**
 * toolShadeListGet:
 *
 * It returns a read-only pointer to the internal shade list. Use toolShadeListAppend()
 * to add new shades to this list.
 *
 * Returns: a pointer to the internal shade list.
 */
GList* toolShadeListGet(void);
/**
 * toolShadeListAppend:
 * @shade: a #Shade object.
 *
 * Add a shape to the internal list. Use the return value or toolShadeListGet() method
 * to look into this list.
 *
 * Returns: a read-only pointer to the internal shade list.
 */
GList* toolShadeListAppend(Shade *shade);
/**
 * toolShadeBuild_presetList:
 *
 * Create a list of preset shades.
 *
 * Returns: a read-only pointer to the internal shade list.
 */
GList* toolShadeBuild_presetList(void);


#endif
